/************************************************************************

  Demo access to HydraHarp 500 Hardware via HH500Lib v.1.0.
  The program performs a measurement in continuous mode based
  on hardcoded settings.
  The resulting data are stored in a file, dependent
  on the value you set for the control variable writeFILE.
  Selected items of the data are extracted for immediate display.
  This is a demo for experts wanting to extract the data under full 
  control over their related code. Other users are probably better off
  with the demo contmode_easy where the extraction is done with
  helper routines fom HH500Lib.

  Michael Wahl, PicoQuant GmbH, August 2025

  Note: This is a console application (i.e. run in Windows cmd box)

  Note: At the API level channel numbers are indexed 0..N-1
        where N is the number of channels the device has.

  Tested with the following compilers:

  - MinGW-W64 4.3.5 (Windows 64 bit)
  - MS Visual C++ 2019 (Windows 64 bit)
  - gcc 9.4.0 and 11.4.0 (Linux 64 bit)

************************************************************************/

#ifdef _WIN32
#include <windows.h>
#include <dos.h>
#include <conio.h>
#else
#include <unistd.h>
#define Sleep(msec) usleep(msec *1000)
#define __int64 long long
# endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "hh500defin.h"
#include "hh500lib.h"
#include "errorcodes.h"

/*
Continuous mode creates data blocks with a header of fixed structure
followed by the histogram data and the histogram sums for each channel.
The following structure represents the continuous mode block header.
The header structure is fixed and cannot not be changed. Should it change 
in future versions, the first structure element 'versioncode' will be
different to make this detectable.
The data following the header changes its size dependent on the
number of enabled channels and the chosen histogram length. It must
therefore be interpreted at runtime. This will be shown further below.
Here we just allocate enough buffer for the max case.
By putting header and data buffer together in a structure we can easily
fill the entire structure and later access the individual items.
*/

#define CONTMODESTRUCT_V01  0xa5a50001

typedef struct
{
    unsigned int versioncode;      // 0xa5a50001 as of HH500Lib v1.0
    unsigned int inpt_group;       // reserved, not relevant as of HH500Lib v1.0
    unsigned int blocknum;         // block number, counting up from 1
    unsigned int enabled_channels; // this is a bitmap, 1=enabled, 0=disabled
    unsigned int histlencode;      // length code as used in HH500_SetHistoLen
    unsigned int flags;            // bit0=stopofl, bit1=contmem_full
    unsigned __int64 starttime;	   // start time of this block in nanosec
    unsigned __int64 duration;	   // duration of this block in nanosec
    unsigned countM1;              // count of marker 1 within this block duration
    unsigned countM2;              // count of marker 2 within this block duration
    unsigned countM3;              // count of marker 3 within this block duration
    unsigned countM4;              // count of marker 4 within this block duration
    unsigned int reserved[2];      // reserved, not relevant as of HH500Lib v1.0

} ContModeBlockHeaderType;

typedef struct
{
    ContModeBlockHeaderType header;
    char data[MAXCONTMODEBUFLEN];
} ContModeBlockBufferType;

#define LENCODE 0	// will control the length of each histogram: 0=1024, 1=2048, 2=4096, 3=8192 bins
#define NBLOCKS 10	// so many continuous blocks we want to collect

// helper macro and associated function for API calls with error check
// the stringize operator # makes a printable string from the macro's input argument
#define APICALL(call) doapicall(call, #call, __LINE__)
int doapicall(int retcode, char* callstr, int line)
{
    if (retcode < 0)
    {
        char errorstring[100] = { 0 };
        HH500_GetErrorString(errorstring, retcode);
        printf("\nThe API call %s at line %d\nreturned error %d (%s)\n", callstr, line, retcode, errorstring);
    }
    return retcode;
}

// for brevity of the main demo code further below we encapsulate the programming
// of the input circuits as a soubroutine here
int SetInputModalities(int devidx, int numchans)
{
    // Dependent on the hardware model, the HydraHarp 500 may have different input
    // circuits: Edge Trigger (ETR) and/or Constant Fraction Discrimonator (CFD)
    // This can even vary from channel to channel. An input that has both ETR and CFD
    // will be programmable as to which trigger mode to use. Here we define default 
    // settings for both variants and the default trigger mode ETR.
    // The latter can be changed to CFD which of course only works with channels
    // that actually have a CFD. The code further below will therefore query the 
    // input capabilities and where necessary fall back to the mode available.

    int SyncTrgMode = TRGMODE_ETR;       // you can change this to TRGMODE_CFD    
    // in case of SyncTrgMode == TRGMODE_ETR this will apply:
    int SyncTiggerEdge = EDGE_FALLING;   // you can change this to EDGE_RISING
    int SyncTriggerLevel = -50;          // in mV, you can change this
    // in case of SyncTrgMode == TRGMODE_CFD this will apply:
    int SyncCFDZeroCross = -20;          // in mV, you can change this
    int SyncCFDLevel = -50;              // in mV, you can change this

    int InputTrgMode = TRGMODE_ETR;     // you can change this to TRGMODE_CFD
    // in case of InputTrgMode == TRGMODE_ETR this will apply:
    int InputTriggerEdge = EDGE_FALLING; // you can change this to EDGE_RISING
    int InputTriggerLevel = -50;         // in mV, you can change this
    // in case of InputTrgMode == TRGMODE_CFD this will apply:
    int InputCFDZeroCross = -20;         // in mV, you can change this
    int InputCFDLevel = -50;             // in mV, you can change this

    int SyncChannelOffset = -5000;       // you can change this (in ps, like a cable delay)
    int InputChannelOffset = 0;          // you can change this (in ps, like a cable delay)

    unsigned ChnFeatures = 0;

    if (APICALL(HH500_GetSyncFeatures(devidx, &ChnFeatures)) < 0)
        return -1;

    // check if the sync channel has the right feature for the requested setting
    if ((SyncTrgMode == TRGMODE_ETR) && ((ChnFeatures & HAS_ETR) == 0))
    {
        printf("\nWarning: Sync channel has no Edge Trigger, switching to CFD");
        SyncTrgMode = TRGMODE_CFD;
    }
    if ((SyncTrgMode == TRGMODE_CFD) && ((ChnFeatures & HAS_CFD) == 0))
    {
        printf("\nWarning: Sync channel has no CFD, switching to Edge Trigger");
        SyncTrgMode = TRGMODE_ETR;
    }

    if (APICALL(HH500_SetSyncTrgMode(devidx, SyncTrgMode)) < 0)
        return -1;

    if (SyncTrgMode == TRGMODE_ETR)
    {
        if (APICALL(HH500_SetSyncEdgeTrg(devidx, SyncTriggerLevel, SyncTiggerEdge)) < 0)
            return -1;
    }

    if (SyncTrgMode == TRGMODE_CFD)
    {
        if (APICALL(HH500_SetSyncCFD(devidx, SyncCFDLevel, SyncCFDZeroCross)) < 0)
            return -1;
    }

    if (APICALL(HH500_SetSyncChannelOffset(devidx, SyncChannelOffset)) < 0)
        return -1;

    for (int i = 0; i < numchans; i++) // for simplicity we use the same settings for all channels
    {
        int RealTrgMode = InputTrgMode;

        if (APICALL(HH500_GetInputFeatures(devidx, i, &ChnFeatures)) < 0)
            return -1;

        // check if the input channel has the right feature for the requested setting
        if ((InputTrgMode == TRGMODE_ETR) && ((ChnFeatures & HAS_ETR) == 0))
        {
            printf("\nWarning: Input channel %1d has no Edge Trigger, switching to CFD", i + 1);
            RealTrgMode = TRGMODE_CFD;
        }
        if ((InputTrgMode == TRGMODE_CFD) && ((ChnFeatures & HAS_CFD) == 0))
        {
            printf("\nWarning: Input channel %1d has no CFD, switching to Edge Trigger", i + 1);
            RealTrgMode = TRGMODE_ETR;
        }

        if (APICALL(HH500_SetInputTrgMode(devidx, i, RealTrgMode)) < 0)
            return -1;

        if (RealTrgMode == TRGMODE_ETR)
        {
            if (APICALL(HH500_SetInputEdgeTrg(devidx, i, InputTriggerLevel, InputTriggerEdge)) < 0)
                return -1;
        }

        if (RealTrgMode == TRGMODE_CFD)
        {
            if (APICALL(HH500_SetInputCFD(devidx, i, InputCFDLevel, InputCFDZeroCross)) < 0)
                return -1;
        }

        if (APICALL(HH500_SetInputChannelOffset(devidx, i, InputChannelOffset)) < 0)
            return -1;
    }
    return 0;
}


int main(int argc, char* argv[])
{
    int dev[MAXDEVNUM];
    int found = 0;
    FILE* fpout = NULL;
    int writeFILE = 0; // set this to one if you want the data written to disk 
    int retcode;
    char LIB_Version[8];
    char HW_Model[32];
    char HW_Partno[8];
    char HW_Version[8];
    char HW_Serial[8];
    char Errorstring[40];
    char debuginfobuffer[16384]; // must have 16384 bytes of text buffer
    int NumChannels;
    int EnabledChannels;
    int HistLen;

    int MeasControl = MEASCTRL_CONT_CTC_RESTART;          // this starts a new histogram time automatically when the previous is over
    //int MeasControl = MEASCTRL_CONT_C1_START_CTC_STOP;  // this would require a TTL pulse at the C1 connector for each new histogram
    //int MeasControl = MEASCTRL_CONT_C1_GATED;           // this would require a TTL pulse at the C1 connector for each new histogram

    int Binning = 5;     // you can change this
    int Offset = 0;      // you can change this, normally 0
    int Tacq = 100;      // measurement time per histogram in millisec, you can change this
    int SyncDivider = 1; // you can change this
    double Resolution;
    int Syncrate;
    int Countrate;
    int i;
    int flags;
    int warnings;
    char warningstext[16384]; // must have 16384 bytes of text buffer
    int nBytesReceived;
    unsigned int Blocknum;
    int expectedblocksize;
    ContModeBlockBufferType block;

    unsigned int* histograms[MAXINPCHAN];	// an array of pointers to access the histograms of each channel
    unsigned __int64 histosums[MAXINPCHAN];	// the histogram sums of each channel


    printf("\nHydraHarp 500 HHLib Demo Application                 PicoQuant GmbH, 2025");
    printf("\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~");

    if (APICALL(HH500_GetLibraryVersion(LIB_Version)) < 0)
        goto ex;
    printf("\nLibrary version is %s", LIB_Version);
    if (strncmp(LIB_Version, LIB_VERSION, sizeof(LIB_VERSION)) != 0)
        printf("\nWarning: The application was built for version %s.", LIB_VERSION);

    if ((fpout = fopen("contmode.out", "wb")) == NULL)
    {
        printf("\ncannot open output file\n");
        goto ex;
    }

    printf("\n");
    printf("\nSearching for HydraHarp 500 devices...");
    printf("\nDevidx     Serial     Status");

    for (i = 0; i < MAXDEVNUM; i++)
    {
        memset(HW_Serial, 0, sizeof(HW_Serial));
        retcode = HH500_OpenDevice(i, HW_Serial); // not using APICALL here as fails must be expected
        if (retcode == 0) // grab any device we can open
        {
            printf("\n  %1d        %7s    open ok", i, HW_Serial);
            dev[found] = i; // keep index to device(s) we want to use
            found++;
        }
        else
        {
            if (retcode == HH500_ERROR_DEVICE_OPEN_FAIL)
                printf("\n  %1d        %7s    no device", i, HW_Serial);
            else
            {
                HH500_GetErrorString(Errorstring, retcode);
                printf("\n  %1d        %7s    %s", i, HW_Serial, Errorstring);
            }
        }
    }

    // In this demo we will use the first device we find, i.e. dev[0].
    // You can also use multiple devices in parallel.
    // You can also check for specific serial numbers, so that you always 
    // know which physical device you are talking to.

    if (found < 1)
    {
        printf("\nNo device available.");
        goto ex;
    }

    printf("\nUsing device #%1d", dev[0]);
    printf("\nInitializing the device...");

    if (APICALL(HH500_Initialize(dev[0], MODE_CONT, 0)) < 0) // Cont. mode with internal clock
    {
        // in case of an obscure error (a hardware error in particular) 
        // it may be helpful to obtain debug information like so:
        HH500_GetDebugInfo(dev[0], debuginfobuffer);
        printf("\nDEBUGINFO:\n%s", debuginfobuffer);
        goto ex;
    }

    if (APICALL(HH500_GetHardwareInfo(dev[0], HW_Model, HW_Partno, HW_Version)) < 0)
        goto ex;
    printf("\nFound Model %s Part no %s Version %s", HW_Model, HW_Partno, HW_Version);

    if (APICALL(HH500_GetNumOfInputChannels(dev[0], &NumChannels)) < 0)
        goto ex;
    printf("\nDevice has %i input channels.", NumChannels);

    if (APICALL(HH500_SetSyncDiv(dev[0], SyncDivider)) < 0)
        goto ex;

    if (SetInputModalities(dev[0], NumChannels) < 0) // see soubroutine above
        goto ex;

    EnabledChannels = 0;
    for (int i = 0; i < NumChannels; i++)	// we enable all channels the device has, could use less
    {
        if (APICALL(HH500_SetInputChannelEnable(dev[0], i, 1)) < 0)
            goto ex;
        EnabledChannels++;
    }

    if (APICALL(HH500_ClearHistMem(dev[0])) < 0)
        goto ex;

    if (APICALL(HH500_SetMeasControl(dev[0], MeasControl, EDGE_RISING, EDGE_RISING)) < 0)
        goto ex;

    if (APICALL(HH500_SetHistoLen(dev[0], LENCODE, &HistLen)) < 0)
        goto ex;
    printf("\nHistogram length is %d", HistLen);

    if (APICALL(HH500_SetBinning(dev[0], Binning)) < 0)
        goto ex;

    if (APICALL(HH500_SetOffset(dev[0], Offset)) < 0)
        goto ex;

    if (APICALL(HH500_GetResolution(dev[0], &Resolution)) < 0)
        goto ex;
    printf("\nResolution is %1.0lfps\n", Resolution);

    // after Initialize or SetSyncDiv allow >100 ms for valid new count rate readings
    Sleep(150);

    if (APICALL(HH500_GetSyncRate(dev[0], &Syncrate)) < 0)
        goto ex;
    printf("\nSyncrate=%1d/s", Syncrate);

    for (i = 0; i < NumChannels; i++) // for all channels
    {
        if (APICALL(HH500_GetCountRate(dev[0], i, &Countrate)) < 0)
            goto ex;
        printf("\nCountrate[%1d]=%1d/s", i, Countrate);
    }

    printf("\n");

    // after getting the count rates you can check for warnings
    if (APICALL(HH500_GetWarnings(dev[0], &warnings)) < 0)
        goto ex;

    if (warnings)
    {
        if (APICALL(HH500_GetWarningsText(dev[0], warningstext, warnings)) < 0)
            goto ex;
        printf("\n\n%s", warningstext);
    }

    printf("\npress RETURN to start\n");
    getchar();
    printf(" #   start/ns  duration/ns   sum[hist]  sum[hist]  sum[hist]   ...\n");

    Blocknum = 1; // starts counting at 1

    if (APICALL(HH500_StartMeas(dev[0], Tacq)) < 0)
        goto ex;

    expectedblocksize = sizeof(ContModeBlockHeaderType) + (HistLen * 4 + 8) * EnabledChannels;

    while (Blocknum <= NBLOCKS)
    {
        if (APICALL(HH500_GetFlags(dev[0], &flags)) < 0)
            goto ex;

        if (flags & FLAG_FIFOFULL)
        {
            printf("\nFiFo Overrun!\n");
            goto stoprun;
        }

        if (APICALL(HH500_GetContModeBlock(dev[0], &block, &nBytesReceived)) < 0)
            goto ex;

        if (nBytesReceived)	// we might have received nothing, then nBytesReceived is 0
        {
            // sanity check: if we did receive something, then it must be the right size
            if (nBytesReceived != expectedblocksize)
            {
                printf("\nError: unexpected block size! Aborted.\n");
                goto stoprun;
            }

            if (writeFILE == 1)
            {
                if (fwrite(&block, 1, nBytesReceived, fpout) != (unsigned)nBytesReceived)
                {
                    printf("\nfile write error\n");
                    goto stoprun;
                }
            }

            // The following shows how to dissect the freshly collected continuous mode data on the fly. 
            // Of course the same processing scheme can be applied on file data.

            if (block.header.versioncode != CONTMODESTRUCT_V01)	// sanity check
            {
                printf("\nUnexpected structure version code 0x%08X! Aborted\n", block.header.versioncode);
                goto stoprun;
            }

            // the header items can be accessed directly via the corresponding structure elements
            printf("%2u %10llu %10llu", block.header.blocknum, block.header.starttime, block.header.duration);

            // count enabled channels, only for sanity check
            unsigned enabled_chans = 0; 
            for (int i = 0; i < 32; i++) 
                enabled_chans += (block.header.enabled_channels >> i) & 1;
            if (enabled_chans != EnabledChannels)
            {
                printf("\nUnexpected number of enabled channels! Aborted\n");
                goto stoprun;
            }

            if (block.header.blocknum != Blocknum)	// another sanity check: blocknum should increment each round
            {
                printf("\nUnexpected blocknum! Aborted\n");
                goto stoprun;
            }

            // the histogram data items must be extracted dynamically as follows:
            for (i = 0; i < EnabledChannels; i++)
            {
                int histlen_hdr = 1024 << (block.header.histlencode - 1);
                if (histlen_hdr != HistLen)	// just another sanity check
                {
                    printf("\nUnexpected histogram length! Aborted\n");
                    goto stoprun;
                }

                histograms[i] = (unsigned*)(&block.data) + i * (histlen_hdr + 2);
                // pointer arithmethic is in DWORDS (size of unsigned int), 
                // +2 is to skip over the sum (8 bytes) following the histogram
                // histograms[i] are actually pointers but they can be used to emulate double indexed arrays.
                // So we could now access and e.g. print  histograms[channel][bin]  without copying the data
                // but we don't print them all here to keep the screen tidy.

                // now we obtain the histogram sums, knowing they immediately follow each histogram
                histosums[i] = *(unsigned __int64*)(histograms[i] + HistLen);
                // these we print as they are just one number per channel
                printf(" %10llu", histosums[i]);

                // note that disabled channels will not appear in the output data. 
                // the index i may then not correspond to actual input channel numbers
            }

            printf("\n");
            Blocknum++;
        }

        // within this loop you can also read the count rates if needed.
    }

stoprun:

    if (APICALL(HH500_StopMeas(dev[0])) < 0)
        goto ex;

ex:

    for (i = 0; i < MAXDEVNUM; i++) // no harm to close all
        HH500_CloseDevice(i);

    if (fpout) fclose(fpout);
    printf("\npress RETURN to exit");
    getchar();

    return 0;
}
