/************************************************************************

  Demo access to HydraHarp 500 hardware via HH500Lib v 1.0

  THIS IS AN ADVANCED DEMO. DO NOT USE FOR YOUR FIRST EXPERIMENTS.
  Look at the variable meascontrol down below to see what it does.

  The program performs a measurement based on hardcoded settings.
  The resulting histogram is stored in an ASCII output file.

  Michael Wahl, PicoQuant GmbH, August 2025

  Note: This is a console application

  Note: At the API level channel numbers are indexed 0..N-1 
    where N is the number of channels the device has.


  Tested with the following compilers:

  - MinGW-W64 4.3.5 (Windows 64 bit)
  - MS Visual C++ 2019 (Windows 64 bit)
  - gcc 9.4.0 and 11.4.0 (Linux 64 bit)

************************************************************************/

#ifndef _WIN32
#include <unistd.h>
#define Sleep(msec) usleep(msec*1000)
#else
#include <windows.h>
#include <dos.h>
#include <conio.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "hh500defin.h"
#include "hh500lib.h"
#include "errorcodes.h"


unsigned int counts[MAXINPCHAN][MAXHISTLEN]; // gobal only because of stack constraints

// helper macro and associated function for API calls with error check
// the stringize operator # makes a printable string from the macro's input argument
#define APICALL(call) doapicall(call, #call, __LINE__)
int doapicall(int retcode, char* callstr, int line)
{
    if (retcode < 0)
    {
        char errorstring[100] = { 0 };
        HH500_GetErrorString(errorstring, retcode);
        printf("\nThe API call %s at line %d\nreturned error %d (%s)\n", callstr, line, retcode, errorstring);
    }
    return retcode;
}

// for brevity of the main demo code further below we encapsulate the programming
// of the input circuits as a soubroutine here
int SetInputModalities(int devidx, int numchans)
{
    // Dependent on the hardware model, the HydraHarp 500 may have different input
    // circuits: Edge Trigger (ETR) and/or Constant Fraction Discrimonator (CFD)
    // This can even vary from channel to channel. An input that has both ETR and CFD
    // will be programmable as to which trigger mode to use. Here we define default 
    // settings for both variants and the default trigger mode ETR.
    // The latter can be changed to CFD which of course only works with channels
    // that actually have a CFD. The code further below will therefore query the 
    // input capabilities and where necessary fall back to the mode available.

    int SyncTrgMode = TRGMODE_ETR;       // you can change this to TRGMODE_CFD    
    // in case of SyncTrgMode == TRGMODE_ETR this will apply:
    int SyncTiggerEdge = EDGE_FALLING;   // you can change this to EDGE_RISING
    int SyncTriggerLevel = -50;          // in mV, you can change this
    // in case of SyncTrgMode == TRGMODE_CFD this will apply:
    int SyncCFDZeroCross = -20;          // in mV, you can change this
    int SyncCFDLevel = -50;              // in mV, you can change this

    int InputTrgMode = TRGMODE_ETR;     // you can change this to TRGMODE_CFD
    // in case of InputTrgMode == TRGMODE_ETR this will apply:
    int InputTriggerEdge = EDGE_FALLING; // you can change this to EDGE_RISING
    int InputTriggerLevel = -50;         // in mV, you can change this
    // in case of InputTrgMode == TRGMODE_CFD this will apply:
    int InputCFDZeroCross = -20;         // in mV, you can change this
    int InputCFDLevel = -50;             // in mV, you can change this

    int SyncChannelOffset = -5000;       // you can change this (in ps, like a cable delay)
    int InputChannelOffset = 0;          // you can change this (in ps, like a cable delay)


    unsigned ChnFeatures = 0;

    if (APICALL(HH500_GetSyncFeatures(devidx, &ChnFeatures)) < 0)
        return -1;

    // check if the sync channel has the right feature for the requested setting
    if ((SyncTrgMode == TRGMODE_ETR) && ((ChnFeatures & HAS_ETR) == 0))
    {
        printf("\nWarning: Sync channel has no Edge Trigger, switching to CFD");
        SyncTrgMode = TRGMODE_CFD;
    }
    if ((SyncTrgMode == TRGMODE_CFD) && ((ChnFeatures & HAS_CFD) == 0))
    {
        printf("\nWarning: Sync channel has no CFD, switching to Edge Trigger");
        SyncTrgMode = TRGMODE_ETR;
    }

    if (APICALL(HH500_SetSyncTrgMode(devidx, SyncTrgMode)) < 0)
        return -1;

    if (SyncTrgMode == TRGMODE_ETR)
    {
        if (APICALL(HH500_SetSyncEdgeTrg(devidx, SyncTriggerLevel, SyncTiggerEdge)) < 0)
            return -1;
    }

    if (SyncTrgMode == TRGMODE_CFD)
    {
        if (APICALL(HH500_SetSyncCFD(devidx, SyncCFDLevel, SyncCFDZeroCross)) < 0)
            return -1;
    }

    if (APICALL(HH500_SetSyncChannelOffset(devidx, SyncChannelOffset)) < 0)
        return -1;

    for (int i = 0; i < numchans; i++) // for simplicity we use the same settings for all channels
    {
        int RealTrgMode = InputTrgMode;

        if (APICALL(HH500_GetInputFeatures(devidx, i, &ChnFeatures)) < 0)
            return -1;

        // check if the input channel has the right feature for the requested setting
        if ((InputTrgMode == TRGMODE_ETR) && ((ChnFeatures & HAS_ETR) == 0))
        {
            printf("\nWarning: Input channel %1d has no Edge Trigger, switching to CFD", i + 1);
            RealTrgMode = TRGMODE_CFD;
        }
        if ((InputTrgMode == TRGMODE_CFD) && ((ChnFeatures & HAS_CFD) == 0))
        {
            printf("\nWarning: Input channel %1d has no CFD, switching to Edge Trigger", i + 1);
            RealTrgMode = TRGMODE_ETR;
        }

        if (APICALL(HH500_SetInputTrgMode(devidx, i, RealTrgMode)) < 0)
            return -1;

        if (RealTrgMode == TRGMODE_ETR)
        {
            if (APICALL(HH500_SetInputEdgeTrg(devidx, i, InputTriggerLevel, InputTriggerEdge)) < 0)
                return -1;
        }

        if (RealTrgMode == TRGMODE_CFD)
        {
            if (APICALL(HH500_SetInputCFD(devidx, i, InputCFDLevel, InputCFDZeroCross)) < 0)
                return -1;
        }

        if (APICALL(HH500_SetInputChannelOffset(devidx, i, InputChannelOffset)) < 0)
            return -1;

        if (APICALL(HH500_SetInputChannelEnable(devidx, i, 1)) < 0) // we enable all channels
            return -1;
    }
    return 0;
}


int main(int argc, char* argv[])
{
    int dev[MAXDEVNUM];
    int found = 0;
    FILE* fpout = 0;
    int retcode;
    int ctcstatus;
    char LIB_Version[8];
    char HW_Model[32];
    char HW_Partno[8];
    char HW_Version[16];
    char HW_Serial[16];
    char Errorstring[40];
    char debuginfobuffer[16384]; // must have 16384 bytes of text buffer
    int NumChannels;
    int HistLen;
    int Binning = 0; // you can change this
    int Offset = 0;
    int Tacq = 100;  // measurement time in millisec, you can change this
    int SyncDivider = 1; // you can change this

    double Resolution;
    int Syncrate;
    int Countrate;
    double Integralcount;
    double elapsed;
    int flags;
    int warnings;
    char warningstext[16384]; // must have 16384 bytes of text buffer
    char cmd = 0;

    int meascontrol
        = MEASCTRL_SINGLESHOT_CTC;    // start by software and stop when CTC expires (default)
     // = MEASCTRL_C1_GATED;          // measure while C1 is active
     // = MEASCTRL_C1_START_CTC_STOP; // start with C1 and stop when CTC expires 
     // = MEASCTRL_C1_START_C2_STOP;  // start with C1 and stop with C2
    int edge1 = EDGE_RISING;  // edge of C1 to start (if applicable in chosen mode)
    int edge2 = EDGE_FALLING; // edge of C2 to stop (if applicable in chosen mode)


    memset(Errorstring, 0x00, sizeof(Errorstring));
    memset(warningstext, 0x00, sizeof(warningstext));

    printf("\nHydraHarp 500 - HH500Lib Demo Application          PicoQuant GmbH, 2025");
    printf("\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~");
   
    if (APICALL(HH500_GetLibraryVersion(LIB_Version)) < 0)
        goto ex;
    printf("\nLibrary version is %s", LIB_Version);
    if (strncmp(LIB_Version, LIB_VERSION, sizeof(LIB_VERSION)) != 0)
        printf("\nWarning: The application was built for version %s.", LIB_VERSION);

    if ((fpout = fopen("histomodeout.txt", "w")) == NULL)
    {
        printf("\ncannot open output file\n");
        goto ex;
    }

    printf("\nSearching for HydraHarp 500 devices...");
    printf("\nDevidx     Serial     Status");

    for (int i = 0; i < MAXDEVNUM; i++)
    {
        memset(HW_Serial, 0x00, sizeof(HW_Serial));
        retcode = HH500_OpenDevice(i, HW_Serial);
        if (retcode == 0) // grab any device we can open
        {
            printf("\n  %1d        %7s    open ok", i, HW_Serial);
            dev[found] = i; // keep index to devices we want to use
            found++;
        }
        else
        {
            if (retcode == HH500_ERROR_DEVICE_OPEN_FAIL)
                printf("\n  %1d        %7s    no device", i, HW_Serial);
            else
            {
                HH500_GetErrorString(Errorstring, retcode);
                printf("\n  %1d        %7s    %s", i, HW_Serial, Errorstring);
            }
        }
    }

    // In this demo we will use the first device we find, i.e. dev[0].
    // You could also use multiple devices in parallel.
    // You can also check for specific serial numbers, so that you know 
    // which physical device you are talking to.

    if (found < 1)
    {
        printf("\nNo device available.");
        goto ex;
    }

    printf("\nUsing device #%1d", dev[0]);
    printf("\nInitializing the device...");
    fflush(stdout);

    if (APICALL(HH500_Initialize(dev[0], MODE_HIST, 0)) < 0) // Histo mode with internal clock
    {
        // in case of an obscure error (a hardware error in particular) 
        // it may be helpful to obtain debug information like so:
        HH500_GetDebugInfo(dev[0], debuginfobuffer);
        printf("\nDEBUGINFO:\n%s", debuginfobuffer);
        goto ex;
    }

    if (APICALL(HH500_GetHardwareInfo(dev[0], HW_Model, HW_Partno, HW_Version)) < 0)
        goto ex;
    printf("\nFound Model %s Part no %s Version %s", HW_Model, HW_Partno, HW_Version);

    if (APICALL(HH500_GetNumOfInputChannels(dev[0], &NumChannels)) < 0)
        goto ex;
    printf("\nDevice has %i input channels.", NumChannels);

    if (APICALL(HH500_SetSyncDiv(dev[0], SyncDivider)) < 0)
        goto ex;

    if (SetInputModalities(dev[0], NumChannels) < 0) // see soubroutine above
        goto ex;

    if (APICALL(HH500_SetHistoLen(dev[0], MAXLENCODE, &HistLen)) < 0)
        goto ex;
    printf("\nHistogram length is %d", HistLen);

    if (APICALL(HH500_SetBinning(dev[0], Binning)) < 0)
        goto ex;

    if (APICALL(HH500_SetOffset(dev[0], Offset)) < 0)
        goto ex;

    if (APICALL(HH500_GetResolution(dev[0], &Resolution)) < 0)
        goto ex;
    printf("\nResolution is %1.0lfps\n", Resolution);

    // after Init allow 150 ms for valid  count rate readings
    // subsequently you get new values after every 100ms
    Sleep(150);

    if (APICALL(HH500_GetSyncRate(dev[0], &Syncrate)) < 0)
        goto ex;
    printf("\nSyncrate=%1d/s", Syncrate);

    for (int i = 0; i < NumChannels; i++) // for all channels
    {
        if (APICALL(HH500_GetCountRate(dev[0], i, &Countrate)) < 0)
            goto ex;
        printf("\nCountrate[%1d]=%1d/s", i, Countrate);
    }

    printf("\n");

    // after getting the count rates you can check for warnings
    if (APICALL(HH500_GetWarnings(dev[0], &warnings)) < 0)
        goto ex;

    if (warnings)
    {
        if (APICALL(HH500_GetWarningsText(dev[0], warningstext, warnings)) < 0)
            goto ex;
        printf("\n\n%s", warningstext);
    }

    if (APICALL(HH500_SetStopOverflow(dev[0], 0, 10000)) < 0) // for example only
        goto ex;

    // important for this demo:
    if (APICALL(HH500_SetMeasControl(dev[0], meascontrol, edge1, edge2)) < 0)
        goto ex;

    while (cmd != 'q')
    {
        if (APICALL(HH500_ClearHistMem(dev[0])) < 0)
            goto ex;

        printf("\npress RETURN to start measurement");
        getchar();

        if (APICALL(HH500_GetSyncRate(dev[0], &Syncrate)) < 0)
            goto ex;
        printf("\nSyncrate=%1d/s", Syncrate);

        for (int i = 0; i < NumChannels; i++) // for all channels
        {
            if (APICALL(HH500_GetCountRate(dev[0], i, &Countrate)) < 0)
                goto ex;
            printf("\nCountrate[%1d]=%1d/s", i, Countrate);
        }

        if (APICALL(HH500_StartMeas(dev[0], Tacq)) < 0)
            goto ex;

        if (meascontrol != MEASCTRL_SINGLESHOT_CTC) 
        {
            printf("\nwaiting for hardware start on C1...");
            fflush(stdout);
            ctcstatus = 1;
            while (ctcstatus == 1)
            {
                if (APICALL(HH500_CTCStatus(dev[0], &ctcstatus)) < 0)
                    goto ex;
            }
        }

        if ((meascontrol == MEASCTRL_SINGLESHOT_CTC) || meascontrol == MEASCTRL_C1_START_CTC_STOP)
            printf("\n\nMeasuring for %1d milliseconds...", Tacq);

        if (meascontrol == MEASCTRL_C1_GATED)
            printf("\n\nMeasuring, waiting for other C1 edge to stop...");

        if (meascontrol == MEASCTRL_C1_START_C2_STOP)
            printf("\n\nMeasuring, waiting for C2 to stop...");

        fflush(stdout);

        ctcstatus = 0;

        while (ctcstatus == 0)
        {
            if (APICALL(HH500_CTCStatus(dev[0], &ctcstatus)) < 0)
                goto ex;
        }

        if (APICALL(HH500_StopMeas(dev[0])) < 0)
            goto ex;

        if (APICALL(HH500_GetElapsedMeasTime(dev[0], &elapsed)) < 0)
            goto ex;
        printf("\n  Elapsed measurement time was %1.0lf ms", elapsed);

        printf("\n");
        for (int i = 0; i < NumChannels; i++) // for all channels
        {
            if (APICALL(HH500_GetHistogram(dev[0], counts[i], i)) < 0)
                goto ex;

            Integralcount = 0;
            for (int j = 0; j < HistLen; j++)
                Integralcount += counts[i][j];

            printf("\n  Integralcount[%1d]=%1.0lf", i, Integralcount);
        }
        printf("\n");

        if (APICALL(HH500_GetFlags(dev[0], &flags)) < 0)
            goto ex;

        if (flags & FLAG_OVERFLOW)
            printf("\n  Overflow.");

        printf("\nEnter c to continue or q to quit and save the count data.");
        cmd = getchar();
        getchar();
    }

    for (int i = 0; i < NumChannels; i++)
        fprintf(fpout, "  ch%02d ", i+1);
    fprintf(fpout, "\n");
    for (int j = 0; j < HistLen; j++)
    {
        for (int i = 0; i < NumChannels; i++)
            fprintf(fpout, "%6d ", counts[i][j]);
        fprintf(fpout, "\n");
    }

ex:
    for (int i = 0; i < MAXDEVNUM; i++) // no harm to close all
        HH500_CloseDevice(i);

    if (fpout)
        fclose(fpout);

    printf("\npress RETURN to exit");
    getchar();

    return 0;
}


