/************************************************************************

  Demo access to HydraHarp 500 hardware via HH500Lib v 1.0
  The program performs a measurement based on hardcoded settings.
  The resulting photon event data is instantly histogrammed. T3 mode only!

  Michael Wahl, PicoQuant GmbH, August 2025

  Note: This is a console application

  Note: At the API level the input channel numbers are indexed 0..N-1
  where N is the number of input channels the device has.
  Upon processing we map this back to 1..N corresponding to the front 
  panel labelling.


  Tested with the following compilers:

  - MinGW-W64 4.3.5 (Windows 64 bit)
  - MS Visual C++ 2019 (Windows 64 bit)
  - gcc 9.4.0 and 11.4.0 (Linux 64 bit)

************************************************************************/

#ifndef _WIN32
#include <unistd.h>
#define Sleep(msec) usleep(msec*1000)
#else
#include <windows.h>
#include <dos.h>
#include <conio.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>

#include "hh500defin.h"
#include "hh500lib.h"
#include "errorcodes.h"


FILE *fpout = NULL;
uint64_t oflcorrection = 0;
double Resolution = 0; // in ps
double Syncperiod = 0; // in s

unsigned int buffer[TTREADMAX];

#define T3HISTBINS 32768 // =2^15, dtime in T3 mode has 15 bits
unsigned int histogram[MAXINPCHAN][T3HISTBINS];


// helper macro and associated function for API calls with error check
// the stringize operator # makes a printable string from the macro's input argument
#define APICALL(call) doapicall(call, #call, __LINE__)
int doapicall(int retcode, char* callstr, int line)
{
    if (retcode < 0)
    {
        char errorstring[100] = { 0 };
        HH500_GetErrorString(errorstring, retcode);
        printf("\nThe API call %s at line %d\nreturned error %d (%s)\n", callstr, line, retcode, errorstring);
    }
    return retcode;
}


//Got PhotonT3
//  NSync: Overflow-corrected arrival time in units of the sync period 
//  DTime: Arrival time of photon after last Sync event in units of the chosen resolution (set by binning)
//  Channel: 1..N where N is the numer of channels the device has
void GotPhotonT3(uint64_t NSync, int Channel, int DTime)
{
    histogram[Channel-1][DTime]++; //histogramming
}


//Got MarkerT3
//  NSync: Overflow-corrected arrival time in units of the sync period 
//  Markers: Bitfield of arrived Markers, different markers can arrive at same time (same record)
void GotMarkerT3(uint64_t NSync, int Markers)
{
    // could switch to a new histogram here, e.g. for pixels in a FLIM application
}


// HydraHarpV2 or TimeHarp260 or MultiHarp or PicoHarp330 T3 record data
void ProcessT3(unsigned int TTTRRecord)
{
    int ch, dt;
    uint64_t truensync;
    const int T3WRAPAROUND = 1024;

    union {
        unsigned allbits;
        struct {
            unsigned nsync : 10;  // number of sync period
            unsigned dtime : 15;  // delay from last sync in units of chosen resolution
            unsigned channel : 6;
            unsigned special : 1;
        } bits;
    } T3Rec;

    T3Rec.allbits = TTTRRecord;

    if (T3Rec.bits.special == 1)
    {
        if (T3Rec.bits.channel == 0x3F) // overflow
        {
            // number of overflows is stored in nsync
            oflcorrection += (uint64_t)T3WRAPAROUND * T3Rec.bits.nsync;
        }
        if ((T3Rec.bits.channel >= 1) && (T3Rec.bits.channel <= 15)) // markers
        {
            truensync = oflcorrection + T3Rec.bits.nsync;
            // the time unit depends on sync period
            GotMarkerT3(truensync, T3Rec.bits.channel);
        }
    }
    else // regular input channel
    {
        truensync = oflcorrection + T3Rec.bits.nsync;
        ch = T3Rec.bits.channel + 1; // we encode the input channels as 1..N
        dt = T3Rec.bits.dtime;
        // truensync indicates the number of the sync period this event was in
        // the dtime unit depends on the chosen resolution (binning)
        GotPhotonT3(truensync, ch, dt);
    }
}

// for brevity of the main demo code further below we encapsulate the programming
// of the input circuits as a soubroutine here
int SetInputModalities(int devidx, int numchans)
{
    // Dependent on the hardware model, the HydraHarp 500 may have different input
    // circuits: Edge Trigger (ETR) and/or Constant Fraction Discrimonator (CFD)
    // This can even vary from channel to channel. An input that has both ETR and CFD
    // will be programmable as to which trigger mode to use. Here we define default 
    // settings for both variants and the default trigger mode ETR.
    // The latter can be changed to CFD which of course only works with channels
    // that actually have a CFD. The code further below will therefore query the 
    // input capabilities and where necessary fall back to the mode available.

    int SyncTrgMode = TRGMODE_ETR;       // you can change this to TRGMODE_CFD    
    // in case of SyncTrgMode == TRGMODE_ETR this will apply:
    int SyncTiggerEdge = EDGE_FALLING;   // you can change this to EDGE_RISING
    int SyncTriggerLevel = -50;          // in mV, you can change this
    // in case of SyncTrgMode == TRGMODE_CFD this will apply:
    int SyncCFDZeroCross = -20;          // in mV, you can change this
    int SyncCFDLevel = -50;              // in mV, you can change this

    int InputTrgMode = TRGMODE_ETR;     // you can change this to TRGMODE_CFD
    // in case of InputTrgMode == TRGMODE_ETR this will apply:
    int InputTriggerEdge = EDGE_FALLING; // you can change this to EDGE_RISING
    int InputTriggerLevel = -50;         // in mV, you can change this
    // in case of InputTrgMode == TRGMODE_CFD this will apply:
    int InputCFDZeroCross = -20;         // in mV, you can change this
    int InputCFDLevel = -50;             // in mV, you can change this

    int SyncChannelOffset = -5000;       // you can change this (in ps, like a cable delay)
    int InputChannelOffset = 0;          // you can change this (in ps, like a cable delay)

    unsigned ChnFeatures = 0;

    if (APICALL(HH500_GetSyncFeatures(devidx, &ChnFeatures)) < 0)
        return -1;

    // check if the sync channel has the right feature for the requested setting
    if ((SyncTrgMode == TRGMODE_ETR) && ((ChnFeatures & HAS_ETR) == 0))
    {
        printf("\nWarning: Sync channel has no Edge Trigger, switching to CFD");
        SyncTrgMode = TRGMODE_CFD;
    }
    if ((SyncTrgMode == TRGMODE_CFD) && ((ChnFeatures & HAS_CFD) == 0))
    {
        printf("\nWarning: Sync channel has no CFD, switching to Edge Trigger");
        SyncTrgMode = TRGMODE_ETR;
    }

    if (APICALL(HH500_SetSyncTrgMode(devidx, SyncTrgMode)) < 0)
        return -1;

    if (SyncTrgMode == TRGMODE_ETR)
    {
        if (APICALL(HH500_SetSyncEdgeTrg(devidx, SyncTriggerLevel, SyncTiggerEdge)) < 0)
            return -1;
    }

    if (SyncTrgMode == TRGMODE_CFD)
    {
        if (APICALL(HH500_SetSyncCFD(devidx, SyncCFDLevel, SyncCFDZeroCross)) < 0)
            return -1;
    }

    if (APICALL(HH500_SetSyncChannelOffset(devidx, SyncChannelOffset)) < 0)
        return -1;

    for (int i = 0; i < numchans; i++) // for simplicity we use the same settings for all channels
    {
        int RealTrgMode = InputTrgMode;

        if (APICALL(HH500_GetInputFeatures(devidx, i, &ChnFeatures)) < 0)
            return -1;

        // check if the input channel has the right feature for the requested setting
        if ((InputTrgMode == TRGMODE_ETR) && ((ChnFeatures & HAS_ETR) == 0))
        {
            printf("\nWarning: Input channel %1d has no Edge Trigger, switching to CFD", i + 1);
            RealTrgMode = TRGMODE_CFD;
        }
        if ((InputTrgMode == TRGMODE_CFD) && ((ChnFeatures & HAS_CFD) == 0))
        {
            printf("\nWarning: Input channel %1d has no CFD, switching to Edge Trigger", i + 1);
            RealTrgMode = TRGMODE_ETR;
        }

        if (APICALL(HH500_SetInputTrgMode(devidx, i, RealTrgMode)) < 0)
            return -1;

        if (RealTrgMode == TRGMODE_ETR)
        {
            if (APICALL(HH500_SetInputEdgeTrg(devidx, i, InputTriggerLevel, InputTriggerEdge)) < 0)
                return -1;
        }

        if (RealTrgMode == TRGMODE_CFD)
        {
            if (APICALL(HH500_SetInputCFD(devidx, i, InputCFDLevel, InputCFDZeroCross)) < 0)
                return -1;
        }

        if (APICALL(HH500_SetInputChannelOffset(devidx, i, InputChannelOffset)) < 0)
            return -1;

        if (APICALL(HH500_SetInputChannelEnable(devidx, i, 1)) < 0) // we enable all channels
            return -1;
    }
    return 0;
}


int main(int argc, char* argv[])
{
    int dev[MAXDEVNUM];
    int found = 0;
    int retcode;
    int ctcstatus;
    char LIB_Version[8];
    char HW_Model[32];
    char HW_Partno[8];
    char HW_Serial[9];
    char HW_Version[16];
    char Errorstring[40];
    char debuginfobuffer[16384]; // must have 16384 bytes of text buffer
    int NumChannels;
    int Mode = MODE_T3;  // keep as is, this demo is only for T3 !
    int Binning = 4;     // you can change this, meaningful only in T3 mode
    int Offset = 0;      // you can change this, meaningful only in T3 mode
    int Tacq = 1000;     // measurement time in millisec, you can change this
    int SyncDivider = 1; // you can change this, observe Mode! READ MANUAL!

    int Syncrate;
    int Countrate;
    int i, j;
    int flags;
    int warnings;
    char warningstext[16384]; //must have 16384 bytes of text buffer
    int nRecords;
    unsigned int Progress;
    int stopretry = 0;

    memset(histogram, 0, sizeof(histogram));

    printf("\nHydraHarp 500 - HH500Lib Demo Application             PicoQuant GmbH, 2025");
    printf("\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~");

    if (Mode == MODE_T2)
    {
        printf("This demo is not for use with T2 mode!\n");
        goto ex;
    }

    if (APICALL(HH500_GetLibraryVersion(LIB_Version)) < 0)
        goto ex;
    printf("\nLibrary version is %s", LIB_Version);
    if (strncmp(LIB_Version, LIB_VERSION, sizeof(LIB_VERSION)) != 0)
        printf("\nWarning: The application was built for version %s.", LIB_VERSION);

    if ((fpout = fopen("t3histout.txt", "w")) == NULL)
    {
        printf("\ncannot open output file\n");
        goto ex;
    }

    printf("\nSearching for HydraHarp 500 devices...");
    printf("\nDevidx     Serial     Status");

    for (i = 0; i < MAXDEVNUM; i++)
    {
        memset(HW_Serial, 0, sizeof(HW_Serial));
        retcode = HH500_OpenDevice(i, HW_Serial);
        // not using APICALL here to allow selective error handling
        if (retcode == 0) // grab any device we can open
        {
            printf("\n  %1d        %7s    open ok", i, HW_Serial);
            dev[found] = i; // keep index to devices we may want to use
            found++;
        }
        else
        {
            if (retcode == HH500_ERROR_DEVICE_OPEN_FAIL)
                printf("\n  %1d        %7s    no device", i, HW_Serial);
            else
            {
                HH500_GetErrorString(Errorstring, retcode);
                printf("\n  %1d        %7s    %s", i, HW_Serial, Errorstring);
            }
        }
    }

    // In this demo we will use the first device we find, i.e. dev[0].
    // You could also use multiple devices in parallel.
    // You can also check for specific serial numbers, so that you know 
    // which physical device you are talking to.

    if (found < 1)
    {
        printf("\nNo device available.");
        goto ex;
    }
    printf("\nUsing device #%1d", dev[0]);
    printf("\nInitializing the device...");

    if (APICALL(HH500_Initialize(dev[0], Mode, 0)) < 0)
    {
        // in case of an obscure error (a hardware error in particular) 
        // it may be helpful to obtain debug information like so:
        HH500_GetDebugInfo(dev[0], debuginfobuffer);
        printf("\nDEBUGINFO:\n%s", debuginfobuffer);
        goto ex;
    }

    if (APICALL(HH500_GetHardwareInfo(dev[0], HW_Model, HW_Partno, HW_Version)) < 0)
        goto ex;
    printf("\nFound Model %s Part no %s Version %s", HW_Model, HW_Partno, HW_Version);

    if (APICALL(HH500_GetNumOfInputChannels(dev[0], &NumChannels)) < 0)
        goto ex;
    printf("\nDevice has %i input channels.", NumChannels);

    if (APICALL(HH500_SetSyncDiv(dev[0], SyncDivider)) < 0)
        goto ex;

    if (SetInputModalities(dev[0], NumChannels) < 0) // see soubroutine above
        goto ex;

    if (Mode != MODE_T2)
    {
        if (APICALL(HH500_SetBinning(dev[0], Binning)) < 0)
            goto ex;

        if (APICALL(HH500_SetOffset(dev[0], Offset)) < 0)
            goto ex;
    }

    if (APICALL(HH500_GetResolution(dev[0], &Resolution)) < 0)
        goto ex;
    printf("\nResolution is %1.0lfps\n", Resolution);

    // after Init allow 150 ms for valid  count rate readings
    // subsequently you get new values after every 100ms
    Sleep(150);

    if (APICALL(HH500_GetSyncRate(dev[0], &Syncrate)) < 0)
        goto ex;
    printf("\nSyncrate=%1d/s", Syncrate);

    for (i = 0; i < NumChannels; i++) // for all channels
    {
        if (APICALL(HH500_GetCountRate(dev[0], i, &Countrate)) < 0)
            goto ex;
        printf("\nCountrate[%1d]=%1d/s", i, Countrate);
    }
    printf("\n");

    // after getting the count rates you can check for warnings
    if (APICALL(HH500_GetWarnings(dev[0], &warnings)) < 0)
        goto ex;

    if (warnings)
    {
        if (APICALL(HH500_GetWarningsText(dev[0], warningstext, warnings)) < 0)
            goto ex;
        printf("\n\n%s", warningstext);
    }
  
    printf("\npress RETURN to start");
    getchar();

    if (APICALL(HH500_StartMeas(dev[0], Tacq)) < 0)
        goto ex;

    if (Mode == MODE_T3)
    {
        // We need the sync period in order to calculate the true times of photon records.
        // This only makes sense in T3 mode and it assumes a stable period like from a laser.
        // Note: Two sync periods must have elapsed after HH500_StartMeas to get proper results.
        // You can also use the inverse of what you read via GetSyncRate but it depends on 
        // the actual sync rate if this is accurate enough.
        // It is OK to use the sync input for a photon detector, e.g. if you want to perform
        // something like an antibunching measurement. In that case the sync rate obviously is
        // not periodic. This means that a) you should set the sync divider to 1 (none) and
        // b) that you cannot meaningfully measure the sync period here, which probaly won't
        // matter as you only care for the time difference (dtime) of the events.
        if (APICALL(HH500_GetSyncPeriod(dev[0], &Syncperiod)) < 0)
            goto ex;
        printf("\nSync period is %lf ns\n", Syncperiod * 1e9);
    }

    printf("\nStarting data collection...\n");

    Progress = 0;
    printf("\nProgress:%12u", Progress);

    oflcorrection = 0;

    while (1)
    {
        if (APICALL(HH500_GetFlags(dev[0], &flags)) < 0)
            goto ex;

        if (flags & FLAG_FIFOFULL)
        {
            printf("\nFIFO Overrun!\n");
            goto stoptttr;
        }

        if (APICALL(HH500_ReadFiFo(dev[0], buffer, &nRecords)) < 0)
            goto ex;

        if (nRecords)
        {
            // Here we process the data. Note that the time this consumes prevents us
            // from getting around the loop quickly for the next FIFO read.
            // In a serious performance critical scenario you would write the data to
            // a software queue and do the processing in another thread reading from 
            // that queue.

            for (i = 0; i < nRecords; i++)
                ProcessT3(buffer[i]);

            Progress += nRecords;
            printf("\b\b\b\b\b\b\b\b\b\b\b\b%12u", Progress);
            fflush(stdout);
        }
        else
        {
            if (APICALL(HH500_CTCStatus(dev[0], &ctcstatus)) < 0)
                goto ex;

            if (ctcstatus)
            {
                stopretry++; // do a few more rounds as there might be some more data in the FIFO
                if (stopretry > 15)
                {
                    printf("\nDone\n");
                    goto stoptttr;
                }
            }
        }

        // within this loop you can also read the count rates if needed
    }

stoptttr:

    if (APICALL(HH500_StopMeas(dev[0])) < 0)
        goto ex;

    for (i = 0; i < NumChannels; i++)
        fprintf(fpout, "  ch%02d ", i+1);
    fprintf(fpout, "\n");   
    for (i = 0; i < T3HISTBINS; i++)
    {
        for (j = 0; j < NumChannels; j++)
            fprintf(fpout, "%6u ", histogram[j][i]);
        fprintf(fpout, "\n");
    }

ex:

    for (i = 0; i < MAXDEVNUM; i++) // no harm to close all
        HH500_CloseDevice(i);

    if (fpout)
        fclose(fpout);

    printf("\npress RETURN to exit");
    getchar();

    return 0;
}


