/************************************************************************

Demo access to the HydraHarp 500 external FPGA interface. The application sets
up the link to the external FPGA using the HydraHarp 500 library functions.
This demo application is compatible with the VHDL code in the file
usr_application_example.vhd, which constructs a basic statistics module
analyzing the TTTR data transmitted from the HydraHarp 500 to an external FPGA.
The statistics are formatted in a 32-bit x 128 array and then transmitted to
the USB FIFO, which will be read by this file. This is a basic example of how
to use the HydraHarp 500 USB connection for your own FPGA implementation. You can
build your own protocol based on this example. You can also retrieve data in
T2 or T3 mode e.g. for filter sczenarios. For more information please refer to
the manual.

Nicolai Adelhoefer, PicoQuant GmbH, July 2025

Tested with the following compilers:
  windows:
    - gcc ( MinGW-W64 12.2.0 )
    - MSVC 2019
  linux:
   - gcc 13.3.0

************************************************************************/

// Include os dependant library headers
#ifndef _WIN32
#include <unistd.h>
#define Sleep(msec) usleep(msec * 1000)
#define __int64 long long
#else
#include <windows.h>
#endif

// Include standard library headers
#include <stdio.h>  
#include <stdlib.h>
#include <string.h>
#include <stdint.h>

#include "hh500defin.h"
#include "hh500lib.h"
#include "errorcodes.h"

// Constants definitions
#define EXTFPGA_USER_PADDING 0xABABABAB
#define APICALL(call) doapicall(call, #call, __LINE__)

// global definitions
uint32_t buffer[TTREADMAX];
FILE *fpout;

// private function prototypes
int extfpga_block_until_link_rdy(int devidx, unsigned int link_index, unsigned int timeout_ms);
int extfpga_demo_cleanup(void);
int doapicall(int retcode, char* callstr, int line);


int main(int argc, char *argv[])
{
  // local variables
  int dev[MAXDEVNUM];
  int found = 0;
  int retcode;
  int ctcstatus;
  char LIB_Version[8];
  char HW_Model[32];
  char HW_Partno[8];
  char HW_Serial[9];
  char HW_Version[16];
  int NumChannels;
  int NumModules;
  int Mode = MODE_T2; // set T2 or T3 here, observe suitable Sync divider and Range!
  int Tacq = 1000;    // Measurement time in millisec, you can change this

  int SyncTiggerEdge = 0;      // you can change this
  int SyncTriggerLevel = -50;  // you can change this
  int InputTriggerEdge = 0;    // you can change this
  int InputTriggerLevel = -50; // you can change this

  int Syncrate;
  int Countrate;
  int i;
  int flags;
  int nRecords;

  // External FPGA Interface variables
  int ext_fpga_mode = EXTFPGA_MODE_T2;
  int ext_fpga_loopback = EXTFPGA_LOOPBACK_T2;
  int ext_fpga_if = EXTFPGA_LINK_ON_FRONT_SFP;
  int ext_fpga_force_reinit = 0;
  uint32_t ext_link_status[9];
  uint32_t ext_usr_addr;
  uint32_t ext_usr_data;

  printf("\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~");
  printf("\nExternal FPGA Interface for HydraHarp 500 Demo Application    PicoQuant GmbH, 2025");
  printf("\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~");
  HH500_GetLibraryVersion(LIB_Version);
  printf("\nLibrary version is %s\n", LIB_Version);

  printf("\nSearching for HydraHarp 500 devices...");
  printf("\nDevidx     Serial     Status");

  for (i = 0; i < MAXDEVNUM; i++)
  {
    retcode = HH500_OpenDevice(i, HW_Serial);
    if (retcode == 0) // Grab any device we can open
    {
      printf("\n  %1d        %7s    open ok", i, HW_Serial);
      dev[found] = i; // keep index to devices we want to use
      found++;
    }
  }

  // general HydraHarp 500 setup and statistics
  if (found < 1)
  {
    printf("\nNo device available.");
    return extfpga_demo_cleanup();
  }
  printf("\nUsing device #%1d", dev[0]);
  printf("\nInitializing the device...");

  APICALL(HH500_Initialize(dev[0], Mode, 0));

  APICALL(HH500_GetHardwareInfo(dev[0], HW_Model, HW_Partno, HW_Version));
  printf("\nFound Model %s Part no %s Version %s", HW_Model, HW_Partno, HW_Version);

  APICALL(HH500_GetNumOfInputChannels(dev[0], &NumChannels));
  printf("\nDevice has %i input channels.", NumChannels);

  APICALL(HH500_GetNumOfModules(dev[0], &NumModules));
  printf("\nDevice has %i Modules.", NumModules);

  printf("\n\nUsing the following settings:\n");
  printf("Mode              : %d\n", Mode);
  printf("extFPGA Mode      : %d\n", ext_fpga_mode);
  printf("extFPGA Loopback  : %d\n", ext_fpga_loopback);
  printf("AcquisitionTime   : %d\n", Tacq);
  printf("SyncTiggerEdge    : %d\n", SyncTiggerEdge);
  printf("SyncTriggerLevel  : %d\n", SyncTriggerLevel);
  printf("InputTriggerEdge  : %d\n", InputTriggerEdge);
  printf("InputTriggerLevel : %d\n", InputTriggerLevel);
//  HH500_ExtFPGAInitLink(dev[0], 0, 0);
 // HH500_WRabbitInitLink(dev[0], 0);
  // external fpga setup
  printf("\nInitializing External FPGA Interface...");
  for (i = 0; i < NumModules; i++)
  {
    printf("\n get M%d Link Status...", i);
    APICALL(HH500_ExtFPGAGetLinkStatus(dev[0], i, &ext_link_status[i])); // check all links for their statu)s
    printf(" %x", ext_link_status[i]);
  }

  if (!ext_fpga_mode && ext_link_status[0] == 0x3) // if mode is off and link is up --> switch off interface
  {
    printf("\n   Link Init M0 Off...");
    APICALL(HH500_ExtFPGAInitLink(dev[0], 0, EXTFPGA_MODE_OFF)); // if mode is zero, switch off external fpga interface
    printf("ready");
  }

  if (ext_fpga_mode)
  { // module M0 must be initialized first, then all other links can optionally be enabled
    if (ext_fpga_force_reinit || ext_link_status[0] != 0x3)
    {
      printf("\n   Link Init M0...");
      APICALL(HH500_ExtFPGAInitLink(dev[0], 0, ext_fpga_if));
      extfpga_block_until_link_rdy(0, 0, 1000); // blocking wait for link up on m0 before interacting with interface!
      printf("OK");
    }
    else
    {
      printf("\n   Link M0 is ready");
    }

    if (ext_fpga_mode == EXTFPGA_MODE_T2RAW)
    {
      for (i = 1; i < NumModules; i++)
      {
        if (ext_fpga_force_reinit || ext_link_status[i] != 0x3)
        {
          printf("\n   Link Init M%d...", i);
          APICALL(HH500_ExtFPGAInitLink(dev[0], i, 1));
          extfpga_block_until_link_rdy(0, i, 1000); // blocking wait for link up on m0 before interacting with interface!
          printf("OK");
        }
        else
        {
          printf("\n   Link M%d is ready", i);
        }
        APICALL(HH500_ExtFPGAResetStreamFifos(dev[0]));
      }
    }
    else // If RAWT2 is not selected, switch off all active, unnecessary links
    {
      for (i = 1; i < NumModules; i++)
      {
        if (ext_link_status[i] != 0x0)
        {
          printf("\n   Link M%d Off...", i);
          APICALL(HH500_ExtFPGAInitLink(dev[0], i, EXTFPGA_MODE_OFF));
          printf("OK");
        }
      }
    }

    // Use user command to set a register in the external FPGA and read back to verify correctness.
    ext_usr_addr = 0;
    ext_usr_data = 1;
    printf("\n HH500_ExtFPGAUserCommand Write: ADDR=%8x DATA=%8x", ext_usr_addr, ext_usr_data);
    APICALL(HH500_ExtFPGAUserCommand(dev[0], 1, ext_usr_addr, &ext_usr_data)); // write

    ext_usr_addr = 0;
    ext_usr_data = 0;
    printf("\n HH500_ExtFPGAUserCommand Read: ADDR=%8x", ext_usr_addr);
    APICALL(HH500_ExtFPGAUserCommand(dev[0], 0, ext_usr_addr, &ext_usr_data)); // read back
    printf("DATA=%8x", ext_usr_data);

    printf("\n Set Ext FPGA Mode=%d Loopback=%d ", ext_fpga_mode, ext_fpga_loopback);
    APICALL(HH500_ExtFPGASetMode(dev[0], ext_fpga_mode, ext_fpga_loopback));
    printf("OK");
  }

  APICALL(HH500_SetSyncEdgeTrg(dev[0], SyncTriggerLevel, SyncTiggerEdge));

  for (i = 0; i < NumChannels; i++) // we use the same input offset for all channels
  {
    APICALL(HH500_SetInputEdgeTrg(dev[0], i, InputTriggerLevel, InputTriggerEdge));

    APICALL(HH500_SetInputChannelEnable(dev[0], i, 1));
  }

  printf("\nMeasuring input rates...\n");

  // After Init allow 150 ms for valid  count rate readings
  // Subsequently you get new values after every 100ms
  Sleep(150);

  // check channel input rates at HydraHarp 500 to verify calculated input rates at the external fpga later
  APICALL(HH500_GetSyncRate(dev[0], &Syncrate));
  printf("\nSyncrate=%1d/s", Syncrate);

  for (i = 0; i < NumChannels; i++) // for all channels
  {
    APICALL(HH500_GetCountRate(dev[0], i, &Countrate));
    printf("\nCountrate[%1d]=%1d/s", i, Countrate);
  }
  printf("\n");

  // Infinite loop for retrieving and displaying statistics data calculated by the external FPGA
  while (1)
  {
   // clear old data which might be in buffers just before starting a new measurement
    APICALL(HH500_ExtFPGAResetStreamFifos(dev[0]));

  
    printf("\nDo you want to start a new measurement? Y/n : ");
    int key = getchar();
    if (key == 'n' || key == 'N')
    {
      return extfpga_demo_cleanup();
    }

    printf("\nStarting data collection...\n");
    APICALL(HH500_StartMeas(dev[0], Tacq));

    int measurement_active = 1;
    while (measurement_active)
    {

      APICALL(HH500_GetFlags(dev[0], &flags));

      if (flags & FLAG_FIFOFULL)
      {
        measurement_active = 0; // when fifo is full, data wasnt read out fast enough --> data gets invalid
        break;
      }

      APICALL(HH500_ReadFiFo(dev[0], buffer, &nRecords)); // may return less)!

      if (nRecords)
      {
  // insert user code here
      }
      else
      {
        APICALL(HH500_CTCStatus(dev[0], &ctcstatus));

        if (ctcstatus)
        {
          measurement_active = 0;
  }
    }
    APICALL(HH500_StopMeas(dev[0]));
  }
}
}
// helper functions

int extfpga_demo_cleanup(void)
{                                     // close file descriptors and all harp devices
  for (int i = 0; i < MAXDEVNUM; i++) // no harm to close all
  {
    HH500_CloseDevice(i);
  }
  if (fpout)
  {
    fclose(fpout);
  }

  return EXIT_SUCCESS;
}

int extfpga_block_until_link_rdy(int devidx, unsigned int link_index, unsigned int timeout_ms)
{
  uint32_t t_ms = 0;
  unsigned int link_status = 0;
  int retcode = -1;

  while (t_ms < timeout_ms && link_status == 0)
  {
    APICALL(HH500_ExtFPGAGetLinkStatus(devidx, link_index, &link_status));
    Sleep(30);
    t_ms += 30;
  }
  return retcode;
}

int doapicall(int retcode, char* callstr, int line)
{
    if (retcode < 0)
    {
        char errorstring[100] = { 0 };
        HH500_GetErrorString(errorstring, retcode);
        printf("\nThe API call %s at line %d\nreturned error %d (%s)\n", callstr, line, retcode, errorstring);
        extfpga_demo_cleanup();
        exit(EXIT_FAILURE);
    }
    return retcode;
}