/************************************************************************

  Demo access to HydraHarp 500 hardware via HH500Lib v 1.0

  THIS IS AN ADVANCED DEMO. DO NOT USE FOR YOUR FIRST EXPERIMENTS.
  Look at the variable meascontrol down below to see what it does.

  The program performs a measurement based on hardcoded settings.
  The resulting histogram is stored in an ASCII output file.

  Michael Wahl, PicoQuant GmbH, August 2025

  Note: This is a console application

  Note: At the API level channel numbers are indexed 0..N-1 
    where N is the number of channels the device has.
 
  Tested with the following compilers:

  - Microsoft Visual C# 2019 (Windows)
  - Mono 6.12.0 (Windows and Linux)

************************************************************************/

using System;  // for Console
using System.Text;  // for StringBuilder 
using System.IO;  // for File
using static HH500LibSpace.HH500Lib;

class HistoMode
{
    // for brevity of the main demo code further below we encapsulate the programming
    // of the input circuits as a soubroutine here
    public static int SetInputModalities(int devidx, int numchans)
    {
        // Dependent on the hardware model, the HydraHarp 500 may have different input
        // circuits: Edge Trigger (ETR) and/or Constant Fraction Discrimonator (CFD)
        // This can even vary from channel to channel. An input that has both ETR and CFD
        // will be programmable as to which trigger mode to use. Here we define default 
        // settings for both variants and the default trigger mode ETR.
        // The latter can be changed to CFD which of course only works with channels
        // that actually have a CFD. The code further below will therefore query the 
        // input capabilities and where necessary fall back to the mode available.

        int SyncTrgMode = TRGMODE_ETR;       // you can change this to TRGMODE_CFD    
                                             // in case of SyncTrgMode == TRGMODE_ETR this will apply:
        int SyncTiggerEdge = EDGE_FALLING;   // you can change this to EDGE_RISING
        int SyncTriggerLevel = -50;          // in mV, you can change this
                                             // in case of SyncTrgMode == TRGMODE_CFD this will apply:
        int SyncCFDZeroCross = -20;          // in mV, you can change this
        int SyncCFDLevel = -50;              // in mV, you can change this

        int InputTrgMode = TRGMODE_ETR;      // you can change this to TRGMODE_CFD
                                             // in case of InputTrgMode == TRGMODE_ETR this will apply:
        int InputTriggerEdge = EDGE_FALLING; // you can change this to EDGE_RISING
        int InputTriggerLevel = -50;         // in mV, you can change this
                                             // in case of InputTrgMode == TRGMODE_CFD this will apply:
        int InputCFDZeroCross = -20;         // in mV, you can change this
        int InputCFDLevel = -50;             // in mV, you can change this

        int SyncChannelOffset = -5000;       // you can change this (in ps, like a cable delay)
        int InputChannelOffset = 0;          // you can change this (in ps, like a cable delay)

        uint ChnFeatures = 0;

        if (APICALL(() => HH500_GetSyncFeatures(devidx, ref ChnFeatures)) < 0)
            return -1;

        // check if the sync channel has the right feature for the requested setting
        if ((SyncTrgMode == TRGMODE_ETR) && ((ChnFeatures & HAS_ETR) == 0))
        {
            Console.WriteLine("Warning: Sync channel has no Edge Trigger, switching to CFD");
            SyncTrgMode = TRGMODE_CFD;
        }
        if ((SyncTrgMode == TRGMODE_CFD) && ((ChnFeatures & HAS_CFD) == 0))
        {
            Console.WriteLine("Warning: Sync channel has no CFD, switching to Edge Trigger");
            SyncTrgMode = TRGMODE_ETR;
        }

        if (APICALL(() => HH500_SetSyncTrgMode(devidx, SyncTrgMode)) < 0)
            return -1;

        if (SyncTrgMode == TRGMODE_ETR)
        {
            if (APICALL(() => HH500_SetSyncEdgeTrg(devidx, SyncTriggerLevel, SyncTiggerEdge)) < 0)
                return -1;
        }

        if (SyncTrgMode == TRGMODE_CFD)
        {
            if (APICALL(() => HH500_SetSyncCFD(devidx, SyncCFDLevel, SyncCFDZeroCross)) < 0)
                return -1;
        }

        if (APICALL(() => HH500_SetSyncChannelOffset(devidx, SyncChannelOffset)) < 0)
            return -1;

        for (int i = 0; i < numchans; i++) // for simplicity we use the same settings for all channels
        {
            int RealTrgMode = InputTrgMode;

            if (APICALL(() => HH500_GetInputFeatures(devidx, i, ref ChnFeatures)) < 0)
                return -1;

            // check if the input channel has the right feature for the requested setting
            if ((InputTrgMode == TRGMODE_ETR) && ((ChnFeatures & HAS_ETR) == 0))
            {
                Console.WriteLine("Warning: Input channel {0} has no Edge Trigger, switching to CFD", i + 1);
                RealTrgMode = TRGMODE_CFD;
            }
            if ((InputTrgMode == TRGMODE_CFD) && ((ChnFeatures & HAS_CFD) == 0))
            {
                Console.WriteLine("Warning: Input channel {0} has no CFD, switching to Edge Trigger", i + 1);
                RealTrgMode = TRGMODE_ETR;
            }

            if (APICALL(() => HH500_SetInputTrgMode(devidx, i, RealTrgMode)) < 0)
                return -1;

            if (RealTrgMode == TRGMODE_ETR)
            {
                if (APICALL(() => HH500_SetInputEdgeTrg(devidx, i, InputTriggerLevel, InputTriggerEdge)) < 0)
                    return -1;
            }

            if (RealTrgMode == TRGMODE_CFD)
            {
                if (APICALL(() => HH500_SetInputCFD(devidx, i, InputCFDLevel, InputCFDZeroCross)) < 0)
                    return -1;
            }

            if (APICALL(() => HH500_SetInputChannelOffset(devidx, i, InputChannelOffset)) < 0)
                return -1;

            if (APICALL(() => HH500_SetInputChannelEnable(devidx, i, 1)) < 0) // we enable all channels
                return -1;
        }

        return 0;
    }

    static void Main()
    {
        //////////////////////// some settings you can change /////////////////////////

        int Binning = 0;      // you can change this, observe limits
        int Offset = 0;       // you can change this, observe limits
        int Tacq = 100;      // Measurement time in millisec, you can change this, observe limits
        int SyncDivider = 1;  // you can change this, observe limits

        int meascontrol
         = MEASCTRL_SINGLESHOT_CTC;     // start by software and stop when CTC expires (default)
        // = MEASCTRL_C1_GATED;           // measure while C1 is active		1
        // = MEASCTRL_C1_START_CTC_STOP;  // start with C1 and stop when CTC expires 
        // = MEASCTRL_C1_START_C2_STOP;   // start with C1 and stop with C2
        int edge1 = EDGE_RISING;          // edge of C1 to start (if applicable in chosen mode)
        int edge2 = EDGE_FALLING;         // edge of C2 to stop (if applicable in chosen mode)

        /////////////////////////////// program variables //////////////////////////////

        int i, j;
        int retcode;
        string cmd = "";
        int[] dev = new int[MAXDEVNUM];
        int found = 0;
        int NumChannels = 0;

        StringBuilder LibVer = new StringBuilder(8);
        StringBuilder Serial = new StringBuilder(8);
        StringBuilder Errstr = new StringBuilder(40);
        StringBuilder Model = new StringBuilder(32);
        StringBuilder Partno = new StringBuilder(16);
        StringBuilder Version = new StringBuilder(16);
        StringBuilder Wtext = new StringBuilder(16384);

        double Resolution = 0;
        int HistLen = 0;

        int Syncrate = 0;
        int Countrate = 0;
        int ctcstatus = 0;
        int flags = 0;
        int warnings = 0;
        double elapsed = 0;

        uint[][] counts = new uint[MAXINPCHAN][];
        for (i = 0; i < MAXINPCHAN; i++)
            counts[i] = new uint[MAXHISTLEN];

        StreamWriter SW = null;

        ///////////////////////////// main program code /////////////////////////////

        Console.WriteLine("HydraHarp 500 - HH500Lib Demo Application              PicoQuant GmbH, 2025");
        Console.WriteLine("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~");

        if (APICALL(() => HH500_GetLibraryVersion(LibVer)) < 0)
            goto ex;

        Console.WriteLine("HH500Lib Version is " + LibVer);

        if (LibVer.ToString() != LIB_VERSION)
        {
            Console.WriteLine("This program requires HH500Lib v." + LIB_VERSION);
            goto ex;
        }

        try
        {
            SW = File.CreateText("histomodeout.txt");
        }
        catch (Exception)
        {
            Console.WriteLine("Error creating file");
            goto ex;
        }

        Console.WriteLine("Searching for HydraHarp 500 devices...");
        Console.WriteLine("Devidx     Status");

        for (i = 0; i < MAXDEVNUM; i++)
        {
            Serial.Clear();
            // not using APICALL here as we wish to handle errors gracefully 
            retcode = HH500_OpenDevice(i, Serial);
            if (retcode == 0) // grab any HydraHarp 500 we can open
            {
                Console.WriteLine("  {0}        S/N {1}", i, Serial);
                dev[found] = i; // keep index to devices we want to use
                found++;
            }
            else
            {
                if (retcode == HH500_ERROR_DEVICE_OPEN_FAIL)
                {
                    Console.WriteLine("  {0}        no device", i);
                }
                else
                {
                    HH500_GetErrorString(Errstr, retcode);
                    Console.WriteLine("  {0}        S/N {1}", i, Errstr);
                }
            }
        }

        // In this demo we will use the first device we find, i.e. dev[0].
        // You can also use multiple devices in parallel.
        // You can also check for specific serial numbers, so that you always know 
        // which physical device you are talking to.

        if (found < 1)
        {
            Console.WriteLine("No device available.");
            goto ex;
        }

        Console.WriteLine("Using device {0}", dev[0]);
        Console.WriteLine();
        Console.WriteLine("Initializing the device...");

        if (APICALL(() => HH500_Initialize(dev[0], MODE_HIST, 0)) < 0)  // Histo mode with internal clock))
            goto ex;

        if (APICALL(() => HH500_GetHardwareInfo(dev[0], Model, Partno, Version)) < 0) // only for information
            goto ex;

        Console.WriteLine("Found Model {0} Part no {1} Version {2}", Model, Partno, Version);

        if (APICALL(() => HH500_GetNumOfInputChannels(dev[0], ref NumChannels)) < 0)
            goto ex;

        Console.WriteLine("Device has {0} input channels.", NumChannels);

        if (APICALL(() => HH500_SetSyncDiv(dev[0], SyncDivider)) < 0)
            goto ex;

        if (SetInputModalities(dev[0], NumChannels) < 0)
            goto ex;

        if (APICALL(() => HH500_SetHistoLen(dev[0], MAXLENCODE, out HistLen)) < 0)
            goto ex;

        Console.WriteLine("Histogram length is {0}", HistLen);

        if (APICALL(() => HH500_SetBinning(dev[0], Binning)) < 0)
            goto ex;

        if (APICALL(() => HH500_SetOffset(dev[0], Offset)) < 0)
            goto ex;

        if (APICALL(() => HH500_GetResolution(dev[0], ref Resolution)) < 0)
            goto ex;

        Console.WriteLine("Resolution is {0} ps", Resolution);

        // after Init allow 150 ms for valid  count rate readings
        // subsequently you get new values after every 100ms
        System.Threading.Thread.Sleep(150);

        Console.WriteLine();

        if (APICALL(() => HH500_GetSyncRate(dev[0], ref Syncrate)) < 0)
            goto ex;

        Console.WriteLine("Syncrate = {0}/s", Syncrate);

        for (i = 0; i < NumChannels; i++) // for all channels
        {
            if (APICALL(() => HH500_GetCountRate(dev[0], i, ref Countrate)) < 0)
                goto ex;

            Console.WriteLine("Countrate[{0}] = {1}/s", i, Countrate);
        }

        Console.WriteLine();

        // after getting the count rates you can check for warnings
        if (APICALL(() => HH500_GetWarnings(dev[0], ref warnings)) < 0)
            goto ex;

        if (warnings != 0)
        {
            if (APICALL(() => HH500_GetWarningsText(dev[0], Wtext, warnings)) < 0)
                goto ex;
            Console.WriteLine("{0}", Wtext);
        }

        if (APICALL(() => HH500_SetStopOverflow(dev[0], 0, 10000)) < 0)  // for example only
            goto ex;

        if (APICALL(() => HH500_SetMeasControl(dev[0], meascontrol, edge1, edge2)) < 0)
            goto ex;
        

        while (cmd != "q")
        {
            if (APICALL(() => HH500_ClearHistMem(dev[0])) < 0)
                goto ex;

            Console.WriteLine("press RETURN to start measurement");
            Console.ReadLine();

            if (APICALL(() => HH500_GetSyncRate(dev[0], ref Syncrate)) < 0)
                goto ex;

            Console.WriteLine("Syncrate = {0}/s", Syncrate);

            for (i = 0; i < NumChannels; i++) // for all channels
            {
                if (APICALL(() => HH500_GetCountRate(dev[0], i, ref Countrate)) < 0)
                    goto ex;

                Console.WriteLine("Countrate[{0}] = {1}/s", i, Countrate);
            }

            Console.WriteLine();

            if (APICALL(() => HH500_StartMeas(dev[0], Tacq)) < 0)
                goto ex;

            #region here the demo differs from 'histomode'

            if (meascontrol != MEASCTRL_SINGLESHOT_CTC)
            {
                Console.WriteLine("waiting for hardware start on C1...");
                ctcstatus = 1;
                while (ctcstatus == 1)
                {
                    if (APICALL(() => HH500_CTCStatus(dev[0], ref ctcstatus)) < 0)
                        goto ex;
                }
            }

            if ((meascontrol == MEASCTRL_SINGLESHOT_CTC) || meascontrol == MEASCTRL_C1_START_CTC_STOP)
                Console.WriteLine("\nMeasuring for {0} milliseconds...", Tacq);

            if (meascontrol == MEASCTRL_C1_GATED)
                Console.WriteLine("\nMeasuring, waiting for other C1 edge to stop...");

            if (meascontrol == MEASCTRL_C1_START_C2_STOP)
                Console.WriteLine("\nMeasuring, waiting for C2 to stop...");


            ctcstatus = 0;

            while (ctcstatus == 0)
            {
                if (APICALL(() => HH500_CTCStatus(dev[0], ref ctcstatus)) < 0)
                    goto ex;
            }

            if (APICALL(() => HH500_StopMeas(dev[0])) < 0)
                goto ex;

            if (APICALL(() => HH500_GetElapsedMeasTime(dev[0], out elapsed)) < 0)
                goto ex;
            Console.WriteLine("\n  Elapsed measurement time was {0} ms", elapsed);

            #endregion

            Console.WriteLine();
            for (i = 0; i < NumChannels; i++)  // for all channels
            {
                if (APICALL(() => HH500_GetHistogram(dev[0], counts[i], i)) < 0)
                    goto ex;

                double Integralcount = 0;
                for (j = 0; j < HistLen; j++)
                {
                    Integralcount += counts[i][j];
                }

                Console.WriteLine("  Integralcount[{0}] = {1}", i, Integralcount);
            }

            Console.WriteLine();

            if (APICALL(() => HH500_GetFlags(dev[0], ref flags)) < 0)
                goto ex;

            if ((flags & FLAG_OVERFLOW) != 0)
            {
                Console.WriteLine("  Overflow.");
            }

            Console.WriteLine("Enter c to continue or q to quit and save the count data.");
            cmd = Console.ReadLine();

        }  // while

        for (i = 0; i < NumChannels; i++)
             SW.Write("  ch{0,2} ", i+1);
        SW.Write("\n");

        for (j = 0; j < HistLen; j++)
        {
            for (i = 0; i < NumChannels; i++)
            {
                SW.Write("{0,6} ", counts[i][j]);
            }
            SW.WriteLine();
        }

    ex:
        for (i = 0; i < MAXDEVNUM; i++)  // no harm to close all
            HH500_CloseDevice(i);

        if (SW != null)
        {
            try
            {
                SW.Flush();
                SW.Close();
                SW.Dispose();
                SW = null;
            }
            catch (Exception e)
            {
                Console.WriteLine("Error closing the file: " + e);
            }
        }

        Console.WriteLine("press RETURN to exit");
        Console.ReadLine();
    }
}

