﻿/************************************************************************

  Demo access to HydraHarp 500 hardware via HH500Lib v 1.0
  The program performs a measurement based on hardcoded settings.
  The resulting photon event data is instantly histogrammed. T3 mode only!

  Michael Wahl, PicoQuant GmbH, August 2025

  Note: This is a console application

  Note: At the API level the input channel numbers are indexed 0..N-1
  where N is the number of input channels the device has.
  Upon processing we map this back to 1..N corresponding to the front 
  panel labelling.

  
  Tested with the following compilers:

  - MS Visual Studio 2019
  - Mono 6.12.0 (Windows and Linux)

************************************************************************/

using System;  // for Console
using System.Text;  // for StringBuilder 
using System.IO;  // for File
using static HH500LibSpace.HH500Lib;

namespace tttrmode
{
    class tttrmode
    {
        static ulong oflcorrection = 0;
        static double Resolution = 0; // in ps
        static double Syncperiod = 0; // in s

        const int T3HISTBINS = 32768; // =2^15, dtime in T3 mode has 15 bits
        static uint[,] histogram = new uint[MAXINPCHAN, T3HISTBINS];

        // for brevity of the main demo code further below we encapsulate the programming
        // of the input circuits as a soubroutine here
        public static int SetInputModalities(int devidx, int numchans)
        {
            // Dependent on the hardware model, the HydraHarp 500 may have different input
            // circuits: Edge Trigger (ETR) and/or Constant Fraction Discrimonator (CFD)
            // This can even vary from channel to channel. An input that has both ETR and CFD
            // will be programmable as to which trigger mode to use. Here we define default 
            // settings for both variants and the default trigger mode ETR.
            // The latter can be changed to CFD which of course only works with channels
            // that actually have a CFD. The code further below will therefore query the 
            // input capabilities and where necessary fall back to the mode available.

            int SyncTrgMode = TRGMODE_ETR;       // you can change this to TRGMODE_CFD    
                                                 // in case of SyncTrgMode == TRGMODE_ETR this will apply:
            int SyncTiggerEdge = EDGE_FALLING;   // you can change this to EDGE_RISING
            int SyncTriggerLevel = -50;          // in mV, you can change this
                                                 // in case of SyncTrgMode == TRGMODE_CFD this will apply:
            int SyncCFDZeroCross = -20;          // in mV, you can change this
            int SyncCFDLevel = -50;              // in mV, you can change this

            int InputTrgMode = TRGMODE_ETR;      // you can change this to TRGMODE_CFD
                                                 // in case of InputTrgMode == TRGMODE_ETR this will apply:
            int InputTriggerEdge = EDGE_FALLING; // you can change this to EDGE_RISING
            int InputTriggerLevel = -50;         // in mV, you can change this
                                                 // in case of InputTrgMode == TRGMODE_CFD this will apply:
            int InputCFDZeroCross = -20;         // in mV, you can change this
            int InputCFDLevel = -50;             // in mV, you can change this

            int SyncChannelOffset = -5000;       // you can change this (in ps, like a cable delay)
            int InputChannelOffset = 0;          // you can change this (in ps, like a cable delay)

            uint ChnFeatures = 0;

            if (APICALL(() => HH500_GetSyncFeatures(devidx, ref ChnFeatures)) < 0)
                return -1;

            // check if the sync channel has the right feature for the requested setting
            if ((SyncTrgMode == TRGMODE_ETR) && ((ChnFeatures & HAS_ETR) == 0))
            {
                Console.WriteLine("Warning: Sync channel has no Edge Trigger, switching to CFD");
                SyncTrgMode = TRGMODE_CFD;
            }
            if ((SyncTrgMode == TRGMODE_CFD) && ((ChnFeatures & HAS_CFD) == 0))
            {
                Console.WriteLine("Warning: Sync channel has no CFD, switching to Edge Trigger");
                SyncTrgMode = TRGMODE_ETR;
            }

            if (APICALL(() => HH500_SetSyncTrgMode(devidx, SyncTrgMode)) < 0)
                return -1;

            if (SyncTrgMode == TRGMODE_ETR)
            {
                if (APICALL(() => HH500_SetSyncEdgeTrg(devidx, SyncTriggerLevel, SyncTiggerEdge)) < 0)
                    return -1;
            }

            if (SyncTrgMode == TRGMODE_CFD)
            {
                if (APICALL(() => HH500_SetSyncCFD(devidx, SyncCFDLevel, SyncCFDZeroCross)) < 0)
                    return -1;
            }

            if (APICALL(() => HH500_SetSyncChannelOffset(devidx, SyncChannelOffset)) < 0)
                return -1;

            for (int i = 0; i < numchans; i++) // for simplicity we use the same settings for all channels
            {
                int RealTrgMode = InputTrgMode;

                if (APICALL(() => HH500_GetInputFeatures(devidx, i, ref ChnFeatures)) < 0)
                    return -1;

                // check if the input channel has the right feature for the requested setting
                if ((InputTrgMode == TRGMODE_ETR) && ((ChnFeatures & HAS_ETR) == 0))
                {
                    Console.WriteLine("Warning: Input channel {0} has no Edge Trigger, switching to CFD", i + 1);
                    RealTrgMode = TRGMODE_CFD;
                }
                if ((InputTrgMode == TRGMODE_CFD) && ((ChnFeatures & HAS_CFD) == 0))
                {
                    Console.WriteLine("Warning: Input channel {0} has no CFD, switching to Edge Trigger", i + 1);
                    RealTrgMode = TRGMODE_ETR;
                }

                if (APICALL(() => HH500_SetInputTrgMode(devidx, i, RealTrgMode)) < 0)
                    return -1;

                if (RealTrgMode == TRGMODE_ETR)
                {
                    if (APICALL(() => HH500_SetInputEdgeTrg(devidx, i, InputTriggerLevel, InputTriggerEdge)) < 0)
                        return -1;
                }

                if (RealTrgMode == TRGMODE_CFD)
                {
                    if (APICALL(() => HH500_SetInputCFD(devidx, i, InputCFDLevel, InputCFDZeroCross)) < 0)
                        return -1;
                }

                if (APICALL(() => HH500_SetInputChannelOffset(devidx, i, InputChannelOffset)) < 0)
                    return -1;

                if (APICALL(() => HH500_SetInputChannelEnable(devidx, i, 1)) < 0) // we enable all channels
                    return -1;
            }

            return 0;
        }

        static void Main(string[] args)
        {
            int[] dev = new int[MAXDEVNUM];
            int found = 0;
            int retcode;
            int ctcstatus = 0;

            StringBuilder LibVer = new StringBuilder(8);
            StringBuilder Serial = new StringBuilder(8);
            StringBuilder Errstr = new StringBuilder(40);
            StringBuilder Model = new StringBuilder(32);
            StringBuilder Partno = new StringBuilder(16);
            StringBuilder Version = new StringBuilder(16);
            StringBuilder Wtext = new StringBuilder(16384);

            int NumChannels = 0;

            int Mode = MODE_T3;   // Leave as is, this demo is only for T3 mode!
            int Binning = 4;      // you can change this, meaningful only in T3 mode, observe limits 
            int Offset = 0;       // you can change this, meaningful only in T3 mode, observe limits 
            int Tacq = 1000;      // Measurement time in millisec, you can change this, observe limits 
            int SyncDivider = 1;  // you can change this, usually 1 in T2 mode 

            int Syncrate = 0;
            int Countrate = 0;
            int flags = 0;
            int warnings = 0;
            long Progress = 0;
            int nRecords = 0;
            int stopretry = 0;

            uint[] buffer = new uint[TTREADMAX];

            int i, j;

            FileStream fs = null;
            StreamWriter sw = null;

            Console.WriteLine("HydraHarp 500 - HH500Lib Demo Application              PicoQuant GmbH, 2025");
            Console.WriteLine("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~");

            if (APICALL(() => HH500_GetLibraryVersion(LibVer)) < 0)
                goto ex;

            Console.WriteLine("HH500Lib Version is " + LibVer);

            if (LibVer.ToString() != LIB_VERSION)
            {
                Console.WriteLine("This program requires HH500Lib v." + LIB_VERSION);
                goto ex;
            }

            try
            {
                fs = new FileStream("t3histout.txt", FileMode.Create, FileAccess.Write);
                sw = new StreamWriter(fs);
            }
            catch (Exception)
            {
                Console.WriteLine("Error creating file");
                goto ex;
            }

            Console.WriteLine("Searching for HydraHarp 500 devices...");
            Console.WriteLine("Devidx     Status");

            for (i = 0; i < MAXDEVNUM; i++)
            {
                Serial.Clear();
                // not using APICALL here as we wish to handle errors gracefully
                retcode = HH500_OpenDevice(i, Serial);
                if (retcode == 0) // grab any device we can open
                {
                    Console.WriteLine("  {0}        S/N {1}", i, Serial);
                    dev[found] = i; // keep index to devices we want to use
                    found++;
                }
                else
                {

                    if (retcode == HH500_ERROR_DEVICE_OPEN_FAIL)
                        Console.WriteLine("  {0}        no device", i);
                    else
                    {
                        HH500_GetErrorString(Errstr, retcode);
                        Console.WriteLine("  {0}        S/N {1}", i, Errstr);
                    }
                }
            }

            // In this demo we will use the first device we find, i.e. dev[0].
            // You can also use multiple devices in parallel.
            // You can also check for specific serial numbers, so that you always know 
            // which physical device you are talking to.

            if (found < 1)
            {
                Console.WriteLine("No device available.");
                goto ex;
            }

            Console.WriteLine("Using device {0}", dev[0]);
            Console.WriteLine("Initializing the device...");

            if (APICALL(() => HH500_Initialize(dev[0], Mode, 0)) < 0)
                goto ex;

            if (APICALL(() => HH500_GetHardwareInfo(dev[0], Model, Partno, Version)) < 0)
                goto ex;
            else
                Console.WriteLine("Found Model {0} Part no {1} Version {2}", Model, Partno, Version);

            if (APICALL(() => HH500_GetNumOfInputChannels(dev[0], ref NumChannels)) < 0)
                goto ex;
            else
                Console.WriteLine("Device has {0} input channels.", NumChannels);

            if (APICALL(() => HH500_SetSyncDiv(dev[0], SyncDivider)) < 0)
                goto ex;

            if (SetInputModalities(dev[0], NumChannels) < 0)
                goto ex;

            if (Mode != MODE_T2)
            {
                if (APICALL(() => HH500_SetBinning(dev[0], Binning)) < 0)
                    goto ex;

                if (APICALL(() => HH500_SetOffset(dev[0], Offset)) < 0)
                    goto ex;
            }

            if (APICALL(() => HH500_GetResolution(dev[0], ref Resolution)) < 0)
                goto ex;

            Console.WriteLine("Resolution is {0} ps", Resolution);

            Console.WriteLine("Measuring input rates...\n");

            // after Init allow 150 ms for valid count rate readings
            // subsequently you get new values after every 100ms
            System.Threading.Thread.Sleep(150);

            if (APICALL(() => HH500_GetSyncRate(dev[0], ref Syncrate)) < 0)
                goto ex;
            Console.WriteLine("Syncrate = {0}/s", Syncrate);

            for (i = 0; i < NumChannels; i++)  // for all channels
            {
                if (APICALL(() => HH500_GetCountRate(dev[0], i, ref Countrate)) < 0)
                    goto ex;
                Console.WriteLine("Countrate[{0}] = {1}/s", i, Countrate);
            }

            Console.WriteLine();

            // after getting the count rates you can check for warnings
            if (APICALL(() => HH500_GetWarnings(dev[0], ref warnings)) < 0)
                goto ex;
            if (warnings != 0)
            {
                if (APICALL(() => HH500_GetWarningsText(dev[0], Wtext, warnings)) < 0)
                    goto ex;
                Console.WriteLine("{0}", Wtext);
            }

            if (Mode == MODE_T2)
                sw.Write("This demo is not for use with T2 mode!\n");
            else
            {
                for (j = 0; j < NumChannels; j++)
                    sw.Write("  ch{0,2} ", j+1);
                sw.Write("\n");
            }

            Console.WriteLine("press RETURN to start");
            Console.ReadLine();

            if (APICALL(() => HH500_StartMeas(dev[0], Tacq)) < 0)
                goto ex;

            #region Here this demo differs from 'classic' tttrmode-demo

            if (Mode == MODE_T3)
            {
                //We need the sync period in order to calculate the true times of photon records.
                //This only makes sense in T3 mode and it assumes a stable period like from a laser.
                //Note: Two sync periods must have elapsed after HH500_StartMeas to get proper results.
                //You can also use the inverse of what you read via GetSyncRate but it depends on 
                //the actual sync rate if this is accurate enough.
                //It is OK to use the sync input for a photon detector, e.g. if you want to perform
                //something like an antibunching measurement. In that case the sync rate obviously is
                //not periodic. This means that a) you should set the sync divider to 1 (none) and
                //b) that you cannot meaningfully measure the sync period here, which probaly won't
                //matter as you only care for the time difference (dtime) of the events.
                if (APICALL(() => HH500_GetSyncPeriod(dev[0], ref Syncperiod)) < 0)
                    goto ex;
                Console.WriteLine("Sync period is {0} ns", Syncperiod * 1e9);
            }

            Console.WriteLine("Starting data collection...");

            Progress = 0;
            Console.Write("Progress: {0,12}", Progress);

            oflcorrection = 0;

            for (i = 0; i < T3HISTBINS; i++)
                for (j = 0; j < MAXINPCHAN; j++)
                    histogram[j, i] = 0;

            #endregion

            while (true)
            {
                if (APICALL(() => HH500_GetFlags(dev[0], ref flags)) < 0)
                    goto ex;

                if ((flags & FLAG_FIFOFULL) != 0)
                {
                    Console.WriteLine();
                    Console.WriteLine("FiFo Overrun!");
                    goto stoptttr;
                }

                if (APICALL(() => HH500_ReadFiFo(dev[0], buffer, ref nRecords)) < 0)  // may return less!
                    goto stoptttr;

                if (nRecords > 0)
                {
                    // Here we process the data. Note that the time this consumes prevents us
                    // from getting around the loop quickly for the next Fifo read.
                    // In a serious performance critical scenario you would write the data to
                    // a software queue and do the processing in another thread reading from 
                    // that queue.

                    if (Mode == MODE_T2)
                        for (i = 0; i < nRecords; i++)
                            ProcessT2(sw, buffer[i]);
                    else
                        for (i = 0; i < nRecords; i++)
                            ProcessT3(sw, buffer[i]);

                    Progress += nRecords;
                    Console.Write("\b\b\b\b\b\b\b\b\b\b\b\b{0,12}", Progress);
                }
                else
                {
                    if (APICALL(() => HH500_CTCStatus(dev[0], ref ctcstatus)) < 0)
                        goto ex;
                    if (ctcstatus > 0)
                    {
                        stopretry++; //do a few more rounds as there might be some more data in the FIFO
                        if (stopretry > 15)
                        {
                            Console.WriteLine("\nDone\n");
                            goto stoptttr;
                        }

                    }
                }

                // within this loop you can also read the count rates if needed.
            }

            stoptttr:
            Console.WriteLine();

            if (APICALL(() => HH500_StopMeas(dev[0])) < 0)
                goto ex;

            for (i = 0; i < T3HISTBINS; i++)
            {
                for (j = 0; j < NumChannels; j++)
                    sw.Write("{0,6} ", histogram[j, i]);
                sw.Write("\n");
            }

            ex:

            for (i = 0; i < MAXDEVNUM; i++)  // no harm to close all
            {
                HH500_CloseDevice(i);
            }

            if (sw != null)
            {
                try
                {
                    sw.Flush();
                    sw.Close();
                    sw.Dispose();
                    sw = null;
                }
                catch (Exception e)
                {
                    Console.WriteLine("Error closing the file: " + e);
                }
            }

            Console.WriteLine("press RETURN to exit");
            Console.ReadLine();
        }

        //Got PhotonT2
        //  TimeTag: Overflow-corrected arrival time in units of the device's base resolution 
        //  Channel: Channel the photon arrived (0 = Sync channel, 1..N = regular timing channel)
        static void GotPhotonT2(StreamWriter fpout, ulong TimeTag, int Channel)
        {
        }

        //Got MarkerT2
        //  TimeTag: Overflow-corrected arrival time in units of the device's base resolution 
        //  Markers: Bitfield of arrived markers, different markers can arrive at same time (same record)
        static void GotMarkerT2(StreamWriter fpout, ulong TimeTag, int Markers)
        {
        }

        //Got PhotonT3
        //  TimeTag: Overflow-corrected arrival time in units of the sync period 
        //  DTime: Arrival time of photon after last Sync event in units of the chosen resolution (set by binning)
        //  Channel: 1..N where N is the numer of channels the device has
        static void GotPhotonT3(StreamWriter fpout, ulong TimeTag, int Channel, int DTime)
        {
            histogram[Channel-1, DTime]++; // histogramming
        }

        //Got MarkerT3
        //  TimeTag: Overflow-corrected arrival time in units of the sync period 
        //  Markers: Bitfield of arrived Markers, different markers can arrive at same time (same record)
        static void GotMarkerT3(StreamWriter fpout, ulong TimeTag, int Markers)
        {
            // could switch to a new histogrm here
        }

        // HydraHarpV2 or TimeHarp260 or PicoHarp330 or MultiHarp T2 record data
        static void ProcessT2(StreamWriter fpout, uint TTTRRecord)
        {
            int ch;
            ulong truetime;
            const int T2WRAPAROUND_V2 = 33554432;

            // shift and mask out the elements of TTTRRecord
            uint timetag = (TTTRRecord >> 00) & (0xFFFFFFFF >> (32 - 25)); //the lowest 25 bits
            uint channel = (TTTRRecord >> 25) & (0xFFFFFFFF >> (32 - 06)); //the next    6 bits
            uint special = (TTTRRecord >> 31) & (0xFFFFFFFF >> (32 - 01)); //the next    1 bit

            if (special == 1)
            {
                if (channel == 0x3F) //an overflow record
                {
                    // number of overflows is stored in timetag
                    oflcorrection += (ulong)T2WRAPAROUND_V2 * timetag;
                }
                if ((channel >= 1) && (channel <= 15)) //markers
                {
                    truetime = oflcorrection + timetag;
                    // note that actual marker tagging accuracy is only some ns
                    ch = (int)channel;
                    GotMarkerT2(fpout, truetime, ch);
                }
                if (channel == 0) //sync
                {
                    truetime = oflcorrection + timetag;
                    ch = 0; // we encode the Sync channel as 0
                    GotPhotonT2(fpout, truetime, ch);
                }
            }
            else // regular input channel
            {
                truetime = oflcorrection + timetag;
                ch = (int)(channel + 1); // we encode the regular channels as 1..N
                GotPhotonT2(fpout, truetime, ch);
            }
        }

        // HydraHarpV2 or TimeHarp260 or PicoHarp330 or MultiHarp T3 record data
        static void ProcessT3(StreamWriter fpout, uint TTTRRecord)
        {
            int ch, dt;
            ulong truensync;
            const int T3WRAPAROUND = 1024;

            uint nsync = (TTTRRecord >> 00) & (0xFFFFFFFF >> (32 - 10));   // the lowest 10 bits
            uint dtime = (TTTRRecord >> 10) & (0xFFFFFFFF >> (32 - 15));   // the next   15 bits
            uint channel = (TTTRRecord >> 25) & (0xFFFFFFFF >> (32 - 06)); // the next   6  bits
            uint special = (TTTRRecord >> 31) & (0xFFFFFFFF >> (32 - 01)); // the next   1  bit

            if (special == 1)
            {
                if (channel == 0x3F) // overflow
                {
                    // number of overflows is stored in nsync
                    oflcorrection += (ulong)T3WRAPAROUND * nsync;
                }
                if ((channel >= 1) && (channel <= 15)) //markers
                {
                    truensync = oflcorrection + nsync;
                    // the time unit depends on sync period
                    GotMarkerT3(fpout, truensync, (int)channel);
                }
            }
            else //regular input channel
            {
                truensync = oflcorrection + nsync;
                ch = (int)(channel + 1); // we encode the input channels as 1..N
                dt = (int)dtime;
                // truensync indicates the number of the sync period this event was in
                // the dtime unit depends on the chosen resolution (binning)
                GotPhotonT3(fpout, truensync, ch, dt);
            }
        }
    }
}
