﻿/************************************************************************

  Demo access to HydraHarp 500 hardware via HH500Lib v 1.0
  The program performs a measurement based on hardcoded settings.
  The resulting event data is stored in a binary output file.

  Michael Wahl, PicoQuant GmbH, August 2025

  Note: This is a console application

  Note: At the API level the input channel numbers are indexed 0..N-1
  where N is the number of input channels the device has.

  Note: This demo writes only raw event data to the output file.
  It does not write a file header as regular .ptu files have it.

  
  Tested with the following compilers:

  - MS Visual Studio 2019
  - Mono 6.12.0 (Windows and Linux)

************************************************************************/

using System;  // for Console
using System.Text;  // for StringBuilder 
using System.IO;  // for File
using static HH500LibSpace.HH500Lib;

namespace tttrmode
{
    class tttrmode
    {
        // for brevity of the main demo code further below we encapsulate the programming
        // of the input circuits as a soubroutine here
        public static int SetInputModalities(int devidx, int numchans)
        {
            // Dependent on the hardware model, the HydraHarp 500 may have different input
            // circuits: Edge Trigger (ETR) and/or Constant Fraction Discrimonator (CFD)
            // This can even vary from channel to channel. An input that has both ETR and CFD
            // will be programmable as to which trigger mode to use. Here we define default 
            // settings for both variants and the default trigger mode ETR.
            // The latter can be changed to CFD which of course only works with channels
            // that actually have a CFD. The code further below will therefore query the 
            // input capabilities and where necessary fall back to the mode available.

            int SyncTrgMode = TRGMODE_ETR;       // you can change this to TRGMODE_CFD    
                                                 // in case of SyncTrgMode == TRGMODE_ETR this will apply:
            int SyncTiggerEdge = EDGE_FALLING;   // you can change this to EDGE_RISING
            int SyncTriggerLevel = -50;          // in mV, you can change this
                                                 // in case of SyncTrgMode == TRGMODE_CFD this will apply:
            int SyncCFDZeroCross = -20;          // in mV, you can change this
            int SyncCFDLevel = -50;              // in mV, you can change this

            int InputTrgMode = TRGMODE_ETR;      // you can change this to TRGMODE_CFD
                                                 // in case of InputTrgMode == TRGMODE_ETR this will apply:
            int InputTriggerEdge = EDGE_FALLING; // you can change this to EDGE_RISING
            int InputTriggerLevel = -50;         // in mV, you can change this
                                                 // in case of InputTrgMode == TRGMODE_CFD this will apply:
            int InputCFDZeroCross = -20;         // in mV, you can change this
            int InputCFDLevel = -50;             // in mV, you can change this

            int SyncChannelOffset = -5000;       // you can change this (in ps, like a cable delay)
            int InputChannelOffset = 0;          // you can change this (in ps, like a cable delay)

            uint ChnFeatures = 0;

            if (APICALL(() => HH500_GetSyncFeatures(devidx, ref ChnFeatures)) < 0)
                return -1;

            // check if the sync channel has the right feature for the requested setting
            if ((SyncTrgMode == TRGMODE_ETR) && ((ChnFeatures & HAS_ETR) == 0))
            {
                Console.WriteLine("Warning: Sync channel has no Edge Trigger, switching to CFD");
                SyncTrgMode = TRGMODE_CFD;
            }
            if ((SyncTrgMode == TRGMODE_CFD) && ((ChnFeatures & HAS_CFD) == 0))
            {
                Console.WriteLine("Warning: Sync channel has no CFD, switching to Edge Trigger");
                SyncTrgMode = TRGMODE_ETR;
            }

            if (APICALL(() => HH500_SetSyncTrgMode(devidx, SyncTrgMode)) < 0)
                return -1;

            if (SyncTrgMode == TRGMODE_ETR)
            {
                if (APICALL(() => HH500_SetSyncEdgeTrg(devidx, SyncTriggerLevel, SyncTiggerEdge)) < 0)
                    return -1;
            }

            if (SyncTrgMode == TRGMODE_CFD)
            {
                if (APICALL(() => HH500_SetSyncCFD(devidx, SyncCFDLevel, SyncCFDZeroCross)) < 0)
                    return -1;
            }

            if (APICALL(() => HH500_SetSyncChannelOffset(devidx, SyncChannelOffset)) < 0)
                return -1;

            for (int i = 0; i < numchans; i++) // for simplicity we use the same settings for all channels
            {
                int RealTrgMode = InputTrgMode;

                if (APICALL(() => HH500_GetInputFeatures(devidx, i, ref ChnFeatures)) < 0)
                    return -1;

                // check if the input channel has the right feature for the requested setting
                if ((InputTrgMode == TRGMODE_ETR) && ((ChnFeatures & HAS_ETR) == 0))
                {
                    Console.WriteLine("Warning: Input channel {0} has no Edge Trigger, switching to CFD", i + 1);
                    RealTrgMode = TRGMODE_CFD;
                }
                if ((InputTrgMode == TRGMODE_CFD) && ((ChnFeatures & HAS_CFD) == 0))
                {
                    Console.WriteLine("Warning: Input channel {0} has no CFD, switching to Edge Trigger", i + 1);
                    RealTrgMode = TRGMODE_ETR;
                }

                if (APICALL(() => HH500_SetInputTrgMode(devidx, i, RealTrgMode)) < 0)
                    return -1;

                if (RealTrgMode == TRGMODE_ETR)
                {
                    if (APICALL(() => HH500_SetInputEdgeTrg(devidx, i, InputTriggerLevel, InputTriggerEdge)) < 0)
                        return -1;
                }

                if (RealTrgMode == TRGMODE_CFD)
                {
                    if (APICALL(() => HH500_SetInputCFD(devidx, i, InputCFDLevel, InputCFDZeroCross)) < 0)
                        return -1;
                }

                if (APICALL(() => HH500_SetInputChannelOffset(devidx, i, InputChannelOffset)) < 0)
                    return -1;

                if (APICALL(() => HH500_SetInputChannelEnable(devidx, i, 1)) < 0) // we enable all channels
                    return -1;
            }

            return 0;
        }

        static void Main(string[] args)
        {
            int i, j;
            int retcode;
            int[] dev = new int[MAXDEVNUM];
            int found = 0;
            int NumChannels = 0;

            StringBuilder LibVer = new StringBuilder(8);
            StringBuilder Serial = new StringBuilder(8);
            StringBuilder Errstr = new StringBuilder(40);
            StringBuilder Model = new StringBuilder(32);
            StringBuilder Partno = new StringBuilder(16);
            StringBuilder Version = new StringBuilder(16);
            StringBuilder Wtext = new StringBuilder(16384);

            int Mode = MODE_T2;   // you can change this, adjust other settings accordingly!
            int Binning = 0;      // you can change this, meaningful only in T3 mode, observe limits 
            int Offset = 0;       // you can change this, meaningful only in T3 mode, observe limits 
            int Tacq = 10000;     // Measurement time in millisec, you can change this, observe limits 

            int SyncDivider = 1;  // you can change this, usually 1 in T2 mode 

            double Resolution = 0;
            int Syncrate = 0; 
            int Countrate = 0;
            int ctcstatus = 0;
            int flags = 0;
            long Progress = 0;
            int stopretry = 0;
            int nRecords = 0;
            int warnings = 0;

            uint[] buffer = new uint[TTREADMAX];

            FileStream fs = null;
            BinaryWriter bw = null;


            Console.WriteLine("HydraHarp 500 - HH500Lib Demo Application              PicoQuant GmbH, 2025");
            Console.WriteLine("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~");

            if (APICALL(() => HH500_GetLibraryVersion(LibVer)) < 0)
                goto ex;

            Console.WriteLine("HH500Lib Version is " + LibVer);

            if (LibVer.ToString() != LIB_VERSION)
            {
                Console.WriteLine("This program requires HH500Lib v." + LIB_VERSION);
                goto ex;
            }

            try
            {
                fs = File.Create("tttrmode.out");
                bw = new BinaryWriter(fs);
            }
            catch (Exception)
            {
                Console.WriteLine("Error creating file");
                goto ex;
            }

            Console.WriteLine("Searching for HydraHarp 500 devices...");
            Console.WriteLine("Devidx     Status");


            for (i = 0; i < MAXDEVNUM; i++)
            {
                Serial.Clear();
                // not using APICALL here as we wish to handle errors gracefully
                retcode = HH500_OpenDevice(i, Serial);
                if (retcode == 0) // grab any device we can open
                {
                    Console.WriteLine("  {0}        S/N {1}", i, Serial);
                    dev[found] = i; // keep index to devices we want to use
                    found++;
                }
                else
                {
                    if (retcode == HH500_ERROR_DEVICE_OPEN_FAIL)
                        Console.WriteLine("  {0}        no device", i);
                    else
                    {
                        HH500_GetErrorString(Errstr, retcode);
                        Console.WriteLine("  {0}        S/N {1}", i, Errstr);
                    }
                }
            }

            // In this demo we will use the first device we find, i.e. dev[0].
            // You can also use multiple devices in parallel.
            // You can also check for specific serial numbers, so that you always know 
            // which physical device you are talking to.

            if (found < 1)
            {
                Console.WriteLine("No device available.");
                goto ex;
            }

            Console.WriteLine("Using device {0}", dev[0]);
            Console.WriteLine("Initializing the device...");

            if (APICALL(() => HH500_Initialize(dev[0], Mode, 0)) < 0)
                goto ex;

            if (APICALL(() => HH500_GetHardwareInfo(dev[0], Model, Partno, Version)) < 0)
                goto ex;
            else
                Console.WriteLine("Found Model {0} Part no {1} Version {2}", Model, Partno, Version);

            if (APICALL(() => HH500_GetNumOfInputChannels(dev[0], ref NumChannels)) < 0)
                goto ex;
            else
                Console.WriteLine("Device has {0} input channels.", NumChannels);

            Console.WriteLine("\n\nUsing the following settings:\n");

            Console.WriteLine("Mode              : " + Mode);
            Console.WriteLine("Binning           : " + Binning);
            Console.WriteLine("Offset            : " + Offset);
            Console.WriteLine("AcquisitionTime   : " + Tacq);
            Console.WriteLine("SyncDivider       : " + SyncDivider);

            if (APICALL(() => HH500_SetSyncDiv(dev[0], SyncDivider)) < 0)
                goto ex;

            if (SetInputModalities(dev[0], NumChannels) < 0)
                goto ex;

            if (Mode != MODE_T2)
            {
                if (APICALL(() => HH500_SetBinning(dev[0], Binning)) < 0)
                    goto ex;

                if (APICALL(() => HH500_SetOffset(dev[0], Offset)) < 0)
                    goto ex;
            }

            if (APICALL(() => HH500_GetResolution(dev[0], ref Resolution)) < 0)
                goto ex;

            Console.WriteLine("Resolution is {0} ps", Resolution);
            Console.WriteLine();

            // after Init allow 150 ms for valid  count rate readings
            // subsequently you get new values after every 100ms
            System.Threading.Thread.Sleep(150);

            if (APICALL(() => HH500_GetSyncRate(dev[0], ref Syncrate)) < 0)
                goto ex;

            Console.WriteLine("Syncrate = {0}/s", Syncrate);

            for (i = 0; i < NumChannels; i++)  // for all channels
            {
                if (APICALL(() => HH500_GetCountRate(dev[0], i, ref Countrate)) < 0)
                    goto ex;

                Console.WriteLine("Countrate[{0}] = {1}/s", i, Countrate);
            }

            Console.WriteLine();

            // after getting the count rates you can check for warnings
            if (APICALL(() => HH500_GetWarnings(dev[0], ref warnings)) < 0)
                goto ex;

            if (warnings != 0)
            {
                if (APICALL(() => HH500_GetWarningsText(dev[0], Wtext, warnings)) < 0)
                    goto ex;

                Console.WriteLine("{0}", Wtext);
            }

            Console.WriteLine("press RETURN to start measurement");
            Console.ReadLine();

            Progress = 0;
            Console.Write("Progress: {0,12}", Progress);

            if (APICALL(() => HH500_StartMeas(dev[0], Tacq)) < 0)
                goto ex;

            while (true)
            {
                if (APICALL(() => HH500_GetFlags(dev[0], ref flags)) < 0)
                    goto ex;

                if ((flags & FLAG_FIFOFULL) != 0)
                {
                    Console.WriteLine();
                    Console.WriteLine("FiFo Overrun!");
                    goto stoptttr;
                }

                if (APICALL(() => HH500_ReadFiFo(dev[0], buffer, ref nRecords)) < 0)
                    goto ex;

                if (nRecords > 0)
                {
                    for (j = 0; j < nRecords; j++)
                        bw.Write(buffer[j]);

                    Progress += nRecords;
                    Console.Write("\b\b\b\b\b\b\b\b\b\b\b\b{0,12}", Progress);
                }
                else
                {
                    if (APICALL(() => HH500_CTCStatus(dev[0], ref ctcstatus)) < 0)
                        goto ex;

                    if (ctcstatus > 0)
                    {
                        stopretry++; // do a few more rounds as there might be some more data in the FIFO
                        if (stopretry > 15)
                        {
                            Console.WriteLine("\nDone\n");
                            goto stoptttr;
                        }
                    }
                }

                // within this loop you can also read the count rates if needed.
            }

            stoptttr:
            Console.WriteLine();

            if (APICALL(() => HH500_StopMeas(dev[0])) < 0)
                goto ex;

            ex:

            for (i = 0; i < MAXDEVNUM; i++)  // no harm to close all
            {
                HH500_CloseDevice(i);
            }

            if (bw != null)
            {
                try
                {
                    bw.Flush();
                    bw.Close();
                    bw.Dispose();
                    bw = null;
                }
                catch (Exception e)
                {
                    Console.WriteLine("Error closing the file: " + e);
                }
            }

            Console.WriteLine("press RETURN to exit");
            Console.ReadLine();
        }
    }
}
