% HH500call  Calls a routine from HH500Lib with error handling
%   
%   [retcode, ...] = HH500call(routinename, ...) calls the routine
%   specified by routinename, where ... are variable lists of arguments 
%   that must match the signature of the routine.    
%   HH500Lib must have been loaded before HH500call can be used.
%   HH500call is essentially a wrapper around the native Matlab function 
%   callib. The usage is therefore very similar, only that the library name
%   is omitted. After invoking calllib HH500call will check for errors (ret<0)
%   and if there was an error translate it to an error string.
%   Upon detecting an error HH500call will invoke error() to report the 
%   error code and error string and then terminate the program. This is 
%   useful for lean demo code but may not be ideal for production code.
%   In that case you may want to modify HH500call and, for instance,
%   replace error(..) by warning(..). Note, however, that then you
%   need to check the return code of HH500call and handle possible errors 
%   at the caller's level.
% 
%   See also LOADLIBRARY, CALLLIB.

function varargout = HH500Lib(varargin)

% see https://nl.mathworks.com/help/matlab/matlab_prog/support-variable-number-of-inputs.html
% and https://nl.mathworks.com/help/matlab/matlab_prog/support-variable-number-of-outputs.html

funcname = varargin{1};

nOutputs = nargout;
if(nOutputs==0) %in case we were called without assigning the return code to a variable
    nOutputs = 1;
end
varargout = cell(1,nOutputs);

[varargout{:}] = calllib('HH500Lib', varargin{:});

retcode = varargout{1};  % this is our retcode from the DLL call

if (retcode<0)
    fprintf('\n\n');
    ErrorStr   = blanks(40); % provide enough length!
    ErrorPtr   = libpointer('cstring', ErrorStr);
    [ret, ErrorStr] = calllib('HH500Lib', 'HH500_GetErrorString', ErrorPtr, retcode);
    % if you do not want the program to abort upon every error here 
    % you may want to replace error(..) by warning(..). Note that then you
    % need to check the return code of HH500call and handle possible errors 
    % at the caller's level.
    if(ret==0)
        error('HH500Lib Error %d (%s) calling %s', retcode, ErrorStr, funcname);
    else
        error('HH500Lib Error %d calling %s\n', retcode, funcname);
    end       
end






