% setinputs    Programs the input circuits of a HydraHarp 500
%   
%   retcode = setinputs(devindex, numchannels)
%   programs the input circuits of a HydraHarp 500 where devindex specifies
%   the device to be programmed and numchannels specifies the number of input 
%   channels to be programmed. The sync channel is always programmed.
%   HH500Lib must be leoaded prior to this call and the the device must have
%   been opended and initialized. The call must not be made while a measurement 
%   is runnig.   

function retcode = setinputs(devindex, numchannels)

  % The following are constants from hh500defin.h and can NOT be changed

  % bitmasks for results from HH500_GetSyncFeatures and HH500_GetInputFeatures
  HAS_ETR =            1;   % this channel has an edge trigger
  HAS_CFD =            2;   % this channel has a constant fraction discriminator
  % note: dependent on the hardware model an input can have both

  % codes for HH500_SetSyncTrgMode and HH500_SetInputTrgMode parameter "mode"
  TRGMODE_ETR =        0;   % edge trigger
  TRGMODE_CFD =        1;   % constant fraction discriminator

  % limits for HH500_SetSyncEdgeTrg and HH500_SetInputEdgeTrg
  TRGLVLMIN   =    -1500;   % mV
  TRGLVLMAX   =     1500;   % mV

  % limits for HH500_SetSyncCFD and HH500_SetInputCFD
  CFDLVLMIN   =    -1500;   % mV
  CFDLVLMAX   =        0;   % mV
  CFDZCMIN    =     -100;   % mV
  CFDZCMAX    =        0;   % mV

  % The following are variables that can be changed, observe the limits
  
  SyncTrgMode   = TRGMODE_ETR; % you can change this to TRGMODE_CFD
  InputTrgMode  = TRGMODE_ETR; % you can change this to TRGMODE_CFD

  SyncTrgLevel  =      -50;    % mV, you can change this
  SyncTrgEdge   =        0;    % mV, you can change this
  InputTrgLevel =      -50;    % mV, you can change this
  InputTrgEdge  =        0;    % mV, you can change this

  SyncCFDLevel      =  -50;    % mV, you can change this
  SyncCDFZeroCross  =  -20;    % mV, you can change this
  InputCFDLevel     =  -50;    % mV, you can change this
  InputCFDZeroCross =  -20;    % mV, you can change this
  
  % The following is simple code for demo purposes, change it if needed

  ChnFeatures = uint32(0);
  ChnFeaturesPtr = libpointer('uint32Ptr', ChnFeatures);
  [ret, ChnFeatures] = HH500call('HH500_GetSyncFeatures', devindex, ChnFeaturesPtr);
        
  % check if the sync channel has the right feature for the requested setting
  if((SyncTrgMode == TRGMODE_ETR) & (bitand(ChnFeatures, HAS_ETR)==0))
    fprintf('\nWarning: Sync channel has no Edge Trigger, switching to CFD');
    SyncTrgMode = TRGMODE_CFD;
  end
  if((SyncTrgMode == TRGMODE_CFD) & (bitand(ChnFeatures & HAS_CFD)==0))
    fprintf('\nWarning: Sync channel has no CFD, switching to Edge Trigger');
    SyncTrgMode = TRGMODE_ETR;
  end
    
  [ret] = HH500call('HH500_SetSyncTrgMode', devindex, SyncTrgMode);

  if(SyncTrgMode == TRGMODE_ETR)
      [ret] = HH500call('HH500_SetSyncEdgeTrg', devindex, SyncTrgLevel, SyncTrgEdge);
  end

  if(SyncTrgMode == TRGMODE_CFD)
      [ret] = HH500call('HH500_SetSyncCFD', devindex, SyncCFDLevel, SyncCFDZeroCross);
  end
   
  [ret] = HH500call('HH500_SetSyncChannelOffset', devindex, 0);

  for i=0:numchannels-1 % we use the same input settings for all channels
    
    RealTrgMode = InputTrgMode;

    ChnFeatures = uint32(0);
    ChnFeaturesPtr = libpointer('uint32Ptr', ChnFeatures);
    [ret, ChnFeatures] = HH500call('HH500_GetInputFeatures', devindex, i, ChnFeaturesPtr);
    % check if the input channel has the right feature for the requested setting
    if((InputTrgMode == TRGMODE_ETR) & (bitand(ChnFeatures, HAS_ETR)==0))
      fprintf('\nWarning: Input channel %1d has no Edge Trigger, switching to CFD', i);
      RealTrgMode = TRGMODE_CFD;
    end
    if((InputTrgMode == TRGMODE_CFD) & (bitand(ChnFeatures & HAS_CFD)==0))
      fprintf('\nWarning: Input channel %1d has no CFD, switching to Edge Trigger', i);
      RealTrgMode = TRGMODE_ETR;
    end

    [ret] = HH500call('HH500_SetInputTrgMode', devindex, i, RealTrgMode);
      
    if(RealTrgMode == TRGMODE_ETR)
      [ret] = HH500call('HH500_SetInputEdgeTrg', devindex, i, InputTrgLevel, InputTrgEdge);
    end
    if(RealTrgMode == TRGMODE_CFD)
      [ret] = HH500call('HH500_SetInputCFD', devindex, i, InputCFDLevel, InputCFDZeroCross);
    end 
    
      [ret] = HH500call('HH500_SetInputChannelOffset', devindex, i, 0);
  
  end

  retcode = 0;

end  
  

    


