
% Demo for access to HydraHarp 500 via HH500Lib.dll or libh500lib.so v.1.0
% The demo performs a TTTR measurement based on hardcoded settings.
% The resulting data stream is stored in a binary output file.
%
% Michael Wahl, PicoQuant, August 2025 

% Constants from hh500defin.h in CAPITALS. You cannot change these!

REQLIBVER   =   '1.0';    % the library version this program expects
MAXDEVNUM   =       8;    % max number of HydraHarp 500 devices
MAXINPCHAN  =      16;    % max number of input channels
TTREADMAX   = 1048576;    % 1M event records max from one call of HH500_ReadFiFo
MAXBINSTEPS =      24;    % max binning steps, irrelevant in T2 mode
 
MODE_HIST   =       0;    % histogramming mode
MODE_T2     =       2;    % Time tagging mode T2
MODE_T3     =       3;    % Time tagging mode T3

FLAG_FIFOFULL = hex2dec('0002');

% limits for tacq in HH500_StartMeas
ACQTMIN     =          1;	   % ms
ACQTMAX     =  360000000;	   % ms  (100*60*60*1000ms = 100h)

% Errorcodes from errorcodes.h

HH500_ERROR_DEVICE_OPEN_FAIL		 = -1;

% Settings for the measurement, adjust them to your setup!
 
Mode          = MODE_T2;  %  you can change this

SyncDiv       =       1;  %  you can change this (observe mode!)
Binning       =       0;  %  you can change this, irrelevant in T2 mode
Tacq          =    1000;  %  ms, you can change this       
    

fprintf('\nHydraHarp 500 - HH500Lib Demo Application          PicoQuant 2025\n');

if (~libisloaded('HH500Lib'))    
    %Attention: The header file name given below is case sensitive and must
    %be spelled exactly the same as the actual name on disk except the file 
    %extension. 
    %Wrong case will apparently do the load successfully but you will not
    %be able to access the library!
    %The alias is used to provide a fixed spelling for any further access via
    %calllib() etc, which is also case sensitive.
    
    %To load the right dll for the given OS we use the mexext command.
    %Note that only x86-64 is supported.
    OS = mexext;
    if strcmp('mexw64', OS)
        DLL = 'HH500Lib.dll';                           % Windows 64 bit
    elseif strcmp('mexa64', OS)
        DLL = '/opt/picoquant/hh500lib/libhh500lib.so'; % Linux 64 bit
    else
        fprintf('\nNo supported operating system\n');
        return;
    end
    loadlibrary(DLL, 'hh500lib.h', 'alias', 'HH500Lib');
else
    fprintf('Note: HH500Lib was already loaded\n');
end

if (libisloaded('HH500Lib'))
    fprintf('HH500Lib opened successfully\n');
    %libfunctionsview HH500Lib; % use this to test for proper loading
else
    fprintf('Could not open HH500Lib\n');
    return;
end
   
% Normally one uses Matlab's native calllib() to invoke DLL
% routines. To simplify things we use HH500call.m where possible.
% Type help HH500call to understand this little helper routine.
% Note that it will terminate the program in case of errors.

LibVersion    = blanks(8); % provide enough length!
LibVersionPtr = libpointer('cstring', LibVersion);
[ret, LibVersion] = HH500call('HH500_GetLibraryVersion', LibVersionPtr);	
fprintf('HH500Lib version is %s\n', LibVersion);
if ~strcmp(LibVersion,REQLIBVER)
    fprintf('This program requires HH500Lib version %s\n', REQLIBVER);
    return;
end

fid = fopen('tttrmode.out','wb');
if (fid<0)
    fprintf('Cannot open output file\n');
    return;
end

fprintf('\nSearching for HydraHarp 500 devices...');

dev = [];
found = 0;
Serial     = blanks(8); % provide enough length!
SerialPtr  = libpointer('cstring', Serial);
ErrorStr   = blanks(40); % provide enough length!
ErrorPtr   = libpointer('cstring', ErrorStr);

% in this loop we DO NOT use HH500call to allow more specific error handling 
for i=0:MAXDEVNUM-1
    [ret, Serial] = calllib('HH500Lib', 'HH500_OpenDevice', i, SerialPtr);
    if (ret==0)       % grab any HydraHarp 500 we successfully opened
        fprintf('\n  %1d        S/N %s', i, Serial);
        found = found+1;            
        dev(found)=i; % keep index to devices we may want to use
    else
        if(ret==HH500_ERROR_DEVICE_OPEN_FAIL)
            fprintf('\n  %1d        no device', i);
        else 
            [ret, ErrorStr] = calllib('HH500Lib', 'HH500_GetErrorString', ErrorPtr, ret);
            fprintf('\n  %1d        %s', i,ErrorStr);
        end
    end
end
    
% in this demo we will use the first HydraHarp 500 device we found, i.e. dev(1)
% you could also check for a specific serial number, so that you always know 
% which physical device you are talking to.

if (found<1)
	fprintf('\nNo device available. Aborted.\n');
	return; 
end

fprintf('\nUsing device #%1d',dev(1));
fprintf('\nInitializing the device...');

[ret] = HH500call('HH500_Initialize', dev(1), Mode, 0);
% Note that here we assign but then actually ignore the value ret.
% Also note the brackets. We do this for consistency with all other
% calls to HH500call where there can be more return values in the bracket. 
% HH500call always requires a correct assignment of return values
% according to the signature of the called function. This is important 
% for the variable aruments lists passed in and out from HH500call 
% (as well as for callib) to work properly. 

% the following library call is only for information
Model      = blanks(24); % provide enough length!
Partno     = blanks(8);  % provide enough length!
Version    = blanks(8);  % provide enough length!
ModelPtr   = libpointer('cstring', Model);
PartnoPtr  = libpointer('cstring', Partno);
VersionPtr = libpointer('cstring', Version);

[ret, Model, Partno, Version] = HH500call('HH500_GetHardwareInfo', dev(1), ModelPtr, PartnoPtr, VersionPtr);
fprintf('\nFound model %s part number %s version %s', Model, Partno, Version);             

NumInpChannels = int32(0);
NumChPtr = libpointer('int32Ptr', NumInpChannels);
[ret,NumInpChannels] = HH500call('HH500_GetNumOfInputChannels', dev(1), NumChPtr); 
fprintf('\nDevice has %i input channels.', NumInpChannels);             

% The following call to setinputs.m sets the modalities for the sync and input 
% channels. It is outsourced here to keep the main demo code brief and readable.
% The values for these settings are set inside setinputs.m, edit there as needed.
setinputs(dev(1), NumInpChannels);
   
[ret] = HH500call('HH500_SetSyncDiv', dev(1), SyncDiv);

if (Mode ~= MODE_T2)    % The next 2 calls make no sense in T2 mode
    [ret] = HH500call('HH500_SetBinning', dev(1), Binning);
    [ret] = HH500call('HH500_SetOffset', dev(1), 0);
end

% HH500_GetResolution is meaningless in T2 mode but it does no harm
Resolution = 0;
ResolutionPtr = libpointer('doublePtr', Resolution);
[ret, Resolution] = HH500call('HH500_GetResolution', dev(1), ResolutionPtr);
fprintf('\nResolution=%1dps', Resolution);

pause(0.15); % after Init or SetSyncDiv you must allow 150 ms for valid new 
             % count rates, otherwise you get new values every 100 ms

% from here you could repeat the measurement (with the same settings)

Syncrate = 0;
SyncratePtr = libpointer('int32Ptr', Syncrate);
[ret, Syncrate] = HH500call('HH500_GetSyncRate', dev(1), SyncratePtr);
fprintf('\nSyncrate=%1d/s', Syncrate);
 
for i=0:NumInpChannels-1
	Countrate = 0;
	CountratePtr = libpointer('int32Ptr', Countrate);
	[ret, Countrate] = HH500call('HH500_GetCountRate', dev(1), i, CountratePtr);
	fprintf('\nCountrate%1d=%1d/s', i, Countrate);
end

Warnings = 0;
WarningsPtr = libpointer('int32Ptr', Warnings);
[ret, Warnings] = HH500call('HH500_GetWarnings', dev(1), WarningsPtr);
if (Warnings~=0)
    Warningstext = blanks(16384); % provide enough length!
    WtextPtr     = libpointer('cstring', Warningstext);
    [ret, Warningstext] = HH500call('HH500_GetWarningsText', dev(1), WtextPtr, Warnings);
    fprintf(2, '\n\n%s',Warningstext);
end

Progress = 0;
fprintf('\nProgress:%11d',Progress);

buffer  = uint32(zeros(1,TTREADMAX));
bufferptr = libpointer('uint32Ptr', buffer);
nactual = int32(0);
nactualptr = libpointer('int32Ptr', nactual);
ctcdone = int32(0);
ctcdonePtr = libpointer('int32Ptr', ctcdone);       

[ret] = HH500call('HH500_StartMeas', dev(1), Tacq); 
       
while(1)  % the data retrieval loop, keep this slim
    
    flags = int32(0);
    flagsPtr = libpointer('int32Ptr', flags);
    [ret,flags] = HH500call('HH500_GetFlags', dev(1), flagsPtr);   
    if (bitand(uint32(flags),FLAG_FIFOFULL)) 
        fprintf('\nFiFo Overrun!\n'); 
        break;
    end

    [ret, buffer, nactual] = HH500call('HH500_ReadFiFo', dev(1), bufferptr, nactualptr);
    % consider that HH500_ReadFiFo may return zero events
    if(nactual) 
        cnt = fwrite(fid, buffer(1:nactual),'uint32');
        if(cnt ~= nactual)
            fprintf('\nfile write error\n');
            break;
        end          
		Progress = Progress + nactual;
		fprintf('\b\b\b\b\b\b\b\b\b\b\b%11d',Progress);
    else
        [ret,ctcdone] = HH500call('HH500_CTCStatus', dev(1), ctcdonePtr);     
        if (ctcdone) 
            fprintf('\nDone\n'); 
            break;
        end
    end
     
    % you can read the count rates here if needed, but do it sparingly

end % while

[ret] = HH500call('HH500_StopMeas', dev(1)); 
            
fprintf('\nBinary output data is in tttrmode.out\n');

for i=0:MAXDEVNUM-1 % no harm to close all
    [ret] = HH500call('HH500_CloseDevice', i);
end

if(fid>0) 
    fclose(fid);
end

