﻿{
  Demo access to HydraHarp 500 via HH500Lib.dll or libhh500.so v.1.0
  The program performs a measurement based on hardcoded settings.
  The resulting photon event data is instantly histogrammed.
  Works in T3 mode only!

  Dr. Marcus Sackrow, PicoQuant, May 2025

  Note: This is a console application

  Note: At the API level the input channel numbers are indexed 0..N-1
  where N is the number of input channels the device has.
  Upon processing we map this back to 1..N corresponding to the front
  panel labelling.

  Tested with the following compilers:

  - Delphi 11 (Windows)
  - Lazarus 3.0 RC1 (Windows)
  - Lazarus 2.2.0 / FPC 3.2.2 (Linux)
}

program tttrmode;

{$IF defined(MSWINDOWS)}
  {$APPTYPE CONSOLE}  // Windows needs this, Linux does not want it
{$ENDIF}

uses
  {$ifdef fpc}
  SysUtils,
  {$else}
  System.SysUtils,
  System.Ansistrings,
  {$endif}
  HH500Lib in 'HH500Lib.pas';

const
  T3HISTBINS = 32768; //=2^15, dtime in T3 mode has 15 bits

var
  Dev             : array [0..MAXDEVNUM - 1] of LongInt;
  Device          : LongInt;
  Found           : Integer = 0;
  RetCode         : LongInt;
  CTCStatus       : LongInt;
  strLibVers      : array[0..7]  of AnsiChar;
  HW_Model        : array[0..31] of AnsiChar;
  HW_Partno       : array[0..8]  of AnsiChar;
  HW_Serial       : array[0..9]  of AnsiChar;
  HW_Version      : array[0..15] of AnsiChar;
  Errorstring     : array[0..39] of AnsiChar;
  DebugInfoBuffer : array[0..16383] of AnsiChar;
  NumChannels     : LongInt;
  Mode            : LongInt = MODE_T3;
  //############### User Settings #########################
  Binning         : LongInt = 0; // you can change this (meaningless in T2 mode)
  Offset          : LongInt = 0; // normally no need to change this
  Tacq            : LongInt = 1000; //  measurement time in millisec, you can change this, unit is millisec
  SyncDivider     : LongInt = 1; // you can change this

  // Dependent on the hardware model, the HydraHarp 500 may have different input
  // circuits: Edge Trigger (ETR) and/or Constant Fraction Discrimonator (CFD)
  // This can even vary from channel to channel. An input that has both ETR and CFD
  // will be programmable as to which trigger mode to use. Here we define default
  // settings for both variants and the default trigger mode ETR.
  // The latter can be changed to CFD which of course only works with channels
  // that actually have a CFD. The code further below will therfore query the
  // input capabilities and where necessary fall back to the mode available.

  // Sync Settings
  SyncChannelOffset : LongInt =       -5000; // in ps, you can change this (like a cable delay)
  SyncTrgMode       : LongInt = TRGMODE_ETR; // you can change this to TRGMODE_CFD

  // in case of SyncTrgMode == TRGMODE_ETR this will apply:
  SyncTrgEdge  : LongInt = EDGE_FALLING; // you can change this to EDGE_RISING
  SyncTrgLevel : LongInt = -50; // in mV, you can change this

  // in case of SyncTrgMode == TRGMODE_CFD this will apply:
  SyncCFDZeroCross : LongInt = -50; // in mV, you can change this
  SyncCFDLevel     : LongInt = -50; // in mV, you can change this

  // Input Settings
  InputChannelOffset : LongInt =           0; // in ps, you can change this (like a cable delay)
  InputTrgMode       : LongInt = TRGMODE_ETR; // you can change this to TRGMODE_CFD

  // in case of InputTrgMode == TRGMODE_ETR this will apply:
  InputTrgEdge  : LongInt = EDGE_FALLING; // you can change this to EDGE_RISING
  InputTrgLevel : LongInt = -50;          // in mV, you can change this

  // in case of InputTrgMode == TRGMODE_CFD this will apply:
  InputCFDZeroCross : LongInt = -20;      // in mV, you can change this
  InputCFDLevel     : LongInt = -50;      // in mV, you can change this

// other variables
  Resolution    : Double;
  SyncRate      : LongInt;
  CountRate     : LongInt;
  Flags         : LongInt;
  Warnings      : LongInt;
  WarningsText  : array[0..16343] of AnsiChar; // must have 16384 bytes of text buffer
  Records       : LongInt;
  Progress      : LongInt = 0;
  StopRetry     : LongInt = 0;
  FiFoFull      : Boolean = False;
  FileError     : Boolean = False;
  ChanIdx       : LongInt;
  Buffer        : array[0..TTREADMAX - 1] of LongWord;
  OutTextFile   : TextFile;
  i,j           : Integer;
  TimeOut       : Boolean = False;
  OflCorrection : Int64 = 0;
  Histogram     : array [0..MAXINPCHAN - 1, 0..T3HISTBINS - 1] of LongWord;

// GotPhotonT3 procedure
// NSync: Overflow-corrected arrival time in units of the sync period
// DTime: Arrival time of photon after last Sync event in units of the chosen resolution (set by binning)
// Channel: 1..N where N is the numer of channels the device has
procedure GotPhotonT3(NSync: Int64; Channel: Integer; DTime: Integer);
begin
  Inc(Histogram[Channel - 1, DTime]); // Histogramming
end;

// GotMarkerT3 rocedure
//  TimeTag: Raw TimeTag from Record * Global resolution = Real Time arrival of Marker
//  Markers: Bitfield of arrived Markers, different markers can arrive at same time (same record)
procedure GotMarkerT3(TimeTag: Int64; Markers: Integer);
begin
  // Could switch to a new Histogram here, e.g. for a FLIM application
end;

// ProcessT3 procedure
procedure ProcessT3(TTTR_RawData: Cardinal);
const
  T3WRAPAROUND = 1024;
type
  TT3DataRecords = record
    Special: Boolean;
    Channel: Byte;
    DTime: Word;
    NSync: Word;
  end;
var
  TTTR_Data: TT3DataRecords;
  TrueNSync: Integer;
begin
  TTTR_Data.NSync := Word(TTTR_RawData and $000003FF); // 10 bit of 32 bit for NSync
  TTTR_Data.DTime := Word((TTTR_RawData shr 10) and $00007FFF); // 15 bit of 32 bit for DTime
  TTTR_Data.Channel := Byte((TTTR_RawData shr 25) and $0000003F); // 6 bit of 32 bit for Channel
  TTTR_Data.Special := Boolean((TTTR_RawData shr 31) and $00000001); // 1 bit of 32 bit for Special
  if TTTR_Data.Special then // This means we have a Special record
    case TTTR_Data.Channel of
      $3F: // Overflow
        begin
          // Number of overflows is stored in timetag
          // If it is zero, it is an old style single overflow {should never happen with new Firmware}
          if TTTR_Data.NSync = 0 then
            OflCorrection := OflCorrection + T3WRAPAROUND
          else
            OflCorrection := OflCorrection + T3WRAPAROUND * TTTR_Data.NSync;
        end;
      1..15: // Markers
        begin
          TrueNSync := OflCorrection + TTTR_Data.NSync; //the time unit depends on sync period
          // Note that actual marker tagging accuracy is only some ns.
          GotMarkerT3(TrueNSync, TTTR_Data.Channel);
        end;
    end
  else // It is a regular photon record
  begin
    TrueNSync := OflCorrection + TTTR_Data.NSync;
    GotPhotonT3(TrueNSync,
    TTTR_Data.Channel + 1, // We encode the input channels as 1..N
    TTTR_Data.DTime // The dtime unit depends on the chosen resolution (binning)
    ); // TruenSync indicates the number of the sync period this event was in
  end;
end;

procedure Ex(RetCode: Integer);
begin
  if RetCode <> HH500_ERROR_NONE then
  begin
    HH500_GetErrorString(Errorstring, RetCode);
    Writeln('Error ', RetCode:3, ' = "', Trim(string(Errorstring)), '"');
  end;
  Writeln;
  {$I-}
    CloseFile(OutTextFile);
    IOResult();
  {$I+}
  Writeln('Press RETURN to exit');
  Readln;
  Halt(RetCode);
end;

procedure StopTTTR;
begin
  RetCode := HH500_StopMeas(Dev[0]);
  if RetCode <> HH500_ERROR_NONE then
  begin
    Writeln('HH500_StopMeas error ', RetCode:3, '. Aborted.');
    Ex(RetCode);
  end;
end;


function SetInputModalities(DevIdx: LongInt; NumChannels: LongInt): LongInt;
var
  ChanIdx: LongInt;
  ChannelFeatures: LongWord;
  RealTrgMode: LongInt;
begin
  Result := HH500_GetSyncFeatures(DevIdx, ChannelFeatures);
  if Result <> HH500_ERROR_NONE then
  begin
    Writeln('HH500_GetSyncFeatures error ', Result:3, '. Aborted.');
    Exit;
  end;
  // check if the sync channel has the right feature for the requested setting
  if (SyncTrgMode = TRGMODE_ETR) and ((ChannelFeatures and HAS_ETR) = 0) then
  begin
    Writeln('Warning: Sync channel as no Edge Trigger, switching to CFD');
    SyncTrgMode := TRGMODE_CFD;
  end;
  if (SyncTrgMode = TRGMODE_CFD) and ((ChannelFeatures and HAS_CFD) = 0) then
  begin
    Writeln('Warning: Sync channel as no CFD, switching to Edge Trigger');
    SyncTrgMode := TRGMODE_ETR;
  end;

  Result := HH500_SetSyncTrgMode(DevIdx, SyncTrgMode);
  if Result <> HH500_ERROR_NONE then
  begin
    Writeln('HH500_SetSyncTrgMode error ', Result:3, '. Aborted.');
    Exit;
  end;

  if SyncTrgMode = TRGMODE_ETR then
  begin
    Result := HH500_SetSyncEdgeTrg(DevIdx, SyncTrgLevel, SyncTrgEdge);
    if Result <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_SetSyncEdgeTrg error ', Result:3, '. Aborted.');
      Exit;
    end;
  end;
  if SyncTrgMode = TRGMODE_CFD then
  begin
    Result := HH500_SetSyncCFD(DevIdx, SyncCFDLevel, SyncCFDZeroCross);
    if Result <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_SetSyncCFD error ', Result:3, '. Aborted.');
      Exit;
    end;
  end;

  Result := HH500_SetSyncChannelOffset(DevIdx, SyncChannelOffset);
  if Result <> HH500_ERROR_NONE then
  begin
    Writeln('HH500_SetSyncChannelOffset error ', Result:3, '. Aborted.');
    Exit;
  end;

  for ChanIdx := 0 to NumChannels - 1 do // we use the same input settings for all channels
  begin
    RealTrgMode := InputTrgMode;
    Result := HH500_GetInputFeatures(DevIdx, ChanIdx, ChannelFeatures);
    if Result <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_GetInputFeatures error ', Result:3, '. Aborted.');
      Exit;
    end;
    // check if the sync channel has the right feature for the requested setting
    if (RealTrgMode = TRGMODE_ETR) and ((ChannelFeatures and HAS_ETR) = 0) then
    begin
      Writeln('Warning: Input channel ', ChanIdx, ' has no Edge Trigger, switching to CFD');
      RealTrgMode := TRGMODE_CFD;
    end;
    if (RealTrgMode = TRGMODE_CFD) and ((ChannelFeatures and HAS_CFD) = 0) then
    begin
      Writeln('Warning: Input channel ', ChanIdx, ' has no CFD, switching to Edge Trigger');
      RealTrgMode := TRGMODE_ETR;
    end;

    Result := HH500_SetInputTrgMode(DevIdx, ChanIdx, RealTrgMode);
    if Result <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_SetInputTrgMode ', ChanIdx:2, ' error ', Result:3, '. Aborted.');
      Exit;
    end;
    if RealTrgMode = TRGMODE_ETR then
    begin
      Result := HH500_SetInputEdgeTrg(DevIdx, ChanIdx, InputTrgLevel, InputTrgEdge);
      if Result <> HH500_ERROR_NONE then
      begin
        Writeln('HH500_SetInputEdgeTrg ', ChanIdx:2, ' error ', Result:3, '. Aborted.');
        Exit;
      end;
    end;
    if RealTrgMode = TRGMODE_CFD then
    begin
      Result := HH500_SetInputCFD(DevIdx, ChanIdx, InputCFDLevel, InputCFDZeroCross);
      if Result <> HH500_ERROR_NONE then
      begin
        Writeln('HH500_SetInputCFD ', ChanIdx:2, ' error ', Result:3, '. Aborted.');
        Exit;
      end;
    end;

    Result := HH500_SetInputChannelOffset(DevIdx, ChanIdx, InputChannelOffset);
    if Result <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_SetInputChannelOffset channel ', ChanIdx:2, ' error ', Result:3, '. Aborted.');
      Exit;
    end;

    Result := HH500_SetInputChannelEnable(DevIdx, ChanIdx, 1);
    if Result <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_SetInputChannelEnable channel ', ChanIdx:2, ' error ', Result:3, '. Aborted.');
      Exit;
    end;
  end;
end;

// Main procedure
begin
  try
    Writeln;
    Writeln('HydraHarp 500 HH500Lib Demo Application             PicoQuant GmbH, 2025');
    Writeln('~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~');

    if Mode = MODE_T2 then
    begin
      Writeln('This demo is not for use with T2 mode!');
      Ex(HH500_ERROR_NONE);
    end;

    RetCode := HH500_GetLibraryVersion(strLibVers);
    if RetCode <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_GetLibraryVersion error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('HH500Lib version is ' + strLibVers);
    if Trim(AnsiString(strLibVers)) <> Trim(AnsiString (LIB_VERSION)) then
      Writeln('Warning: The application was built for version ' + LIB_VERSION);

    AssignFile(OutTextFile, 't3histout.txt');
    {$I-}
      Rewrite(OutTextFile);
    {$I+}
    if IOResult <> 0 then
    begin
      Writeln('Cannot open output file');
      Ex(HH500_ERROR_NONE);
    end;

    Writeln;
    Writeln('Searching for HydraHarp 500 devices...');
    Writeln('DevIdx       Serial       Status');

    for Device := 0 to MAXDEVNUM - 1 do
    begin
      RetCode := HH500_OpenDevice(Device, HW_Serial);
      //
      case RetCode of
        HH500_ERROR_NONE:
          begin
            // Grab any device we can open
            Dev[Found] := Device; // Keep index to devices we want to use
            Inc(Found);
            Writeln(Device, '            ', HW_Serial,'      open ok');
          end;
        HH500_ERROR_DEVICE_OPEN_FAIL:
          Writeln(Device, '                         no device')
        else
          begin
            HH500_GetErrorString(Errorstring, RetCode);
            Writeln(Device, '       ', HW_Serial, '    ', Trim(string(ErrorString)));
          end;
      end;
    end;

    // In this demo we will use the first device we find, i.e. dev[0].
    // You can also use multiple devices in parallel.
    // You can also check for specific serial numbers, so that you always know
    // which physical device you are talking to.

    if Found < 1 then
    begin
      Writeln('No device available.');
      Ex(HH500_ERROR_NONE);
    end;

    Writeln('Using device ', Dev[0]);
    Writeln('Initializing the device...');

    RetCode := HH500_Initialize (Dev[0], Mode, 0); // with internal clock
    if RetCode <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_Initialize error ', RetCode:3, '. Aborted.');
      HH500_GetDebugInfo(Dev[0], DebugInfoBuffer);
      Writeln('Debug info: ', DebugInfoBuffer);
      Ex(RetCode);
    end;

    RetCode := HH500_GetHardwareInfo (Dev[0], HW_Model, HW_PartNo, HW_Version);
    if RetCode <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_GetHardwareInfo error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end
    else
      Writeln('Found Model ', HW_Model,'  Part no ', HW_PartNo,'  Version ', HW_Version);

    RetCode := HH500_GetNumOfInputChannels (Dev[0], NumChannels);
    if RetCode <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_GetNumOfInputChannels error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end
    else
      Writeln('Device has ', NumChannels, ' input channels.');

    Writeln;

    RetCode := HH500_SetSyncDiv (Dev[0], SyncDivider);
    if RetCode <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_SetSyncDiv error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    if SetInputModalities(Dev[0], NumChannels) < 0 then
      Ex(RetCode);

    if (Mode = MODE_T3) then
    begin
      RetCode := HH500_SetBinning (Dev[0], Binning);
      if RetCode <> HH500_ERROR_NONE then
      begin
        Writeln('HH500_SetBinning error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;

      RetCode := HH500_SetOffset(Dev[0], Offset);
      if RetCode <> HH500_ERROR_NONE then
      begin
        Writeln('HH500_SetOffset error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
    end;

    RetCode := HH500_GetResolution (Dev[0], Resolution);
    if RetCode <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_GetResolution error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('Resolution is ', Resolution:7:3, 'ps');
    Writeln('Measuring input rates...');

    // After Init allow 150 ms for valid  count rate readings
    // Subsequently you get new values after every 100ms
    Sleep (150);

    Writeln;

    RetCode := HH500_GetSyncRate(Dev[0], SyncRate);
    if RetCode <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_GetSyncRate error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('SyncRate = ', SyncRate, '/s');

    Writeln;

    for ChanIdx := 0 to NumChannels - 1 do // For all channels
    begin
      RetCode := HH500_GetCountRate(Dev[0], ChanIdx, CountRate);
      if RetCode <> HH500_ERROR_NONE then
      begin
        Writeln('HH500_GetCountRate error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      Writeln('Countrate [', ChanIdx:2, '] = ', CountRate:8, '/s');
    end;

    Writeln;

    RetCode := HH500_GetWarnings(Dev[0], Warnings);   // After getting the count rates you can check for warnings
    if RetCode <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_GetWarnings error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    if Warnings <> 0 then
    begin
      HH500_GetWarningsText(Dev[0], WarningsText, Warnings);
      Writeln(WarningsText);
    end;

    if Mode = Mode_T2 then
    begin
      Writeln('This demo is only for T3 mode!');
      Ex(HH500_ERROR_NONE);
    end
    else
    begin
      for ChanIdx := 0 to NumChannels - 1 do
        Write(OutTextFile, '   ch', (ChanIdx + 1):2, ' ');
    end;
    Write(OutTextFile, chr(10));

    Writeln('Press RETURN to start measurement');
    Readln;

    RetCode := HH500_StartMeas(Dev[0], Tacq);
    if RetCode <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_StartMeas error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('Starting data collection...');

    Progress := 0;
    OflCorrection := 0;

    for i := 0 to T3HISTBINS - 1 do
        for j := 0 to MAXINPCHAN - 1 do
            Histogram[j, i] := 0;

    repeat
      RetCode := HH500_GetFlags(Dev[0], Flags);
      if RetCode <> HH500_ERROR_NONE then
      begin
        Writeln('HH500_GetFlags error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      FiFoFull := (Flags and FLAG_FIFOFULL) > 0;

      if FiFoFull then
        Writeln('  FiFo Overrun!')
      else
      begin
        RetCode := HH500_ReadFiFo(Dev[0], Buffer[0], Records); // may return less!
        if RetCode <> HH500_ERROR_NONE then
        begin
          Writeln('HH500_TTReadData error ', RetCode:3, '. Aborted.');
          Ex(RetCode);
        end;

        // Here we process the data. Note that the time this consumes prevents us
        // from getting around the loop quickly for the next Fifo read.
        // In a serious performance critical scenario you would write the data to
        // a software queue and do the processing in another thread reading from
        // that queue.
        if Records > 0 then
        begin
          for i := 0 to Records - 1 do
              ProcessT3(Buffer[i]);
          Progress := Progress + Records;
          Write(#8#8#8#8#8#8#8#8#8, Progress:9);
        end
        else // Do the following only when there was no data
        begin
          RetCode := HH500_CTCStatus(Dev[0], CTCStatus);
          if RetCode <> HH500_ERROR_NONE then
          begin
            Writeln;
            Writeln('HH500_CTCStatus error ', RetCode:3, '. Aborted.');
            Ex(RetCode);
          end;
          TimeOut := (CTCStatus <> 0);
          if TimeOut then
          begin
            Inc(StopRetry); // Do a few more rounds as there might be some more in the FiFo
            if StopRetry > 5 then
            begin
              StopTTTR;
              Writeln;
              Writeln('Done');
            end;
          end;
        end;
      end;
      // Within this loop you can also read the count rates if needed.
      // Do it sparingly and use HH500_GetAllCountRates for speed.
    until FiFoFull or TimeOut or FileError;

    Writeln;

    RetCode := HH500_StopMeas(Dev[0]);
    if RetCode <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_StopMeas error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    for i := 0 to T3HISTBINS - 1 do
    begin
      for ChanIdx := 0 to NumChannels - 1 do
        Write(OutTextFile, Histogram[ChanIdx, i]:7, ' ');
      Write(OutTextFile, chr(10));
    end;

    Writeln;

    HH500_CloseAllDevices;
    Ex(HH500_ERROR_NONE);

  except
  on E: Exception do
    Writeln(E.ClassName, ': ', E.Message);
  end;
end.

