unit HH500Lib;
{                                                               }
{    HH500Lib programming library for HydraHarp 500             }
{    Dr. Marcus Sackrow, PicoQuant, May 2025                    }
{                                                               }

interface

const
  LIB_VERSION    =      '1.0';
{$UNDEF PLATFORM_OK}
{$IFDEF WIN64}
  LIB_NAME       =      'HH500Lib.dll'; // Windows
  {$DEFINE PLATFORM_OK}
{$ENDIF}
{$IFDEF LINUX}
  LIB_NAME       =      'libhh500lib.so';  // Linux
  {$DEFINE PLATFORM_OK}
{$ENDIF}
{$IFNDEF PLATFORM_OK}
  {$FATAL OS platform not supported}
{$ENDIF}

  MAXDEVNUM   = 8;   // max num of USB devices
  MAXINPCHAN  = 16;  // max num of physical input channels
  MAXBINSTEPS = 24;  // max number of binning steps,
                     // get actual number via HH500_GetBaseResolution()

  MAXHISTLEN  = 131072;   // max number of histogram bins
  DFLTHISTLEN = 65536;    // default number of histogram bins

  MAXHISTLEN_CONT =	65536;      // max number of histo bins in continuous histogramming mode
  CONTMODEBLKHDRSIZE = 64;      // continuous histogramming mode block header size in bytes
  MAXCONTMODEBUFLEN = 262272;   // max bytes of buffer needed for HH500_GetContModeBlock

  TTREADMAX   = 1048576; // max number of event records that can be read by HH500_ReadFiFo
                         // buffer must provide space for this number of dwords

  // symbolic constants for HH500_Initialize parameter "refsource"
  REFSRC_INTERNAL          = 0; // use internal clock
  REFSRC_EXTERNAL_10MHZ    = 1; // use 10MHz external clock
  REFSRC_WR_MASTER_GENERIC = 2; // White Rabbit master with generic partner
  REFSRC_WR_SLAVE_GENERIC  = 3; // White Rabbit slave with generic partner
  REFSRC_WR_GRANDM_GENERIC = 4; // White Rabbit grand master with generic partner
  REFSRC_EXTN_GPS_PPS      = 5; // use 10 MHz + PPS from GPS
  REFSRC_EXTN_GPS_PPS_UART = 6; // use 10 MHz + PPS + time via UART from GPS
  REFSRC_WR_MASTER_MHARP   = 7; // White Rabbit master with MultiHarp or HydraHarp 500 as partner
  REFSRC_WR_SLAVE_MHARP    = 8; // White Rabbit slave with MultiHarp or HydraHarp 500 as partner
  REFSRC_WR_GRANDM_MHARP   = 9; // White Rabbit grand master with MultiHarp or HydraHarp 500 as partner

  // symbolic constants for HH500_Initialize parameter "mode"
  MODE_HIST = 0;
  MODE_T2   = 2;
  MODE_T3   = 3;
  MODE_CONT = 8;  //continuous histogramming mode

  // symbolic constants for HH500_SetMeasControl parameter "control"
  MEASCTRL_SINGLESHOT_CTC     = 0; //default
  MEASCTRL_C1_GATED           = 1;
  MEASCTRL_C1_START_CTC_STOP  = 2;
  MEASCTRL_C1_START_C2_STOP   = 3;
  MEASCTRL_WR_M2S             = 4;
  MEASCTRL_WR_S2M             = 5;
  MEASCTRL_SW_START_SW_STOP   = 6;
  // continuous hitogramming mode only:
  MEASCTRL_CONT_C1_GATED          = 7;
  MEASCTRL_CONT_C1_START_CTC_STOP = 8;
  MEASCTRL_CONT_CTC_RESTART       = 9;

  // symb. const. for HH500_SetMeasControl, HH500_SetMarkerEdges,
  // HH500_SetSycEdgeTrg and HH500_SetInputEdgeTrg parameter "..edge"
  EDGE_RISING  = 1;
  EDGE_FALLING = 0;

  // bitmasks for results from HH500_GetSyncFeatures and HH500_GetInputFeatures
  HAS_ETR = 1; // this input has an edge trigger
  HAS_CFD = 2; // this input has a CFD
  // note: dependent on the hardware model an input can have both

  // bitmasks for results from HH500_GetFeatures
  FEATURE_DLL       = $0001; // DLL License available
  FEATURE_TTTR      = $0002; // TTTR mode available
  FEATURE_MARKERS   = $0004; // Markers available
  FEATURE_LOWRES    = $0008; // Long range mode available
  FEATURE_TRIGOUT   = $0010; // Trigger output available
  FEATURE_PROG_TD   = $0020; // Programmable deadtime available
  FEATURE_EXT_FPGA  = $0040; // Interface for External FPGA available
  FEATURE_PROG_HYST = $0080; // Programmable input hysteresis available
  FEATURE_EVNT_FILT = $0100; // Coincidence filtering available
  FEATURE_INPT_MODE = $0200; // Programmable input mode (CFD vs. edge trigger)

  // bitmasks for results from HH500_GetFlags
  FLAG_OVERFLOW     = $0001; // histo mode only
  FLAG_FIFOFULL     = $0002; // TTTR mode only
  FLAG_SYNC_LOST    = $0004;
  FLAG_REF_LOST     = $0008;
  FLAG_SYSERROR     = $0010; // hardware error, must contact support
  FLAG_ACTIVE       = $0020; // measurement is running
  FLAG_CNTS_DROPPED = $0040; // events dropped

  // limits for HH500_SetHistoLen
  //note: length codes 0 and 1 will not work with MH_GetHistogram
  //if you need these short lengths then use MH_GetAllHistograms
  MINLENCODE = 0;  // 1024 histogram bins
  MAXLENCODE = 7;  // max length code in histogramming mode
  MAXLENCODE_CONT =	6;    // max length code in continuous mode
  DFLTLENCODE = 6;        // default length code, see DFLTHISTLEN

  // limits for HH500_SetSyncDiv
  SYNCDIVMIN = 1;
  SYNCDIVMAX = 16;

  // codes for HH500_SetSyncTrgMode and HH500_SetInputTrgMode parameter "mode"
  TRGMODE_ETR = 0; // edge trigger
  TRGMODE_CFD = 1; // constant fraction discriminator

  //limits for HH500_SetSyncCFD and HH500_SetInputCFD
  CFDLVLMIN = -1200; // mV
  CFDLVLMAX =  0;    // mV
  CFDZCMIN  = -100;  // mV
  CFDZCMAX  =  0;    // mV

  // limits for HH500_SetSyncEdgeTrg and HH500_SetInputEdgeTrg
  TRGLVLMIN	      =      -1200; // mV
  TRGLVLMAX	      =       1200; // mV

  // limits for HH500_SetSyncChannelOffset and HH500_SetInputChannelOffset
  CHANOFFSMIN = -99999; // ps
  CHANOFFSMAX =  99999; // ps

  // limits for HH500_SetSyncDeadTime and HH500_SetInputDeadTime
  EXTDEADMIN  = 800;    // ps
  EXTDEADMAX  = 160000; // ps

  // limits for HH500_SetOffset
  OFFSETMIN = 0;         // ns
  OFFSETMAX = 100000000; // ns

  // limits for HH500_StartMeas
  ACQTMIN = 1;         // ms
  ACQTMAX = 360000000; // ms  (100*60*60*1000ms = 100h)

  // limits for HH500_SetStopOverflow
  STOPCNTMIN = 1;
  STOPCNTMAX = 4294967295; // 32 bit is mem max

  // limits for HH500_SetTriggerOutput
  TRIGOUTMIN = 0;        // 0 = off
  TRIGOUTMAX = 16777215; // in units of 100ns

  // limits for HH500_SetMarkerHoldoffTime
  HOLDOFFMIN = 0;     // ns
  HOLDOFFMAX = 25500; // ns

  // limits for HH500_SetInputHysteresis
  HYSTCODEMIN = 0; // approx. 3mV
  HYSTCODEMAX = 1; // approx. 35mV

  // limits for HH500_SetOflCompression
  HOLDTIMEMIN = 0;   // ms
  HOLDTIMEMAX = 255; // ms

  // constants for brightness in HH500_SetLEDBrightness
  LEDS_OFF = 0;
  LEDS_DIM = 1;
  LEDS_ON  = 2;

  // limits for HH500_SetEventFilterParams and HH500_SetEventFilterChannels
  GROUPIDXMIN  =      0;
  GROUPIDXMAX  =      7;     // actual upper limit is smaller, dep. on rows present
  BLOCKIDXMIN  =      0;
  BLOCKIDXMAX  =      7;     // actual upper limit is smaller, dep. on rows present
  MATCHCNTMIN  =      1;
  MATCHCNTMAX  =      6;
  INVERSEMIN   =      0;
  INVERSEMAX   =      1;
  TIMERANGEMIN =      0; // ps
  TIMERANGEMAX =  32000; // ps
  USECHANSMIN  =   $000; // no channels used
  USECHANSMAX  =   $1FF; // note: sync bit 0x100 will be ignored in T3 mode and in row filter
  PASSCHANSMIN =   $000; // no channels passed
  PASSCHANSMAX =   $1FF; // note: sync bit 0x100 will be ignored in T3 mode and in row filter

  // bitmasks for results from HH500_GetWarnings
  WARNING_SYNC_RATE_ZERO      = $0001;
  WARNING_SYNC_RATE_VERY_LOW  = $0002;
  WARNING_SYNC_RATE_TOO_HIGH  = $0004;

  WARNING_INPT_RATE_ZERO      = $0010;
  WARNING_INPT_RATE_TOO_HIGH  = $0040;

  WARNING_INPT_RATE_RATIO     = $0100;
  WARNING_DIVIDER_GREATER_ONE = $0200;
  WARNING_TIME_SPAN_TOO_SMALL = $0400;
  WARNING_OFFSET_UNNECESSARY  = $0800;

  WARNING_DIVIDER_TOO_SMALL   = $1000;
  WARNING_COUNTS_DROPPED      = $2000;
  WARNING_USB20_SPEED_ONLY    = $4000;

//The following is only for use with White Rabbit

  WR_STATUS_LINK_ON          = $00000001;  // WR link is switched on
  WR_STATUS_LINK_UP          = $00000002;  // WR link is established

  WR_STATUS_MODE_BITMASK     = $0000000C;  // mask for the mode bits
  WR_STATUS_MODE_OFF         = $00000000;  // mode is "off"
  WR_STATUS_MODE_SLAVE       = $00000004;  // mode is "slave"
  WR_STATUS_MODE_MASTER      = $00000008;  // mode is "master"
  WR_STATUS_MODE_GMASTER     = $0000000C;  // mode is "grandmaster"

  WR_STATUS_LOCKED_CALIBD    = $00000010;  // locked and calibrated

  WR_STATUS_PTP_BITMASK      = $000000E0;  // mask for the PTP bits
  WR_STATUS_PTP_LISTENING    = $00000020;
  WR_STATUS_PTP_UNCLWRSLCK   = $00000040;
  WR_STATUS_PTP_SLAVE        = $00000060;
  WR_STATUS_PTP_MSTRWRMLCK   = $00000080;
  WR_STATUS_PTP_MASTER       = $000000A0;

  WR_STATUS_SERVO_BITMASK    = $00000700;  // mask for the servo bits
  WR_STATUS_SERVO_UNINITLZD  = $00000100;  //
  WR_STATUS_SERVO_SYNC_SEC   = $00000200;  //
  WR_STATUS_SERVO_SYNC_NSEC  = $00000300;  //
  WR_STATUS_SERVO_SYNC_PHASE = $00000400;  //
  WR_STATUS_SERVO_WAIT_OFFST = $00000500;  //
  WR_STATUS_SERVO_TRCK_PHASE = $00000600;  //

  WR_STATUS_MAC_SET          = $00000800;  // user defined mac address is set
  WR_STATUS_IS_NEW           = $80000000;  // status updated since last check


//The following is only for use with an external FPGA

// for parameter "on" in HH500_ExtFPGAInitLink
  EXTFPGA_LINK_ON_FRONT_SFP = 2;
  EXTFPGA_LINK_ON_BACK_EXT  = 1;
  EXTFPGA_LINK_OFF          = 0;

//for parameter "mode" in HH500_ExtFPGASetMode
  EXTFPGA_MODE_OFF   = 0;
  EXTFPGA_MODE_T2RAW = 1;
  EXTFPGA_MODE_T2    = 2;
  EXTFPGA_MODE_T3    = 3;

//for parameter "loopback" in HH500_ExtFPGASetMode
  EXTFPGA_LOOPBACK_OFF    = 0;
  EXTFPGA_LOOPBACK_CUSTOM = 1;
  EXTFPGA_LOOPBACK_T2     = 2;
  EXTFPGA_LOOPBACK_T3     = 3;

//The following is only for use with continuous histogramming mode
  CONTMODE_FLAG_STOP_ON_OVFL = 1;
  CONTMODE_FLAG_CONTMEM_FULL = 2;

// functions from the shared lib/DLL

function HH500_GetLibraryVersion(Vers: PAnsiChar): LongInt; external LIB_NAME;
function HH500_GetErrorString(ErrString: PAnsiChar; ErrCode: LongInt): LongInt; external LIB_NAME;

function HH500_OpenDevice(DevIdx: LongInt; Serial: PAnsiChar): LongInt; external LIB_NAME;
function HH500_CloseDevice(Devidx: LongInt): LongInt; external LIB_NAME;
function HH500_Initialize(Devidx: LongInt; Mode: LongInt; RefSource: LongInt): LongInt; external LIB_NAME;

// all functions below can only be used after HH500_Initialize

function HH500_GetHardwareInfo(DevIdx: LongInt; Model: PAnsiChar; PartNo: PAnsiChar; Version: PAnsiChar): LongInt; external LIB_NAME;
function HH500_GetSerialNumber(DevIdx: LongInt; Serial: PAnsiChar): LongInt; external LIB_NAME;
function HH500_GetFeatures(DevIdx: LongInt; var Features: LongInt): LongInt; external LIB_NAME;
function HH500_GetBaseResolution(DevIdx: LongInt; var Resolution: Double; var BinSteps: LongInt): LongInt; external LIB_NAME;
function HH500_GetNumOfInputChannels(DevIdx: LongInt; var NChannels: LongInt): LongInt; external LIB_NAME;

function HH500_GetSyncFeatures(DevIdx: LongInt; var Features: LongWord): LongInt; external LIB_NAME;
function HH500_SetSyncDiv(DevIdx: LongInt; SyncDiv: LongInt): LongInt; external LIB_NAME;
function HH500_SetSyncTrgMode(DevIdx: LongInt; Mode: LongInt): LongInt; external LIB_NAME;
function HH500_SetSyncEdgeTrg(DevIdx: LongInt; Level: LongInt; Edge: LongInt): LongInt; external LIB_NAME;
function HH500_SetSyncCFD(DevIdx: LongInt; Level: LongInt; ZeroCross: LongInt): LongInt; external LIB_NAME;
function HH500_SetSyncChannelOffset(DevIdx: LongInt; Value: LongInt): LongInt; external LIB_NAME;
function HH500_SetSyncChannelEnable(DevIdx: LongInt; Enable: LongInt): LongInt; external LIB_NAME;
function HH500_SetSyncDeadTime(Devidx: LongInt; IsOn: LongInt; DeadTime: LongInt): LongInt; external LIB_NAME;

function HH500_GetInputFeatures(DevIdx: LongInt; Channel: LongInt; var Features: LongWord): LongInt; external LIB_NAME;
function HH500_SetInputTrgMode(DevIdx: LongInt; Channel: LongInt; Mode: LongInt): LongInt; external LIB_NAME;
function HH500_SetInputEdgeTrg(DevIdx: LongInt; channel: LongInt; level: LongInt; edge: LongInt): LongInt; external LIB_NAME;
function HH500_SetInputCFD(DevIdx: LongInt; Channel: LongInt; Level: LongInt; ZeroCross: LongInt): LongInt; external LIB_NAME;
function HH500_SetInputChannelOffset(DevIdx: LongInt; channel: LongInt; value: LongInt): LongInt; external LIB_NAME;
function HH500_SetInputDeadTime(DevIdx: LongInt; Channel: LongInt; IsOn: LongInt; DeadTime: LongInt): LongInt; external LIB_NAME;
function HH500_SetInputHysteresis(DevIdx: LongInt; HystCode: LongInt): LongInt; external LIB_NAME;
function HH500_SetInputChannelEnable(DevIdx: LongInt; channel: LongInt; enable: LongInt): LongInt; external LIB_NAME;

function HH500_SetStopOverflow(DevIdx: LongInt; stop_ovfl: LongInt; stopcount: LongWord): LongInt; external LIB_NAME;
function HH500_SetBinning(DevIdx: LongInt; binning: LongInt): LongInt; external LIB_NAME;
function HH500_SetOffset(DevIdx: LongInt; offset: LongInt): LongInt; external LIB_NAME;
function HH500_SetHistoLen(DevIdx: LongInt; lencode: LongInt; var actuallen: LongInt): LongInt; external LIB_NAME;
function HH500_SetMeasControl(DevIdx: LongInt; control: LongInt; startedge: LongInt; stopedge: LongInt): LongInt; external LIB_NAME;
function HH500_SetTriggerOutput(DevIdx: LongInt; period: LongInt): LongInt; external LIB_NAME;
function HH500_SetLEDBrightness(DevIdx: LongInt; brightness: LongInt): LongInt; external LIB_NAME;

function HH500_ClearHistMem(DevIdx: LongInt): LongInt; external LIB_NAME;
function HH500_StartMeas(DevIdx: LongInt; tacq: LongInt): LongInt; external LIB_NAME;
function HH500_StopMeas(DevIdx: LongInt): LongInt; external LIB_NAME;
function HH500_CTCStatus(DevIdx: LongInt; var ctcstatus: LongInt): LongInt; external LIB_NAME;

function HH500_GetHistogram(DevIdx: LongInt; var chcount: LongWord; channel: LongInt): LongInt; external LIB_NAME;
function HH500_GetAllHistograms(DevIdx: LongInt; var chcount: LongWord): LongInt; external LIB_NAME;
function HH500_GetResolution(DevIdx: LongInt; var Resolution: Double): LongInt; external LIB_NAME;
function HH500_GetSyncPeriod(DevIdx: LongInt; var period: Double): LongInt; external LIB_NAME;
function HH500_GetSyncRate(DevIdx: LongInt; var syncrate: LongInt): LongInt; external LIB_NAME;
function HH500_GetCountRate(DevIdx: LongInt; channel: LongInt; var cntrate: LongInt): LongInt; external LIB_NAME;
function HH500_GetAllCountRates(DevIdx: LongInt; var syncrate: LongInt; var cntrates: LongInt): LongInt; external LIB_NAME;
function HH500_GetFlags(DevIdx: LongInt; var flags: LongInt): LongInt; external LIB_NAME;
function HH500_GetElapsedMeasTime(DevIdx: LongInt; var elapsed: Double): LongInt; external LIB_NAME;
function HH500_GetStartTime(DevIdx: LongInt; var TimeDw2: LongWord; var TimeDw1: LongWord; var TimeDw0: LongWord): LongInt; external LIB_NAME;

function HH500_GetWarnings(DevIdx: LongInt; var warnings: LongInt): LongInt; external LIB_NAME;
function HH500_GetWarningsText(DevIdx: LongInt; text: PAnsiChar; warnings: LongInt): LongInt; external LIB_NAME;

// for the time tagging modes only
function HH500_SetOflCompression(DevIdx: LongInt; HoldTime: LongInt): LongInt; external LIB_NAME;
function HH500_SetMarkerHoldoffTime(DevIdx: LongInt; holdofftime: LongInt): LongInt; external LIB_NAME;
function HH500_SetMarkerEdges(DevIdx: LongInt; me1: LongInt; me2: LongInt; me3: LongInt; me4: LongInt): LongInt; external LIB_NAME;
function HH500_SetMarkerEnable(DevIdx: LongInt; en1: LongInt; en2: LongInt; en3: LongInt; en4: LongInt): LongInt; external LIB_NAME;
function HH500_ReadFiFo(DevIdx: LongInt; var buffer: LongWord; var nactual: LongInt): LongInt; external LIB_NAME;

// for event filtering, time tagging modes only
function HH500_SetGroupEventFilter(DevIdx: LongInt; groupidx: LongInt; timerange: LongInt; matchcnt: LongInt; inverse: LongInt; usechannels: LongInt; passchannels: LongInt): LongInt; external LIB_NAME;
function HH500_EnableGroupEventFilter(DevIdx: LongInt;  groupidx: LongInt; enable: LongInt): LongInt; external LIB_NAME;
function HH500_SetMainEventFilterParams(DevIdx: LongInt; timerange: LongInt; matchcnt: LongInt; inverse: LongInt): LongInt; external LIB_NAME;
function HH500_SetMainEventFilterChannels(DevIdx: LongInt;  blockidx: LongInt; usechannels: LongInt; passchannels: LongInt): LongInt; external LIB_NAME;
function HH500_EnableMainEventFilter(DevIdx: LongInt; enable: LongInt): LongInt; external LIB_NAME;
function HH500_SetFilterTestMode(DevIdx: LongInt; testmode: LongInt): LongInt; external LIB_NAME;
function HH500_GetGroupFilteredRates(DevIdx: LongInt; var syncrate: LongInt; var cntrates:  LongInt): LongInt; external LIB_NAME;
function HH500_GetMainFilteredRates(DevIdx: LongInt; var syncrate: LongInt; var cntrates: LongInt): LongInt; external LIB_NAME;

// for debugging only
function HH500_GetDebugInfo(DevIdx: LongInt; debuginfo: PAnsiChar): LongInt; external LIB_NAME;
function HH500_GetNumOfModules(DevIdx: LongInt; var NumMod: LongInt): LongInt; external LIB_NAME;
function HH500_GetModuleInfo(DevIdx: LongInt; ModIdx: LongInt; var ModelCode: LongInt; var VersionCode: LongInt): LongInt; external LIB_NAME;
function HH500_SaveDebugDump(DevIdx: LongInt; filepath: PAnsiChar): LongInt; external LIB_NAME;

//for White Rabbit only
function HH500_WRabbitGetMAC(DevIdx: LongInt; mac_addr: PByte): LongInt; external LIB_NAME;
function HH500_WRabbitSetMAC(DevIdx: LongInt; mac_addr: PByte): LongInt; external LIB_NAME;
function HH500_WRabbitGetInitScript(DevIdx: LongInt; initscript: PAnsiChar): LongInt; external LIB_NAME;
function HH500_WRabbitSetInitScript(DevIdx: LongInt; initscript: PAnsiChar): LongInt; external LIB_NAME;
function HH500_WRabbitGetSFPData(DevIdx: LongInt; sfpnames: PAnsiChar; var dTxs: LongInt; var dRxs: LongInt; var alphas: LongInt): LongInt; external LIB_NAME;
function HH500_WRabbitSetSFPData(DevIdx: LongInt; sfpnames: PAnsiChar; var dTxs: LongInt; var dRxs: LongInt; var alphas: LongInt): LongInt; external LIB_NAME;
function HH500_WRabbitInitLink(DevIdx: LongInt; link_on: LongInt): LongInt; external LIB_NAME;
function HH500_WRabbitSetMode(DevIdx: LongInt; bootfromscript: LongInt; reinit_with_mode: LongInt; mode: LongInt): LongInt; external LIB_NAME;
function HH500_WRabbitSetTime(DevIdx: LongInt; timehidw: LongWord; timelodw: LongWord): LongInt; external LIB_NAME;
function HH500_WRabbitGetTime(DevIdx: LongInt; var timehidw: LongWord; var timelodw: LongWord; var subsec16ns: LongWord): LongInt; external LIB_NAME;
function HH500_WRabbitGetStatus(DevIdx: LongInt; var wrstatus: LongInt): LongInt; external LIB_NAME;
function HH500_WRabbitGetTermOutput(DevIdx: LongInt; buffer: PAnsiChar; var nchar: LongInt): LongInt; external LIB_NAME;

//for external FPGA interface only
function HH500_ExtFPGAInitLink(DevIdx: LongInt; linknumber: LongInt; on: LongInt): LongInt; external LIB_NAME;
function HH500_ExtFPGAGetLinkStatus(DevIdx: LongInt; linknumber: LongInt; var status: LongWord): LongInt; external LIB_NAME;
function HH500_ExtFPGASetMode(DevIdx: LongInt; mode: LongInt; loopback: LongInt): LongInt; external LIB_NAME;
function HH500_ExtFPGAResetStreamFifos(DevIdx: LongInt): LongInt; external LIB_NAME;
function HH500_ExtFPGAUserCommand(DevIdx: LongInt; write: LongInt; addr: LongWord; var data: LongWord): LongInt; external LIB_NAME;

//for Continuous Mode only
function HH500_GetContModeBlock(DevIdx: LongInt; blockbuffer: Pointer; var nbytesreceived: Integer): LongInt; external LIB_NAME;
function HH500_DissectContModeBlkHdr(
  blockbuffer: Pointer;                 //IN  buffer with raw cont mode data to dissect
  var blocknum: LongWord;               //OUT counting up from 1, can be used for sanity check
  var nchannels: LongWord;              //OUT number of active input channels (= nhistograms) in this data block
  var histlen: LongWord;			          //OUT number of histogram bins as set by HH500_SetHistoLen
  var flags: LongWord;                  //OUT bit0: stop on overflow occurred, bit1: contmode_mem full
  var starttime: UInt64;                //OUT start time of this block of histograms in nanosec
  var ctctime: UInt64;                  //OUT effective histogram duration in nanosec
  markercounts: PLongWord	              //OUT array[4], count of occurrences of markers 0..3 within this time
  ): LongInt; external LIB_NAME;
function HH500_ExtractContModeBlkData(
  blockbuffer: Pointer;                 //IN  buffer with raw cont mode data to dissect
  var chanindex: LongWord;              //IN  index of channel to extract, must not exceed nchan-1 from block header
  var histosum: UInt64;                 //OUT sum of counts in histogram of selected channel
  histogram: PLongWord                  //OUT histogram of that channel as array[histlen] with histlen from header
  ): LongInt; external LIB_NAME;

// for final cleanup
procedure HH500_CloseAllDevices;

// error codes (return codes from the functions above)
const
  HH500_ERROR_NONE                      =   0;

  HH500_ERROR_DEVICE_OPEN_FAIL          =  -1;
  HH500_ERROR_DEVICE_BUSY               =  -2;
  HH500_ERROR_DEVICE_HEVENT_FAIL        =  -3;
  HH500_ERROR_DEVICE_CALLBSET_FAIL      =  -4;
  HH500_ERROR_DEVICE_BARMAP_FAIL        =  -5;
  HH500_ERROR_DEVICE_CLOSE_FAIL         =  -6;
  HH500_ERROR_DEVICE_RESET_FAIL         =  -7;
  HH500_ERROR_DEVICE_GETVERSION_FAIL    =  -8;
  HH500_ERROR_DEVICE_VERSION_MISMATCH   =  -9;
  HH500_ERROR_DEVICE_NOT_OPEN           = -10;
  HH500_ERROR_DEVICE_LOCKED             = -11;
  HH500_ERROR_DEVICE_DRIVERVER_MISMATCH = -12;

  HH500_ERROR_INSTANCE_RUNNING          = -16;
  HH500_ERROR_INVALID_ARGUMENT          = -17;
  HH500_ERROR_INVALID_MODE              = -18;
  HH500_ERROR_INVALID_OPTION            = -19;
  HH500_ERROR_INVALID_MEMORY            = -20;
  HH500_ERROR_INVALID_RDATA             = -21;
  HH500_ERROR_NOT_INITIALIZED           = -22;
  HH500_ERROR_NOT_CALIBRATED            = -23;
  HH500_ERROR_DMA_FAIL                  = -24;
  HH500_ERROR_XTDEVICE_FAIL             = -25;
  HH500_ERROR_FPGACONF_FAIL             = -26;
  HH500_ERROR_IFCONF_FAIL               = -27;
  HH500_ERROR_FIFORESET_FAIL            = -28;
  HH500_ERROR_THREADSTATE_FAIL          = -29;
  HH500_ERROR_THREADLOCK_FAIL           = -30;

  HH500_ERROR_USB_GETDRIVERVER_FAIL     = -32;
  HH500_ERROR_USB_DRIVERVER_MISMATCH    = -33;
  HH500_ERROR_USB_GETIFINFO_FAIL        = -34;
  HH500_ERROR_USB_HISPEED_FAIL          = -35;
  HH500_ERROR_USB_VCMD_FAIL             = -36;
  HH500_ERROR_USB_BULKRD_FAIL           = -37;
  HH500_ERROR_USB_RESET_FAIL            = -38;

  HH500_ERROR_LANEUP_TIMEOUT            = -40;
  HH500_ERROR_DONEALL_TIMEOUT           = -41;
  HH500_ERROR_MB_ACK_TIMEOUT            = -42;
  HH500_ERROR_MACTIVE_TIMEOUT           = -43;
  HH500_ERROR_MEMCLEAR_FAIL             = -44;
  HH500_ERROR_MEMTEST_FAIL              = -45;
  HH500_ERROR_CALIB_FAIL                = -46;
  HH500_ERROR_REFSEL_FAIL               = -47;
  HH500_ERROR_STATUS_FAIL               = -48;
  HH500_ERROR_MODNUM_FAIL               = -49;
  HH500_ERROR_DIGMUX_FAIL               = -50;
  HH500_ERROR_MODMUX_FAIL               = -51;
  HH500_ERROR_MODFWPCB_MISMATCH         = -52;
  HH500_ERROR_MODFWVER_MISMATCH         = -53;
  HH500_ERROR_MODPROPERTY_MISMATCH      = -54;
  HH500_ERROR_INVALID_MAGIC             = -55;
  HH500_ERROR_INVALID_LENGTH            = -56;
  HH500_ERROR_RATE_FAIL                 = -57;
  HH500_ERROR_MODFWVER_TOO_LOW          = -58;
  HH500_ERROR_MODFWVER_TOO_HIGH         = -59;
  HH500_ERROR_MB_ACK_FAIL               = -60;

  HH500_ERROR_EEPROM_F01                = -64;
  HH500_ERROR_EEPROM_F02                = -65;
  HH500_ERROR_EEPROM_F03                = -66;
  HH500_ERROR_EEPROM_F04                = -67;
  HH500_ERROR_EEPROM_F05                = -68;
  HH500_ERROR_EEPROM_F06                = -69;
  HH500_ERROR_EEPROM_F07                = -70;
  HH500_ERROR_EEPROM_F08                = -71;
  HH500_ERROR_EEPROM_F09                = -72;
  HH500_ERROR_EEPROM_F10                = -73;
  HH500_ERROR_EEPROM_F11                = -74;
  HH500_ERROR_EEPROM_F12                = -75;
  HH500_ERROR_EEPROM_F13                = -76;
  HH500_ERROR_EEPROM_F14                = -77;
  HH500_ERROR_EEPROM_F15                = -78;

  HH500_ERROR_UNSUPPORTED_FUNCTION      = -80;
  HH500_ERROR_WRONG_TRGMODE             = -81;
  HH500_ERROR_BULKRDINIT_FAIL           = -82;
  HH500_ERROR_CREATETHREAD_FAIL         = -83;
  HH500_ERROR_FILEOPEN_FAIL             = -84;
  HH500_ERROR_FILEWRITE_FAIL            = -85;
  HH500_ERROR_FILEREAD_FAIL             = -86;

  HH500_ERROR_SFP_BUSY                  = -90;

  HH500_ERROR_INVALID_ARGUMENT_1       = -201;
  HH500_ERROR_INVALID_ARGUMENT_2       = -202;
  HH500_ERROR_INVALID_ARGUMENT_3       = -203;
  HH500_ERROR_INVALID_ARGUMENT_4       = -204;
  HH500_ERROR_INVALID_ARGUMENT_5       = -205;
  HH500_ERROR_INVALID_ARGUMENT_6       = -206;
  HH500_ERROR_INVALID_ARGUMENT_7       = -207;
  HH500_ERROR_INVALID_ARGUMENT_8       = -208;
  HH500_ERROR_INVALID_ARGUMENT_9       = -209;

implementation

  procedure HH500_CloseAllDevices;
  var
    Dev : Integer;
  begin
    for Dev := 0 to MAXDEVNUM - 1 do // no harm closing all
      HH500_CloseDevice(Dev);
  end;

initialization

finalization
  HH500_CloseAllDevices;
end.
