﻿{
  Demo access to HydraHarp 500 via HH500Lib.dll or libhh500.so v.1.0
  The program performs a measurement based on hardcoded settings.
  The resulting event data is instantly processed.
  Processing consists here only of dissecting the binary event record
  data and writing it to a text file. This is only for demo purposes.
  In a real application this makes no sense as it limits throughput and
  creates very large files. In practice you would more sensibly perform
  some meaningful processing such as counting coincidences on the fly.

  Dr. Marcus Sackrow, PicoQuant, May 2025

  Note: This is a console application.

  Note: At the API level the input channel numbers are indexed 0..N-1
  where N is the number of input channels the device has.
  Upon processing we map this back to 1..N corresponding to the front
  panel labelling.

  Tested with the following compilers:

  - Delphi 11.1 (Windows)
  - Lazarus 3.0 RC1 (Windows)
  - Lazarus 2.2.0 / FPC 3.2.2 (Linux)
}

program tttrmode;

{$IF defined(MSWINDOWS)}
  {$APPTYPE CONSOLE}  // Windows needs this, Linux does not want it
{$ENDIF}

uses
  {$ifdef fpc}
  SysUtils,
  {$else}
  System.SysUtils,
  System.Ansistrings,
  {$endif}
  HH500Lib in 'HH500Lib.pas';


var
  Dev             : array [0..MAXDEVNUM - 1] of LongInt;
  Device          : LongInt;
  Found           : Integer = 0;
  RetCode         : LongInt;
  CTCStatus       : LongInt;
  strLibVers      : array[0..7]  of AnsiChar;
  HW_Model        : array[0..31] of AnsiChar;
  HW_Partno       : array[0..8]  of AnsiChar;
  HW_Serial       : array[0..9]  of AnsiChar;
  HW_Version      : array[0..15] of AnsiChar;
  Errorstring     : array[0..39] of AnsiChar;
  DebugInfoBuffer : array[0..16383] of AnsiChar;
  NumChannels     : LongInt;

  //############### User Settings #########################
  Mode            : LongInt = MODE_T2; // set T2 or T3 here, observe suitable Sync divider and Range!
  Binning         : LongInt = 0; // You can change this, meaningful only in T3 mode
  Offset          : LongInt = 0; // You can change this, meaningful only in T3 mode, normally 0
  Tacq            : LongInt = 1000; // Measurement time in millisec, you can change this
  SyncDivider     : LongInt = 1; // you can change this

  // Dependent on the hardware model, the HydraHarp 500 may have different input
  // circuits: Edge Trigger (ETR) and/or Constant Fraction Discrimonator (CFD)
  // This can even vary from channel to channel. An input that has both ETR and CFD
  // will be programmable as to which trigger mode to use. Here we define default
  // settings for both variants and the default trigger mode ETR.
  // The latter can be changed to CFD which of course only works with channels
  // that actually have a CFD. The code further below will therfore query the
  // input capabilities and where necessary fall back to the mode available.

  // Sync Settings
  SyncChannelOffset : LongInt =           0; // in ps, you can change this (like a cable delay)
  SyncTrgMode       : LongInt = TRGMODE_ETR; // you can change this to TRGMODE_CFD

  // in case of SyncTrgMode == TRGMODE_ETR this will apply:
  SyncTrgEdge  : LongInt = EDGE_FALLING; // you can change this to EDGE_RISING
  SyncTrgLevel : LongInt = -50; // in mV, you can change this

  // in case of SyncTrgMode == TRGMODE_CFD this will apply:
  SyncCFDZeroCross : LongInt = -50; // in mV, you can change this
  SyncCFDLevel     : LongInt = -50; // in mV, you can change this

  // Input Settings
  InputChannelOffset : LongInt = 0; // in ps, you can change this (like a cable delay)
  InputTrgMode       : LongInt = TRGMODE_ETR; // you can change this to TRGMODE_CFD

  // in case of InputTrgMode == TRGMODE_ETR this will apply:
  InputTrgEdge  : LongInt = EDGE_FALLING; // you can change this to EDGE_RISING
  InputTrgLevel : LongInt = -50;          // in mV, you can change this

  // in case of InputTrgMode == TRGMODE_CFD this will apply:
  InputCFDZeroCross : LongInt = -20;      // in mV, you can change this
  InputCFDLevel     : LongInt = -50;      // in mV, you can change this

  // other variables
  Resolution    : Double;
  SyncRate      : LongInt;
  CountRate     : LongInt;
  Flags         : LongInt;
  Warnings      : LongInt;
  WarningsText  : array[0..16343] of AnsiChar; // must have 16384 bytes of text buffer
  Records       : LongInt;
  Progress      : LongInt = 0;
  FiFoFull      : Boolean = False;
  FileError     : Boolean = False;
  ChanIdx       : LongInt;
  Buffer        : array[0..TTREADMAX - 1] of LongWord;
  OutputFile    : TextFile;
  i             : Integer;
  TimeOut       : Boolean = False;
  OflCorrection : Int64 = 0;
  SyncPeriod    : Double = 0;

  //GotPhotonT2 procedure
  //  TimeTag: Raw TimeTag from Record * Resolution = Real Time arrival of Photon
  procedure GotPhotonT2(TimeTag: Int64; Channel: Integer);
  begin
    Writeln(OutputFile, 'CH ', Channel, ' ', Round(TimeTag * Resolution):9);
  end;

  //GotPhotonT3 procedure
  //  DTime: Arrival time of Photon after last Sync event (T3 only)
  //    DTime * Resolution = Real time arrival of Photon after last Sync event
  //  Channel: Channel the Photon arrived (0 = Sync channel for T2 measurements)
  procedure GotPhotonT3(NSync: Int64; DTime: Integer; Channel: Integer);
  begin
    Writeln(OutputFile, 'CH ', Channel, ' ', (NSync * SyncPeriod):3:9, ' ',
      Round(DTime * Resolution)); //NSync * SyncPeriod/s
  end;

  //GotMarker
  //  TimeTag: Raw TimeTag from Record * Global resolution = Real Time arrival of Marker
  //  Markers: Bitfield of arrived Markers, different markers can arrive at same time (same record)
  procedure GotMarker(TimeTag: Int64; Markers: Integer);
  begin
    Writeln(OutputFile, ' MAR ', TimeTag, ' ', Markers);
  end;

  //ProcessT2 procedure
  //HydraHarpV2 (Version 2) or TimeHarp260, MultiHarp, PicoHarp330 and HydraHarp500 record data
  procedure ProcessT2(TTTR_RawData: Cardinal);
  const
    T2WRAPAROUND_V2 = 33554432;
  type
    TT2DataRecords = record
      Special: Boolean;
      Channel: Byte;
      TimeTag: Cardinal;
    end;
  var
    TTTR_Data: TT2DataRecords;
    TrueTime: Cardinal;
  begin
    {split "RawData" into its parts}
    TTTR_Data.TimeTag := Cardinal(TTTR_RawData and $01FFFFFF); // 25 bit of 32 bit for TimeTag
    TTTR_Data.Channel := Byte((TTTR_RawData shr 25) and $0000003F); // 6 bit of 32 bit for Channel
    TTTR_Data.Special := Boolean((TTTR_RawData shr 31) and $00000001); // 1 bit of 32 bit for Special
    if TTTR_Data.Special then                  // this means we have a Special record
      case TTTR_Data.Channel of
        $3F:        // overflow
          begin
            // number of overflows is stored in timetag
            // if it is zero, it is an old style single overflow {should never happen with new Firmware}
            if TTTR_Data.TimeTag = 0 then
              OflCorrection := OflCorrection + T2WRAPAROUND_V2
            else
              OflCorrection := OflCorrection + T2WRAPAROUND_V2 * TTTR_Data.TimeTag;
          end;
        1..15:   // markers
          begin
            TrueTime := OflCorrection + TTTR_Data.TimeTag;
            // Note that actual marker tagging accuracy is only some ns.
            GotMarker(TrueTime, TTTR_Data.Channel);
          end;
        0: //sync
          begin
            TrueTime := OflCorrection + TTTR_Data.TimeTag;
            GotPhotonT2(TrueTime, 0); //we encode the sync channel as 0
          end;
      end
    else
    begin // it is a regular photon record
      TrueTime := OflCorrection + TTTR_Data.TimeTag;
      GotPhotonT2(TrueTime,
        TTTR_Data.Channel + 1 //we encode the regular channels as 1..N
        );
    end;
  end;

  //ProcessT3 procedure
  //HydraHarp or TimeHarp260, MultiHarp, PicoHarp330 and HydraHarp500 record data
  procedure ProcessT3(TTTR_RawData: Cardinal);
  const
    T3WRAPAROUND = 1024;
  type
    TT3DataRecords = record
      Special: Boolean;
      Channel: Byte;
      DTime: Word;
      NSync: Word;
    end;
  var
    TTTR_Data: TT3DataRecords;
    TrueNSync: Integer;
  begin
    {split "RawData" into its parts}
    TTTR_Data.NSync := Word(TTTR_RawData and $000003FF); // 10 bit of 32 bit for NSync
    TTTR_Data.DTime := Word((TTTR_RawData shr 10) and $00007FFF); // 15 bit of 32 bit for DTime
    TTTR_Data.Channel := Byte((TTTR_RawData shr 25) and $0000003F); // 6 bit of 32 bit for Channel
    TTTR_Data.Special := Boolean((TTTR_RawData shr 31) and $00000001); // 1 bit of 32 bit for Special
    if TTTR_Data.Special then                  // this means we have a Special record
      case TTTR_Data.Channel of
        $3F:        // overflow
          begin
            // number of overflows is stored in timetag
            // if it is zero, it is an old style single overflow {should never happen with new Firmware}
            if TTTR_Data.NSync = 0 then
              OflCorrection := OflCorrection + T3WRAPAROUND
            else
              OflCorrection := OflCorrection + T3WRAPAROUND * TTTR_Data.NSync;
          end;
        1..15:   // markers
          begin
            TrueNSync := OflCorrection + TTTR_Data.NSync; //the time unit depends on sync period
            // Note that actual marker tagging accuracy is only some ns.
            GotMarker(TrueNSync, TTTR_Data.Channel);
          end;
      end
    else
    begin // it is a regular photon record
      TrueNSync := OflCorrection + TTTR_Data.NSync;
      //TruenSync indicates the number of the sync period this event was in
      GotPhotonT3(TrueNSync,
        TTTR_Data.DTime, //the dtime unit depends on the chosen resolution (binning)
        TTTR_Data.Channel + 1 //we encode the regular channels as 1..N
        );
    end;
  end;


procedure Ex(RetCode: Integer);
begin
  if RetCode <> HH500_ERROR_NONE then
  begin
    HH500_GetErrorString(ErrorString, RetCode);
    Writeln('Error ', RetCode:3, ' = "', Trim (string(ErrorString)), '"');
  end;
  Writeln;
  {$I-}
    CloseFile(OutputFile);
    IOResult();
  {$I+}
  Writeln('Press RETURN to exit');
  Readln;
  Halt(RetCode);
end;

procedure StopTTTR;
begin
  RetCode := HH500_StopMeas(Dev[0]);
  if RetCode <> HH500_ERROR_NONE then
  begin
    Writeln('HH500_StopMeas error ', RetCode:3, '. Aborted.');
    Ex(RetCode);
  end;
end;

function SetInputModalities(DevIdx: LongInt; NumChannels: LongInt): LongInt;
var
  ChanIdx: LongInt;
  ChannelFeatures: LongWord;
  RealTrgMode: LongInt;
begin
  Result := HH500_GetSyncFeatures(DevIdx, ChannelFeatures);
  if Result <> HH500_ERROR_NONE then
  begin
    Writeln('HH500_GetSyncFeatures error ', Result:3, '. Aborted.');
    Exit;
  end;
  // check if the sync channel has the right feature for the requested setting
  if (SyncTrgMode = TRGMODE_ETR) and ((ChannelFeatures and HAS_ETR) = 0) then
  begin
    Writeln('Warning: Sync channel as no Edge Trigger, switching to CFD');
    SyncTrgMode := TRGMODE_CFD;
  end;
  if (SyncTrgMode = TRGMODE_CFD) and ((ChannelFeatures and HAS_CFD) = 0) then
  begin
    Writeln('Warning: Sync channel as no CFD, switching to Edge Trigger');
    SyncTrgMode := TRGMODE_ETR;
  end;

  Result := HH500_SetSyncTrgMode(DevIdx, SyncTrgMode);
  if Result <> HH500_ERROR_NONE then
  begin
    Writeln('HH500_SetSyncTrgMode error ', Result:3, '. Aborted.');
    Exit;
  end;

  if SyncTrgMode = TRGMODE_ETR then
  begin
    Result := HH500_SetSyncEdgeTrg(DevIdx, SyncTrgLevel, SyncTrgEdge);
    if Result <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_SetSyncEdgeTrg error ', Result:3, '. Aborted.');
      Exit;
    end;
  end;
  if SyncTrgMode = TRGMODE_CFD then
  begin
    Result := HH500_SetSyncCFD(DevIdx, SyncCFDLevel, SyncCFDZeroCross);
    if Result <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_SetSyncCFD error ', Result:3, '. Aborted.');
      Exit;
    end;
  end;

  Result := HH500_SetSyncChannelOffset(DevIdx, SyncChannelOffset);
  if Result <> HH500_ERROR_NONE then
  begin
    Writeln('HH500_SetSyncChannelOffset error ', Result:3, '. Aborted.');
    Exit;
  end;

  for ChanIdx := 0 to NumChannels - 1 do // we use the same input settings for all channels
  begin
    RealTrgMode := InputTrgMode;
    Result := HH500_GetInputFeatures(DevIdx, ChanIdx, ChannelFeatures);
    if Result <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_GetInputFeatures error ', Result:3, '. Aborted.');
      Exit;
    end;
    // check if the sync channel has the right feature for the requested setting
    if (RealTrgMode = TRGMODE_ETR) and ((ChannelFeatures and HAS_ETR) = 0) then
    begin
      Writeln('Warning: Input channel ', ChanIdx, ' has no Edge Trigger, switching to CFD');
      RealTrgMode := TRGMODE_CFD;
    end;
    if (RealTrgMode = TRGMODE_CFD) and ((ChannelFeatures and HAS_CFD) = 0) then
    begin
      Writeln('Warning: Input channel ', ChanIdx, ' has no CFD, switching to Edge Trigger');
      RealTrgMode := TRGMODE_ETR;
    end;

    Result := HH500_SetInputTrgMode(DevIdx, ChanIdx, RealTrgMode);
    if Result <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_SetInputTrgMode ', ChanIdx:2, ' error ', Result:3, '. Aborted.');
      Exit;
    end;
    if RealTrgMode = TRGMODE_ETR then
    begin
      Result := HH500_SetInputEdgeTrg(DevIdx, ChanIdx, InputTrgLevel, InputTrgEdge);
      if Result <> HH500_ERROR_NONE then
      begin
        Writeln('HH500_SetInputEdgeTrg ', ChanIdx:2, ' error ', Result:3, '. Aborted.');
        Exit;
      end;
    end;
    if RealTrgMode = TRGMODE_CFD then
    begin
      Result := HH500_SetInputCFD(DevIdx, ChanIdx, InputCFDLevel, InputCFDZeroCross);
      if Result <> HH500_ERROR_NONE then
      begin
        Writeln('HH500_SetInputCFD ', ChanIdx:2, ' error ', Result:3, '. Aborted.');
        Exit;
      end;
    end;

    Result := HH500_SetInputChannelOffset(DevIdx, ChanIdx, InputChannelOffset);
    if Result <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_SetInputChannelOffset channel ', ChanIdx:2, ' error ', Result:3, '. Aborted.');
      Exit;
    end;

    Result := HH500_SetInputChannelEnable(DevIdx, ChanIdx, 1);
    if Result <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_SetInputChannelEnable channel ', ChanIdx:2, ' error ', Result:3, '. Aborted.');
      Exit;
    end;
  end;
end;


// Main procedure
begin
  try
    Writeln;
    Writeln('HydraHarp 500 HH500Lib Demo Application             PicoQuant GmbH, 2025');
    Writeln('~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~');

    RetCode := HH500_GetLibraryVersion(strLibVers);
    if RetCode <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_GetLibraryVersion error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('HH500Lib version is ' + strLibVers);
    if Trim(Ansistring(strLibVers)) <> Trim(AnsiString (LIB_VERSION)) then
      Writeln('Warning: The application was built for version ' + LIB_VERSION);

    AssignFile(OutputFile, 'tttrmodeout.txt');
    {$I-}
      Rewrite(OutputFile);
    {$I+}
    if IOResult <> 0 then
    begin
      Writeln('cannot open output file');
      Ex(HH500_ERROR_NONE);
    end;

    Writeln;
    Writeln('Searching for HydraHarp 500 devices...');
    Writeln('DevIdx     Serial     Status');

    for Device := 0 to MAXDEVNUM - 1 do
    begin
      RetCode := HH500_OpenDevice(Device, HW_Serial);
      //
      if RetCode = HH500_ERROR_NONE then
      begin
        // Grab any HydraHarp 500 we can open
        Writeln(Device, '          ', HW_Serial, '    open ok');
        Dev[Found] := Device; // Keep index to devices we want to use
        Inc(Found);
      end
      else
      begin
        if RetCode = HH500_ERROR_DEVICE_OPEN_FAIL then
          Writeln(Device, HW_Serial, '                     no device')
        else
        begin
          HH500_GetErrorString(ErrorString, RetCode);
          Writeln(Device,HW_Serial, '       ' + Trim(string(ErrorString)));
        end;
      end;
    end;

    // In this demo we will use the first device we found,
    // i.e. iDevIdx[0].  You can also use multiple devices in parallel.
    // You could also check for a specific serial number, so that you
    // always know which physical device you are talking to.

    if Found < 1 then
    begin
      Writeln('No device available.');
      Ex(HH500_ERROR_NONE);
    end;

    Writeln('Using device ', Dev[0]);
    Writeln('Initializing the device...');

    RetCode := HH500_Initialize (Dev[0], Mode, 0); //with internal clock
    if RetCode <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_Initialize error ', RetCode:3, '. Aborted.');
      HH500_GetDebugInfo(Dev[0], DebugInfoBuffer);
      Writeln('Debug info: ', DebugInfoBuffer);
      Ex(RetCode);
    end;

    RetCode := HH500_GetHardwareInfo (Dev[0], HW_Model, HW_Partno, HW_Version); // this is only for information
    if RetCode <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_GetHardwareInfo error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end
    else
      Writeln('Found Model ', HW_Model,'  Part no ', HW_Partno,'  Version ', HW_Version);

    RetCode := HH500_GetNumOfInputChannels (Dev[0], NumChannels);
    if RetCode <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_GetNumOfInputChannels error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end
    else
      Writeln('Device has ', NumChannels, ' input channels.');

    Writeln;

    RetCode := HH500_SetSyncDiv (Dev[0], SyncDivider);
    if RetCode <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_SetSyncDiv error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    if SetInputModalities(Dev[0], NumChannels) < 0 then
      Ex(RetCode);

    if (Mode <> MODE_T2) then
    begin
      RetCode := HH500_SetBinning (Dev[0], Binning);
      if RetCode <> HH500_ERROR_NONE then
      begin
        Writeln('HH500_SetBinning error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;

      RetCode := HH500_SetOffset(Dev[0], Offset);
      if RetCode <> HH500_ERROR_NONE then
      begin
        Writeln('HH500_SetOffset error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
    end;

    RetCode := HH500_GetResolution (Dev[0], Resolution);
    if RetCode <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_GetResolution error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('Resolution is ', Resolution:7:3, 'ps');

    // after Init allow 150 ms for valid  count rate readings
    // subsequently you get new values after every 100ms
    Sleep (150);

    Writeln;

    Writeln('Measuring input rates...');

    RetCode := HH500_GetSyncRate (Dev[0], SyncRate);
    if RetCode <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_GetSyncRate error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('SyncRate = ', SyncRate, '/s');

    Writeln;

    for ChanIdx := 0 to NumChannels - 1 do // for all channels
    begin
      RetCode := HH500_GetCountRate (Dev[0], ChanIdx, CountRate);
      if RetCode <> HH500_ERROR_NONE then
      begin
        Writeln('HH500_GetCountRate error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      Writeln('Countrate [', ChanIdx:2, '] = ', CountRate:8, '/s');
    end;

    Writeln;

    RetCode := HH500_GetWarnings(Dev[0], Warnings);   //after getting the count rates you can check for warnings
    if RetCode <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_GetWarnings error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    if Warnings <> 0 then
    begin
      HH500_GetWarningsText(Dev[0], WarningsText, Warnings);
      Writeln(WarningsText);
    end;

    if Mode = Mode_T2 then
    begin
      Writeln(OutputFile, 'ev chn time/ps');
    end
    else
    begin
      Writeln(OutputFile, 'ev chn ttag/s dtime/ps');
    end;

    Writeln('Press RETURN to start measurement');
    Readln;

    RetCode := HH500_StartMeas(Dev[0], Tacq);
    if RetCode <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_StartMeas error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('Measuring for ', Tacq, ' milliseconds...');

    if (Mode = MODE_T3) then
    begin
      // We need the sync period in order to calculate the true times of photon records.
      // This only makes sense in T3 mode and it assumes a stable period like from a laser.
      // Note: Two sync periods must have elapsed after HH500_StartMeas to get proper results.
      // You can also use the inverse of what you read via GetSyncRate but it depends on
      // the actual sync rate if this is accurate enough.
      // It is OK to use the sync input for a photon detector, e.g. if you want to perform
      // something like an antibunching measurement. In that case the sync rate obviously is
      // not periodic. This means that a) you should set the sync divider to 1 (none) and
      // b) that you cannot meaningfully measure the sync period here, which probaly won't
      // matter as you only care for the time difference (dtime) of the events.

      RetCode := HH500_GetSyncPeriod(Dev[0], SyncPeriod);
      if RetCode <> HH500_ERROR_NONE then
      begin
        Writeln('HH500_GetSyncPeriod error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      Writeln('Sync period is ', (SyncPeriod * 1e9):3:2,' ns');
    end;

    Progress := 0;
    OflCorrection := 0;

    Writeln;
    Writeln('Starting data collection...');

    repeat
      RetCode := HH500_GetFlags(Dev[0], Flags);
      if RetCode <> HH500_ERROR_NONE then
      begin
        Writeln('HH500_GetFlags error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      FiFoFull := (Flags and FLAG_FIFOFULL) > 0;

      if FiFoFull then
        Writeln('  FiFo Overrun!')
      else
      begin
        RetCode := HH500_ReadFiFo(Dev[0], Buffer[0], Records); // may return less!
        if RetCode <> HH500_ERROR_NONE then
        begin
          Writeln('HH500_ReadFiFo error ', RetCode:3, '. Aborted.');
          Ex(RetCode);
        end;

        // Here we process the data. Note that the time this consumes prevents us
        // from getting around the loop quickly for the next Fifo read.
        // In a serious performance critical scenario you would write the data to
        // a software queue and do the processing in another thread reading from
        // that queue.
        if Records > 0 then
        begin
          if Mode = Mode_T2 then
            for i := 0 to Records - 1 do
              ProcessT2(Buffer[i])
          else
            for i := 0 to Records - 1 do
              ProcessT3(Buffer[i]);
          Progress := Progress + Records;
          Write(#8#8#8#8#8#8#8#8#8, Progress:9);
        end
        else // Do the following only when there was no data
        begin
          RetCode := HH500_CTCStatus(Dev[0], CTCStatus);
          if RetCode <> HH500_ERROR_NONE then
          begin
            Writeln;
            Writeln('HH500_CTCStatus error ', RetCode:3, '. Aborted.');
            Ex(RetCode);
          end;
          TimeOut := (CTCStatus <> 0);
          if TimeOut then
          begin
            Writeln;
            Writeln('Done');
            StopTTTR;
          end;
        end;
      end;
      // Within this loop you can also read the count rates if needed.
      // Do it sparingly and use HH500_GetAllCountRates for speed.
    until FiFoFull or TimeOut or FileError;

    Writeln;

    RetCode := HH500_StopMeas(Dev[0]);
    if RetCode <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_StopMeas error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    HH500_CloseAllDevices;
    Ex(HH500_ERROR_NONE);

  except
    on E: Exception do
      Writeln(E.ClassName, ': ', E.Message);
  end;
end.

