{
  Demo access to HydraHarp 500 via HH500Lib.dll or libhh500.so
  The program performs a measurement based on hardcoded settings.
  The resulting event data is stored in a binary output file.

  Dr. Marcus Sackrow, PicoQuant, May 2025

  Note: This is a console application

  Note: At the API level the input channel numbers are indexed 0..N-1
  where N is the number of input channels the device has.

  Note: This demo writes only raw event data to the output file.
  It does not write a file header as regular .ptu files have it.

  Tested with the following compilers:

  - Delphi 12.2 (Windows)
  - Lazarus 3.0 RC1 (Windows)
  - Lazarus 2.0.8 / FPC 3.0.4 (Linux)
}

program tttrmode;

{$IF defined(MSWINDOWS)}
  {$APPTYPE CONSOLE}  // Windows needs this, Linux does not want it
{$ENDIF}

uses
  {$ifdef fpc}
  SysUtils,
  {$else}
  System.SysUtils,
  System.Ansistrings,
  {$endif}
  HH500Lib in 'HH500Lib.pas';

var
  Dev: array[0..MAXDEVNUM - 1] of LongInt;  // list of all found devices
  Device: LongInt;
  Found: Integer = 0;
  OutFile: File;
  RetCode: LongInt;
  CTCStatus: LongInt;
  strLibVers: array[0..7]  of AnsiChar;
  HW_Model: array[0..31] of AnsiChar;
  HW_Partno: array[0..8] of AnsiChar;
  HW_Serial: array[0..9] of AnsiChar;
  HW_Version: array[0..15] of AnsiChar;
  Errorstring: array[0..39] of AnsiChar;
  NumChannels: LongInt;

  //############### User Settings #########################
  Mode            : LongInt = MODE_T3; // set T2 or T3 here, observe suitable Syncdivider and Range!
  Binning         : LongInt = 0; // you can change this (meaningless in T2 mode)
  Offset          : LongInt = 0; // normally no need to change this
  Tacq            : LongInt = 1000; //  measurement time in millisec, you can change this, unit is millisec
  SyncDivider     : LongInt = 1; // you can change this

  // Dependent on the hardware model, the HydraHarp 500 may have different input
  // circuits: Edge Trigger (ETR) and/or Constant Fraction Discrimonator (CFD)
  // This can even vary from channel to channel. An input that has both ETR and CFD
  // will be programmable as to which trigger mode to use. Here we define default
  // settings for both variants and the default trigger mode ETR.
  // The latter can be changed to CFD which of course only works with channels
  // that actually have a CFD. The code further below will therfore query the
  // input capabilities and where necessary fall back to the mode available.

  // Sync Settings
  SyncChannelOffset: LongInt = 0; // in ps, you can change this (like a cable delay)
  SyncTrgMode: LongInt       = TRGMODE_ETR; // you can change this to TRGMODE_CFD (if your device supports it)

  // in case of SyncTrgMode = TRGMODE_ETR this will apply:
  SyncTrgEdge: LongInt  = EDGE_FALLING; // you can change this to EDGE_RISING
  SyncTrgLevel: LongInt = -50; // in mV, you can change this

  // in case of SyncTrgMode = TRGMODE_CFD this will apply:
  SyncCFDZeroCross: LongInt = -50; // in mV, you can change this
  SyncCFDLevel: LongInt     = -50; // in mV, you can change this
  
  // Input Settings
  InputChannelOffset: LongInt = 0;           // in ps, you can change this (like a cable delay)
  InputTrgMode: LongInt       = TRGMODE_ETR; // you can change this to TRGMODE_CFD

  // in case of InputTrgMode = TRGMODE_ETR this will apply:
  InputTrgEdge: LongInt  = EDGE_FALLING; // you can change this to EDGE_RISING
  InputTrgLevel: LongInt = -50;          // in mV, you can change this

  // in case of InputTrgMode = TRGMODE_CFD this will apply:
  InputCFDZeroCross: LongInt = -20;      // in mV, you can change this
  InputCFDLevel: LongInt     = -50;      // in mV, you can change this

procedure Ex(RetCode: Integer);
begin
  if RetCode <> HH500_ERROR_NONE then
  begin
    HH500_GetErrorString(Errorstring, RetCode);
    Writeln('Error ', RetCode:3, ' = "', Trim(string(Errorstring)), '"');
  end;
  Writeln;
  {$I-}
    CloseFile(OutFile);
    IOResult();
  {$I+}
  Writeln('Press RETURN to exit');
  Readln;
  Halt(RetCode);
end;

function SetInputModalities(DevIdx: LongInt; NumChannels: LongInt): LongInt;
var
  ChanIdx: LongInt;
  ChannelFeatures: LongWord;
  RealTrgMode: LongInt;
begin
  Result := HH500_GetSyncFeatures(DevIdx, ChannelFeatures);
  if Result <> HH500_ERROR_NONE then
  begin
    Writeln('HH500_GetSyncFeatures error ', Result:3, '. Aborted.');
    Exit;
  end;
  // check if the sync channel has the right feature for the requested setting
  if (SyncTrgMode = TRGMODE_ETR) and ((ChannelFeatures and HAS_ETR) = 0) then
  begin
    Writeln('Warning: Sync channel as no Edge Trigger, switching to CFD');
    SyncTrgMode := TRGMODE_CFD;
  end;
  if (SyncTrgMode = TRGMODE_CFD) and ((ChannelFeatures and HAS_CFD) = 0) then
  begin
    Writeln('Warning: Sync channel as no CFD, switching to Edge Trigger');
    SyncTrgMode := TRGMODE_ETR;
  end;

  Result := HH500_SetSyncTrgMode(DevIdx, SyncTrgMode);
  if Result <> HH500_ERROR_NONE then
  begin
    Writeln('HH500_SetSyncTrgMode error ', Result:3, '. Aborted.');
    Exit;
  end;

  if SyncTrgMode = TRGMODE_ETR then
  begin
    Result := HH500_SetSyncEdgeTrg(DevIdx, SyncTrgLevel, SyncTrgEdge);
    if Result <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_SetSyncEdgeTrg error ', Result:3, '. Aborted.');
      Exit;
    end;
  end;
  if SyncTrgMode = TRGMODE_CFD then
  begin
    Result := HH500_SetSyncCFD(DevIdx, SyncCFDLevel, SyncCFDZeroCross);
    if Result <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_SetSyncCFD error ', Result:3, '. Aborted.');
      Exit;
    end;
  end;

  Result := HH500_SetSyncChannelOffset(DevIdx, SyncChannelOffset);
  if Result <> HH500_ERROR_NONE then
  begin
    Writeln('HH500_SetSyncChannelOffset error ', Result:3, '. Aborted.');
    Exit;
  end;

  for ChanIdx := 0 to NumChannels - 1 do // we use the same input settings for all channels
  begin
    RealTrgMode := InputTrgMode;
    Result := HH500_GetInputFeatures(DevIdx, ChanIdx, ChannelFeatures);
    if Result <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_GetInputFeatures error ', Result:3, '. Aborted.');
      Exit;
    end;
    // check if the sync channel has the right feature for the requested setting
    if (RealTrgMode = TRGMODE_ETR) and ((ChannelFeatures and HAS_ETR) = 0) then
    begin
      Writeln('Warning: Input channel ', ChanIdx, ' has no Edge Trigger, switching to CFD');
      RealTrgMode := TRGMODE_CFD;
    end;
    if (RealTrgMode = TRGMODE_CFD) and ((ChannelFeatures and HAS_CFD) = 0) then
    begin
      Writeln('Warning: Input channel ', ChanIdx, ' has no CFD, switching to Edge Trigger');
      RealTrgMode := TRGMODE_ETR;
    end;

    Result := HH500_SetInputTrgMode(DevIdx, ChanIdx, RealTrgMode);
    if Result <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_SetInputTrgMode ', ChanIdx:2, ' error ', Result:3, '. Aborted.');
      Exit;
    end;
    if RealTrgMode = TRGMODE_ETR then
    begin
      Result := HH500_SetInputEdgeTrg(DevIdx, ChanIdx, InputTrgLevel, InputTrgEdge);
      if Result <> HH500_ERROR_NONE then
      begin
        Writeln('HH500_SetInputEdgeTrg ', ChanIdx:2, ' error ', Result:3, '. Aborted.');
        Exit;
      end;
    end;
    if RealTrgMode = TRGMODE_CFD then
    begin
      Result := HH500_SetInputCFD(DevIdx, ChanIdx, InputCFDLevel, InputCFDZeroCross);
      if Result <> HH500_ERROR_NONE then
      begin
        Writeln('HH500_SetInputCFD ', ChanIdx:2, ' error ', Result:3, '. Aborted.');
        Exit;
      end;
    end;

    Result := HH500_SetInputChannelOffset(DevIdx, ChanIdx, InputChannelOffset);
    if Result <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_SetInputChannelOffset channel ', ChanIdx:2, ' error ', Result:3, '. Aborted.');
      Exit;
    end;

    Result := HH500_SetInputChannelEnable(DevIdx, ChanIdx, 1);
    if Result <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_SetInputChannelEnable channel ', ChanIdx:2, ' error ', Result:3, '. Aborted.');
      Exit;
    end;
  end;
end;

var
  // other variables
  Resolution: Double;
  SyncRate: LongInt;
  CountRate: LongInt;
  Flags: LongInt;
  Warnings: LongInt;
  WarningsText: array[0..16343] of AnsiChar; // must have 16384 bytes of text buffer
  Records: LongInt;
  Progress: LongInt = 0;
  Written: LongInt;
  FiFoFull: Boolean =   False;
  MeasDone: Boolean =   False;
  FileError: Boolean =   False;
  ChanIdx: LongInt;
  Buffer: array[0..TTREADMAX - 1] of LongWord;

procedure StopTTTR;
begin
  RetCode := HH500_StopMeas(Dev[0]);
  if RetCode <> HH500_ERROR_NONE then
  begin
    Writeln('HH500_StopMeas error ', RetCode:3, '. Aborted.');
    Ex(RetCode);
  end;
end;

begin
  try
    Writeln;
    Writeln('HydraHarp 500 HH500Lib Demo Application             PicoQuant GmbH, 2025');
    Writeln('~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~');
    RetCode := HH500_GetLibraryVersion (strLibVers);
    if RetCode <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_GetLibraryVersion error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('HH500Lib version is ' + strLibVers);
    if Trim(AnsiString(strLibVers)) <> Trim(AnsiString(LIB_VERSION)) then
      Writeln('Warning: The application was built for version ' + LIB_VERSION);

    AssignFile(OutFile, 'tttrmode.out');
    {$I-}
      Rewrite(OutFile, 4);
    {$I+}
    if IOResult <> 0 then
    begin
      Writeln('cannot open output file');
      Ex(HH500_ERROR_NONE);
    end;

    Writeln;
    Writeln('Searching for HydraHarp 500 devices...');
    Writeln('DevIdx     Serial     Status');

    for Device := 0 to MAXDEVNUM - 1 do
    begin
      RetCode := HH500_OpenDevice(Device, HW_Serial);
      //
      if RetCode = HH500_ERROR_NONE then
      begin
        // grab any HydraHarp 500 we can open
        Writeln(Device, '          ', HW_Serial, '    open ok');
        Dev[Found] := Device; // keep index to devices we want to use
        Inc(Found);
      end
      else
      begin
        if RetCode = HH500_ERROR_DEVICE_OPEN_FAIL then
          Writeln(Device, '          ', HW_Serial, '           no device')
        else
        begin
          HH500_GetErrorString(ErrorString, RetCode);
          Writeln(Device, HW_Serial, '      ' + Trim(string(ErrorString)));
        end;
      end;
    end;

    // in this demo we will use the first HydraHarp500 device we found,
    // i.e. iDevIdx[0].  You can also use multiple devices in parallel.
    // you could also check for a specific serial number, so that you
    // always know which physical device you are talking to.

    if Found < 1 then
    begin
      Writeln('No device available.');
      Ex(HH500_ERROR_NONE);
    end;

    Writeln('Using device ', Dev[0]);
    Writeln('Initializing the device...');

    RetCode := HH500_Initialize(Dev[0], Mode, 0); // with internal clock
    if RetCode <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_Initialize error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    RetCode := HH500_GetHardwareInfo(Dev[0], HW_Model, HW_Partno, HW_Version);
    if RetCode <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_GetHardwareInfo error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end
    else
    Writeln('Found Model ', HW_Model,'  Part no ', HW_Partno,'  Version ', HW_Version);

    RetCode := HH500_GetNumOfInputChannels(Dev[0], NumChannels);
    if RetCode <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_GetNumOfInputChannels error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end
    else
    Writeln('Device has ', NumChannels, ' input channels.');
    Writeln;

    RetCode := HH500_SetSyncDiv(Dev[0], SyncDivider);
    if RetCode <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_SetSyncDiv error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    if SetInputModalities(Dev[0], NumChannels) < 0 then
      Ex(RetCode);

    if (Mode = MODE_T3) then // the following functions are meaningless in T2 mode
    begin
      RetCode := HH500_SetBinning(Dev[0], Binning);
      if RetCode <> HH500_ERROR_NONE then
      begin
        Writeln('HH500_SetBinning error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;

      RetCode := HH500_SetOffset(Dev[0], Offset);
      if RetCode <> HH500_ERROR_NONE then
      begin
        Writeln('HH500_SetOffset error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;

      RetCode := HH500_GetResolution(Dev[0], Resolution);
      if RetCode <> HH500_ERROR_NONE then
      begin
        Writeln('HH500_GetResolution error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      Writeln('Resolution is ', Resolution:7:3, 'ps');
    end;

    Writeln('Measuring input rates...');

    // after Init allow 150 ms for valid  count rate readings
    // subsequently you get new values after every 100ms
    Sleep(150);

    Writeln;

    RetCode := HH500_GetSyncRate(Dev[0], SyncRate);
    if RetCode <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_GetSyncRate error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('SyncRate = ', SyncRate, '/s');

    Writeln;

    for ChanIdx := 0 to NumChannels - 1 do // for all channels
    begin
      RetCode := HH500_GetCountRate(Dev[0], ChanIdx, CountRate);
      if RetCode <> HH500_ERROR_NONE then
      begin
        Writeln('HH500_GetCountRate error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      Writeln('Countrate [', ChanIdx:2, '] = ', CountRate:8, '/s');
    end;

    Writeln;

    RetCode := HH500_GetWarnings(Dev[0], Warnings);
    if RetCode <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_GetWarnings error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    if Warnings <> 0 then
    begin
      HH500_GetWarningsText(Dev[0], WarningsText, Warnings);
      Writeln(WarningsText);
    end;

    Writeln('Press RETURN to start');
    Readln;

    Writeln('Starting data collection...');

    RetCode := HH500_StartMeas(Dev[0], Tacq);
    if RetCode <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_StartMeas error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('Measuring for ', Tacq, ' milliseconds...');

    Progress := 0;
    Write(#8#8#8#8#8#8#8#8#8, Progress:9);

    repeat
      RetCode := HH500_GetFlags(Dev[0], Flags);
      if RetCode <> HH500_ERROR_NONE then
      begin
        Writeln('HH500_GetFlags error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      FiFoFull := (Flags and FLAG_FIFOFULL) > 0;

      if FiFoFull then
      begin
        Writeln ('  FiFo Overrun!');
        StopTTTR;
      end
      else
      begin
        RetCode := HH500_ReadFiFo(Dev[0], Buffer[0], Records); // may return less!
        if RetCode <> HH500_ERROR_NONE then
        begin
          Writeln('HH500_TTReadData error ', RetCode:3, '. Aborted.');
          Ex(RetCode);
        end;

        if Records > 0 then
        begin
          BlockWrite(OutFile, Buffer[0], Records, Written);
          if Records <> Written then
          begin
            Writeln;
            Writeln('File write error');
            FileError := True;
            StopTTTR;
          end;

          Progress := Progress + Written;
          Write(#8#8#8#8#8#8#8#8#8, Progress:9);
        end
        else // do the following only when there was no data
        begin
          RetCode := HH500_CTCStatus(Dev[0], CTCStatus);
          if RetCode <> HH500_ERROR_NONE then
          begin
            Writeln;
            Writeln('HH500_CTCStatus error ', RetCode:3, '. Aborted.');
            Ex(RetCode);
          end;
          MeasDone := (CTCStatus <> 0);
          if MeasDone then
          begin
            Writeln;
            Writeln('Done');
            StopTTTR;
          end;
        end;
      end;
      // Within this loop you can also read the count rates if needed.
      // Do it sparingly and use HH500_GetAllCountRates for speed.
    until FiFoFull or MeasDone or FileError;

    Writeln;

    RetCode := HH500_StopMeas(Dev[0]);
    if RetCode <> HH500_ERROR_NONE then
    begin
      Writeln('HH500_StopMeas error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    Writeln;

    HH500_CloseAllDevices;
    Ex(HH500_ERROR_NONE);

  except
  on E: Exception do
    Writeln(E.ClassName, ': ', E.Message);
  end;
end.

