# Demo access to HydraHarp 500 via HH500Lib.dll or libhh500.so
# The program performs a measurement based on hardcoded settings.
# The resulting histogram is stored in an ASCII output file.

# Stefan Eilers, Michael Wahl, Markus Goetz, PicoQuant, August 2025

# Note: This is a console application.

# Note: At the API level the input channel numbers are indexed 0..N-1
# where N is the number of input channels the device has.

# Tested with the following compilers:
# - Python 3.11 (Windows)

import time
import ctypes as ct
from ctypes import byref
import os
import sys

if sys.version_info[0] < 3:
    print("[Warning] Python 2 is not fully supported. It might work, but "
          "use Python 3 if you encounter errors.\n")
    raw_input("press RETURN to continue"); print

if os.name == "nt":
    hhlib = ct.WinDLL("HH500Lib.dll")
else:
    hhlib = ct.CDLL("libhh500lib.so")


# Constants from hh500defin.h, do not edit these!

LIB_VERSION = "1.0"
MAXDEVNUM = 8
MODE_HIST = 0
MAXLENCODE = 7
MAXINPCHAN = 16
MAXHISTLEN = 131072
FLAG_OVERFLOW = 0x0001
TRGMODE_ETR = 0 # edge trigger
TRGMODE_CFD = 1 # constant fraction discriminator
EDGE_FALLING = 0
EDGE_RISING = 1

# bitmasks for results from HH500_GetSyncFeatures and HH500_GetInputFeatures
HAS_ETR = 1           # this channel has an edge trigger
HAS_CFD = 2           # this channel has a constant fraction discriminator
# note: dependent on the hardware model an input can have both

# Measurement parameters, hard coded here since this is just a demo

binning = 0 # You can change this
offset = 0 # You can change this, normally zero, see manual
tacq = 1000 # Measurement time in millisec, you can change this
sync_divider = 1 # You can change this
sync_channel_offset = -5000 # You can change this (like a cable delay)
input_channel_offset = 0 # You can change this (like a cable delay)

"""
Dependent on the hardware model, the HydraHarp 500 may have different input
circuits: Edge Trigger (ETR) and/or Constant Fraction Discrimonator (CFD)
This can even vary from channel to channel. An input that has both ETR and CFD
will be programmable as to which trigger mode to use. Here we define default
settings for both variants and the default trigger mode ETR.
The latter can be changed to CFD which of course only works with channels
that actually have a CFD. The code further below will therfore query the
input capabilities and where necessary fall back to the mode available.
"""
sync_trg_mode = TRGMODE_ETR # You can change this to TRGMODE_CFD
# In case of SyncTrgMode is TRGMODE_ETR this will apply:
sync_trg_edge = EDGE_FALLING # You can change this to EDGE_RISING
sync_trg_level = -50 # You can change this (in mV)
# In case of SyncTrgMode is TRGMODE_CFD this will apply:
sync_cfd_zero_cross = -20 # In mV, you can change this
sync_cfd_level = -50 # In mV, you can change this

input_trg_mode = TRGMODE_ETR # You can change this to TRGMODE_CFD
# In case of InputTrgMode is TRGMODE_ETR this will apply:
input_trg_edge = EDGE_FALLING # You can change this to EDGE_RISING
input_trg_level = -50 # In mV, you can change this
# In case of InputTrgMode is TRGMODE_CFD this will apply:
input_cfd_zero_cross = -20 # In mV, you can change this
input_cfd_level = -50 # In mV, you can change this

# Variables for flow contol and results from library calls

dev_idx = []
ctc_status = ct.c_int()
lib_version = ct.create_string_buffer(b"", 8)
hw_model = ct.create_string_buffer(b"", 32)
hw_partno = ct.create_string_buffer(b"", 8)
hw_serial = ct.create_string_buffer(b"", 32)
hw_version = ct.create_string_buffer(b"", 16)
error_string = ct.create_string_buffer(b"", 40)
debug_info_puffer = ct.create_string_buffer(b"", 16384)
num_channels = ct.c_int()
resolution = ct.c_double()
sync_rate = ct.c_int()
count_rate = ct.c_int()
flags = ct.c_int()
warnings = ct.c_int()
warnings_text = ct.create_string_buffer(b"", 16384)
hist_len = ct.c_int()
integral_count = ct.c_double()
cmd = 0
counts = [(ct.c_uint * MAXHISTLEN)() for i in range(0, MAXINPCHAN)]

# Subroutines

def closeDevices():
    histogramfile.close()
    for i in range(0, MAXDEVNUM):
        hhlib.HH500_CloseDevice(ct.c_int(i))
    if sys.version_info[0] < 3:
        raw_input("\nPress RETURN to exit"); print
    else:
        input("\nPress RETURN to exit"); print
    sys.exit(0)

# Helper for showing error texts and shutdown in case of error
def tryfunc(ret_code, func_name):
    if ret_code < 0:
        hhlib.HH500_GetErrorString(error_string, ret_code)
        print("HH500_%s error %d (%s). Aborted." % (func_name, ret_code,
              error_string.value.decode("utf-8")))
        closeDevices()

def SetInputModalities(dev_idx: int, num_channels: int) -> None:
    global sync_trg_mode
    channel_features = ct.c_uint()
    tryfunc(hhlib.HH500_GetSyncFeatures(ct.c_int(dev_idx), byref(channel_features)),
            "GetSyncFeatures")
    # check if the sync channel has the right feature for the requested setting
    if (sync_trg_mode == TRGMODE_ETR) and ((channel_features.value & HAS_ETR) == 0):
        print("Warning: Sync channel has no Edge Trigger, switching to CFD")
        sync_trg_mode = TRGMODE_CFD
    if (sync_trg_mode == TRGMODE_CFD) and ((channel_features & HAS_CFD) == 0):
        print("Warning: Sync channel has no CFD, switching to Edge Trigger")
        sync_trg_mode = TRGMODE_ETR
    tryfunc(hhlib.HH500_SetSyncTrgMode(ct.c_int(dev_idx), ct.c_int(sync_trg_mode)),
            "SetSyncTrgMode")
    if sync_trg_mode == TRGMODE_ETR:
        tryfunc(hhlib.HH500_SetSyncEdgeTrg(ct.c_int(dev_idx), ct.c_int(sync_trg_level), ct.c_int(sync_trg_edge)),
                "SetSyncEdgeTrg")
    if sync_trg_mode == TRGMODE_CFD:
        tryfunc(hhlib.HH500_SetSyncCFD(ct.c_int(dev_idx), ct.c_int(sync_cfd_level), ct.c_int(sync_cfd_zero_cross)),
                "SetSyncCFD")
    tryfunc(hhlib.HH500_SetSyncChannelOffset(ct.c_int(dev_idx), ct.c_int(sync_channel_offset)),
            "SetSyncChannelOffset")
    # We use the same input settings for all channels, you can change this
    for chan_idx in range(num_channels):
        real_trg_mode = input_trg_mode
        tryfunc(hhlib.HH500_GetInputFeatures(ct.c_int(dev_idx), ct.c_int(chan_idx), byref(channel_features)),
                "GetInputFeatures channel %d" % (chan_idx))
        # check if the channel has the right feature for the requested setting
        if (real_trg_mode == TRGMODE_ETR) and ((channel_features.value & HAS_ETR) == 0):
            print("Warning: Input channel %d has no Edge Trigger, switching to CFD" % (chan_idx))
            real_trg_mode = TRGMODE_CFD
        if (real_trg_mode == TRGMODE_CFD) and ((channel_features.value & HAS_CFD) == 0):
            print("Warning: Input channel %d has no CFD, switching to Edge Trigger" % (chan_idx))
            real_trg_mode = TRGMODE_ETR
        tryfunc(hhlib.HH500_SetInputTrgMode(ct.c_int(dev_idx), ct.c_int(chan_idx), ct.c_int(real_trg_mode)),
                "SetInputTrgMode channel %d" % (chan_idx))
        if real_trg_mode == TRGMODE_ETR:
            tryfunc(hhlib.HH500_SetInputEdgeTrg(ct.c_int(dev_idx), ct.c_int(chan_idx), ct.c_int(input_trg_level), ct.c_int(input_trg_edge)),
                    "SetInputEdgeTrg channel %d" % (chan_idx))
        if real_trg_mode == TRGMODE_CFD:
            tryfunc(hhlib.HH500_SetInputCFD(ct.c_int(dev_idx), ct.c_int(chan_idx), ct.c_int(input_cfd_level), ct.c_int(input_cfd_zero_cross)),
                    "SetInputCFD %d" % (chan_idx))
        tryfunc(hhlib.HH500_SetInputChannelOffset(ct.c_int(dev_idx), ct.c_int(chan_idx), ct.c_int(input_channel_offset)),
                "SetInputChannelOffset channel %d")
        tryfunc(hhlib.HH500_SetInputChannelEnable(ct.c_int(dev_idx), ct.c_int(chan_idx), ct.c_int(1)),
                "SetInputChannelEnable channel %d" % (chan_idx))


# The main demo code starts here

print("\nHydraHarp 500 HH500Lib Demo Application               PicoQuant GmbH, 2025")
print("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~")

hhlib.HH500_GetLibraryVersion(lib_version)
print("Library version is %s" % lib_version.value.decode("utf-8"))
if lib_version.value.decode("utf-8") != LIB_VERSION:
    print("Warning: The application was built for version %s" % LIB_VERSION)

histogramfile = open("histomodeout.txt", "w+")

print("\nSearching for HydraHarp 500 devices...")
print("Devidx     Serial        Status")

for i in range(0, MAXDEVNUM):
    ret_code = hhlib.HH500_OpenDevice(ct.c_int(i), hw_serial)
    if ret_code == 0:
        print("  %1d        %s       open ok" % (i, hw_serial.value.decode("utf-8")))
        dev_idx.append(i)
    else:
        if ret_code == -1: # HH500_ERROR_DEVICE_OPEN_FAIL
            print("  %1d        %s              no device" % (i, hw_serial.value.decode("utf-8")))
        else:
            hhlib.HH500_GetErrorString(error_string, ct.c_int(ret_code))
            print("  %1d        %s       %s" % (i, error_string.value.decode("utf8"), error_string))

# In this demo we will use the first device we found,
# i.e. DevIdx[0]. You can also use multiple devices in parallel, see the manual
# You could also check for a specific serial number, so that you
# always know which physical device you are talking to.

if len(dev_idx) < 1:
    print("No device available.")
    closeDevices()
print("Using device #%1d" % dev_idx[0])
print("\nInitializing the device...")

# Initialize for histo mode with internal clock
tryfunc(hhlib.HH500_Initialize(ct.c_int(dev_idx[0]), ct.c_int(MODE_HIST), ct.c_int(0)),
        "Initialize")

# Only for information
tryfunc(hhlib.HH500_GetHardwareInfo(dev_idx[0], hw_model, hw_partno, hw_version),
        "GetHardwareInfo")
print("Found Model %s Part no %s Version %s" % (hw_model.value.decode("utf-8"),
      hw_partno.value.decode("utf-8"), hw_version.value.decode("utf-8")))

tryfunc(hhlib.HH500_GetNumOfInputChannels(ct.c_int(dev_idx[0]), byref(num_channels)),
        "GetNumOfInputChannels")
print("Device has %i input channels." % num_channels.value)

tryfunc(hhlib.HH500_SetSyncDiv(ct.c_int(dev_idx[0]), ct.c_int(sync_divider)), "SetSyncDiv")

SetInputModalities(dev_idx[0], num_channels.value)

tryfunc(hhlib.HH500_SetBinning(ct.c_int(dev_idx[0]), ct.c_int(binning)), "SetBinning")

tryfunc(hhlib.HH500_SetOffset(ct.c_int(dev_idx[0]), ct.c_int(offset)), "SetOffset")

tryfunc(hhlib.HH500_SetHistoLen(ct.c_int(dev_idx[0]), ct.c_int(MAXLENCODE), byref(hist_len)),
        "SetHistoLen")
print("Histogram length is %d" % hist_len.value)

tryfunc(hhlib.HH500_GetResolution(ct.c_int(dev_idx[0]), byref(resolution)), "GetResolution")
print("Resolution is %1.1lfps" % resolution.value)

# Note: after Init or SetSyncDiv you must allow >100 ms for valid count rate readings
time.sleep(0.15)

tryfunc(hhlib.HH500_GetSyncRate(ct.c_int(dev_idx[0]), byref(sync_rate)), "GetSyncRate")
print("\nSyncrate=%1d/s" % sync_rate.value)

for i in range(0, num_channels.value):
    tryfunc(hhlib.HH500_GetCountRate(ct.c_int(dev_idx[0]), ct.c_int(i), byref(count_rate)),
            "GetCountRate")
    print("Countrate[%1d]=%1d/s" % (i, count_rate.value))

# After getting the count rates you can check for warnings
tryfunc(hhlib.HH500_GetWarnings(ct.c_int(dev_idx[0]), byref(warnings)), "GetWarnings")
if warnings.value != 0:
    tryfunc(hhlib.HH500_GetWarningsText(ct.c_int(dev_idx[0]), warnings_text, warnings),
            "GetWarningsText")
    print("\n\n%s" % warnings_text.value.decode("utf-8"))

tryfunc(hhlib.HH500_SetStopOverflow(ct.c_int(dev_idx[0]), ct.c_int(0), ct.c_int(10000)),
        "SetStopOverflow") # for example only

while cmd != "q":
    tryfunc(hhlib.HH500_ClearHistMem(ct.c_int(dev_idx[0])), "ClearHistMem")

    input("\nPress RETURN to start measurement\n")
    print

    tryfunc(hhlib.HH500_GetSyncRate(ct.c_int(dev_idx[0]), byref(sync_rate)), "GetSyncRate")
    print("Syncrate=%1d/s" % sync_rate.value)

    for i in range(0, num_channels.value):
        tryfunc(hhlib.HH500_GetCountRate(ct.c_int(dev_idx[0]), ct.c_int(i), byref(count_rate)),
                "GetCountRate")
        print("Countrate[%1d]=%1d/s" % (i, count_rate.value))

    # Here you could check for warnings again

    tryfunc(hhlib.HH500_StartMeas(ct.c_int(dev_idx[0]), ct.c_int(tacq)), "StartMeas")
    print("\nMeasuring for %1d milliseconds...\n" % tacq)

    ctc_status = ct.c_int(0)
    while ctc_status.value == 0:
        tryfunc(hhlib.HH500_CTCStatus(ct.c_int(dev_idx[0]), byref(ctc_status)),
                "CTCStatus")

    tryfunc(hhlib.HH500_StopMeas(ct.c_int(dev_idx[0])), "StopMeas")

    for i in range(0, num_channels.value):

        tryfunc(hhlib.HH500_GetHistogram(ct.c_int(dev_idx[0]), byref(counts[i]),
                ct.c_int(i)), "GetHistogram")

        integral_count = 0
        for j in range(0, hist_len.value):
            integral_count += counts[i][j]

        print("  Integralcount[%1d]=%1.0lf" % (i, integral_count))

    tryfunc(hhlib.HH500_GetFlags(ct.c_int(dev_idx[0]), byref(flags)), "GetFlags")

    if flags.value & FLAG_OVERFLOW > 0:
        print("  Overflow.")

    print("\nEnter c to continue or q to quit and save the count data.")
    cmd = input()

print("\nSaving data to file...")

for i in range(0, num_channels.value):
    histogramfile.write("    Ch%1d " % (i + 1))
histogramfile.write("\n")

for j in range(0, hist_len.value):
    for i in range(0, num_channels.value):
        histogramfile.write("%7u " % counts[i][j])
    histogramfile.write("\n")

print("Saved.\n")

closeDevices()

