/*********************************************************************
 * PicoQuant HH500 Rust Demo
 *
 * This file demonstrates how to wrap the PicoQuant Hydraharp 500 DLL using Rust.
 * Its based on the histomode C demo, but in a more rustanian way.
 *
 * The example code opens and configures the device, performs a short
 * measurement, and writes the raw data to a file.
 *
 * Each function from the C DLL is wrapped with Rust data types. Access to
 * the C DLL functions and C data types is marked as unsafe; they are used
 * briefly before converting to Rust code.
 *
 * For more information, please refer to:
 * https://doc.rust-lang.org/book/ch19-01-unsafe-rust.html
 *
 * Using the Rust approach, all functions return a Result type, which can be
 * either Ok or Err. The Ok variant contains the return value of the specific
 * function, enforcing error handling for any potential issues.
 *
 * For quick prototyping, it is common to use the unwrap() function, which either
 * returns the value from the function or panics. This is used frequently in the
 * code below for better readability. For production code, it is recommended to handle
 * all possible failure cases.
 *
 * Example:
 *
 * hh500_function(arg1, arg2).unwrap();
 *
 * or for production code:
 *
 * match hh500_function(arg1, arg2) {
 *     Ok(()) => handle_ok(),
 *     Err(_) => handle_err(),
 * }
 *
 *********************************************************************/

use std::fs::File;
use std::io::Write;
use std::thread::sleep;
use std::time::Duration;

use hh500_rust_integration::hh500::{self, TriggerMode};
use hh500_rust_integration::hh500::Edge;

fn main() {
    println!("~~~~~~~~~~~~~~~~~~~~~~~~");
    println!("~~~ HH500 Rust Demo  ~~~");
    println!("~~~     Histomode    ~~~");
    println!("~~~~~~~~~~~~~~~~~~~~~~~~");

    // Set basic decvice settings
    let devidx: i32 = 0;
    let mode: hh500::Mode = hh500::Mode::Hist;
    let ref_source: hh500::Refsource = hh500::Refsource::Internal;

    const ACQUISITION_TIME_MS: i32 = 5000;
    const TRIGGER_EDGE: Edge = hh500::Edge::Falling;
    const SYNC_TRIGGER_LEVEL: i32 = -50; //mV
    const CHAN_TRIGGER_LEVEL: i32 = -50; //mV
    const CHANNEL_IDX: i32 = -1; //-1 for all channels
    const TRIGGER_MODE : TriggerMode = TriggerMode::ETR;
    let output_file_path: &str = "histograms.txt";

    // in Rust, files get automatically closed when it goes out of scope
    let mut file = File::create(output_file_path).unwrap();
    
    // get lib version
    let r_str = hh500::get_library_version();
    println!("Library version: {:?}", r_str);
    
    // open device, based on its index and print out its serial number
    let serial = hh500::open_device(devidx);
    println!("Serial number = {:?}", serial);
    
    
    match hh500::initialize(devidx, mode, ref_source) {
        Ok(()) => println!("Initialization Done!"),
        Err(_) => println!("Initialization not possible!"),
    }
    
    // retreive hardware info and print it out
    let info = hh500::get_hardware_info(devidx);
    match info {
        Ok(info) => println!(
            "HH hardware info: Model: {:?}, Part Number: {:?}, Version: {:?}",
            info.model, info.part_number, info.version
        ),
        Err(_) => println!("Failed to get hardware info."),
    }

    match hh500::get_base_resolution(devidx) {
        Ok((steps, resolution)) => {
            println!("Device base resolution steps: {}", steps);
            println!("Device base resolution: {} ps", resolution);
        }
        Err(error_code) => {
            eprintln!(
                "Failed to get base resolution. Error code: {:?}",
                error_code
            );
        }
    }

    match hh500::get_features(devidx) {
        Ok(features) => {
            println!("Device features: {}", features);
        }
        Err(error_code) => {
            eprintln!(
                "Failed to get device features. Error code: {:?}",
                error_code
            );
        }
    }

    let nchannels: i32 = hh500::get_number_of_input_channels(devidx).unwrap();
    println!("Number of input channels: {}", nchannels);

    // setup measurement
    hh500::set_sync_edge_trigger(devidx, SYNC_TRIGGER_LEVEL, TRIGGER_EDGE).unwrap();

    hh500::set_input_edge_trigger(devidx, CHANNEL_IDX, CHAN_TRIGGER_LEVEL, TRIGGER_EDGE)
        .unwrap();

    hh500::set_sync_trg_mode(devidx, TRIGGER_MODE).unwrap();
    hh500::set_input_trg_mode(devidx, CHANNEL_IDX, TRIGGER_MODE).unwrap();

    // After Init allow 150 ms for valid  count rate readings
    sleep(Duration::from_millis(150));

    let (sync_rate, channel_rates) = hh500::get_all_count_rates(devidx).unwrap();
    println!("Syncrate =  {:?}", sync_rate);
    println!("Countrate CH =  {:?}", channel_rates);

    println!("\nCheck for warnings:");
    let warning : i32 = hh500::get_warnings(devidx).unwrap();
    println!("{}", hh500::get_warnings_text(devidx, warning).unwrap() );

    // hh500::set_histo_len(devidx, hh500::MAXLENCODE).unwrap();
    hh500::clear_hist_mem(devidx).unwrap();
    // start real measurement
    hh500::start_meas(devidx, ACQUISITION_TIME_MS).unwrap();

    let mut measurement_done = false;

    while !measurement_done {
        // check if fifo has not overrun
        let flags = hh500::get_flags(devidx).unwrap();
        if 0 != (hh500::Flags::FlagFifoFull as i32 & flags) {
            panic!("Fifo must never overrun!!!");
        }

        let ctc_status = hh500::get_ctc_status(devidx).unwrap();
        if ctc_status != 0 {
            measurement_done = true; // measurement time is over
        }
    }
    // stop the measurement
    hh500::stop_meas(devidx).unwrap();

    // getting histograms by copy, if you need more performence consider using get_all_histograms_mut
    let histograms: Vec<Vec<u32>> = hh500::get_all_histograms(devidx).unwrap();

    // Write header: x0,x1,...,x13
    for i in 0..=histograms.len() {
        if i < histograms.len() {
            write!(file, "CH{} ", i).unwrap();
        } else {
            writeln!(file, "CH{}", i).unwrap();
        }
    }
    // Iterate through the outer Vec
    let num_rows = histograms[0].len();

    for i in 0..num_rows {
        let line = histograms.iter()
            .map(|hist| hist[i].to_string())
            .collect::<Vec<_>>()
            .join(" ");
        writeln!(file, "{line}").unwrap();
    }

    // ensure all data is written to file, non is in any buffer
    file.flush().unwrap();

    let mut events_captured: u32 = 0;
    for hist in &histograms{
        for elem in hist{
            events_captured += elem;
        }
    }

    println!("{} Events captured in histogram!", events_captured);

    
    // close device at the end //
    hh500::close_device(devidx).unwrap();


}
