/*********************************************************************
 * PicoQuant HH500 Rust Crate
 *
 * This file demonstrates wrapping the PicoQuant HH500 DLL in Rust.
 * Each function from the C DLL is wrapped and provided with Rust data types.
 * The access to the C DLL functions is unsafe, and C data types are used
 * briefly before converting them to Rust code.
 *
 * For more information, please refer to:
 * https://doc.rust-lang.org/book/ch19-01-unsafe-rust.html
 *
 * Using the Rust approach, all functions return a Result type, which can be
 * either Ok or Err. The Ok variant contains the return value of the specific function.
 * This enforces error handling for any potential issues when calling the functions.
 *
 * For quick prototyping, you can use the unwrap() function, which either returns the
 * value from the function or panics. Here’s an example of how to use it:
 *
 * function(arg1, arg2).unwrap();
 *
 * This is a common method in Rust for writing prototyping code.
 *********************************************************************/

extern crate libc;

use libc::{
    c_uint, 
    c_int, 
    c_char,
    c_double
};
use core::fmt;
use std::ffi::CStr;
use std::ffi::CString;
use crate::error_codes::ErrorCode;
use std::convert::TryFrom;
use std::sync::Mutex;

// Declare the C function
unsafe extern "C" {
    fn HH500_GetLibraryVersion(vers: *mut c_char) -> i32;
    fn HH500_GetErrorString(errstring: *mut c_char, errcode: i32) -> i32;

    fn HH500_OpenDevice(devidx: i32, serial: *mut c_char) -> i32;
    fn HH500_CloseDevice(devidx: i32) -> i32;
    fn HH500_Initialize(devidx: i32, mode: i32, refsource: i32) -> i32;

    fn HH500_GetHardwareInfo(devidx: i32, model: *mut c_char, partno: *mut c_char, version: *mut c_char, ) -> i32;
    fn HH500_GetSerialNumber(devidx: i32, serial: *mut c_char) -> i32;
    fn HH500_GetFeatures(devidx: i32, features: *mut c_int) -> i32;
    fn HH500_GetBaseResolution(devidx: i32, resolution: *mut c_double, binsteps: *mut c_int) -> i32;
    fn HH500_GetNumOfInputChannels(devidx: i32, nchannels: *mut c_int) -> i32;

    fn HH500_SetSyncDiv(devidx: i32, div: i32) -> i32;
    fn HH500_SetSyncTrgMode(devidx: i32, mode: i32) -> i32;
    fn HH500_SetSyncEdgeTrg(devidx: i32, level: i32, edge: i32) -> i32;
    fn HH500_SetSyncCFD(devidx: i32, level: i32, zerocross: i32) -> i32;
    fn HH500_SetSyncChannelOffset(devidx: i32, offset: i32) -> i32;
    fn HH500_SetSyncChannelEnable(devidx: i32, enable: i32) -> i32;
    fn HH500_SetSyncDeadTime(devidx: i32, on: i32, deadtime: i32) -> i32;

    fn HH500_SetInputTrgMode(devidx: i32, channel: i32, mode: i32) -> i32;
    fn HH500_SetInputCFD(devidx: i32, channel: i32, level: i32, zerocross: i32) -> i32;
    fn HH500_SetInputEdgeTrg(devidx: i32, channel: i32, level: i32, edge: i32) -> i32;
    fn HH500_SetInputChannelOffset(devidx: i32, channel: i32, offset: i32) -> i32;
    fn HH500_SetInputDeadTime(devidx: i32, channel: i32, on: i32, deadtime: i32) -> i32;
    fn HH500_SetInputHysteresis(devidx: i32, hystcode: i32) -> i32;
    fn HH500_SetInputChannelEnable(devidx: i32, channel: i32, enable: i32) -> i32;

    fn HH500_SetStopOverflow(devidx: i32, stop_ovfl: i32, stopcount: u32) -> i32;
    fn HH500_SetBinning(devidx: i32, binning: i32) -> i32;
    fn HH500_SetOffset(devidx: i32, offset: i32) -> i32;
    fn HH500_SetHistoLen(devidx: i32, lencode: i32, actuallen: *mut i32) -> i32;
    fn HH500_SetMeasControl(devidx: i32, control: i32, startedge: i32, stopedge: i32) -> i32;
    fn HH500_SetTriggerOutput(devidx: i32, period: i32) -> i32;

    fn HH500_ClearHistMem(devidx: i32) -> i32;
    fn HH500_StartMeas(devidx: i32, tacq: i32) -> i32;
    fn HH500_StopMeas(devidx: i32) -> i32;
    fn HH500_CTCStatus(devidx: i32, ctcstatus: *mut i32) -> i32;

    fn HH500_GetHistogram(devidx: i32, chcount: *mut u32, channel: i32) -> i32;
    fn HH500_GetAllHistograms(devidx: i32, chcount: *mut u32) -> i32;
    fn HH500_GetResolution(devidx: i32, resolution: *mut f64) -> i32;
    fn HH500_GetSyncPeriod(devidx: i32, period: *mut f64) -> i32;
    fn HH500_GetSyncFeatures(devidx: i32, features: *mut u32) -> i32;
    fn HH500_GetSyncRate(devidx: i32, syncrate: *mut i32) -> i32;
    fn HH500_GetCountRate(devidx: i32, channel: i32, cntrate: *mut i32) -> i32;
    fn HH500_GetAllCountRates(devidx: i32, syncrate: *mut i32, cntrates: *mut i32) -> i32;
    fn HH500_GetFlags(devidx: i32, flags: *mut i32) -> i32;
    fn HH500_GetElapsedMeasTime(devidx: i32, elapsed: *mut f64) -> i32;
    fn HH500_GetInputFeatures(devidx: i32, channel: i32, features: *mut u32) -> i32;
    fn HH500_GetStartTime(devidx: i32, timedw2: *mut u32, timedw1: *mut u32, timedw0: *mut u32) -> i32;

    fn HH500_GetWarnings(devidx: i32, warnings: *mut i32) -> i32;
    fn HH500_GetWarningsText(devidx: i32, text: *mut i8, warnings: i32) -> i32;

    fn HH500_SetOflCompression(devidx: i32, holdtime: i32) -> i32;
    fn HH500_SetMarkerHoldoffTime(devidx: i32, holdofftime: i32) -> i32;
    fn HH500_SetMarkerEdges(devidx: i32, me1: i32, me2: i32, me3: i32, me4: i32) -> i32;
    fn HH500_SetMarkerEnable(devidx: i32, en1: i32, en2: i32, en3: i32, en4: i32) -> i32;
    fn HH500_ReadFiFo(devidx: i32, buffer: *mut u32, nactual: *mut i32) -> i32;

    fn HH500_GetDebugInfo(devidx: i32, debuginfo: *mut i8) -> i32;
    fn HH500_GetNumOfModules(devidx: i32, nummod: *mut i32) -> i32;
    fn HH500_GetModuleInfo(devidx: i32, modidx: i32, modelcode: *mut i32, versioncode: *mut i32) -> i32;
    fn HH500_SaveDebugDump(devidx: i32, filepath: *mut c_char) -> i32;
    
    fn HH500_WRabbitGetMAC(devidx: i32, mac_addr: *mut c_char) -> i32;
    fn HH500_WRabbitSetMAC(devidx: i32, mac_addr: *mut u8) -> i32;
    fn HH500_WRabbitGetInitScript(devidx: i32, initscript: *mut c_char) -> i32;
    fn HH500_WRabbitSetInitScript(devidx: i32, initscript: *mut c_char) -> i32;
    fn HH500_WRabbitGetSFPData(devidx: i32, sfpnames: *mut c_char, dTxs: *mut i32, dRxs: *mut i32, alphas: *mut i32) -> i32;
    fn HH500_WRabbitSetSFPData(devidx: i32, sfpnames: *mut c_char, dTxs: *mut i32, dRxs: *mut i32, alphas: *mut i32) -> i32;
    fn HH500_WRabbitInitLink(devidx: i32, link_on: i32) -> i32;
    fn HH500_WRabbitSetMode(devidx: i32, bootfromscript: i32, reinit_with_mode: i32, mode: i32) -> i32;
    fn HH500_WRabbitSetTime(devidx: i32, timehidw: u32, timelodw: u32) -> c_int;
    fn HH500_WRabbitGetTime(devidx: i32, timehidw: *mut c_uint, timelodw: *mut c_uint, subsec16ns: *mut c_uint) -> i32;
    fn HH500_WRabbitGetStatus(devidx: i32, wrstatus: *mut i32) -> i32;
    fn HH500_WRabbitGetTermOutput(devidx: i32, buffer: *mut c_char, nchar: *mut i32) -> i32;
    
    fn HH500_ExtFPGAInitLink(devidx: i32, linknumber: i32, on: i32) -> i32;
    fn HH500_ExtFPGAGetLinkStatus(devidx: i32, linknumber: i32, status: *mut c_uint) -> i32;
    fn HH500_ExtFPGASetMode(devidx: i32, mode: i32, loopback: i32) -> i32;
    fn HH500_ExtFPGAResetStreamFifos(devidx: i32) -> i32;
    fn HH500_ExtFPGAUserCommand(devidx: i32, write: i32, addr: c_uint, data: *mut c_uint) -> i32;

    fn HH500_SetMainEventFilterParams(devidx: i32, timerange: i32, matchcnt: i32, inverse: i32) -> i32;
    fn HH500_SetMainEventFilterChannels(devidx: i32, rowidx: i32, usechannels: i32, passchannels: i32) -> i32;
    fn HH500_SetFilterTestMode(devidx: i32, testmode: i32) -> i32;
    fn HH500_EnableMainEventFilter(devidx: i32, enable: i32) -> i32;
    fn HH500_GetMainFilteredRates(devidx: i32, syncrate: *mut i32, cntrates: *mut i32) -> i32;

    fn HH500_GetContModeBlock(devidx: i32, buffer: *mut core::ffi::c_void, nbytesreceived: *mut i32) -> i32;
    fn HH500_DissectContModeBlkHdr(blockbuffer: *const core::ffi::c_void, nchannels: *mut u32, histlen: *mut u32, blocknum: *mut u32, starttime: *mut u64, ctctime: *mut u64, markertimes: *mut u64, markercounts: *mut u32) -> i32;
    fn HH500_ExtractContModeBlkData(blockbuffer: *const core::ffi::c_void, chanindex: u32, histosum: *mut u64, histogram: *mut u32) -> i32;
}

pub enum Mode {
    Hist = 0,
    T2 = 2,
    T3 = 3,
}

pub enum Refsource {
    Internal = 0,
    External10mhz = 1,
    External100mhz = 2,
    External500mhz = 3,
}

pub enum TriggerMode {
    ETR = 0,
    CFD = 1
}


pub struct HardwareInfo {
    pub model: String,
    pub part_number: String,
    pub version: String,
}

#[derive(Default)] // implements default values
pub struct WrSfpData {
    pub name : String,
    pub dtx : i32,
    pub drx : i32, 
    pub alpha : i32
}

impl fmt::Display for WrSfpData {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        write!(f, "{} : {} {} {}", self.name, self.dtx, self.drx, self.alpha)
    }
}

pub enum MeasCtrl {
    MeasctrlSingleshotCtc = 0,
    MeasctrlC1Gated = 1,
    MeasctrlC1StartCtcStop = 2,
    MeasctrlC1StartC2Stop = 3,
    MeasctrlSWStartSwStop = 6,
}

pub enum Flags {
    FlagOverflow = 0x0001, // histo mode only
    FlagFifoFull = 0x0002, // TTTR mode only
    FlagSyncLost = 0x0004,
    FlagRefLost = 0x0008,
    FlagSysError = 0x0010,    // hardware error, must contact support
    FlagActive = 0x0020,      // measurement is running
    FlagCntsDropped = 0x0040, // counts were dropped
    FlagSoftError = 0x0080,   // software error, must contact support
}

//bitmasks for results from HH500_GetWarnings
pub enum Warnings {
    WarningSyncRateZero = 0x0001,
    WarningSyncRateVeryLow = 0x0002,
    WarningSyncRateTooHigh = 0x0004,
    WarningInptRateZero = 0x0010,
    WarningInptRateTooHigh = 0x0040,
    WarningInptRateRatio = 0x0100,
    WarningDividerGreaterOne = 0x0200,
    WarningTimeSpanTooSmall = 0x0400,
    WarningOffsetUnnecessary = 0x0800,
    WarningDividerTooSmall = 0x1000,
    WarningCountsDropped = 0x2000,
    WarningUsb20SpeedOnly = 0x4000,
}


pub enum Edge {
    Rising = 1,
    Falling = 0,
}

// limits
pub const MAXDEVNUM: i32 = 8; // max number of USB devices
pub const MAXINPCHAN: i32 = 2; // max number of physical input channels
pub const BINSTEPSMAX: i32 = 24; // max number of binning steps;
                                 // get actual number via HH500_GetBaseResolution()

pub const MAXHISTLEN: i32 = 524288; // max number of histogram bins
pub const DFLTHISTLEN: i32 = 65536; // default number of histogram bins

pub const TTREADMAX: i32 = 1048576; // number of event records that can be read by HH500_ReadFiFo
pub const TRGLVLMIN: i32 = -1500; // mV
pub const TRGLVLMAX: i32 = 1500; // mV

//limits for HH500_SetSyncChannelOffset and HH500_SetInputChannelOffset
pub const CHANOFFSMIN: i32 = -99999; // ps
pub const CHANOFFSMAX: i32 = 99999; // ps

//limits for HH500_SetSyncDeadTime and HH500_SetInputDeadTime
pub const EXTDEADMIN: i32 = 800; // ps
pub const EXTDEADMAX: i32 = 160000; // ps

//limits for HH500_SetOffset
pub const OFFSETMIN: i32 = 0; // ns
pub const OFFSETMAX: i32 = 100000000; // ns

//limits for HH500_StartMeas
pub const ACQTMIN: i32 = 1; // ms
pub const ACQTMAX: i32 = 360000000; // ms  (100*60*60*1000ms = 100h)

//limits for HH500_SetStopOverflow
pub const STOPCNTMIN: i32 = 1;
pub const STOPCNTMAX: u32 = 4294967295; // 32 bit is mem max

//limits for HH500_SetTriggerOutput
pub const TRIGOUTMIN: i32 = 0; // 0=off
pub const TRIGOUTMAX: i32 = 16777215; // in units of 100ns

//limits for HH500_SetMarkerHoldoffTime
pub const HOLDOFFMIN: i32 = 0; // ns
pub const HOLDOFFMAX: i32 = 25500; // ns

//limits for HH500_SetInputHysteresis
pub const HYSTCODEMIN: i32 = 0; // approx. 3mV
pub const HYSTCODEMAX: i32 = 1; // approx. 35mV

//limits for HH500_SetOflCompression
pub const HOLDTIMEMIN: i32 = 0; // ms
pub const HOLDTIMEMAX: i32 = 255; // ms

//limits for HH500_SetEventFilterParams and HH500_SetEventFilterChannels
pub const MATCHCNTMIN: i32 = 1;
pub const MATCHCNTMAX: i32 = 6;
pub const INVERSEMIN: i32 = 0;
pub const INVERSEMAX: i32 = 1;
pub const TIMERANGEMIN: i32 = 0; // ps
pub const TIMERANGEMAX: i32 = 160000; // ps
pub const USECHANSMIN: i32 = 0x000; // no channels used
pub const USECHANSMAX: i32 = 0x10F; // note: sync bit 0x100 will be ignored in T3 mode
pub const PASSCHANSMIN: i32 = 0x000; // no channels passed
pub const PASSCHANSMAX: i32 = 0x10F; // note: sync bit 0x100 will be ignored in T3 mode

//limits for HH500_SetHistoLen
//note: length codes 0 and 1 will not work with HH500_GetHistogram
//if you need these short lengths then use HH500_GetAllHistograms
pub const MINLENCODE: i32 = 0;
pub const MAXLENCODE: i32 = 7;    // max length code in histogramming mode
pub const MAXLENCODE_CONT: i32 = 6;    // max length code in continuous mode
pub const DFLTLENCODE: i32 = 6;    // default length code, see DFLTHISTLEN

//private variables used to store information
static HIST_LEN : Mutex<i32> = Mutex::new(DFLTHISTLEN);

pub fn get_library_version() -> Result<String, ErrorCode> {
    // Define the size of the buffer
    let buffer_size = 10;
    // Allocate a buffer of the defined size
    let mut buffer: Vec<c_char> = vec![0; buffer_size];

    unsafe {
        let ret_code = HH500_GetLibraryVersion(buffer.as_mut_ptr());
        match ret_code {
            0 => {
                // Convert the buffer to a Rust string
                let c_str = CStr::from_ptr(buffer.as_ptr());
                let r_str = c_str.to_str().expect("Invalid UTF-8 string");
                Ok(r_str.to_string())
            }
            _ => {
                match ErrorCode::try_from(ret_code) {
                    Ok(error_code) => Err(error_code),
                    Err(_) => Err(ErrorCode::InvalidArgument), // cannot parse error, should not happen
                }
            }
        }
    }
}

pub fn get_error_string(errcode: i32) -> Result<String, ErrorCode> {
    // Define the size of the buffer
    let buffer_size = 256;
    // Allocate a buffer of the defined size
    let mut buffer: Vec<c_char> = vec![0; buffer_size];
    unsafe {
        let ret_code = HH500_GetErrorString(buffer.as_mut_ptr(), errcode);
        match ret_code {
            0 => {
                let c_str = CStr::from_ptr(buffer.as_ptr());
                let r_str = c_str.to_str().expect("Invalid UTF-8 string");
                Ok(r_str.to_string())
            }
            _ => {
                match ErrorCode::try_from(ret_code) {
                    Ok(error_code) => Err(error_code),
                    Err(_) => Err(ErrorCode::InvalidArgument), // cannot parse error, should not happen
                }
            }
        }
    }
}

pub fn open_device(devidx: i32) -> Result<String, ErrorCode> {
    // Define the size of the buffer
    let buffer_size = 256;
    // Allocate a buffer of the defined size
    let mut buffer: Vec<c_char> = vec![0; buffer_size];
    unsafe {
        let ret_code = HH500_OpenDevice(devidx, buffer.as_mut_ptr());
        match ret_code {
            0 => {
                let c_str = CStr::from_ptr(buffer.as_ptr());
                let r_str = c_str.to_str().expect("Invalid UTF-8 string");
                Ok(r_str.to_string())
            }
            _ => {
                match ErrorCode::try_from(ret_code) {
                    Ok(error_code) => Err(error_code),
                    Err(_) => Err(ErrorCode::InvalidArgument), // cannot parse error, should not happen
                }
            }
        }
    }
}

pub fn close_device(devidx: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_CloseDevice(devidx);
        match ret_code {
            0 => Ok(()),
            _ => {
                match ErrorCode::try_from(ret_code) {
                    Ok(error_code) => Err(error_code),
                    Err(_) => Err(ErrorCode::InvalidArgument), // cannot parse error, should not happen
                }
            }
        }
    }
}

pub fn initialize(devidx: i32, mode: Mode, ref_source: Refsource) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_Initialize(devidx, mode as i32, ref_source as i32);
        match ret_code {
            0 => Ok(()),
            _ => {
                match ErrorCode::try_from(ret_code) {
                    Ok(error_code) => Err(error_code),
                    Err(_) => Err(ErrorCode::InvalidArgument), // cannot parse error, should not happen
                }
            }
        }
    }
}

pub fn get_hardware_info(devidx: i32) -> Result<HardwareInfo, ErrorCode> {
    let buffer_size = 256;
    let mut info = HardwareInfo {
        model: String::new(),
        part_number: String::new(),
        version: String::new(),
    };
    let mut buffers: Vec<Vec<c_char>> = vec![vec![0; buffer_size]; 3]; // initialize inner vectors

    unsafe {
        let ret_code = HH500_GetHardwareInfo(
            devidx,
            buffers[0].as_mut_ptr(),
            buffers[1].as_mut_ptr(),
            buffers[2].as_mut_ptr(),
        );
        match ret_code {
            0 => {
                for (i, buffer) in buffers.iter().enumerate() {
                    let c_str = CStr::from_ptr(buffer.as_ptr());
                    let r_str = c_str.to_str().expect("Invalid UTF-8 string");

                    match i {
                        0 => info.model = r_str.to_string(),
                        1 => info.part_number = r_str.to_string(),
                        2 => info.version = r_str.to_string(),
                        _ => unreachable!(),
                    }
                }
                Ok(info)
            }
            _ => {
                match ErrorCode::try_from(ret_code) {
                    Ok(error_code) => Err(error_code),
                    Err(_) => Err(ErrorCode::InvalidArgument), // cannot parse error, should not happen
                }
            }
        }
    }
}

pub fn get_serial_number(devidx: i32) -> Result<String, ErrorCode> {
    let buffer_size = 256;
    let mut buffer: Vec<c_char> = vec![0; buffer_size];
    unsafe {
        let ret_code = HH500_GetSerialNumber(devidx, buffer.as_mut_ptr());
        match ret_code {
            0 => {
                let c_str = CStr::from_ptr(buffer.as_ptr());
                let r_str = c_str.to_str().expect("Invalid UTF-8 string");
                Ok(r_str.to_string())
            }
            _ => {
                match ErrorCode::try_from(ret_code) {
                    Ok(error_code) => Err(error_code),
                    Err(_) => Err(ErrorCode::InvalidArgument), // cannot parse error, should not happen
                }
            }
        }
    }
}

pub fn get_features(devidx: i32) -> Result<i32, ErrorCode> {
    let mut buffer_features: i32 = 0;
    unsafe {
        let ret_code = HH500_GetFeatures(devidx, &mut buffer_features as *mut i32);
        match ret_code {
            0 => Ok(buffer_features),
            _ => {
                match ErrorCode::try_from(ret_code) {
                    Ok(error_code) => Err(error_code),
                    Err(_) => Err(ErrorCode::InvalidArgument), // cannot parse error, should not happen
                }
            }
        }
    }
}

pub fn get_base_resolution(devidx: i32) -> Result<(i32, f64), ErrorCode> {
    let mut buffer_res: f64 = 0.0;
    let mut buffer_steps: i32 = 0;

    unsafe {
        let ret_code = HH500_GetBaseResolution(
            devidx,
            &mut buffer_res as *mut f64,
            &mut buffer_steps as *mut i32,
        );
        match ret_code {
            0 => Ok((buffer_steps, buffer_res)),
            _ => {
                match ErrorCode::try_from(ret_code) {
                    Ok(error_code) => Err(error_code),
                    Err(_) => Err(ErrorCode::InvalidArgument), // cannot parse error, should not happen
                }
            }
        }
    }
}

pub fn get_number_of_input_channels(devidx: i32) -> Result<i32, ErrorCode> {
    let mut nchannels: i32 = 0;
    unsafe {
        let ret_code = HH500_GetNumOfInputChannels(devidx, &mut nchannels as *mut i32);
        match ret_code {
            0 => Ok(nchannels),
            _ => {
                match ErrorCode::try_from(ret_code) {
                    Ok(error_code) => Err(error_code),
                    Err(_) => Err(ErrorCode::InvalidArgument), // cannot parse error, should not happen
                }
            }
        }
    }
}

pub fn set_sync_divider(devidx: i32, divider: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_SetSyncDiv(devidx, divider);
        match ret_code {
            0 => Ok(()),
            _ => {
                match ErrorCode::try_from(ret_code) {
                    Ok(error_code) => Err(error_code),
                    Err(_) => Err(ErrorCode::InvalidArgument), // cannot parse error, should not happen
                }
            }
        }
    }
}

pub fn set_sync_edge_trigger(devidx: i32, level: i32, edge: Edge) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_SetSyncEdgeTrg(devidx, level, edge as i32);
        match ret_code {
            0 => Ok(()),
            _ => {
                match ErrorCode::try_from(ret_code) {
                    Ok(error_code) => Err(error_code),
                    Err(_) => Err(ErrorCode::InvalidArgument), // cannot parse error, should not happen
                }
            }
        }
    }
}

pub fn get_sync_rate(devidx: i32) -> Result<i32, ErrorCode> {
    let mut syncrate: i32 = 0;
    unsafe {
        let ret_code = HH500_GetSyncRate(devidx, &mut syncrate);
        match ret_code {
            0 => Ok(syncrate),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn get_elapsed_meas_time(devidx: i32) -> Result<f64, ErrorCode> {
    let mut elapsed: f64 = 0.0;
    unsafe {
        let ret_code = HH500_GetElapsedMeasTime(devidx, &mut elapsed);
        match ret_code {
            0 => Ok(elapsed),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn get_warnings(devidx: i32) -> Result<i32, ErrorCode> {
    let mut warnings: i32 = 0;
    unsafe {
        let ret_code = HH500_GetWarnings(devidx, &mut warnings);
        match ret_code {
            0 => Ok(warnings),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn set_sync_cfd(devidx: i32, level: i32, zerocross: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_SetSyncCFD(devidx, level, zerocross);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn set_sync_channel_offset(devidx: i32, offset: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_SetSyncChannelOffset(devidx, offset);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn set_sync_channel_enable(devidx: i32, enable: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_SetSyncChannelEnable(devidx, enable);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn set_sync_dead_time(devidx: i32, on: i32, deadtime: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_SetSyncDeadTime(devidx, on, deadtime);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn set_input_edge_trigger(
    devidx: i32,
    channel: i32,
    level: i32,
    edge: Edge,
) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_SetInputEdgeTrg(devidx, channel, level, edge as i32);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}


pub fn set_input_channel_offset(
    devidx: i32,
    channel: i32,
    offset: i32,
) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_SetInputChannelOffset(devidx, channel, offset);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn set_input_dead_time(
    devidx: i32,
    channel: i32,
    on: i32,
    deadtime: i32,
) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_SetInputDeadTime(devidx, channel, on, deadtime);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn set_input_hysteresis(devidx: i32, hystcode: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_SetInputHysteresis(devidx, hystcode);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn set_input_channel_enable(
    devidx: i32,
    channel: i32,
    enable: i32,
) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_SetInputChannelEnable(devidx, channel, enable);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn set_stop_overflow(
    devidx: i32,
    stop_ovfl: i32,
    stopcount: u32,
) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_SetStopOverflow(devidx, stop_ovfl, stopcount);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn set_binning(devidx: i32, binning: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_SetBinning(devidx, binning);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn set_offset(devidx: i32, offset: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_SetOffset(devidx, offset);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn set_histo_len(devidx: i32, lencode: i32) -> Result<i32, ErrorCode> {
    let mut actuallen: i32 = 0;
    unsafe {
        let ret_code = HH500_SetHistoLen(devidx, lencode, &mut actuallen);
        match ret_code {
            0 => {
                let mut guard = HIST_LEN.lock().unwrap();
                *guard = actuallen;
                Ok(actuallen)
            },
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn set_meas_control(
    devidx: i32,
    control: MeasCtrl,
    startedge: Edge,
    stopedge: Edge,
) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code =
            HH500_SetMeasControl(devidx, control as i32, startedge as i32, stopedge as i32);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn set_trigger_output(devidx: i32, period: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_SetTriggerOutput(devidx, period);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn clear_hist_mem(devidx: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_ClearHistMem(devidx);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn start_meas(devidx: i32, tacq: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_StartMeas(devidx, tacq);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn stop_meas(devidx: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_StopMeas(devidx);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn get_ctc_status(devidx: i32) -> Result<i32, ErrorCode> {
    let mut ctcstatus: i32 = 0;
    unsafe {
        let ret_code = HH500_CTCStatus(devidx, &mut ctcstatus);
        match ret_code {
            0 => Ok(ctcstatus),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

// this function returns a new allocated buffer for histogram data
pub fn get_histogram(devidx: i32, channel: i32) -> Result<Vec<u32>, ErrorCode> {
    let mut buffer: Vec<u32> = vec![0; MAXHISTLEN as usize];
    
    unsafe {
        let ret_code = HH500_GetHistogram(devidx, buffer.as_mut_ptr(), channel);
        match ret_code {
            0 => Ok(buffer),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

// This function works like the one above, but you must create the buffer yourself.
// This can improve performance by avoiding repeated memory allocations 
// when calling this function multiple times.
pub fn get_histogram_mut(devidx: i32, channel: i32, buffer: &mut [u32]) -> Result<(), ErrorCode> {
    if buffer.len() < MAXHISTLEN as usize {
        return Err(ErrorCode::InvalidArgument); //Ensure buffer is large enough
    }

    unsafe {
        let ret_code = HH500_GetHistogram(devidx, buffer.as_mut_ptr(), channel);
        match ret_code {
            0 => Ok(()), // No need to return buffer, since it's modified in-place
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}


// this function returns a new allocated buffer for every channel histogram, make sure you can handle this memory allocation
// every time you call this function
pub fn get_all_histograms(devidx: i32) -> Result<Vec<Vec<u32>>, ErrorCode> {
    let ret_code = get_number_of_input_channels(devidx);
    let guard = HIST_LEN.lock().unwrap();

    match ret_code {
        Ok(number_of_input_channels) => {
            let mut buffer: Vec<u32> = vec![0; (*guard * number_of_input_channels) as usize];
            unsafe {
                let ret_code = HH500_GetAllHistograms(devidx, buffer.as_mut_ptr());
                if ret_code != 0 {
                    return match ErrorCode::try_from(ret_code) {
                        Ok(error_code) => Err(error_code),
                        Err(_) => Err(ErrorCode::InvalidArgument),
                    };
                }
            }

            let histograms: Vec<Vec<u32>> = buffer
                .chunks(*guard as usize) //Split buffer into `MAXHISTLEN`-sized chunks
                .map(|chunk| chunk.to_vec()) //Convert each chunk into its own Vec<u32>
                .collect();

            Ok(histograms)
        }
        Err(error_code) => Err(error_code),
    }
}

// This function works like the one above, but you must create the buffer yourself.
// This can improve performance by avoiding repeated memory allocations 
// when calling this function multiple times.
pub fn get_all_histograms_mut(devidx: i32, buffer: &mut [u32]) -> Result<(), ErrorCode> {
    let ret = get_number_of_input_channels(devidx);
    match ret {
        Ok(number_of_input_channels) => {
            if buffer.len() < ( MAXHISTLEN * number_of_input_channels ) as usize {
                return Err(ErrorCode::InvalidArgument); //Ensure buffer is large enough
            }
            unsafe {
                let ret_code = HH500_GetAllHistograms(devidx, buffer.as_mut_ptr());
                match ret_code {
                    0 => Ok(()), // No need to return buffer, since it's modified in-place
                    _ => match ErrorCode::try_from(ret_code) {
                        Ok(error_code) => Err(error_code),
                        Err(_) => Err(ErrorCode::InvalidArgument),
                    },
                }
            }
        }
        Err(error_code) => Err(error_code),
    }
}


pub fn get_resolution(devidx: i32) -> Result<f64, ErrorCode> {
    let mut resolution: f64 = 0.0;
    unsafe {
        let ret_code = HH500_GetResolution(devidx, &mut resolution);
        match ret_code {
            0 => Ok(resolution),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn get_sync_period(devidx: i32) -> Result<f64, ErrorCode> {
    let mut period: f64 = 0.0;
    unsafe {
        let ret_code = HH500_GetSyncPeriod(devidx, &mut period);
        match ret_code {
            0 => Ok(period),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn get_count_rate(devidx: i32, channel: i32) -> Result<i32, ErrorCode> {
    let mut cntrate: i32 = 0;
    unsafe {
        let ret_code = HH500_GetCountRate(devidx, channel, &mut cntrate);
        match ret_code {
            0 => Ok(cntrate),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn get_all_count_rates(devidx: i32) -> Result<(i32, Vec<i32>), ErrorCode> {
    let mut syncrate: i32 = 0;
    let ret = get_number_of_input_channels(devidx);

    match ret {
        Ok(number_of_channels) => {
            let mut cntrates: Vec<i32> = vec![0; number_of_channels as usize]; // Corrected to use `number_of_channels`
            unsafe {
                let ret_code = HH500_GetAllCountRates(devidx, &mut syncrate, cntrates.as_mut_ptr());
                match ret_code {
                    0 => Ok((syncrate, cntrates)),
                    _ => match ErrorCode::try_from(ret_code) {
                        Ok(error_code) => Err(error_code),
                        Err(_) => Err(ErrorCode::InvalidArgument),
                    },
                }
            }
        }
        Err(error_code) => Err(error_code), // Return the specific error code
    }
}

pub fn get_flags(devidx: i32) -> Result<i32, ErrorCode> {
    let mut flags: i32 = 0;
    unsafe {
        let ret_code = HH500_GetFlags(devidx, &mut flags);
        match ret_code {
            0 => Ok(flags),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn get_warnings_text(devidx: i32, warnings: i32) -> Result<String, ErrorCode> {
    let mut buffer: Vec<c_char> = vec![0; 16384];
    unsafe {
        let ret_code = HH500_GetWarningsText(devidx, buffer.as_mut_ptr(), warnings);
        match ret_code {
            0 => {
                let c_str = CStr::from_ptr(buffer.as_ptr());
                let r_str = c_str.to_str().expect("Invalid UTF-8 string");
                Ok(r_str.to_string())
            }
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn set_ofl_compression(devidx: i32, holdtime: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_SetOflCompression(devidx, holdtime);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn set_marker_holdoff_time(devidx: i32, holdofftime: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_SetMarkerHoldoffTime(devidx, holdofftime);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn set_marker_edges(
    devidx: i32,
    me1: Edge,
    me2: Edge,
    me3: Edge,
    me4: Edge,
) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_SetMarkerEdges(devidx, me1 as i32, me2 as i32, me3 as i32, me4 as i32);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn set_marker_enable(
    devidx: i32,
    en1: i32,
    en2: i32,
    en3: i32,
    en4: i32,
) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_SetMarkerEnable(devidx, en1, en2, en3, en4);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn read_fifo(devidx: i32) -> Result<Vec<u32>, ErrorCode> {
    let mut nactual: i32 = 0;
    let mut buffer: Vec<u32> = Vec::with_capacity(TTREADMAX as usize);
    unsafe {
        let ret_code = HH500_ReadFiFo(devidx, buffer.as_mut_ptr(), &mut nactual);
        match ret_code {
            0 => {
                buffer.set_len(nactual as usize);
                Ok(buffer)
            }
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

// pub fn set_event_filter_params(
//     devidx: i32,
//     timerange: i32,
//     matchcnt: i32,
//     inverse: i32,
// ) -> Result<(), ErrorCode> {
//     unsafe {
//         let ret_code = HH500_SetEventFilterParams(devidx, timerange, matchcnt, inverse);
//         match ret_code {
//             0 => Ok(()),
//             _ => match ErrorCode::try_from(ret_code) {
//                 Ok(error_code) => Err(error_code),
//                 Err(_) => Err(ErrorCode::InvalidArgument),
//             },
//         }
//     }
// }

// pub fn set_event_filter_channels(
//     devidx: i32,
//     usechannels: i32,
//     passchannels: i32,
// ) -> Result<(), ErrorCode> {
//     unsafe {
//         let ret_code = HH500_SetEventFilterChannels(devidx, usechannels, passchannels);
//         match ret_code {
//             0 => Ok(()),
//             _ => match ErrorCode::try_from(ret_code) {
//                 Ok(error_code) => Err(error_code),
//                 Err(_) => Err(ErrorCode::InvalidArgument),
//             },
//         }
//     }
// }

// pub fn enable_event_filter(devidx: i32, enable: i32) -> Result<(), ErrorCode> {
//     unsafe {
//         let ret_code = HH500_EnableEventFilter(devidx, enable);
//         match ret_code {
//             0 => Ok(()),
//             _ => match ErrorCode::try_from(ret_code) {
//                 Ok(error_code) => Err(error_code),
//                 Err(_) => Err(ErrorCode::InvalidArgument),
//             },
//         }
//     }
// }

pub fn set_filter_test_mode(devidx: i32, testmode: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_SetFilterTestMode(devidx, testmode);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

// pub fn get_filter_input_rates(devidx: i32) -> Result<(i32, Vec<i32>), ErrorCode> {
//     let mut syncrate: i32 = 0;
//     let ret_code = get_number_of_input_channels(devidx);

//     match ret_code {
//         Ok(number_of_channels) => {
//             let mut cntrates: Vec<i32> = vec![0; number_of_channels as usize];
//             unsafe {
//                 let ret_code =
//                     HH500_GetFilterInputRates(devidx, &mut syncrate, cntrates.as_mut_ptr());
//                 match ret_code {
//                     0 => Ok((syncrate, cntrates)),
//                     _ => match ErrorCode::try_from(ret_code) {
//                         Ok(error_code) => Err(error_code),
//                         Err(_) => Err(ErrorCode::InvalidArgument),
//                     },
//                 }
//             }
//         }
//         Err(error_code) => Err(error_code),
//     }
// }

// pub fn get_filter_output_rates(devidx: i32) -> Result<(i32, Vec<i32>), ErrorCode> {
//     let mut syncrate: i32 = 0;
//     let ret_code = get_number_of_input_channels(devidx);

//     match ret_code {
//         Ok(number_of_channels) => {
//             let mut cntrates: Vec<i32> = vec![0; number_of_channels as usize];
//             unsafe {
//                 let ret_code =
//                     HH500_GetFilterOutputRates(devidx, &mut syncrate, cntrates.as_mut_ptr());
//                 match ret_code {
//                     0 => Ok((syncrate, cntrates)),
//                     _ => match ErrorCode::try_from(ret_code) {
//                         Ok(error_code) => Err(error_code),
//                         Err(_) => Err(ErrorCode::InvalidArgument),
//                     },
//                 }
//             }
//         }
//         Err(error_code) => Err(error_code),
//     }
// }

pub fn get_debug_info(devidx: i32) -> Result<String, ErrorCode> {
    let mut buffer: Vec<c_char> = vec![0; 256];
    unsafe {
        let ret_code = HH500_GetDebugInfo(devidx, buffer.as_mut_ptr());
        match ret_code {
            0 => {
                let c_str = CStr::from_ptr(buffer.as_ptr());
                let r_str = c_str.to_str().expect("Invalid UTF-8 string");
                Ok(r_str.to_string())
            }
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn get_module_info(devidx: i32, mod_index : i32) -> Result<(i32, i32), ErrorCode> {
    let mut modelcode: i32 = 0;
    let mut versioncode: i32 = 0;
    unsafe {
        let ret_code = HH500_GetModuleInfo(devidx, mod_index, &mut modelcode, &mut versioncode);
        match ret_code {
            0 => Ok((modelcode, versioncode)),  // No semicolon here!
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn get_number_of_modules(devidx : i32) -> Result<i32, ErrorCode> {
    let mut number_of_modules : i32 = 0;
    unsafe{
     let ret_code = HH500_GetNumOfModules(devidx, &mut number_of_modules);
        match ret_code {
            0 => Ok(number_of_modules),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }   
    }
}

pub fn white_rabbit_get_mac(devidx: i32) -> Result<[u8; 6], ErrorCode> {
    let mut mac_addr: [u8; 6] = [0; 6];

    unsafe {
        let ret_code = HH500_WRabbitGetMAC(devidx, mac_addr.as_mut_ptr() as *mut i8);

        match ret_code {
            0 => Ok(mac_addr),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ext_fgpa_init_link(devidx: i32, link_number : i32, on : i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_ExtFPGAInitLink(devidx, link_number, on);
        match ret_code {
            0 => Ok(()),
            _ => {
                match ErrorCode::try_from(ret_code) {
                    Ok(error_code) => Err(error_code),
                    Err(_) => Err(ErrorCode::InvalidArgument), // cannot parse error, should not happen
                }
            }
        }
    }
}

pub fn ext_fpga_get_link_status(devidx: i32, link_number : i32) -> Result<u32, ErrorCode> {
    let mut link_status : u32 = 0;
    unsafe {
        let ret_code = HH500_ExtFPGAGetLinkStatus(devidx, link_number, &mut link_status);
        match ret_code {
            0 => Ok(link_status),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ext_fpga_set_mode(devidx: i32, mode : i32, loopback : i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_ExtFPGASetMode(devidx, mode, loopback);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ext_fpga_reset_stream_fifos(devidx: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_ExtFPGAResetStreamFifos(devidx);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ext_fpga_write_usr_cmd(devidx: i32, addr : u32, data : u32) -> Result<(), ErrorCode> {
    let mut write_data : u32 = data; 
    unsafe {
        let ret_code = HH500_ExtFPGAUserCommand(devidx, 1, addr, &mut write_data);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ext_fpga_read_usr_cmd(devidx: i32, addr : u32) -> Result<u32, ErrorCode> {
    let mut read_data : u32 = 0; 
    unsafe {
        let ret_code = HH500_ExtFPGAUserCommand(devidx, 0, addr, &mut read_data);
        match ret_code {
            0 => Ok(read_data),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn white_rabbit_get_status(devidx : i32) -> Result<i32, ErrorCode> {
    let mut status : i32 = 0; 
    unsafe {
        let ret_code = HH500_WRabbitGetStatus(devidx, &mut status);
        match ret_code {
            0 => Ok(status),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}


pub fn white_rabbit_init_link(devidx : i32, link_on : i32) -> Result<(), ErrorCode>{
    unsafe {
        let ret_code = HH500_WRabbitInitLink(devidx, link_on);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}


pub fn white_rabbit_set_mac(devidx: i32, mac_addr: [u8; 6]) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_WRabbitSetMAC(devidx, mac_addr.as_ptr() as *mut u8);

        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

//     fn HH500_WRabbitGetTime(devidx: i32, timehidw: *mut c_uint, timelodw: *mut c_uint, subsec16ns: *mut c_uint) -> i32;
pub fn white_rabbit_get_time(devidx: i32) -> Result<(u32, u32, u32), ErrorCode> {
    let mut time_hidw: u32 = 0;
    let mut time_lodw: u32 = 0;
    let mut subsec16ns: u32 = 0;

    unsafe {
        let ret_code = HH500_WRabbitGetTime(
            devidx,
            &mut time_hidw,
            &mut time_lodw,
            &mut subsec16ns,
        );

        match ret_code {
            0 => Ok((time_hidw, time_lodw, subsec16ns)),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}


pub fn white_rabbit_get_init_script(devidx: i32) -> Result<String, ErrorCode> {
    // Define the size of the buffer
    let buffer_size = 256;
    // Allocate a buffer of the defined size
    let mut initscript: Vec<c_char> = vec![0; buffer_size];
    unsafe {
        let ret_code = HH500_WRabbitGetInitScript(devidx, initscript.as_mut_ptr());
        match ret_code {
            0 => {
                let c_str = CStr::from_ptr(initscript.as_ptr());
                let r_str = c_str.to_str().expect("Invalid UTF-8 string");
                Ok(r_str.to_string())
            }
            _ => {
                match ErrorCode::try_from(ret_code) {
                    Ok(error_code) => Err(error_code),
                    Err(_) => Err(ErrorCode::InvalidArgument), // cannot parse error, should not happen
                }
            }
        }
    }
}


pub fn white_rabbit_get_sfp_data(devidx: i32) -> Result<[WrSfpData; 4], ErrorCode> {
    let buffer_size = 256;

    let mut sfp_data: [WrSfpData; 4] = [
        WrSfpData { name: String::new(), dtx: 0, drx: 0, alpha: 0 },
        WrSfpData { name: String::new(), dtx: 0, drx: 0, alpha: 0 },
        WrSfpData { name: String::new(), dtx: 0, drx: 0, alpha: 0 },
        WrSfpData { name: String::new(), dtx: 0, drx: 0, alpha: 0 },
    ];

    let mut names: Vec<c_char> = vec![0; buffer_size];
    let mut dtxs: Vec<i32> = vec![0; 4];
    let mut drxs: Vec<i32> = vec![0; 4];
    let mut alphas: Vec<i32> = vec![0; 4];

    unsafe {
        let ret_code = HH500_WRabbitGetSFPData(
            devidx,
            names.as_mut_ptr(),
            dtxs.as_mut_ptr(),
            drxs.as_mut_ptr(),
            alphas.as_mut_ptr(),
        );

        match ret_code {
            0 => {

                for i in 0..4 {
                    let name = &names[i * 20..(i + 1) * 20]; // slice names
                    let c_str: &CStr = CStr::from_ptr(name.as_ptr());
                    let r_str = c_str.to_str().expect("Invalid UTF-8 string");
                    sfp_data[i].name = r_str.to_string();
                    sfp_data[i].dtx = dtxs[i];
                    sfp_data[i].drx = drxs[i];
                    sfp_data[i].alpha = alphas[i];
                }

                Ok(sfp_data)
            }
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn save_debug_dump(devidx: i32, filepath : &str) -> Result<(), ErrorCode> {
    let c_filepath = CString::new(filepath).expect("CString conversion failed"); // Keep CString alive
    let c_str_ptr = c_filepath.into_raw();

    unsafe {
        let ret_code = HH500_SaveDebugDump(devidx, c_str_ptr); 
        // retake pointer to free memory --> https://doc.rust-lang.org/std/ffi/struct.CString.html#method.into_raw
        let _ = CString::from_raw(c_str_ptr); 

        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    } 
}

pub fn white_rabbit_set_init_script(devidx: i32, script : &str) -> Result< () , ErrorCode> {
    let c_script = CString::new(script).expect("CString conversion failed"); // Keep CString alive
    let c_str_ptr = c_script.into_raw();

    unsafe {
        let ret_code = HH500_WRabbitSetInitScript(devidx, c_str_ptr); 
        // retake pointer to free memory --> https://doc.rust-lang.org/std/ffi/struct.CString.html#method.into_raw
        let _ = CString::from_raw(c_str_ptr); 

        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    } 
}

pub fn white_rabbit_get_terminal_output(devidx: i32) -> Result<String, ErrorCode> {
    let buffer_size = 513;
    let mut buffer: Vec<c_char> = vec![0; buffer_size];
    let mut number_characters : i32 = 0;

    unsafe {
        let ret_code = HH500_WRabbitGetTermOutput(devidx, buffer.as_mut_ptr(),&mut number_characters as *mut i32);
        match ret_code {
            0 => {
                let valid_end : usize = usize::min(number_characters as usize, buffer_size);
                let buffer = &buffer[0..valid_end]; // Safe slicing
                let c_str = CStr::from_ptr(buffer.as_ptr());
                let r_str = c_str.to_str().expect("Invalid UTF-8 string");
                Ok(r_str.to_string())
            }
            _ => {
                match ErrorCode::try_from(ret_code) {
                    Ok(error_code) => Err(error_code),
                    Err(_) => Err(ErrorCode::InvalidArgument), // cannot parse error, should not happen
                }
            }
        }
    }

}

pub fn white_rabbit_set_sfp_data(devidx: i32, sfp_data: &[WrSfpData; 4]) -> Result<(), ErrorCode> {

    // Allocate memory for writing names (C-compatible)
    let mut dtxs: Vec<i32> = vec![0; 4];
    let mut drxs: Vec<i32> = vec![0; 4];
    let mut alphas: Vec<i32> = vec![0; 4];


    for (i, sfp) in sfp_data.iter().enumerate() {
        dtxs[i] = sfp.dtx;
        drxs[i] = sfp.drx;
        alphas[i] = sfp.alpha;
    }

    let c_strings: Vec<CString> = sfp_data
        .iter()
        .map(|sfp| {
            CString::new(sfp.name.clone()).expect("CString conversion failed")
        })
        .collect();

    let c_str_pointers: Vec<*const c_char> = c_strings.iter()
        .map(|s| s.as_ptr()) // Use *const c_char, no cast to *mut
        .collect();
    
    unsafe {
        let ret_code = HH500_WRabbitSetSFPData(
            devidx,
            c_str_pointers.as_ptr() as *mut i8,
            dtxs.as_mut_ptr(),
            drxs.as_mut_ptr(),
            alphas.as_mut_ptr(),
        );

        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn white_rabbit_set_time(devidx: i32, time_hidw: u32, time_lodw: u32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_WRabbitSetTime(
            devidx, 
            time_hidw, 
            time_lodw
        );

        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn white_rabbit_set_mode(devidx: i32, bootfromscript: i32, reinit_with_mode: i32, mode: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_WRabbitSetMode(devidx, bootfromscript, reinit_with_mode, mode);

        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

// todo validate function
/// Returns sync features for the device 
pub fn get_sync_features(devidx: i32) -> Result<u32, ErrorCode> {
    let mut features: u32 = 0;
    unsafe {
        let ret_code = HH500_GetSyncFeatures(devidx, &mut features as *mut u32);
        match ret_code {
            0 => Ok(features),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

/// Sets sync trigger mode
pub fn set_sync_trg_mode(devidx: i32, mode: TriggerMode) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_SetSyncTrgMode(devidx, mode as i32);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

/// Gets input features for a channel
pub fn get_input_features(devidx: i32, channel: i32) -> Result<u32, ErrorCode> {
    let mut features: u32 = 0;
    unsafe {
        let ret_code = HH500_GetInputFeatures(devidx, channel, &mut features as *mut u32);
        match ret_code {
            0 => Ok(features),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

/// Sets input trigger mode for a channel
pub fn set_input_trg_mode(devidx: i32, channel: i32, mode: TriggerMode) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_SetInputTrgMode(devidx, channel, mode as i32);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

/// Sets input CFD for a channel
pub fn set_input_cfd(devidx: i32, channel: i32, level: i32, zerocross: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_SetInputCFD(devidx, channel, level, zerocross);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

/// Gets the start time (returns tuple of 3 u32 values)
pub fn get_start_time(devidx: i32) -> Result<(u32, u32, u32), ErrorCode> {
    let mut timedw2: u32 = 0;
    let mut timedw1: u32 = 0;
    let mut timedw0: u32 = 0;
    unsafe {
        let ret_code = HH500_GetStartTime(devidx, &mut timedw2, &mut timedw1, &mut timedw0);
        match ret_code {
            0 => Ok((timedw2, timedw1, timedw0)),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn set_main_event_filter_params(devidx: i32, timerange: i32, matchcnt: i32, inverse: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_SetMainEventFilterParams(devidx, timerange, matchcnt, inverse);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            }
        }
    }
}

pub fn set_main_event_filter_channels(devidx: i32, rowidx: i32, usechannels: i32, passchannels: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_SetMainEventFilterChannels(devidx, rowidx, usechannels, passchannels);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            }
        }
    }
}

pub fn enable_main_event_filter(devidx: i32, enable: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = HH500_EnableMainEventFilter(devidx, enable);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            }
        }
    }
}

pub fn get_main_filtered_rates(devidx: i32, number_of_channels: usize) -> Result<(i32, Vec<i32>), ErrorCode> {
    let mut syncrate: i32 = 0;
    let mut cntrates: Vec<i32> = vec![0; number_of_channels];
    unsafe {
        let ret_code = HH500_GetMainFilteredRates(devidx, &mut syncrate, cntrates.as_mut_ptr());
        match ret_code {
            0 => Ok((syncrate, cntrates)),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            }
        }
    }
}

/// Reads a continuous mode block into the provided buffer, returns number of bytes received.
pub fn get_cont_mode_block(devidx: i32, buffer: &mut [u8]) -> Result<usize, ErrorCode> {
    let mut nbytesreceived: i32 = 0;
    unsafe {
        let ret_code = HH500_GetContModeBlock(devidx, buffer.as_mut_ptr() as *mut core::ffi::c_void, &mut nbytesreceived);
        match ret_code {
            0 => Ok(nbytesreceived as usize),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            }
        }
    }
}

// Dissect block header. You can design a Rust struct for the outputs!
pub fn dissect_cont_mode_blk_hdr(
    blockbuffer: &[u8],
) -> Result<(u32, u32, u32, u64, u64, [u64;4], [u32;4]), ErrorCode> {
    let mut nchannels: u32 = 0;
    let mut histlen: u32 = 0;
    let mut blocknum: u32 = 0;
    let mut starttime: u64 = 0;
    let mut ctctime: u64 = 0;
    let mut markertimes: [u64; 4] = [0; 4];
    let mut markercounts: [u32; 4] = [0; 4];
    unsafe {
        let ret_code = HH500_DissectContModeBlkHdr(
            blockbuffer.as_ptr() as *const core::ffi::c_void,
            &mut nchannels,
            &mut histlen,
            &mut blocknum,
            &mut starttime,
            &mut ctctime,
            markertimes.as_mut_ptr(),
            markercounts.as_mut_ptr(),
        );
        match ret_code {
            0 => Ok((nchannels, histlen, blocknum, starttime, ctctime, markertimes, markercounts)),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            }
        }
    }
}

// Extract data for one channel from cont mode block
pub fn extract_cont_mode_blk_data(
    blockbuffer: &[u8],
    chanindex: u32,
    histogram: &mut [u32],
) -> Result<u64, ErrorCode> {
    let mut histosum: u64 = 0;
    unsafe {
        let ret_code = HH500_ExtractContModeBlkData(
            blockbuffer.as_ptr() as *const core::ffi::c_void,
            chanindex,
            &mut histosum,
            histogram.as_mut_ptr(),
        );
        match ret_code {
            0 => Ok(histosum),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            }
        }
    }
}
