/************************************************************************

  Demo access to MultiHarp 150/160 hardware via MHLIB v.4.0
  The program performs a TTTR mode measurement simultaneously on mutiple 
  devices, using hardcoded settings. The resulting event data is stored in 
  multiple binary output files.

  Michael Wahl, PicoQuant GmbH, January 2025

  Note: This is a console application

  Note: At the API level the input channel numbers are indexed 0..N-1
  where N is the number of input channels the device has.

  Note: This demo writes only raw event data to the output file.
  It does not write a file header as regular .ptu files have it.

  Tested with the following compilers:

  - MinGW-W64 4.3.5 (Windows 64 bit)
  - MS Visual C++ 2019 (Windows 64 bit)
  - gcc 9.4.0 and 11.4.0 (Linux 64 bit)

************************************************************************/

#ifndef _WIN32
#include <unistd.h>
#define Sleep(msec) usleep(msec*1000)
#else
#include <windows.h>
#include <dos.h>
#include <conio.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>

#include "mhdefin.h"
#include "mhlib.h"
#include "errorcodes.h"

#define NDEVICES 2  // this specifies how many devices we want to use in parallel

unsigned int buffer[NDEVICES][TTREADMAX];


// helper macro and associated function for API calls with error check
// the stringize operator # makes a printable string from the macro's input argument
#define APICALL(call) doapicall(call, #call, __LINE__)
int doapicall(int retcode, char* callstr, int line)
{
    if (retcode < 0)
    {
        char errorstring[100] = { 0 };
        MH_GetErrorString(errorstring, retcode);
        printf("\nThe API call %s at line %d\nreturned error %d (%s)\n", callstr, line, retcode, errorstring);
    }
    return retcode;
}


int main(int argc, char* argv[])
{
    int dev[MAXDEVNUM];
    int found = 0;
    FILE* fpout[NDEVICES] = {NULL};
    int retcode;
    int ctcstatus;
    char LIB_Version[8];
    char HW_Model[32];
    char HW_Partno[8];
    char HW_Serial[9];
    char HW_Version[16];
    char Errorstring[40];
    char debuginfobuffer[16384]; // must have 16384 bytes of text buffer
    int NumChannels;
    int Mode = MODE_T2;  // set T2 or T3 here, observe suitable Sync divider and Range!
    int Binning = 0;     // you can change this, meaningful only in T3 mode
    int Offset = 0;      // you can change this, meaningful only in T3 mode
    int Tacq = 10000;    // measurement time in millisec, you can change this
    int SyncDivider = 1; // you can change this, observe Mode! READ MANUAL!

    int SyncTiggerEdge = 0;      // 0 or 1, you can change this
    int SyncTriggerLevel = -50;  // in mV,  you can change this
    int InputTriggerEdge = 0;    // 0 or 1, you can change this
    int InputTriggerLevel = -50; // in mV,  you can change this

    double Resolution;
    int Syncrate;
    int Countrate;
    int i, j, n;
    int flags;
    int warnings;
    char warningstext[16384]; // must have 16384 bytes of text buffer
    int nRecords;
    unsigned int Progress;
    char filename[40];
    int done[NDEVICES];
    int alldone;


    printf("\nMultiHarp MHLib Demo Application                      PicoQuant GmbH, 2025");
    printf("\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~");

    if (APICALL(MH_GetLibraryVersion(LIB_Version)) < 0)
        goto ex;
    printf("\nLibrary version is %s", LIB_Version);
    if (strncmp(LIB_Version, LIB_VERSION, sizeof(LIB_VERSION)) != 0)
        printf("\nWarning: The application was built for version %s.", LIB_VERSION);

    for (n = 0; n < NDEVICES; n++)
    {
        sprintf(filename, "tttrmode_%1d.out", n);
        if ((fpout[n] = fopen(filename, "wb")) == NULL)
        {
            printf("\ncannot open output file %s\n", filename);
            goto ex;
        }
    }

    printf("\nSearching for MultiHarp devices...");
    printf("\nDevidx     Serial     Status");


    for (i = 0; i < MAXDEVNUM; i++)
    {
        // not using APICALL here to allow selective error handling
        retcode = MH_OpenDevice(i, HW_Serial);
        if (retcode == 0) // grab any device we can open
        {
            printf("\n  %1d        %7s    open ok", i, HW_Serial);
            dev[found] = i; // keep index to devices we may want to use
            found++;
        }
        else
        {
            if (retcode == MH_ERROR_DEVICE_OPEN_FAIL)
                printf("\n  %1d        %7s    no device", i, HW_Serial);
            else
            {
                MH_GetErrorString(Errorstring, retcode);
                printf("\n  %1d        %7s    %s", i, HW_Serial, Errorstring);
            }
        }
    }

    // In this demo we will use the first NDEVICES devices we find.
    // You can also check for specific serial numbers, so that you know 
    // which physical device you are talking to.

    if (found < NDEVICES)
    {
        printf("\nNot enough devices available.");
        goto ex;
    }

    printf("\n");
    for (n = 0; n < NDEVICES; n++)
        printf("\nUsing device #%1d", dev[n]);
    printf("\n");

    for (n = 0; n < NDEVICES; n++)
    {
        printf("\nInitializing device #%1d", dev[n]);

        if (APICALL(MH_Initialize(dev[n], Mode, 0)) < 0)
        {
            // in case of an obscure error (a hardware error in particular) 
            // it may be helpful to obtain debug information like so:
            MH_GetDebugInfo(dev[n], debuginfobuffer);
            printf("\nDEBUGINFO:\n%s", debuginfobuffer);
            goto ex;
        }

        if (APICALL(MH_GetHardwareInfo(dev[n], HW_Model, HW_Partno, HW_Version)) < 0)
            goto ex;
        else
            printf("\nFound Model %s Part no %s Version %s", HW_Model, HW_Partno, HW_Version);

        if (APICALL(MH_GetNumOfInputChannels(dev[n], &NumChannels)) < 0)
            goto ex;
        else
            printf("\nDevice has %i input channels.", NumChannels);

        if (APICALL(MH_SetSyncDiv(dev[n], SyncDivider)) < 0)
            goto ex;

        if (APICALL(MH_SetSyncEdgeTrg(dev[n], SyncTriggerLevel, SyncTiggerEdge)) < 0)
            goto ex;

        if (APICALL(MH_SetSyncChannelOffset(dev[n], 0)) < 0) // can emulate a cable delay
            goto ex;

        for (i = 0; i < NumChannels; i++) // we use the same input settings for all channels
        {
            if (APICALL(MH_SetInputEdgeTrg(dev[n], i, InputTriggerLevel, InputTriggerEdge)) < 0)
                goto ex;

            if (APICALL(MH_SetInputChannelOffset(dev[n], i, 0)) < 0) // can emulate a cable delay
                goto ex;

            if (APICALL(MH_SetInputChannelEnable(dev[n], i, 1)) < 0)
                goto ex;
        }

        if (Mode != MODE_T2)
        {
            if (APICALL(MH_SetBinning(dev[n], Binning)) < 0)
                goto ex;

            if (APICALL(MH_SetOffset(dev[n], Offset)) < 0)
                goto ex;
        }

        if (APICALL(MH_GetResolution(dev[n], &Resolution)) < 0)
            goto ex;
        printf("\nResolution is %1.0lfps\n", Resolution);

    }

    // after Init allow 150 ms for valid count rate readings
    // subsequently you get new values after every 100ms
    Sleep(150);

    for (n = 0; n < NDEVICES; n++)
    {
        printf("\nMeasuring input rates...\n");

        if (APICALL(MH_GetSyncRate(dev[n], &Syncrate)) < 0)
            goto ex;
        printf("\nSyncrate[%1d]=%1d/s", n, Syncrate);

        for (i = 0; i < NumChannels; i++) // for all channels
        {
            if (APICALL(MH_GetCountRate(dev[n], i, &Countrate)) < 0)
                goto ex;
            printf("\nCountrate[%1d][%1d]=%1d/s", n, i, Countrate);
        }
        printf("\n");
    }

    // after getting the count rates you can check for warnings
    for (n = 0; n < NDEVICES; n++)
    {
        if (APICALL(MH_GetWarnings(dev[n], &warnings)) < 0)
            goto ex;
        if (warnings)
        {
            MH_GetWarningsText(dev[n], warningstext, warnings);
            printf("\n\nDevice %1d:", n);
            printf("\n\n%s", warningstext);
        }
    }

    printf("\npress RETURN to start");
    getchar();

    printf("\nStarting data collection...\n");

    Progress = 0;
    printf("\nProgress:%12u", Progress);

    // Starting the measurement on multiple devices via software will inevitably
    // introduce some ms of delay, so you cannot rely on an exact agreement
    // of the starting points of the TTTR streams. If you need this, you will 
    // have to use hardware synchronization.
    for (n = 0; n < NDEVICES; n++)
    {
        if (APICALL(MH_StartMeas(dev[n], Tacq)) < 0)
            goto ex;
        done[n] = 0;
    }

    while (1) //the overall measurement loop
    {
        // In this demo we loop over NDEVICES to fetch the data.
        // For efficiency this should be done in parallel
        for (n = 0; n < NDEVICES; n++)
        {
            if (APICALL(MH_GetFlags(dev[n], &flags)) < 0)
                goto ex;

            if (flags & FLAG_FIFOFULL)
            {
                printf("\nFiFo Overrun!\n");
                goto stoptttr;
            }

            if (APICALL(MH_ReadFiFo(dev[n], buffer[n], &nRecords)) < 0)
                goto ex;
            
            if (nRecords)
            {
                if (fwrite(buffer[n], 4, nRecords, fpout[n]) != (unsigned)nRecords)
                {
                    printf("\nfile write error\n");
                    goto stoptttr;
                }
                Progress += nRecords;
                if (n == NDEVICES - 1)
                {
                    printf("\b\b\b\b\b\b\b\b\b\b\b\b%12u", Progress);
                    fflush(stdout);
                }
            }
            else
            {
                if (APICALL(MH_CTCStatus(dev[n], &ctcstatus)) < 0)
                    goto ex;
                if (ctcstatus)
                {
                    done[n] = 1;
                    alldone = 0;
                    for (j = 0; j < NDEVICES; j++)
                        alldone += done[j];
                    if (alldone == NDEVICES)
                    {
                        printf("\nDone\n");
                        goto stoptttr;
                    }
                }
                // here you can also read the count rates if needed.
            }
        }
    }

stoptttr:

    for (n = 0; n < NDEVICES; n++)
        if (APICALL(MH_StopMeas(dev[n])) < 0)
            goto ex;

ex:

    for (i = 0; i < MAXDEVNUM; i++) // no harm to close all
        MH_CloseDevice(i);

    for (n = 0; n < NDEVICES; n++)
        if (fpout[n])
            fclose(fpout[n]);

    printf("\npress RETURN to exit");
    getchar();

    return 0;
}


