/************************************************************************

Demo access to MultiHarp Hardware via MHLib v.4.0
The demo consists of two items: wr_master.c and wr_slave.c
Together they perform a joint measurement of two MultiHarp devices
connected via White Rabbit. The master will remote-start the slave so
that the measurement of the two devices is closely aligned in time.
Please refer to the MHLib manual section on White Rabbit for concept 
and hardware setup.

The demo uses hardcoded settings for the serial numbers of the
used devices, for the input trigger levels, etc.
You need to change these to match your specific setup.

The executable of wr_slave must be started first. When it waits for
remote start by the master then the executable of wr_master can also
be started.

Nicolai Adelhoefer, Michael Wahl, PicoQuant GmbH, January 2025

Note: This is a console application (i.e. run in Windows cmd box)

Note: At the API level the input channel numbers are indexed 0..N-1
where N is the number of input channels the device has.

Note: This demo writes only raw event data to the output files.
It does not write a file header as regular .PTU files have it.


Tested with the following compilers:

  - gcc 14.2.0
  - MS Visual C++ 2019
  - MS Visual Studio Code MSVC 19.29.30147.0

************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>

#include "mhdefin.h"
#include "mhlib.h"
#include "errorcodes.h"
#ifndef _WIN32
#include <unistd.h>
#define Sleep(msec) usleep(msec * 1000)
#define __int64 long long
#else
#include <windows.h>
#include <assert.h>
#include <dos.h>
#include <conio.h>
#endif

// We will be using two devices, WR_Master and WR_Slave. In order to make this
// independent from the order of device enumeration we use their serials to identify them.
// In principle you could use multiple slaves.

#define WR_Master_Serial "1045483" // adjust this to your master device's serial number

// Helper routines for data evaluation
void process_data(FILE *fp, unsigned int *T2_RAW_DATA, size_t records, double resolution);
void load_binary_file(const char *filename, unsigned char **buffer, size_t *size);

// buffer for calls of MH_ReadFiFo
unsigned int buffer[TTREADMAX];

// helper macro and associated function for API calls with error check
// the stringize operator # makes a printable string from the macro's input argument
#define APICALL(call) doapicall(call, #call, __LINE__)
int doapicall(int retcode, char *callstr, int line)
{
  if (retcode < 0)
  {
    char errorstring[100] = {0};
    MH_GetErrorString(errorstring, retcode);
    printf("\nThe API call %s at line %d\nreturned error %d (%s)\n", callstr, line, retcode, errorstring);
  }
  return retcode;
}

int main(int argc, char *argv[])
{
  int devNum_WR_Master = -1;
  FILE *fpout = NULL;
  int retcode;
  int ctcstatus = 0;
  char LIB_Version[8] = {0};
  char HW_Serial[9] = {0};
  char Errorstring[40] = {0};
  int NumChannels = {0};

  int Mode = MODE_T2;  // We must use T2 here because data processing below assumes it.
                       // If you want T3 then you need to adapt the data analysis code.
  int Binning = 0;     // you can change this, meaningful only in T3 mode
  int Offset = 0;      // you can change this, meaningful only in T3 mode
  int Tacq = 1000;     // Measurement time in millisec, you can change this
  int SyncDivider = 1; // you can change this, observe Mode! READ MANUAL!

  int SyncTiggerEdge = 0;       // you can change this
  int SyncTriggerLevel = -100;  // you can change this
  int InputTriggerEdge = 1;     // you can change this
  int InputTriggerLevel = -100; // you can change this

  int Syncrate = 0;
  int Countrate = 0;
  int i;
  int flags;
  int nRecordsRead;
  int nRecordsTotal;
  double Resolution;

  printf("\nMultiHarp White Rabbit Remote Start Demo                PicoQuant GmbH, 2025");
  printf("\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~");

  if (APICALL(MH_GetLibraryVersion(LIB_Version)) < 0)
    goto ex;
  printf("\nLibrary version is %s\n", LIB_Version);
  if (strncmp(LIB_Version, LIB_VERSION, sizeof(LIB_VERSION)) != 0)
    printf("\nWarning: The application was built for version %s.", LIB_VERSION);

  printf("\nSearching for MultiHarp devices...");
  printf("\nDevidx     Serial     Status");

  for (i = 0; i < MAXDEVNUM; i++)
  {
    retcode = MH_OpenDevice(i, HW_Serial); // not using APICALL here as fails must be expected
    if (retcode == 0)
    {
      printf("\n  %1d        %7s    open ok", i, HW_Serial);
      if (strcmp(HW_Serial, WR_Master_Serial) == 0)
      {
        devNum_WR_Master = i;
        fpout = fopen("wr_master.out", "wb");
        if (fpout == NULL)
        {
          printf("\ncannot open output file\n");
          goto ex;
        }
      }
      else
        MH_CloseDevice(i);
    }
    else
    {
      if (retcode == MH_ERROR_DEVICE_OPEN_FAIL)
      {
        printf("\n  %1d        %7s    no device", i, HW_Serial);
      }
      else
      {
        MH_GetErrorString(Errorstring, retcode);
        printf("\n  %1d        %7s    %s", i, HW_Serial, Errorstring);
      }
    }
  }

  if (devNum_WR_Master < 0)
  {
    printf("\nThe required device S/N %s could not be opened. Aborting.", WR_Master_Serial);
    goto ex;
  }

  // Note that the  devices must be initialized to establish a WR connection beforehand.
  // This can be done by means of the regular MultiHarp software for Windows.
  // Here we only initialize the master for a suitable reference clock mode.
  // This will fail if the WR connetion is not up and running.

  printf("\nInitializing WR_Master ...");
  if (APICALL(MH_Initialize(devNum_WR_Master, Mode, REFSRC_WR_MASTER_MHARP)) < 0)
    goto ex;

  printf("\n\nUsing the following settings:\n");

  printf("Mode              : %d\n", Mode);
  printf("Binning           : %d\n", Binning);
  printf("Offset            : %d\n", Offset);
  printf("AcquisitionTime   : %d\n", Tacq);
  printf("SyncDivider       : %d\n", SyncDivider);
  printf("SyncTiggerEdge    : %d\n", SyncTiggerEdge);
  printf("SyncTriggerLevel  : %d\n", SyncTriggerLevel);
  printf("InputTriggerEdge  : %d\n", InputTriggerEdge);
  printf("InputTriggerLevel : %d\n", InputTriggerLevel);

  if (APICALL(MH_GetNumOfInputChannels(devNum_WR_Master, &NumChannels)) < 0)
    goto ex;
  if (APICALL(MH_GetResolution(devNum_WR_Master, &Resolution)) < 0)
    goto ex;
  if (APICALL(MH_SetSyncDiv(devNum_WR_Master, SyncDivider)) < 0)
    goto ex;
  if (APICALL(MH_SetSyncEdgeTrg(devNum_WR_Master, SyncTriggerLevel, SyncTiggerEdge)) < 0)
    goto ex;
  if (APICALL(MH_SetSyncChannelOffset(devNum_WR_Master, 0)) < 0)
    goto ex;

  for (i = 0; i < NumChannels; i++) // we use the same input settings for all channels
  {
    if (APICALL(MH_SetInputEdgeTrg(devNum_WR_Master, i, InputTriggerLevel, InputTriggerEdge)) < 0)
      goto ex;
    if (APICALL(MH_SetInputChannelOffset(devNum_WR_Master, i, 0)) < 0)
      goto ex;
    if (APICALL(MH_SetInputChannelEnable(devNum_WR_Master, i, 1)) < 0)
      goto ex;
  }

  if (Mode != MODE_T2)
  {
    if (APICALL(MH_SetBinning(devNum_WR_Master, Binning)) < 0)
      goto ex;
    if (APICALL(MH_SetOffset(devNum_WR_Master, Offset)) < 0)
      goto ex;
  }

  printf("\nMeasuring input rates...\n");
  // After Init allow 150 ms for valid  count rate readings
  // Subsequently you get new values after every 100ms
  Sleep(150);
  if (APICALL(MH_GetSyncRate(devNum_WR_Master, &Syncrate)) < 0)
    goto ex;
  printf("\n Syncrate=%1d/s", Syncrate);
  printf("\n Countrates");
  for (i = 0; i < NumChannels; i++) // for all channels
  {
    if (APICALL(MH_GetSyncRate(devNum_WR_Master, &Syncrate)) < 0)
      goto ex;
    // format output catering for a device with many channels
    if (i == 0 || i == 8 || i == 16 || i == 24 || i == 32 || i == 40 || i == 48 || i == 56)
    {
      printf("\n [%2d-%2d]", i, i + 7);
    }
    printf(" %7d/s", Countrate);
  }
  printf("\n");

  // In the following call we specify which device starts measurements on the other.
  // Note well: This is independent from which device is WR master and which is slave.
  // Either one can start the other. Here we specify MEASCTRL_WR_M2S which means that
  // the WR master shall remote-start measurements on the WR slave.
  // The slave must use the same setting.
  if (APICALL(MH_SetMeasControl(devNum_WR_Master, MEASCTRL_WR_M2S, 0, 0)) < 0)
    goto ex;

  // Meanwhile (here at the latest) the WR slave must be ready to wait for a remote start.
  // I.e. it must also be configured for MEASCTRL_WR_M2S and have called MH_StartMeas. 
  // It is important that this is done BEFORE calling MH_StartMeas here on the master.

  // Now we start a measurement on the master, which in turn will remote-start the slave via WR
  if (APICALL(MH_StartMeas(devNum_WR_Master, Tacq)) < 0)
    goto ex;

  // Prepare for data collection
  nRecordsTotal = 0;
  printf("\nRecords read: %12d", nRecordsTotal);

  while (1) // This is the data collection loop
  {
    if (APICALL(MH_GetFlags(devNum_WR_Master, &flags)) < 0)
      goto ex;
    if (flags & FLAG_FIFOFULL)
    {
      printf("\nFiFo Overrun!\n");
      goto stoptttr;
    }

    if (APICALL(MH_ReadFiFo(devNum_WR_Master, buffer, &nRecordsRead)) < 0)
      goto ex;
    if (nRecordsRead)
    {
      nRecordsTotal += nRecordsRead;
      if (fwrite(buffer, 4, nRecordsRead, fpout) != (unsigned)nRecordsRead)
      {
        printf("\nfile write error\n");
        goto stoptttr;
      } 
      printf("\b\b\b\b\b\b\b\b\b\b\b\b%12d", nRecordsTotal);
      fflush(stdout);
    }
    else
    {
      if (APICALL(MH_CTCStatus(devNum_WR_Master, &ctcstatus)) < 0)
        goto ex;
      if (ctcstatus)
      {        
        // There may still be some data in the FIFO, so do a few more reads
        for (int i = 0; i < 20; i++)
        {
          if (APICALL(MH_ReadFiFo(devNum_WR_Master, buffer, &nRecordsRead)) < 0)
            goto ex;
          if (nRecordsRead)
          {
            nRecordsTotal += nRecordsRead;
            if (fwrite(buffer, 4, nRecordsRead, fpout) != (unsigned)nRecordsRead)
            {
              printf("\nfile write error\n");
              goto stoptttr;
            }
          }
        }
        printf("\b\b\b\b\b\b\b\b\b\b\b\b%12d", nRecordsTotal);
        printf("\nMeasurement Done\n");
        fflush(stdout);
        goto stoptttr; // data collection finished
      }
    }
  }

stoptttr:

  // for internal cleanup it is important that MH_StopMeas is finally called
  if (APICALL(MH_StopMeas(devNum_WR_Master)) < 0)
    goto ex;

  if (fpout)
  {
    fclose(fpout);
    fpout=NULL;
  }

  // now start evaluating the data

  FILE *fm_log = fopen("wr_master_dec.txt", "w");

  unsigned char *file_buffer = NULL;
  size_t size = 0;

  printf("Processing data \n");

  load_binary_file("wr_master.out", &file_buffer, &size);
  process_data(fm_log, (unsigned int *)file_buffer, size / 4, Resolution);

  // the start of measurements on the two devices is synchronized to within the same WR TAI cycle (16ns)
  // in the following we use MH_GetStartTime to retrieve the start time of the master at an even higher resolution
  // doing the same on the slave this can subsequently be used to align the two event streams to within +-3ns
  uint32_t m_buffer_wr_start_time[3] = {0};
  MH_GetStartTime(devNum_WR_Master, &m_buffer_wr_start_time[2], &m_buffer_wr_start_time[1], &m_buffer_wr_start_time[0]);
  printf("Master start time : 0x%08x%08x%08x\n", m_buffer_wr_start_time[2], m_buffer_wr_start_time[1], m_buffer_wr_start_time[0]);

  // cleanup files and allocated memory
  if(file_buffer)
  {
    free(file_buffer);
    file_buffer = NULL;
  }  
  if(fm_log)
  {
    fclose(fm_log);
    fm_log=NULL;
  }

ex:

  for (i = 0; i < MAXDEVNUM; i++) // no harm to close all
  {
    MH_CloseDevice(i);
  }
  if (fpout)
  {
    fclose(fpout);
    fpout=NULL;
  }
  printf("\npress RETURN to exit");
  getchar();
  return 0;
}

// the following are routines for data evaluation

void GotPhotonT2(FILE *stream, uint64_t TimeTag, int Channel)
{
  fprintf(stream, "CH %d %llu\n", Channel, TimeTag);
}

//   TimeTag: Overflow-corrected arrival time in units of the device's base resolution
//   Markers: Bitfield of arrived markers, different markers can arrive at same time (same record)
void GotMarkerT2(FILE *stream, uint64_t TimeTag, int Markers)
{
  // fprintf(stream, "MK %d llu\n", Markers, TimeTag);
}

void process_data(FILE *fp, unsigned int *T2_RAW_DATA, size_t records, double resolution)
{
  int ch;
  uint64_t truetime;
  const int T2WRAPAROUND_V2 = 33554432;
  uint64_t oflcorrection = 0;
  union
  {
    unsigned allbits;
    struct
    {
      unsigned timetag : 25;
      unsigned channel : 6;
      unsigned special : 1; // or sync, if channel==0
    } bits;
  } T2Rec;

  for (size_t i = 0; i < records; i++)
  {
    T2Rec.allbits = *(T2_RAW_DATA + i);

    if (T2Rec.bits.special == 1)
    {
      if (T2Rec.bits.channel == 0x3F) // an overflow record
      {
        // number of overflows is stored in timetag
        oflcorrection += (uint64_t)T2WRAPAROUND_V2 * T2Rec.bits.timetag;
      }
      if ((T2Rec.bits.channel >= 1) && (T2Rec.bits.channel <= 15)) // markers
      {
        truetime = (uint64_t)resolution * (oflcorrection + T2Rec.bits.timetag);
        // note that actual marker tagging accuracy is only some ns
        ch = T2Rec.bits.channel;
        GotMarkerT2(fp, truetime, ch);
      }
      if (T2Rec.bits.channel == 0) // sync
      {
        truetime = 5 * (oflcorrection + T2Rec.bits.timetag);
        ch = 0; // we encode the Sync channel as 0
        GotPhotonT2(fp, truetime, ch);
      }
    }
    else // regular input channel
    {
      truetime = oflcorrection + T2Rec.bits.timetag;
      ch = T2Rec.bits.channel;
      GotPhotonT2(fp, truetime, ch);
    }
  }
}

void load_binary_file(const char *filename, unsigned char **buffer, size_t *size)
{
  FILE *file = fopen(filename, "rb");
  if (!file)
  {
    perror("Failed to open file");
    exit(EXIT_FAILURE);
  }

  // get the file size
  fseek(file, 0, SEEK_END);
  *size = ftell(file);
  fseek(file, 0, SEEK_SET);

  // allocate memory for the buffer
  *buffer = (unsigned char *)malloc(*size);
  if (!*buffer)
  {
    perror("Failed to allocate memory");
    fclose(file);
    exit(EXIT_FAILURE);
  }

  // read the file into the buffer
  size_t read_size = fread(*buffer, 1, *size, file);
  if (read_size != *size)
  {
    perror("Failed to read the file");
    if(*buffer)
    {
      free(*buffer);
      *buffer = NULL;
    }
    if(file)
    {
      fclose(file);
      file=NULL;
    }
    exit(EXIT_FAILURE);
  }

  if(file)
  {
    fclose(file);
    file=NULL;
  }
}
