﻿/* 
    MHLib programming library for MultiHarp 150/160
    PicoQuant GmbH

    Ver. 4.0.0.0   Michael Wahl, January 2025

    You shouldn't normally edit this file!
*/

using System;
using System.Text;
using System.Runtime.InteropServices;  // for DllImport
using System.Runtime.CompilerServices; // for CallerMemberName

namespace MHLibSpace
{
    public static class MHLib
    {
        // unfortunately WINDOWS is not predefined, must define it as a compiler directive
        // MS Visual C#: find it defined in the .csproj-file
#if WINDOWS
        public const string LibName = "MHLib64.dll";
#else
        public const string LibName = "libmhlib"; 
#endif

        //////////////////////////// PH330Lib Constants //////////////////////////////

        public const string LIB_VERSION = "4.0";        // expected library version (major.minor)

        public const int MAXDEVNUM = 8;                 // max number of MultiHarp devices

        public const int MAXINPCHAN = 64;               // max number of physical input channels

        public const int MAXBINSTEPS = 24;              // max number of binning steps, 
                                                        // get actual number via MH_GetBaseResolution()

        public const int MAXHISTLEN = 65536;            // max number of histogram bins

        public const int TTREADMAX = 1048576;           // number of event records that can be read by MH_ReadFiFo
                                                        // buffer must provide space for this number of dwords

        // constants for MH_Initialize parameter "refsource"
        public const int REFSRC_INTERNAL          = 0;  // use internal clock
        public const int REFSRC_EXTERNAL_10MHZ    = 1;  // use 10MHz external clock
        public const int REFSRC_WR_MASTER_GENERIC = 2;  // White Rabbit master with generic partner
        public const int REFSRC_WR_SLAVE_GENERIC  = 3;  // White Rabbit slave with generic partner
        public const int REFSRC_WR_GRANDM_GENERIC = 4;  // White Rabbit grand master with generic partner
        public const int REFSRC_EXTN_GPS_PPS      = 5;  // use 10 MHz + PPS from GPS
        public const int REFSRC_EXTN_GPS_PPS_UART = 6;  // use 10 MHz + PPS + time via UART from GPS
        public const int REFSRC_WR_MASTER_MHARP   = 7;  // White Rabbit master with MultiHarp as partner
        public const int REFSRC_WR_SLAVE_MHARP    = 8;  // White Rabbit slave with MultiHarp as partner
        public const int REFSRC_WR_GRANDM_MHARP   = 9;  // White Rabbit grand master with MultiHarp as partner

        // constants for MH_Initialize parameter "mode"
        public const int MODE_HIST = 0;
        public const int MODE_T2 = 2;
        public const int MODE_T3 = 3;

        // constants for MH_SetMeasControl parameter "control"
        public const int MEASCTRL_SINGLESHOT_CTC = 0; //default
        public const int MEASCTRL_C1_GATED = 1;
        public const int MEASCTRL_C1_START_CTC_STOP = 2;
        public const int MEASCTRL_C1_START_C2_STOP = 3;
        public const int MEASCTRL_WR_M2S = 4;
        public const int MEASCTRL_WR_S2M = 5;
        public const int MEASCTRL_SW_START_SW_STOP = 6;

        // codes for MH_SetMeasControl, MH_SetMarkerEdges, MH_SetSyncEdgeTrg and MH_SetInputEdgeTrg parameters "..edge"
        public const int EDGE_RISING = 1;
        public const int EDGE_FALLING = 0;

        // limits for MH_SetHistoLen
        // note: length codes 0 and 1 will not work with MH_GetHistogram
        // if you need these short lengths then use MH_GetAllHistograms
        public const int MINLENCODE = 0;     // 1024 histogram bins
        public const int MAXLENCODE = 6;     // 65536 histogram bins, default

        // limits for MH_SetSyncDiv
        public const int SYNCDIVMIN = 1;
        public const int SYNCDIVMAX = 16;

        // limits for MH_SetSyncEdgeTrg and MH_SetInputEdgeTrg
        public const int TRGLVLMIN = -1200;    // mV
        public const int TRGLVLMAX = 1200;     // mV

        // limits for MH_SetSyncChannelOffset and MH_SetInputChannelOffset
        public const int CHANOFFSMIN = -99999; // ps
        public const int CHANOFFSMAX = 99999;  // ps

        // limits for MH_SetSyncDeadTime and MH_SetInputDeadTime
        public const int EXTDEADMIN = 800;     // ps
        public const int EXTDEADMAX = 160000;  // ps

        // limits for MH_SetOffset
        public const int OFFSETMIN = 0;         // ns
        public const int OFFSETMAX = 100000000; // ns

        // limits for MH_StartMeas
        public const int ACQTMIN = 1;           // ms
        public const int ACQTMAX = 360000000;   // ms  (100*60*60*1000ms = 100h)

        // limits for MH_SetStopOverflow
        public const int STOPCNTMIN = 1;
        public const uint STOPCNTMAX = 4294967295;  // 32 bit is mem max

        // limits for MH_SetTriggerOutput
        public const int TRIGOUTMIN = 0;         // 0=off
        public const int TRIGOUTMAX = 16777215;  // in units of 100ns

        // limits for MH_SetMarkerHoldoffTime
        public const int HOLDOFFMIN = 0;         // ns
        public const int HOLDOFFMAX = 25500;     // ns

        // limits for MH_SetInputHysteresis
        public const int HYSTCODEMIN = 0;        // approx. 3mV
        public const int HYSTCODEMAX = 1;        // approx. 35mV

        // limits for MH_SetOflCompression
        public const int HOLDTIMEMIN = 0;        // ms
        public const int HOLDTIMEMAX = 255;      // ms

        // limits for MH_SetRowEventFilter and MH_SetMainEventFilter
        public const int ROWIDXMIN = 0;
        public const int ROWIDXMAX = 8;           // actual upper limit is smaller, dep. on rows present
        public const int MATCHCNTMIN = 1;
        public const int MATCHCNTMAX = 6;
        public const int INVERSEMIN = 0;
        public const int INVERSEMAX = 1;
        public const int TIMERANGEMIN = 0;        // ps
        public const int TIMERANGEMAX = 160000;   // ps
        public const int USECHANSMIN = 0x000;     // no channels used 
        public const int USECHANSMAX = 0x1FF;     // note: sync bit 0x100 will be ignored in T3 mode
        public const int PASSCHANSMIN = 0x000;    // no channels passed 
        public const int PASSCHANSMAX = 0x1FF;    // note: sync bit 0x100 will be ignored in T3 mode

        // bitmasks for results from MH_GetFeatures
        public const int FEATURE_DLL = 0x0001;       // DLL License available
        public const int FEATURE_TTTR = 0x0002;      // TTTR mode available
        public const int FEATURE_MARKERS = 0x0004;   // Markers available
        public const int FEATURE_LOWRES = 0x0008;    // Long range mode available 
        public const int FEATURE_TRIGOUT = 0x0010;   // Trigger output available
        public const int FEATURE_PROG_TD = 0x0020;   // Programmable deadtime available
        public const int FEATURE_EXT_FPGA = 0x0040;  // Interface for external FPGA available
        public const int FEATURE_PROG_HYST = 0x0080; // Programmable input hysteresis available
        public const int FEATURE_EVNT_FILT = 0x0100; // Coincidence filtering available
        public const int FEATURE_INPT_MODE = 0x0200; // Programmable input mode (CFD vs. edge trigger)

        // bitmasks for results from MH_GetFlags
        public const int FLAG_OVERFLOW = 0x0001;      // histo mode only
        public const int FLAG_FIFOFULL = 0x0002;      // TTTR mode only
        public const int FLAG_SYNC_LOST = 0x0004;     // Sync signal was lost
        public const int FLAG_REF_LOST = 0x0008;      // Reference clock was lost
        public const int FLAG_SYSERROR = 0x0010;      // hardware error, must contact support
        public const int FLAG_ACTIVE = 0x0020;        // measurement is running
        public const int FLAG_CNTS_DROPPED = 0x0040;  // counts were dropped
        public const int FLAG_SOFTERROR = 0x0080;     // software error, must contact support

        // bitmasks for results from MH_GetWarnings
        public const int WARNING_SYNC_RATE_ZERO = 0x0001;
        public const int WARNING_SYNC_RATE_VERY_LOW = 0x0002;
        public const int WARNING_SYNC_RATE_TOO_HIGH = 0x0004;
        public const int WARNING_INPT_RATE_ZERO = 0x0010;
        public const int WARNING_INPT_RATE_TOO_HIGH = 0x0040;
        public const int WARNING_INPT_RATE_RATIO = 0x0100;
        public const int WARNING_DIVIDER_GREATER_ONE = 0x0200;
        public const int WARNING_TIME_SPAN_TOO_SMALL = 0x0400;
        public const int WARNING_OFFSET_UNNECESSARY = 0x0800;
        public const int WARNING_DIVIDER_TOO_SMALL = 0x1000;
        public const int WARNING_COUNTS_DROPPED = 0x2000;
        public const int WARNING_USB20_SPEED_ONLY = 0x4000;

        // bitmasks for use with White Rabbit
        public const uint WR_STATUS_LINK_ON          = 0x00000001;  // WR link is switched on
        public const uint WR_STATUS_LINK_UP          = 0x00000002;  // WR link is established
        public const uint WR_STATUS_MODE_BITMASK     = 0x0000000C;  // mask for the mode bits
        public const uint WR_STATUS_MODE_OFF         = 0x00000000;  // mode is "off"
        public const uint WR_STATUS_MODE_SLAVE       = 0x00000004;  // mode is "slave"
        public const uint WR_STATUS_MODE_MASTER      = 0x00000008;  // mode is "master" 
        public const uint WR_STATUS_MODE_GMASTER     = 0x0000000C;  // mode is "grandmaster"
        public const uint WR_STATUS_LOCKED_CALIBD    = 0x00000010;  // locked and calibrated
        public const uint WR_STATUS_PTP_BITMASK      = 0x000000E0;  // mask for the PTP bits
        public const uint WR_STATUS_PTP_LISTENING    = 0x00000020;
        public const uint WR_STATUS_PTP_UNCLWRSLCK   = 0x00000040;
        public const uint WR_STATUS_PTP_SLAVE        = 0x00000060;
        public const uint WR_STATUS_PTP_MSTRWRMLCK   = 0x00000080;
        public const uint WR_STATUS_PTP_MASTER       = 0x000000A0;
        public const uint WR_STATUS_SERVO_BITMASK    = 0x00000700;  // mask for the servo bits
        public const uint WR_STATUS_SERVO_UNINITLZD  = 0x00000100;  
        public const uint WR_STATUS_SERVO_SYNC_SEC   = 0x00000200;  
        public const uint WR_STATUS_SERVO_SYNC_NSEC  = 0x00000300;  
        public const uint WR_STATUS_SERVO_SYNC_PHASE = 0x00000400;  
        public const uint WR_STATUS_SERVO_WAIT_OFFST = 0x00000500;
        public const uint WR_STATUS_SERVO_TRCK_PHASE = 0x00000600;  
        public const uint WR_STATUS_MAC_SET          = 0x00000800;  // user defined mac address is set
        public const uint WR_STATUS_IS_NEW           = 0x80000000;  // status updated since last check

        // constants for use with an external FPGA connected to a MultiHarp 160
        public const int EXTFPGA_MODE_OFF = 0;
        public const int EXTFPGA_MODE_T2RAW = 1;
        public const int EXTFPGA_MODE_T2 = 2;
        public const int EXTFPGA_MODE_T3 = 3;
        public const int EXTFPGA_LOOPBACK_OFF = 0;
        public const int EXTFPGA_LOOPBACK_CUSTOM = 1;
        public const int EXTFPGA_LOOPBACK_T2 = 2;
        public const int EXTFPGA_LOOPBACK_T3 = 3;

        //////////////////////////// MHLib Methods //////////////////////////////

        // Helper method for elegant one-line API calls with error reporting
        public static int APICALL(Func<int> apiCall, [CallerMemberName] string caller = null, [CallerLineNumber] int lineNumber = 0)
        {
            int retcode = 0;
            try
            {
                retcode = apiCall();
            }
            catch (Exception e)
            {
                Console.WriteLine($"\nThe API call in {caller} at line {lineNumber} raised an exception: ");
                Console.WriteLine(e.Message);
                Console.WriteLine("press RETURN to exit");
                Console.ReadLine();
                Environment.Exit(0);
            }
            if (retcode < 0)
            {
                StringBuilder errstr = new StringBuilder(40);
                if (MH_GetErrorString(errstr, retcode) == 0)
                    Console.WriteLine($"\nThe API call in {caller} at line {lineNumber} returned error {retcode} ({errstr})\n");
                else
                    Console.WriteLine($"\nThe API call in {caller} at line {lineNumber} returned error {retcode}\n");
            }
            return retcode;
        }

        // for version check and debugging, no device required
        [DllImport(LibName)]
        extern public static int MH_GetLibraryVersion(StringBuilder vers);
        [DllImport(LibName)]
        extern public static int MH_GetErrorString(StringBuilder errstring, int errcode);

        // open, initialize, close

        [DllImport(LibName)]
        extern public static int MH_OpenDevice(int devidx, StringBuilder serial);
        [DllImport(LibName)]
        extern public static int MH_CloseDevice(int devidx);
        [DllImport(LibName)]
        extern public static int MH_Initialize(int devidx, int mode, int refsource);

        // all functions below can only be used after MH_Initialize

        [DllImport(LibName)]
        extern public static int MH_GetHardwareInfo(int devidx, StringBuilder model, StringBuilder partno, StringBuilder version);
        [DllImport(LibName)]
        extern public static int MH_GetSerialNumber(int devidx, StringBuilder serial);
        [DllImport(LibName)]
        extern public static int MH_GetFeatures(int devidx, ref int features);
        [DllImport(LibName)]
        extern public static int MH_GetBaseResolution(int devidx, ref double resolution, ref int binsteps);
        [DllImport(LibName)]
        extern public static int MH_GetNumOfInputChannels(int devidx, ref int nchannels);

        [DllImport(LibName)]
        extern public static int MH_SetSyncDiv(int devidx, int div);
        [DllImport(LibName)]
        extern public static int MH_SetSyncEdgeTrg(int devidx, int level, int edge);
        [DllImport(LibName)]
        extern public static int MH_SetSyncChannelOffset(int devidx, int offset);
        [DllImport(LibName)]
        extern public static int MH_SetSyncChannelEnable(int devidx, int enable);
        [DllImport(LibName)]
        extern public static int MH_SetSyncDeadTime(int devidx, int on, int deadtime);

        [DllImport(LibName)]
        extern public static int MH_SetInputEdgeTrg(int devidx, int channel, int level, int edge);
        [DllImport(LibName)]
        extern public static int MH_SetInputChannelOffset(int devidx, int channel, int offset);
        [DllImport(LibName)]
        extern public static int MH_SetInputDeadTime(int devidx, int channel, int on, int deadtime);
        [DllImport(LibName)]
        extern public static int MH_SetInputHysteresis(int devidx, int hystcode);
        [DllImport(LibName)]
        extern public static int MH_SetInputChannelEnable(int devidx, int channel, int enable);

        [DllImport(LibName)]
        extern public static int MH_SetStopOverflow(int devidx, int stop_ovfl, uint stopcount);
        [DllImport(LibName)]
        extern public static int MH_SetBinning(int devidx, int binning);
        [DllImport(LibName)]
        extern public static int MH_SetOffset(int devidx, int offset);
        [DllImport(LibName)]
        extern public static int MH_SetHistoLen(int devidx, int lencode, out int actuallen);
        [DllImport(LibName)]
        extern public static int MH_SetMeasControl(int devidx, int control, int startedge, int stopedge);
        [DllImport(LibName)]
        extern public static int MH_SetTriggerOutput(int devidx, int period);

        [DllImport(LibName)]
        extern public static int MH_ClearHistMem(int devidx);
        [DllImport(LibName)]
        extern public static int MH_StartMeas(int devidx, int tacq);
        [DllImport(LibName)]
        extern public static int MH_StopMeas(int devidx);
        [DllImport(LibName)]
        extern public static int MH_CTCStatus(int devidx, ref int ctcstatus);

        [DllImport(LibName)]
        extern public static int MH_GetHistogram(int devidx, uint[] chcount, int channel);
        [DllImport(LibName)]
        extern public static int MH_GetAllHistograms(int devidx, uint[] chcount);
        [DllImport(LibName)]
        extern public static int MH_GetResolution(int devidx, ref double resolution);
        [DllImport(LibName)]
        extern public static int MH_GetSyncPeriod(int devidx, ref double period);
        [DllImport(LibName)]
        extern public static int MH_GetSyncRate(int devidx, ref int syncrate);
        [DllImport(LibName)]
        extern public static int MH_GetCountRate(int devidx, int channel, ref int cntrate);
        [DllImport(LibName)]
        extern public static int MH_GetAllCountRates(int devidx, ref int syncrate, int[] cntrates);
        [DllImport(LibName)]
        extern public static int MH_GetFlags(int devidx, ref int flags);
        [DllImport(LibName)]
        extern public static int MH_GetElapsedMeasTime(int devidx, out double elapsed);
        [DllImport(LibName)]
        extern public static int MH_GetStartTime(int devidx, ref uint timedw2, ref uint timedw1, ref uint timedw0);

        [DllImport(LibName)]
        extern public static int MH_GetWarnings(int devidx, ref int warnings);
        [DllImport(LibName)]
        extern public static int MH_GetWarningsText(int devidx, StringBuilder text, int warnings);

        // for the time tagging modes only
        [DllImport(LibName)]
        extern public static int MH_SetOflCompression(int devidx, int holdtime);
        [DllImport(LibName)]
        extern public static int MH_SetMarkerHoldoffTime(int devidx, int holdofftime);
        [DllImport(LibName)]
        extern public static int MH_SetMarkerEdges(int devidx, int me1, int me2, int me3, int me4);
        [DllImport(LibName)]
        extern public static int MH_SetMarkerEnable(int devidx, int en1, int en2, int en3, int en4);
        [DllImport(LibName)]
        extern public static int MH_ReadFiFo(int devidx, uint[] buffer, ref int nactual);

        // for event filtering, time tagging modes only
        [DllImport(LibName)]
        extern public static int MH_SetRowEventFilter(int devidx, int rowidx, int timerange, int matchcnt, int inverse, int usechannels, int passchannels);
        [DllImport(LibName)]
        extern public static int MH_EnableRowEventFilter(int devidx, int rowidx, int enable);
        [DllImport(LibName)]
        extern public static int MH_SetMainEventFilterParams(int devidx, int timerange, int matchcnt, int inverse);
        [DllImport(LibName)]
        extern public static int MH_SetMainEventFilterChannels(int devidx, int rowidx, int usechannels, int passchannels);
        [DllImport(LibName)]
        extern public static int MH_EnableMainEventFilter(int devidx, int enable);
        [DllImport(LibName)]
        extern public static int MH_SetFilterTestMode(int devidx, int testmode);
        [DllImport(LibName)]
        extern public static int MH_GetRowFilteredRates(int devidx, ref int syncrate, int[] cntrates);
        [DllImport(LibName)]
        extern public static int MH_GetMainFilteredRates(int devidx, ref int syncrate, int[] cntrates);

        // for debugging only
        [DllImport(LibName)]
        extern public static int MH_GetDebugInfo(int devidx, StringBuilder debuginfo);
        [DllImport(LibName)]
        extern public static int MH_GetNumOfModules(int devidx, ref int nummod);
        [DllImport(LibName)]
        extern public static int MH_GetModuleInfo(int devidx, ref int modelcode, ref int versioncode);
        [DllImport(LibName)]
        extern public static int MH_SaveDebugDump(int devidx, StringBuilder filepath);

        // for White Rabbit only
        [DllImport(LibName)]
        extern public static int MH_WRabbitGetMAC(int devidx, byte[] mac_addr);                                         //changed in v4.0
        [DllImport(LibName)]
        extern public static int MH_WRabbitSetMAC(int devidx, byte[] mac_addr);                                         //changed in v4.0
        [DllImport(LibName)]
        extern public static int MH_WRabbitGetInitScript(int devidx, StringBuilder initscript);
        [DllImport(LibName)]
        extern public static int MH_WRabbitSetInitScript(int devidx, StringBuilder initscript);
        [DllImport(LibName)]
        extern public static int MH_WRabbitGetSFPData(int devidx, StringBuilder[] sfpnames, int[] dTxs, int[] dRxs, int[] alphas);
        [DllImport(LibName)]
        extern public static int MH_WRabbitSetSFPData(int devidx, StringBuilder[] sfpnames, int[] dTxs, int[] dRxs, int[] alphas);
        [DllImport(LibName)]
        extern public static int MH_WRabbitInitLink(int devidx, int link_on);
        [DllImport(LibName)]
        extern public static int MH_WRabbitSetMode(int devidx, int bootfromscript, int reinit_with_mode, int mode);
        [DllImport(LibName)]
        extern public static int MH_WRabbitSetTime(int devidx, uint timehidw, uint timelodw);
        [DllImport(LibName)]
        extern public static int MH_WRabbitGetTime(int devidx, ref uint timehidw, ref uint timelodw, ref uint subsec16ns);
        [DllImport(LibName)]
        extern public static int MH_WRabbitGetStatus(int devidx, ref int wrstatus);
        [DllImport(LibName)]
        extern public static int MH_WRabbitGetTermOutput(int devidx, StringBuilder buffer, ref int nchar);

        // for use with external FPGA only
        [DllImport(LibName)]
        extern public static int MH_ExtFPGAInitLink(int devidx, int linknumber, int on);
        [DllImport(LibName)]
        extern public static int MH_ExtFPGAGetLinkStatus(int devidx, int linknumber, ref uint status);
        [DllImport(LibName)]
        extern public static int MH_ExtFPGASetMode(int devidx, int mode, int loopback);
        [DllImport(LibName)]
        extern public static int MH_ExtFPGAResetStreamFifos(int devidx);
        [DllImport(LibName)]
        extern public static int MH_ExtFPGAUserCommand(int devidx, int write, uint addr, ref uint data);


        //////////////////////////// MHLib Error Codes //////////////////////////////

        public const int MH_ERROR_NONE                      =    0;

        public const int MH_ERROR_DEVICE_OPEN_FAIL          =   -1;
        public const int MH_ERROR_DEVICE_BUSY               =   -2;
        public const int MH_ERROR_DEVICE_HEVENT_FAIL        =   -3;
        public const int MH_ERROR_DEVICE_CALLBSET_FAIL      =   -4;
        public const int MH_ERROR_DEVICE_BARMAP_FAIL        =   -5;
        public const int MH_ERROR_DEVICE_CLOSE_FAIL         =   -6;
        public const int MH_ERROR_DEVICE_RESET_FAIL         =   -7;
        public const int MH_ERROR_DEVICE_GETVERSION_FAIL    =   -8;
        public const int MH_ERROR_DEVICE_VERSION_MISMATCH   =   -9;
        public const int MH_ERROR_DEVICE_NOT_OPEN           =  -10;
        public const int MH_ERROR_DEVICE_LOCKED             =  -11;
        public const int MH_ERROR_DEVICE_DRIVERVER_MISMATCH =  -12;

        public const int MH_ERROR_INSTANCE_RUNNING          =  -16;
        public const int MH_ERROR_INVALID_ARGUMENT          =  -17;
        public const int MH_ERROR_INVALID_MODE              =  -18;
        public const int MH_ERROR_INVALID_OPTION            =  -19;
        public const int MH_ERROR_INVALID_MEMORY            =  -20;
        public const int MH_ERROR_INVALID_RDATA             =  -21;
        public const int MH_ERROR_NOT_INITIALIZED           =  -22;
        public const int MH_ERROR_NOT_CALIBRATED            =  -23;
        public const int MH_ERROR_DMA_FAIL                  =  -24;
        public const int MH_ERROR_XTDEVICE_FAIL             =  -25;
        public const int MH_ERROR_FPGACONF_FAIL             =  -26;
        public const int MH_ERROR_IFCONF_FAIL               =  -27;
        public const int MH_ERROR_FIFORESET_FAIL            =  -28;
        public const int MH_ERROR_THREADSTATE_FAIL          =  -29;
        public const int MH_ERROR_THREADLOCK_FAIL           =  -30;

        public const int MH_ERROR_USB_GETDRIVERVER_FAIL     =  -32;
        public const int MH_ERROR_USB_DRIVERVER_MISMATCH    =  -33;
        public const int MH_ERROR_USB_GETIFINFO_FAIL        =  -34;
        public const int MH_ERROR_USB_HISPEED_FAIL          =  -35;
        public const int MH_ERROR_USB_VCMD_FAIL             =  -36;
        public const int MH_ERROR_USB_BULKRD_FAIL           =  -37;
        public const int MH_ERROR_USB_RESET_FAIL            =  -38;

        public const int MH_ERROR_LANEUP_TIMEOUT            =  -40;
        public const int MH_ERROR_DONEALL_TIMEOUT           =  -41;
        public const int MH_ERROR_MB_ACK_TIMEOUT            =  -42;
        public const int MH_ERROR_MACTIVE_TIMEOUT           =  -43;
        public const int MH_ERROR_MEMCLEAR_FAIL             =  -44;
        public const int MH_ERROR_MEMTEST_FAIL              =  -45;
        public const int MH_ERROR_CALIB_FAIL                =  -46;
        public const int MH_ERROR_REFSEL_FAIL               =  -47;
        public const int MH_ERROR_STATUS_FAIL               =  -48;
        public const int MH_ERROR_MODNUM_FAIL               =  -49;
        public const int MH_ERROR_DIGMUX_FAIL               =  -50;
        public const int MH_ERROR_MODMUX_FAIL               =  -51;
        public const int MH_ERROR_MODFWPCB_MISMATCH         =  -52;
        public const int MH_ERROR_MODFWVER_MISMATCH         =  -53;
        public const int MH_ERROR_MODPROPERTY_MISMATCH      =  -54;
        public const int MH_ERROR_INVALID_MAGIC             =  -55;
        public const int MH_ERROR_INVALID_LENGTH            =  -56;
        public const int MH_ERROR_RATE_FAIL                 =  -57;
        public const int MH_ERROR_MODFWVER_TOO_LOW          =  -58;
        public const int MH_ERROR_MODFWVER_TOO_HIGH         =  -59;
        public const int MH_ERROR_MB_ACK_FAIL               =  -60;

        public const int MH_ERROR_EEPROM_F01                =  -64;
        public const int MH_ERROR_EEPROM_F02                =  -65;
        public const int MH_ERROR_EEPROM_F03                =  -66;
        public const int MH_ERROR_EEPROM_F04                =  -67;
        public const int MH_ERROR_EEPROM_F05                =  -68;
        public const int MH_ERROR_EEPROM_F06                =  -69;
        public const int MH_ERROR_EEPROM_F07                =  -70;
        public const int MH_ERROR_EEPROM_F08                =  -71;
        public const int MH_ERROR_EEPROM_F09                =  -72;
        public const int MH_ERROR_EEPROM_F10                =  -73;
        public const int MH_ERROR_EEPROM_F11                =  -74;
        public const int MH_ERROR_EEPROM_F12                =  -75;
        public const int MH_ERROR_EEPROM_F13                =  -76;
        public const int MH_ERROR_EEPROM_F14                =  -77;
        public const int MH_ERROR_EEPROM_F15                =  -78;

        public const int MH_ERROR_UNSUPPORTED_FUNCTION      =  -80;
        public const int MH_ERROR_WRONG_TRGMODE             =  -81;
        public const int MH_ERROR_BULKRDINIT_FAIL           =  -82;
        public const int MH_ERROR_CREATETHREAD_FAIL         =  -83;
        public const int MH_ERROR_FILEOPEN_FAIL             =  -84;
        public const int MH_ERROR_FILEWRITE_FAIL            =  -85;
        public const int MH_ERROR_FILEREAD_FAIL             =  -86;

        public const int MH_ERROR_SFP_BUSY                  =  -90;

        public const int MH_ERROR_INVALID_ARGUMENT_1        = -201;
        public const int MH_ERROR_INVALID_ARGUMENT_2        = -202;
        public const int MH_ERROR_INVALID_ARGUMENT_3        = -203;
        public const int MH_ERROR_INVALID_ARGUMENT_4        = -204;
        public const int MH_ERROR_INVALID_ARGUMENT_5        = -205;
        public const int MH_ERROR_INVALID_ARGUMENT_6        = -206;
        public const int MH_ERROR_INVALID_ARGUMENT_7        = -207;
        public const int MH_ERROR_INVALID_ARGUMENT_8        = -208;
    
    }
}

