/************************************************************************

  Demo access to MultiHarp 150/160 hardware via MHLIB v 4.0
  The program performs a measurement based on hardcoded settings.
  The resulting histogram is stored in an ASCII output file.

  Michael Wahl, PicoQuant GmbH, January 2025

  Note: This is a console application

  Note: At the API level channel numbers are indexed 0..N-1 
    where N is the number of channels the device has.
 
  Tested with the following compilers:

  - Microsoft Visual C# 2019 (Windows)
  - Mono 6.12.0 (Windows and Linux)

************************************************************************/

using System;  // for Console
using System.Text;  // for StringBuilder 
using System.IO;  // for File
using static MHLibSpace.MHLib;

class HistoMode
{
    static void Main()
    {
        //////////////////////// some settings you can change /////////////////////////

        int Binning = 0;  // you can change this, observe limits
        int Offset = 0;   // you can change this, observe limits
        int Tacq = 1000;  // Measurement time in millisec, you can change this, observe limits

        int SyncDivider = 1;  // you can change this, observe limits

        int SyncTriggerEdge = EDGE_FALLING;  // you can change this to EDGE_RISING
        int SyncTriggerLevel = -50;          // in mV, you can change this, observe limits 

        int InputTriggerEdge = EDGE_FALLING; // you can change this to EDGE_RISING
        int InputTriggerLevel = -50;         // in mV, you can change this, observe limits 

        /////////////////////////////// program variables //////////////////////////////

        int i, j;
        int retcode;
        string cmd = "";
        int[] dev = new int[MAXDEVNUM];
        int found = 0;
        int NumChannels = 0;

        StringBuilder LibVer = new StringBuilder(8);
        StringBuilder Serial = new StringBuilder(8);
        StringBuilder Errstr = new StringBuilder(40);
        StringBuilder Model = new StringBuilder(32);
        StringBuilder Partno = new StringBuilder(16);
        StringBuilder Version = new StringBuilder(16);
        StringBuilder Wtext = new StringBuilder(16384);

        double Resolution = 0;
        int HistLen = 0;

        int Syncrate = 0;
        int Countrate = 0;
        int ctcstatus = 0;
        int flags = 0;
        int warnings = 0;


        uint[][] counts = new uint[MAXINPCHAN][];
        for (i = 0; i < MAXINPCHAN; i++)
            counts[i] = new uint[MAXHISTLEN];

        StreamWriter SW = null;

        ///////////////////////////// main program code /////////////////////////////

        Console.WriteLine("MultiHarp MHLib Demo Application                       PicoQuant GmbH, 2025");
        Console.WriteLine("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~");


        if (APICALL(() => MH_GetLibraryVersion(LibVer)) < 0)
            goto ex;

        Console.WriteLine("MHLib Version is " + LibVer);

        if (LibVer.ToString() != LIB_VERSION)
        {
            Console.WriteLine("This program requires MHLib v." + LIB_VERSION);
            goto ex;
        }

        try
        {
            SW = File.CreateText("histomodeout.txt");
        }
        catch (Exception)
        {
            Console.WriteLine("Error creating file");
            goto ex;
        }

        Console.WriteLine("Searching for MultiHarp devices...");
        Console.WriteLine("Devidx     Status");

        for (i = 0; i < MAXDEVNUM; i++)
        {
            Serial.Clear();
            // not using APICALL here as we wish to handle errors gracefully 
            retcode = MH_OpenDevice(i, Serial);
            if (retcode == 0) // grab any MultiHarp we can open
            {
                Console.WriteLine("  {0}        S/N {1}", i, Serial);
                dev[found] = i; // keep index to devices we want to use
                found++;
            }
            else
            {
                if (retcode == MH_ERROR_DEVICE_OPEN_FAIL)
                {
                    Console.WriteLine("  {0}        no device", i);
                }
                else
                {
                    MH_GetErrorString(Errstr, retcode);
                    Console.WriteLine("  {0}        S/N {1}", i, Errstr);
                }
            }
        }

        // In this demo we will use the first device we find, i.e. dev[0].
        // You can also use multiple devices in parallel.
        // You can also check for specific serial numbers, so that you always know 
        // which physical device you are talking to.

        if (found < 1)
        {
            Console.WriteLine("No device available.");
            goto ex;
        }

        Console.WriteLine("Using device {0}", dev[0]);
        Console.WriteLine();
        Console.WriteLine("Initializing the device...");

        if (APICALL(() => MH_Initialize(dev[0], MODE_HIST, 0)) < 0)  // Histo mode with internal clock))
            goto ex;

        if (APICALL(() => MH_GetHardwareInfo(dev[0], Model, Partno, Version)) < 0) // only for information
            goto ex;

        Console.WriteLine("Found Model {0} Part no {1} Version {2}", Model, Partno, Version);

        if (APICALL(() => MH_GetNumOfInputChannels(dev[0], ref NumChannels)) < 0)
            goto ex;

        Console.WriteLine("Device has {0} input channels.", NumChannels);

        if (APICALL(() => MH_SetSyncDiv(dev[0], SyncDivider)) < 0)
            goto ex;

        if (APICALL(() => MH_SetSyncEdgeTrg(dev[0], SyncTriggerLevel, SyncTriggerEdge)) < 0)
            goto ex;

        if (APICALL(() => MH_SetSyncChannelOffset(dev[0], 0)) < 0) // in ps, can emulate a cable delay
            goto ex;

        for (i = 0; i < NumChannels; i++) // we use the same input settings for all channels
        {

            if (APICALL(() => MH_SetInputEdgeTrg(dev[0], i, InputTriggerLevel, InputTriggerEdge)) < 0)
                goto ex;

            if (APICALL(() => MH_SetInputChannelOffset(dev[0], i, 0)) < 0)  // in ps, can emulate a cable delay
                goto ex;

            if (APICALL(() => MH_SetInputChannelEnable(dev[0], i, 1)) < 0)
                goto ex;
        }

        if (APICALL(() => MH_SetHistoLen(dev[0], MAXLENCODE, out HistLen)) < 0)
            goto ex;

        Console.WriteLine("Histogram length is {0}", HistLen);

        if (APICALL(() => MH_SetBinning(dev[0], Binning)) < 0)
            goto ex;

        if (APICALL(() => MH_SetOffset(dev[0], Offset)) < 0)
            goto ex;

        if (APICALL(() => MH_GetResolution(dev[0], ref Resolution)) < 0)
            goto ex;

        Console.WriteLine("Resolution is {0} ps", Resolution);

        // after Init allow 150 ms for valid  count rate readings
        // subsequently you get new values after every 100ms
        System.Threading.Thread.Sleep(150);

        Console.WriteLine();

        if (APICALL(() => MH_GetSyncRate(dev[0], ref Syncrate)) < 0)
            goto ex;

        Console.WriteLine("Syncrate = {0}/s", Syncrate);

        for (i = 0; i < NumChannels; i++) // for all channels
        {
            if (APICALL(() => MH_GetCountRate(dev[0], i, ref Countrate)) < 0)
                goto ex;

            Console.WriteLine("Countrate[{0}] = {1}/s", i, Countrate);
        }

        Console.WriteLine();

        // after getting the count rates you can check for warnings
        if (APICALL(() => MH_GetWarnings(dev[0], ref warnings)) < 0)
            goto ex;

        if (warnings != 0)
        {
            if (APICALL(() => MH_GetWarningsText(dev[0], Wtext, warnings)) < 0)
                goto ex;
            Console.WriteLine("{0}", Wtext);
        }

        if (APICALL(() => MH_SetStopOverflow(dev[0], 0, 10000)) < 0)  // for example only
            goto ex;

        while (cmd != "q")
        {
            if (APICALL(() => MH_ClearHistMem(dev[0])) < 0)
                goto ex;

            Console.WriteLine("press RETURN to start measurement");
            Console.ReadLine();

            if (APICALL(() => MH_GetSyncRate(dev[0], ref Syncrate)) < 0)
                goto ex;

            Console.WriteLine("Syncrate = {0}/s", Syncrate);

            for (i = 0; i < NumChannels; i++) // for all channels
            {
                if (APICALL(() => MH_GetCountRate(dev[0], i, ref Countrate)) < 0)
                    goto ex;

                Console.WriteLine("Countrate[{0}] = {1}/s", i, Countrate);
            }

            if (APICALL(() => MH_StartMeas(dev[0], Tacq)) < 0)
                goto ex;

            Console.WriteLine("Measuring for {0} milliseconds...", Tacq);

            ctcstatus = 0;
            while (ctcstatus == 0)  // wait until measurement is completed
            {
                if (APICALL(() => MH_CTCStatus(dev[0], ref ctcstatus)) < 0)
                    goto ex;
            }

            if (APICALL(() => MH_StopMeas(dev[0])) < 0)
                goto ex;

            Console.WriteLine();
            for (i = 0; i < NumChannels; i++)  // for all channels
            {
                if (APICALL(() => MH_GetHistogram(dev[0], counts[i], i)) < 0)
                    goto ex;

                double Integralcount = 0;
                for (j = 0; j < HistLen; j++)
                {
                    Integralcount += counts[i][j];
                }

                Console.WriteLine("  Integralcount[{0}] = {1}", i, Integralcount);
            }

            Console.WriteLine();

            if (APICALL(() => MH_GetFlags(dev[0], ref flags)) < 0)
                goto ex;

            if ((flags & FLAG_OVERFLOW) != 0)
            {
                Console.WriteLine("  Overflow.");
            }

            Console.WriteLine("Enter c to continue or q to quit and save the count data.");
            cmd = Console.ReadLine();

        }  // while

        for (i = 0; i < NumChannels; i++)
             SW.Write("  ch{0,2} ", i+1);
        SW.Write("\n");

        for (j = 0; j < HistLen; j++)
        {
            for (i = 0; i < NumChannels; i++)
            {
                SW.Write("{0,6} ", counts[i][j]);
            }
            SW.WriteLine();
        }

    ex:
        for (i = 0; i < MAXDEVNUM; i++)  // no harm to close all
            MH_CloseDevice(i);

        if (SW != null)
        {
            try
            {
                SW.Flush();
                SW.Close();
                SW.Dispose();
                SW = null;
            }
            catch (Exception e)
            {
                Console.WriteLine("Error closing the file: " + e);
            }
        }

        Console.WriteLine("press RETURN to exit");
        Console.ReadLine();
    }
}

