% MHLibcall  Calls a routine from MHLib with error handling
%   
%   [retcode, ...] = MHLibcall(routinename, ...) calls the routine
%   specified by routinename, where ... are variable lists of arguments 
%   that must match the signature of the routine.    
%   MHLib must have been loaded before MHLibcall can be used.
%   MHLibcall is essentially a wrapper around the native Matlab function 
%   callib. The usage is therefore very similar, only that the library name
%   is omitted. After invoking calllib MHLibcall will check for errors (ret<0)
%   and if there was an error translate it to an error string.
%   Upon detecting an error MHLibcall will invoke error() to report the 
%   error code and error string and then terminate the program. This is 
%   useful for lean demo code but may not be ideal for production code.
%   In that case you may want to modify MHLibcall and, for instance,
%   replace error(..) by warning(..). Note, however, that then you
%   need to check the return code of MHLibcall and handle possible errors 
%   at the caller's level.
% 
%   See also LOADLIBRARY, CALLLIB.

function varargout = MHLibcall(varargin)

% see https://nl.mathworks.com/help/matlab/matlab_prog/support-variable-number-of-inputs.html
% and https://nl.mathworks.com/help/matlab/matlab_prog/support-variable-number-of-outputs.html

funcname = varargin{1};

nOutputs = nargout;
if(nOutputs==0) %in case we were called without assigning the return code to a variable
    nOutputs = 1;
end
varargout = cell(1,nOutputs);

[varargout{:}] = calllib('MHLib', varargin{:});

retcode = varargout{1};  % this is our retcode from the DLL call

if (retcode<0)
    ErrorStr   = blanks(40); % provide enough length!
    ErrorPtr   = libpointer('cstring', ErrorStr);
    [ret, ErrorStr] = calllib('MHLib', 'MH_GetErrorString', ErrorPtr, retcode);
    % if you do not want the program to abort upon every error here 
    % you may want to replace error(..) by warning(..). Note that then you
    % need to check the return code of MHLibcall and handle possible errors 
    % at the caller's level.
    if(ret==0)
        error('MHLib Error %d (%s) calling %s', retcode, ErrorStr, funcname);
    else
        error('MHLib Error %d calling %s\n', retcode, funcname);
    end       
end






