
% Demo for access to MultiHarp 150/160 hardware via MHLib v 4.0.
% The program performs a TTTR measurement based on hardcoded settings.
% The resulting data stream is stored in a binary output file.
%
% Michael Wahl, PicoQuant, January 2025

% Constants from mhdefin.h in capitals. You cannot change these!

REQLIBVER   =   '4.0';    % the library version this program expects
MAXDEVNUM   =       8;    % max number of MultiHarp devices
TTREADMAX   = 1048576;    % up to 1M event records may be received at a time 
MODE_T2     =       2;    % Time tagging mode T2
MODE_T3     =       3;    % Time tagging mode T3

FLAG_FIFOFULL = hex2dec('0002');

% limits for MH_SetSyncEdgeTrg and MH_SetInputEdgeTrg
TRGLVLMIN	  =       -1200;   % mV
TRGLVLMAX	  =        1200;   % mV

% limits for MH_SetOffset
OFFSETMIN	  =           0;   % ns
OFFSETMAX	  =   100000000;   % ns

% limits for Tacq in MH_StartMeas
ACQTMIN		  =           1;   % ms
ACQTMAX		  =   360000000;   % ms  (100*60*60*1000ms = 100h)

% Selected Errorcode from errorcodes.h
MH_ERROR_DEVICE_OPEN_FAIL		 = -1;

% Settings for the measurement, adapt to your setup!
 
Mode          = MODE_T3; %  you can change this
SyncTrgEdge   =       0; %  you can change this
SyncTrgLevel  =    -100; %  you can change this
InputTrgEdge  =       0; %  you can change this
InputTrgLevel =    -100; %  you can change this
SyncDiv       =       1; %  you can change this (observe mode!)
Binning       =       0; %  you can change this
Tacq          =   10000; %  you can change this       
    

fprintf('\nMultiHarp 150/160 MHLib Demo Application             PicoQuant 2025\n');

if (~libisloaded('MHLib'))    
    % Attention: The header file name given below is case sensitive and must
    % be spelled exactly the same as the actual name on disk except the file 
    % extension. 
    % Wrong case will apparently do the load successfully but you will not
    % be able to access the library!
    % The alias is used to provide a fixed spelling for any further access via
    % calllib() etc, which is also case sensitive.
    
    % To load the right dll for the given OS we use the mexext command.
    % Note that only x86-64 is supported.
    OS = mexext;   
    if strcmp('mexw64', OS)  % Windows
        DLL = 'MHLib64.dll';                              
    elseif strcmp('mexa64', OS)  % Linux
        DLL = '/opt/picoquant/mhlib/libmhlib.so';  
    else
        fprintf('\nNo supported OS\n');
        return;
    end
    
    % Load the library
    try
        loadlibrary(DLL, 'mhlib.h', 'alias', 'MHLib');
    catch ME
        fprintf('Error loading the library: %s\n', ME.message);
        return;
    end

else
    fprintf('Note: MHLib was already loaded\n');
end

if (libisloaded('MHLib'))
    fprintf('MHlib opened successfully\n');
    % libfunctionsview MHlib; % use this to test for proper loading
else
    fprintf('Could not open MHLib\n');
    return;
end

% Normally one uses Matlab's native calllib() to invoke DLL
% routines. To simplify things we use MHLibcall.m where possible.
% Type help MHLibcall to understand this little helper routine.
% Note that it will terminate the program in case of errors.
        
LibVersion    = blanks(8); % provide enough length!
LibVersionPtr = libpointer('cstring', LibVersion);
[ret, LibVersion] = MHLibcall('MH_GetLibraryVersion', LibVersionPtr);
fprintf('MHLib version is %s\n', LibVersion);
if ~strcmp(LibVersion,REQLIBVER)
    fprintf('This program requires MHLib version %s\n', REQLIBVER);
    return;
end

fid = fopen('tttrmode.out','wb'); % binary file
if (fid<0)
    fprintf('Cannot open output file\n');
    return;
end

fprintf('\nSearching for MultiHarp devices...');

dev = [];
found = 0;
Serial     = blanks(8); %enough length!
SerialPtr  = libpointer('cstring', Serial);
ErrorStr   = blanks(40); %enough length!
ErrorPtr   = libpointer('cstring', ErrorStr);

for i=0:MAXDEVNUM-1
    % here we do not use MHLibcall to allow more specific error handling  
    [ret, Serial] = calllib('MHLib', 'MH_OpenDevice', i, SerialPtr);
    if (ret==0)       % Grab any MultiHarp we successfully opened
        fprintf('\n  %1d        S/N %s', i, Serial);
        found = found+1;            
        dev(found)=i; % keep index to devices we may want to use
    else
        if(ret==MH_ERROR_DEVICE_OPEN_FAIL)
            fprintf('\n  %1d        no device', i);
        else 
            [ret, ErrorStr] = calllib('MHLib', 'MH_GetErrorString', ErrorPtr, ret);
            fprintf('\n  %1d        %s', i,ErrorStr);
        end
    end
end
    
% in this demo we will use the first MultiHarp device we found, i.e. dev(1)
% you could also check for a specific serial number, so that you always know 
% which physical device you are talking to.

if (found<1)
	fprintf('\nNo device available. Aborted.\n');
	return; 
end

fprintf('\nUsing device #%1d',dev(1));
fprintf('\nInitializing the device...');

[ret] = MHLibcall('MH_Initialize', dev(1), Mode, 0); 
% Note that here we assign but then actually ignore the value ret.
% Also note the brackets. We do this for consistency with all other
% calls to MHLibcall where there can be more return values in the bracket. 
% MHLibcall always requires a correct assignment of return values
% according to the signature of the called function. This is important 
% for the variable aruments lists passed in and out from MHLibcall 
% (as well as for callib) to work properly.

% the following library call is only for gathering device information
Model      = blanks(24); % provide enough length!
Partno     = blanks(8);  % provide enough length!
Version    = blanks(8);  % provide enough length!
ModelPtr   = libpointer('cstring', Model);
PartnoPtr  = libpointer('cstring', Partno);
VersionPtr = libpointer('cstring', Version);
[ret, Model, Partno] = MHLibcall('MH_GetHardwareInfo', dev(1), ModelPtr, PartnoPtr, VersionPtr);
fprintf('\nFound model %s part number %s version %s', Model, Partno, Version);

NumInpChannels = int32(0);
NumChPtr = libpointer('int32Ptr', NumInpChannels);
[ret, NumInpChannels] = MHLibcall('MH_GetNumOfInputChannels', dev(1), NumChPtr); 
fprintf('\nDevice has %i input channels.', NumInpChannels);             

[ret] = MHLibcall('MH_SetSyncDiv', dev(1), SyncDiv);

[ret] = MHLibcall('MH_SetSyncEdgeTrg', dev(1), SyncTrgLevel, SyncTrgEdge);
 
[ret] = MHLibcall('MH_SetSyncChannelOffset', dev(1), 0);

for i=0:NumInpChannels-1 % we use the same input settings for all channels
    [ret] = MHLibcall('MH_SetInputEdgeTrg', dev(1), i, InputTrgLevel, InputTrgEdge);
    [ret] = MHLibcall('MH_SetInputChannelOffset', dev(1), i, 0);
end

if (Mode ~= MODE_T2)    % MH_SetBinning is meaningless in T2 mode
    [ret] = MHLibcall('MH_SetBinning', dev(1), Binning);
end
if (Mode ~= MODE_T2)    % MH_SetOffset is meaningless in T2 mode
    [ret] = MHLibcall('MH_SetOffset', dev(1), 0);
end

% MH_GetResolution is meaningless in T2 mode but it does no harm
Resolution = 0;
ResolutionPtr = libpointer('doublePtr', Resolution);
[ret, Resolution] = MHLibcall('MH_GetResolution', dev(1), ResolutionPtr);
fprintf('\nResolution=%1dps', Resolution);

pause(0.2); % after Init or SetSyncDiv you must allow 150 ms for valid new count rates
            % otherwise you get new values after every 100 ms

% from here you can repeat the measurement (with the same settings)

Syncrate = 0;
SyncratePtr = libpointer('int32Ptr', Syncrate);
[ret, Syncrate] = MHLibcall('MH_GetSyncRate', dev(1), SyncratePtr);
fprintf('\nSyncrate=%1d/s', Syncrate);
 
for i=0:NumInpChannels-1
	Countrate = 0;
	CountratePtr = libpointer('int32Ptr', Countrate);
	[ret, Countrate] = MHLibcall('MH_GetCountRate', dev(1), i, CountratePtr);
	fprintf('\nCountrate%1d=%1d/s', i, Countrate);
end

Warnings = 0;
WarningsPtr = libpointer('int32Ptr', Warnings);
[ret, Warnings] = MHLibcall('MH_GetWarnings', dev(1), WarningsPtr);
if (Warnings~=0)
    Warningstext = blanks(16384); % provide enough length!
    WtextPtr     = libpointer('cstring', Warningstext);
    [ret, Warningstext] = MHLibcall('MH_GetWarningsText', dev(1), WtextPtr, Warnings);
    fprintf('\n\n%s',Warningstext);
end

% prepare buffer and pointers we will need repeatedly in the loop below
buffer  = uint32(zeros(1,TTREADMAX));
bufferptr = libpointer('uint32Ptr', buffer);
nactual = int32(0);
nactualptr = libpointer('int32Ptr', nactual);
ctcdone = int32(0);
ctcdonePtr = libpointer('int32Ptr', ctcdone);
flags = int32(0);
flagsPtr = libpointer('int32Ptr', flags);
    
Progress = 0;
fprintf('\nProgress:%9d',Progress);
       
[ret] = MHLibcall('MH_StartMeas', dev(1),Tacq); % start meassurement
       
while(1)  % the data collection loop, keep it lean for speed
   
    [ret,flags] = MHLibcall('MH_GetFlags', dev(1), flagsPtr);   
    if (bitand(uint32(flags),FLAG_FIFOFULL)) 
        fprintf('\nFiFo Overrun!\n'); 
        break;
    end
		
    [ret, buffer, nactual] = MHLibcall('MH_ReadFiFo', dev(1), bufferptr, nactualptr);
    % consider that MH_ReadFiFo may return zero events  
    if(nactual) 
        cnt = fwrite(fid, buffer(1:nactual),'uint32');
        if(cnt ~= nactual)
            fprintf('\nfile write error\n');
            break;
        end          
        Progress = Progress + nactual;
        fprintf('\b\b\b\b\b\b\b\b\b%9d',Progress);
    else
        [ret,ctcdone] = MHLibcall('MH_CTCStatus', dev(1), ctcdonePtr);      
        if (ctcdone) 
            fprintf('\nDone\n'); 
            break; % note: there may still be some data in the FIFO, so it may be
                   % advisable to do a few more reads, see the demos in C or C#
        end
    end
    
	 % you can read the count rates here if needed

end % ebd of the data collection loop


[ret] = MHLibcall('MH_StopMeas', dev(1)); 
        
closedev; % close devices
    
fprintf('\nBinary output data is in tttrmode.out\n');

if(fid>0) 
    fclose(fid);
end

