{
  MultiHarp 150/160  MHLIB v4.0  Usage Demo with Delphi or Lazarus
  
  Tested with
   - Delphi 11.1 on Windows 11
   - Lazarus 3.2 / fpc 3.2.2 on Windows 11
   - Lazarus 2.0.12 / fpc 3.2.0 on Windows 10
   - Lazarus 2.2.0 / fpc 3.2.2 on Ubuntu 22.04

  Demo access to MultiHarp 150/160 Hardware via MHLIB.
  The program performs a histogram measurement based on hardcoded settings.
  The resulting histogram (65536 time bins) is stored in an ASCII output file.

  Axel Hagen, PicoQuant GmbH, August 2018
  Marcus Sackrow, PicoQuant GmbH, July 2019
  Michael Wahl, PicoQuant GmbH, May 2020, March 2021
  Revised Matthias Patting, Stefan Eilers, PicoQuantGmbH, March 2022
  Revised for v4.0, Michael Wahl, PicoQuant GmbH, January 2025

  Note: This is a console application (i.e. to be run from the command line)

  Note: At the API level channel numbers are indexed 0..N-1
        where N is the number of channels the device has.
}

program histomode;

{$IF defined(MSWINDOWS)}
  {$APPTYPE CONSOLE}  //windows needs this, Linux does not want it
{$ENDIF}

uses
  {$ifdef fpc}
  SysUtils,
  {$else}
  System.SysUtils,
  System.Ansistrings,
  {$endif}
  MHLib in 'mhlib.pas';

type
  THistogramCounts   = array [0..MAXHISTLEN - 1] of LongWord;


var
  RetCode            : LongInt;
  OutFile            : Text;
  i                  : Integer;
  Found              : Integer =   0;

  Mode               : LongInt =    MODE_HIST;
  Binning            : LongInt =            0; // you can change this (meaningless in T2 mode)
  Offset             : LongInt =            0; // normally no need to change this
  TAcq               : LongInt =         1000; // you can change this, unit is millisec
  SyncDivider        : LongInt =            1; // you can change this
  SyncTrgEdge        : LongInt = EDGE_FALLING; // you can change this
  SyncTrgLevel       : LongInt =          -50; // you can change this (mV)
  SyncChannelOffset  : LongInt =       -10000; // you can change this (like a cable delay)
  InputTrgEdge       : LongInt = EDGE_FALLING; // you can change this (mV)
  InputTrgLevel      : LongInt =          -50; // you can change this (mV)
  InputChannelOffset : LongInt =            0; // you can change this (like a cable delay)

  NumChannels        : LongInt;
  HistoBin           : LongInt;
  ChanIdx            : LongInt;
  HistLen            : LongInt;
  Resolution         : Double;
  SyncRate           : LongInt;
  CountRate          : LongInt;
  CTCStatus          : LongInt;
  IntegralCount      : Double;
  Flags              : LongInt;
  Warnings           : LongInt;
  Cmd                : Char    = #0;

  Counts             : array[0..MAXINPCHAN - 1] of THistogramCounts;

procedure Ex(RetCode: Integer);
begin
  if RetCode <> MH_ERROR_NONE then
  begin
    MH_GetErrorString(pcErrText, RetCode);
    Writeln('Error ', RetCode:3, ' = "', Trim (string(strErrText)), '"');
  end;
  Writeln;
  {$I-}
    CloseFile(OutFile);
    IOResult();
  {$I+}
  Writeln('press RETURN to exit');
  Readln;
  Halt(RetCode);
end;

begin
  Writeln;
  Writeln('MultiHarp MHLib   Usage Demo                        PicoQuant GmbH, 2025');
  Writeln('~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~');
  RetCode := MH_GetLibraryVersion(pcLibVers);
  if RetCode <> MH_ERROR_NONE then
  begin
    Writeln('MH_GetLibraryVersion error ', RetCode:3, '. Aborted.');
    Ex(RetCode);
  end;
  Writeln('MHLIB version is ' + strLibVers);
  if Trim(string(strLibVers)) <> Trim(AnsiString(LIB_VERSION)) then
    Writeln('Warning: The application was built for version ' + LIB_VERSION);

  AssignFile(OutFile, 'histomodeout.txt');
  {$I-}
    Rewrite(OutFile);
  {$I+}
  if IOResult <> 0 then
  begin
    Writeln('cannot open output file');
    Ex(MH_ERROR_NONE);
  end;

  Writeln;
  Writeln('Searching for MultiHarp devices...');
  Writeln('Devidx     Status');

  for i := 0 to MAXDEVNUM - 1 do
  begin
    RetCode := MH_OpenDevice(i, pcHWSerNr);
    //
    if RetCode = MH_ERROR_NONE then
    begin
      // Grab any MultiHarp we can open
      DevIdx[Found] := i; // keep index to devices we want to use
      Inc(Found);
      Writeln('   ', i, '      S/N ', strHWSerNr);
    end
    else
    begin
      if RetCode = MH_ERROR_DEVICE_OPEN_FAIL then
        Writeln('   ', i, '       no device')
      else
      begin
        MH_GetErrorString(pcErrText, RetCode);
        Writeln('   ', i, '       ', Trim(string(strErrText)));
      end;
    end;
  end;

  // in this demo we will use the first MultiHarp device we found,
  // i.e. iDevIdx[0].  You can also use multiple devices in parallel.
  // you could also check for a specific serial number, so that you
  // always know which physical device you are talking to.

  if Found < 1 then
  begin
    Writeln('No device available.');
    Ex(MH_ERROR_NONE);
  end;

  Writeln('Using device ', DevIdx[0]);
  Writeln('Initializing the device...');

  RetCode := MH_Initialize(DevIdx[0], Mode, MEASCTRL_SINGLESHOT_CTC); //Histo mode with internal clock
  if RetCode <> MH_ERROR_NONE then
  begin
    Writeln('MH_Initialize error ', RetCode:3, '. Aborted.');
    Ex(RetCode);
  end;

  RetCode := MH_GetHardwareInfo(DevIdx[0], pcHWModel, pcHWPartNo, pcHWVersion); // this is only for information
  if RetCode <> MH_ERROR_NONE then
  begin
    Writeln('MH_GetHardwareInfo error ', RetCode:3, '. Aborted.');
    Ex(RetCode);
  end
  else
    Writeln('Found Model ', strHWModel,'  Part no ', strHWPartNo,'  Version ', strHWVersion);

  RetCode := MH_GetNumOfInputChannels(DevIdx[0], NumChannels);
  if RetCode <> MH_ERROR_NONE then
  begin
    Writeln('MH_GetNumOfInputChannels error ', RetCode:3, '. Aborted.');
    Ex(RetCode);
  end
  else
    Writeln('Device has ', NumChannels, ' input channels.');

  Writeln;

  RetCode := MH_SetSyncDiv(DevIdx[0], SyncDivider);
  if RetCode <> MH_ERROR_NONE then
  begin
    Writeln('MH_SetSyncDiv error ', RetCode:3, '. Aborted.');
    Ex(RetCode);
  end;

  RetCode := MH_SetSyncEdgeTrg(DevIdx[0], SyncTrgLevel, SyncTrgEdge);
  if RetCode <> MH_ERROR_NONE then
  begin
    Writeln('MH_SetSyncEdgeTrg error ', RetCode:3, '. Aborted.');
    Ex(RetCode);
  end;

  RetCode := MH_SetSyncChannelOffset(DevIdx[0], SyncChannelOffset);
  if RetCode <> MH_ERROR_NONE then
  begin
    Writeln('MH_SetSyncChannelOffset error ', RetCode:3, '. Aborted.');
    Ex(RetCode);
  end;

  for ChanIdx := 0 to NumChannels - 1 do // we use the same input settings for all channels
  begin
    RetCode := MH_SetInputEdgeTrg(DevIdx[0], ChanIdx, InputTrgLevel, InputTrgEdge);
    if RetCode <> MH_ERROR_NONE then
    begin
      Writeln('MH_SetInputEdgeTrg ', ChanIdx:2, ' error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    RetCode := MH_SetInputChannelOffset(DevIdx[0], ChanIdx, InputChannelOffset);
    if RetCode <> MH_ERROR_NONE then
    begin
      Writeln('MH_SetInputChannelOffset channel ', ChanIdx:2, ' error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    RetCode := MH_SetInputChannelEnable(DevIdx[0], ChanIdx, 1);
    if RetCode <> MH_ERROR_NONE then
    begin
      Writeln('MH_SetInputChannelEnable channel ', ChanIdx:2, ' error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
  end;


  RetCode := MH_SetHistoLen(DevIdx[0], MAXLENCODE, HistLen);
  if RetCode <> MH_ERROR_NONE then
  begin
    Writeln('MH_SetHistoLen error ', RetCode:3, '. Aborted.');
    Ex(RetCode);
  end;
  Writeln('Histogram length is ', HistLen);

  RetCode := MH_SetBinning(DevIdx[0], Binning);
  if RetCode <> MH_ERROR_NONE then
  begin
    Writeln('MH_SetBinning error ', RetCode:3, '. Aborted.');
    Ex(RetCode);
  end;

  RetCode := MH_SetOffset(DevIdx[0], Offset);
  if RetCode <> MH_ERROR_NONE then
  begin
    Writeln('MH_SetOffset error ', RetCode:3, '. Aborted.');
    Ex(RetCode);
  end;

  RetCode := MH_GetResolution(DevIdx[0], Resolution);
  if RetCode <> MH_ERROR_NONE then
  begin
    Writeln('MH_GetResolution error ', RetCode:3, '. Aborted.');
    Ex(RetCode);
  end;
  Writeln('Resolution is ', Resolution:7:3, 'ps');

  // Note: After Init or SetSyncDiv you must allow > 150 ms for valid new count rate readings
  // otherwise you get new values every 100 ms
  Sleep(150);

  Writeln;

  RetCode := MH_GetSyncRate(DevIdx[0], SyncRate);
  if RetCode <> MH_ERROR_NONE then
  begin
    Writeln('MH_GetSyncRate error ', RetCode:3, '. Aborted.');
    Ex(RetCode);
  end;
  Writeln('SyncRate = ', SyncRate, '/s');

  Writeln;

  for ChanIdx := 0 to NumChannels - 1 do // for all channels
  begin
    RetCode := MH_GetCountRate(DevIdx[0], ChanIdx, CountRate);
    if RetCode <> MH_ERROR_NONE then
    begin
      Writeln('MH_GetCountRate error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('Countrate [', ChanIdx:2, '] = ', CountRate:8, '/s');
  end;

  Writeln;

  // after getting the count rates you can check for warnings
  RetCode := MH_GetWarnings(DevIdx[0], Warnings);
  if RetCode <> MH_ERROR_NONE then
  begin
    Writeln('MH_GetWarnings error ', RetCode:3, '. Aborted.');
    Ex(RetCode);
  end;
  if Warnings <> 0 then
  begin
    MH_GetWarningsText(DevIdx[0], pcWtext, Warnings);
    Writeln(strWtext);
  end;

  RetCode := MH_SetStopOverflow (DevIdx[0], 0, 10000); // for example only
  if RetCode <> MH_ERROR_NONE then
  begin
    Writeln('MH_SetStopOverflow error ', RetCode:3, '. Aborted.');
    Ex(RetCode);
  end;

  repeat
    MH_ClearHistMem(DevIdx[0]);
    if RetCode <> MH_ERROR_NONE then
    begin
      Writeln('MH_ClearHistMem error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    Writeln('press RETURN to start measurement');
    Readln;

    Writeln;

    RetCode := MH_GetSyncRate(DevIdx[0], SyncRate);
    if RetCode <> MH_ERROR_NONE then
    begin
      Writeln('MH_GetSyncRate error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('SyncRate = ', SyncRate, '/s');

    Writeln;

    for ChanIdx := 0 to NumChannels - 1 do // for all channels
    begin
      RetCode := MH_GetCountRate(DevIdx[0], ChanIdx, CountRate);
      if RetCode <> MH_ERROR_NONE then
      begin
        Writeln('MH_GetCountRate error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      Writeln('Countrate [', ChanIdx:2, '] = ', CountRate:8, '/s');
    end;

    Writeln;
    RetCode := MH_StartMeas(DevIdx[0], TAcq);
    if RetCode <> MH_ERROR_NONE then
    begin
      Writeln('MH_StartMeas error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('Measuring for ', TAcq, ' milliseconds...');

    repeat
      RetCode := MH_CTCStatus(DevIdx[0], CTCStatus);
      if RetCode <> MH_ERROR_NONE then
      begin
        Writeln('MH_CTCStatus error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;

    until CTCStatus <> 0;

    RetCode := MH_StopMeas(DevIdx[0]);
    if RetCode <> MH_ERROR_NONE then
    begin
      Writeln('MH_StopMeas error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    Writeln;

    for ChanIdx := 0 to NumChannels - 1 do // for all channels
    begin
      RetCode := MH_GetHistogram(DevIdx[0], Counts[ChanIdx, 0], ChanIdx);
      if RetCode <> MH_ERROR_NONE then
      begin
        Writeln('MH_GetHistogram error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;

      IntegralCount := 0;

      for HistoBin := 0 to HistLen - 1 do
        IntegralCount := IntegralCount + Counts[ChanIdx, HistoBin];

      Writeln('Integralcount [', ChanIdx:2, '] = ', IntegralCount:9:0);
    end;

    Writeln;

    RetCode := MH_GetFlags(DevIdx[0], Flags);
    if RetCode <> MH_ERROR_NONE then
    begin
      Writeln('MH_GetFlags error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    if (Flags and FLAG_OVERFLOW) > 0 then
      Writeln('  Overflow.');

    Writeln('Enter c to continue or q to quit and save the count data.');
    Readln(Cmd);
  until Cmd = 'q';

  Writeln;
  Writeln('Saving data to file...');

  for ChanIdx := 0 to NumChannels - 1 do
    Write(OutFile, '  ch', ChanIdx+1:2, ' ');
  Writeln(OutFile);
  for HistoBin := 0 to HistLen - 1 do
  begin
    for ChanIdx := 0 to NumChannels - 1 do
      Write(OutFile, Counts[ChanIdx, HistoBin]:6, ' ');
    Writeln(OutFile);
  end;

  MH_CloseAllDevices;

  Ex(MH_ERROR_NONE);
end.

