/*********************************************************************
 * PicoQuant MH Rust crate
 *
 * This file demonstrates how to wrap the PicoQuant MH DLL using Rust.
 *
 * The example code opens and configures the device, performs a short
 * measurement, and writes the raw data to a file.
 *
 * Each function from the C DLL is wrapped with Rust data types. Access to
 * the C DLL functions and C data types is marked as unsafe; they are used
 * briefly before converting to Rust code.
 *
 * For more information, please refer to:
 * https://doc.rust-lang.org/book/ch19-01-unsafe-rust.html
 *
 * Using the Rust approach, all functions return a Result type, which can be
 * either Ok or Err. The Ok variant contains the return value of the specific
 * function, enforcing error handling for any potential issues.
 *
 * For quick prototyping, it is common to use the unwrap() function, which either
 * returns the value from the function or panics. This is used frequently in the
 * code below for better readability. For production code, it is recommended to handle
 * all possible failure cases.
 *
 * Example:
 *
 * mh_function(arg1, arg2).unwrap();
 *
 * or for production code:
 *
 * match mh_function(arg1, arg2) {
 *     Ok(()) => handle_ok(),
 *     Err(_) => handle_err(),
 * }
 *
 *********************************************************************/

use std::fs::File;
use std::io::Write;
use std::thread::sleep;
use std::time::Duration;

use mh::{Edge};

pub mod error_codes;
pub mod mh;

fn main() {
    println!("~~~~~~~~~~~~~~~~~~~~~~~~");
    println!("~~~ MH Rust Demo  ~~~");
    println!("~~~~~~~~~~~~~~~~~~~~~~~~");

    // Set basic decvice settings
    let devidx: i32 = 0;
    let mode: mh::Mode = mh::Mode::T2;
    let ref_source: mh::Refsource = mh::Refsource::Internal;

    const ACQUISITION_TIME_MS: i32 = 1000;
    const TRIGGER_EDGE: Edge = mh::Edge::Falling;
    const SYNC_TRIGGER_LEVEL: i32 = -50; //mV
    const CHAN_TRIGGER_LEVEL: i32 = -50; //mV
    const CHANNEL_IDX: i32 = -1; //-1 for all channels
    let output_file_path: &str = "tttr_data_log.bin";

    // in Rust, files get automatically closed when it goes out of scope
    let mut file = File::create(output_file_path).unwrap();
    
    // get lib version
    let r_str = mh::mh_get_library_version();
    println!("Library version: {:?}", r_str);
    
    // open device, based on its index and print out its serial number
    let serial = mh::mh_open_device(devidx);
    println!("Serial number = {:?}", serial);
    
    
    match mh::mh_initialize(devidx, mode, ref_source) {
        Ok(()) => println!("Initialization Done!"),
        Err(_) => println!("Initialization not possible!"),
    }
    
    // retreive hardware info and print it out
    let info = mh::mh_get_hardware_info(devidx);
    match info {
        Ok(info) => println!(
            "MH hardware info: Model: {:?}, Part Number: {:?}, Version: {:?}",
            info.model, info.part_number, info.version
        ),
        Err(_) => println!("Failed to get hardware info."),
    }

    match mh::mh_get_base_resolution(devidx) {
        Ok((steps, resolution)) => {
            println!("Device base resolution steps: {}", steps);
            println!("Device base resolution: {} ps", resolution);
        }
        Err(error_code) => {
            eprintln!(
                "Failed to get base resolution. Error code: {:?}",
                error_code
            );
        }
    }

    match mh::mh_get_features(devidx) {
        Ok(features) => {
            println!("Device features: {}", features);
        }
        Err(error_code) => {
            eprintln!(
                "Failed to get device features. Error code: {:?}",
                error_code
            );
        }
    }

    let nchannels: i32 = mh::mh_get_number_of_input_channels(devidx).unwrap();
    println!("Number of input channels: {}", nchannels);

    // setup measurement
    mh::mh_set_sync_edge_trigger(devidx, SYNC_TRIGGER_LEVEL, TRIGGER_EDGE).unwrap();

    mh::mh_set_input_edge_trigger(devidx, CHANNEL_IDX, CHAN_TRIGGER_LEVEL, TRIGGER_EDGE)
        .unwrap();

    // After Init allow 150 ms for valid  count rate readings
    sleep(Duration::from_millis(150));

    let (sync_rate, channel_rates) = mh::mh_get_all_count_rates(devidx).unwrap();
    println!("Syncrate =  {:?}", sync_rate);
    println!("Countrate CH =  {:?}", channel_rates);

    println!("\nCheck for warnings:");
    let warning : i32 = mh::mh_get_warnings(devidx).unwrap();
    println!("{}", mh::mh_get_warnings_text(devidx, warning).unwrap() );

    // start real measurement
    mh::mh_start_meas(devidx, ACQUISITION_TIME_MS).unwrap();

    let mut measurement_done = false;
    let mut events_written: u64 = 0;

    while !measurement_done {
        // check if fifo has not overrun
        let flags = mh::mh_get_flags(devidx).unwrap();
        if 0 != (mh::Flags::FlagFifoFull as i32 & flags) {
            panic!("Fifo must never overrun!!!");
        }

        let buffer = mh::mh_read_fifo(devidx).unwrap();
        // Convert Vec<u32> to Vec<u8> using bytemuck crate
        let byte_slice: &[u8] = bytemuck::cast_slice(&buffer);
        // write data into file
        file.write_all(byte_slice).unwrap();
        events_written += buffer.len() as u64;

        // if no data is avialable, check ctc status to see if measurement is done
        if buffer.is_empty() {
            let ctc_status = mh::mh_get_ctc_status(devidx).unwrap();
            if ctc_status != 0 {
                measurement_done = true;
            }
        }
    }
    // ensure all data is written to file, non is in any buffer
    file.flush().unwrap();

    println!("{} events written to file", events_written);

    // stop the measurement
    mh::mh_stop_meas(devidx).unwrap();
    // close device at the end //
    mh::mh_close_device(devidx).unwrap();


}
