#ifndef __PPSI_PP_INSTANCE_H__
#define __PPSI_PP_INSTANCE_H__

/*
 * The "instance structure is the most important object in ppsi: is is
 * now part of a separate header because ppsi.h must refer to it, and
 * to diagnostic macros, but also diag-macros.h needs to look at
 * ppi->flags.
 */
#ifndef __PPSI_PPSI_H__
#warning "Please include <ppsi/ppsi.h>, don't refer to pp-instance.h"
#endif

/*
 * Runtime options. Default values can be overridden by command line.
 */
struct pp_runtime_opts {
	ClockQuality clock_quality;
	Integer32 ttl;
	int flags;		/* see below */
	Integer16 ap, ai;
	Integer16 s;
	Integer8 announce_intvl;
	int sync_intvl;
	int prio1;
	int prio2;
	int domain_number;
	void *arch_opts;
};

/*
 * Flags for the above structure
 */
#define PP_FLAG_NO_ADJUST  0x01
/* I'd love to use inlines, but we still miss some structure at this point*/
#define pp_can_adjust(ppi)      (!(OPTS(ppi)->flags & PP_FLAG_NO_ADJUST))

/* slave_only:1, -- moved to ppi, no more global */
/* master_only:1, -- moved to ppi, no more global */
/* ethernet_mode:1, -- moved to ppi, no more global */



/* We need a globally-accessible structure with preset defaults */
extern struct pp_runtime_opts __pp_default_rt_opts;

/*
 * Communication channel. Is the abstraction of a unix socket, so that
 * this struct is platform independent
 */
struct pp_channel {
	union {
		int fd;		/* Posix wants fid descriptor */
		void *custom;	/* Other archs want other stuff */
	};
	void *arch_data;	/* Other arch-private info, if any */
	unsigned char addr[6];	/* Our own MAC address */
	int pkt_present;
};


/*
 * Foreign master record. Used to manage Foreign masters. In the specific
 * it is called foreignMasterDS, see 9.3.2.4
 */
struct pp_frgn_master {
	PortIdentity port_id;	/* used to identify old/new masters */

	/* We don't need all fields of the following ones */
	MsgAnnounce ann;
	MsgHeader hdr;
};

/*
 * Servo. Structs which contain filters for delay average computation. They
 * are used in servo.c src, where specific function for time setting of the
 * machine are implemented.
 *
 * pp_avg_fltr: It is a variable cutoff/delay low-pass, infinite impulse
 * response (IIR) filter. The meanPathDelay filter has the difference equation:
 * s*y[n] - (s-1)*y[n-1] = x[n]/2 + x[n-1]/2,
 * where increasing the stiffness (s) lowers the cutoff and increases the delay.
 */
struct pp_avg_fltr {
	int64_t m; /* magnitude */
	int64_t y;
	int64_t s_exp;
};

struct pp_servo {
	struct pp_time m_to_s_dly;
	struct pp_time s_to_m_dly;
	long long obs_drift;
	struct pp_avg_fltr mpd_fltr;
};

enum { /* The two sockets. They are called "net path" for historical reasons */
	PP_NP_GEN =	0,
	PP_NP_EVT,
	__NR_PP_NP,
};

/*
 * Struct containg the result of ppsi.conf parsing: one for each link
 * (see lib/conf.c). Actually, protocol and role are in the main ppi.
 */
struct pp_instance_cfg {
	char port_name[16];
	char iface_name[16];
	int ext;   /* 0: none, 1: whiterabbit. 2: HA */
	int mech;   /* 0: E2E, 1: P2P */
};

/*
 * Structure for the individual ppsi link
 */
struct pp_instance {
	int state;
	int next_state, next_delay, is_new_state; /* set by state processing */
	struct pp_state_table_item *current_state_item;
	void *arch_data;		/* if arch needs it */
	void *ext_data;			/* if protocol ext needs it */
	unsigned long d_flags;		/* diagnostics, ppi-specific flags */
	unsigned char flags;		/* protocol flags (see below) */
	int	role,			/* same as in config file */
		proto,			/* same as in config file */
		mech;			/* same as in config file */

	/* Pointer to global instance owning this pp_instance*/
	struct pp_globals *glbs;

	/* Operations that may be different in each instance */
	struct pp_network_operations *n_ops;
	struct pp_time_operations *t_ops;

	/*
	 * The buffer for this fsm are allocated. Then we need two
	 * extra pointer to track separately the frame and payload.
	 * So send/recv use the frame, pack/unpack use the payload.
	 */
	void *__tx_buffer, *__rx_buffer;
	void *tx_frame, *rx_frame, *tx_ptp, *rx_ptp;

	/* The net_path used to be allocated separately, but there's no need */
	struct pp_channel ch[__NR_PP_NP];	/* general and event ch */
	Integer32 mcast_addr[2];		/* only ipv4/udp */
	int tx_offset, rx_offset;		/* ptp payload vs send/recv */
	unsigned char peer[6];	/* Our peer's MAC address */
	uint16_t peer_vid;	/* Our peer's VID (for PROTO_VLAN) */

	/* Times, for the various offset computations */
	struct pp_time t1, t2, t3, t4, t5, t6;		/* *the* stamps */
	uint64_t syncCF;				/* transp. clocks */
	struct pp_time last_rcv_time, last_snt_time;	/* two temporaries */

	/* Page 85: each port shall maintain an implementation-specific
	 * foreignMasterDS data set for the purposes of qualifying Announce
	 * messages */
	UInteger16 frgn_rec_num;
	Integer16  frgn_rec_best;
	struct pp_frgn_master frgn_master[PP_NR_FOREIGN_RECORDS];

	DSPort *portDS;				/* page 72 */

	unsigned long timeouts[__PP_TO_ARRAY_SIZE];
	UInteger16 recv_sync_sequence_id;

	UInteger16 sent_seq[__PP_NR_MESSAGES_TYPES]; /* last sent this type */
	MsgHeader received_ptp_header;

	char *iface_name; /* for direct actions on hardware */
	char *port_name; /* for diagnostics, mainly */
	int port_idx;
	int vlans_array_len; /* those looking at shared mem must check */
	int vlans[CONFIG_VLAN_ARRAY_SIZE];
	int nvlans; /* according to configuration */
	struct pp_instance_cfg cfg;

	unsigned long ptp_tx_count;
	unsigned long ptp_rx_count;
};
/* The following things used to be bit fields. Other flags are now enums */
#define PPI_FLAG_FROM_CURRENT_PARENT	0x01
#define PPI_FLAG_WAITING_FOR_F_UP	0x02
#define PPI_FLAG_WAITING_FOR_RF_UP	0x04
#define PPI_FLAGS_WAITING		0x06 /* both of the above */

struct pp_globals_cfg {
	int cfg_items;			/* Remember how many we parsed */
	int cur_ppi_n;	/* Remember which instance we are configuring */
};

/*
 * Structure for the multi-port ppsi instance.
 */
struct pp_globals {
	struct pp_instance *pp_instances;

	struct pp_servo *servo;

	/* Real time options */
	struct pp_runtime_opts *rt_opts;

	/* Data sets */
	DSDefault *defaultDS;			/* page 65 */
	DSCurrent *currentDS;			/* page 67 */
	DSParent *parentDS;			/* page 68 */
	DSTimeProperties *timePropertiesDS;	/* page 70 */

	/* Index of the pp_instance receiving the "Ebest" clock */
	int ebest_idx;
	int ebest_updated; /* set to 1 when ebest_idx changes */

	int nlinks;
	int max_links;
	struct pp_globals_cfg cfg;

	int rxdrop, txdrop;		/* fault injection, per thousand */

	void *arch_data;		/* if arch needs it */
	void *global_ext_data;		/* if protocol ext needs it */
	/* FIXME Here include all is common to many interfaces */
};

#endif /* __PPSI_PP_INSTANCE_H__ */
