/************************************************************************

  Demo access to PicoHarp 330 via PH330Lib.dll or libph330.so v.2.0
  The program performs a TTTR measurement based on hard-coded settings.
  Event data is filtered by the device's event filter in the FPGA.
  The demo shows the software calls for setting the filter parameters
  and activating the filter. Resulting event data is instantly processed.
  Processing consists here of simple histogramming. This allows replicating
  and verifying the filter demonstration as shown in the section on event
  filtering in the PicoHarp 330 manual. If you use identical or at least
  similar input signals and setings you should obtain similar histograms
  as shown in the manual. Observe the offset of -10000ps in the call of
  PH330_SetSyncChannelOffset. Under the assumption that the sync channel
  and the first input channel are receiving the same signal through identical
  cable lengths this is necessary in order to shift the region of interest
  where coincidences occur into the usable histogram range.
  Note also that in order to implement the easiest way of histogramming,
  this demo is implemented for T3 mode only, although the filtering as such
  works in T2 mode as well.

  Michael Wahl, PicoQuant GmbH, August 2024

  Note: This is a console application

  Note: At the API level the input channel numbers are indexed 0..N-1
  where N is the number of input channels the device has.
  Upon processing we map this back to 1..N corresponding to the front
  panel labelling.


  Tested with the following compilers:

  - MinGW-W64 4.3.5 (Windows)
  - MS Visual C++ 2019 (Windows)
  - gcc 9.4.0 and 11.4.0 (Linux)

************************************************************************/

#ifndef _WIN32
#include <unistd.h>
#define Sleep(msec) usleep(msec*1000)
#define __int64 long long
#define uint64_t unsigned long long
#else
#include <windows.h>
#include <dos.h>
#include <conio.h>
#define uint64_t  unsigned __int64
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ph330defin.h"
#include "ph330lib.h"
#include "errorcodes.h"

FILE* fpout = NULL;
uint64_t oflcorrection = 0;
double Resolution = 0; // in ps
double Syncperiod = 0; // in s

unsigned int buffer[TTREADMAX];

#define T3HISTBINS 32768 //=2^15, dtime in T3 mode has 15 bits
unsigned int histogram[MAXINPCHAN][T3HISTBINS]; 

// helper macro and associated function for API calls with error check
// the stringize operator # makes a printable string from the macro's input argument
#define APICALL(call) doapicall(call, #call, __LINE__)
int doapicall(int retcode, char* callstr, int line)
{
    if (retcode < 0)
    {
        char errorstring[100] = { 0 };
        PH330_GetErrorString(errorstring, retcode);
        printf("\nThe API call %s at line %d\nreturned error %d (%s)\n", callstr, line, retcode, errorstring);
    }
    return retcode;
}

//Got PhotonT3
//  NSync: Overflow-corrected arrival time in units of the sync period 
//  DTime: Arrival time of photon after last Sync event in units of the chosen resolution (set by binning)
//  Channel: 1..N where N is the numer of channels the device has
void GotPhotonT3(uint64_t NSync, int Channel, int DTime)
{
    histogram[Channel - 1][DTime]++; //histogramming
}

//Got MarkerT3
//  NSync: Overflow-corrected arrival time in units of the sync period 
//  Markers: Bitfield of arrived Markers, different markers can arrive at same time (same record)
void GotMarkerT3(uint64_t NSync, int Markers)
{
    //  Could switch to a new Histogram here, e.g. for a FLIM application
}

// HydraHarpV2 or TimeHarp260 or MultiHarp ot PicoHarp 330 - T3 record data
void ProcessT3(unsigned int TTTRRecord)
{
    int ch, dt;
    uint64_t truensync;
    const int T3WRAPAROUND = 1024;

    union {
        unsigned allbits;
        struct {
            unsigned nsync : 10;  // numer of sync period
            unsigned dtime : 15;  // delay from last sync in units of chosen resolution
            unsigned channel : 6;
            unsigned special : 1;
        } bits;
    } T3Rec;

    T3Rec.allbits = TTTRRecord;

    if (T3Rec.bits.special == 1)
    {
        if (T3Rec.bits.channel == 0x3F) //overflow
        {
            //number of overflows is stored in nsync
            oflcorrection += (uint64_t)T3WRAPAROUND * T3Rec.bits.nsync;
        }
        if ((T3Rec.bits.channel >= 1) && (T3Rec.bits.channel <= 15)) //markers
        {
            truensync = oflcorrection + T3Rec.bits.nsync;
            //the time unit depends on sync period
            GotMarkerT3(truensync, T3Rec.bits.channel);
        }
    }
    else //regular input channel
    {
        truensync = oflcorrection + T3Rec.bits.nsync;
        ch = T3Rec.bits.channel + 1; //we encode the input channels as 1..N
        dt = T3Rec.bits.dtime;
        //truensync indicates the number of the sync period this event was in
        //the dtime unit depends on the chosen resolution (binning)
        GotPhotonT3(truensync, ch, dt);
    }
}

int main(int argc, char* argv[])
{
    int dev[MAXDEVNUM];
    int found = 0;
    int retcode;
    int ctcstatus;
    char LIB_Version[8];
    char HW_Model[32];
    char HW_Partno[8];
    char HW_Serial[9];
    char HW_Version[16];
    char Errorstring[40];
    char debuginfobuffer[16384];
    int NumChannels;
    int Mode = MODE_T3; //you can change this, observe suitable sync divider and range!
    int Binning = 6; //you can change this, meaningful only in T3 mode
    int Offset = 0;  //you can change this, meaningful only in T3 mode, normally 0
    int Tacq = 1000; //measurement time in millisec, you can change this
    int SyncDivider = 1; //you can change this, observe mode! READ MANUAL!

    int SyncTrgMode = TRGMODE_ETR;  //you can change this to TRGMODE_CFD    
    //in case of SyncTrgMode == TRGMODE_ETR this will apply:
    int SyncTiggerEdge = EDGE_FALLING;   //you can change this to EDGE_RISING
    int SyncTriggerLevel = -50;          //in mV, you can change this
    //in case of SyncTrgMode == TRGMODE_CFD this will apply:
    int SyncCFDZeroCross = -20;          //in mV, you can change this
    int SyncCFDLevel = -50;              //in mV, you can change this

    int InputTrgMode = TRGMODE_ETR; //you can change this to TRGMODE_CFD
    //in case of InputTrgMode == TRGMODE_ETR this will apply:
    int InputTriggerEdge = EDGE_FALLING; //you can change this to EDGE_RISING
    int InputTriggerLevel = -50;         //in mV, you can change this
     //in case of InputTrgMode == TRGMODE_CFD this will apply:
    int InputCFDZeroCross = -20;         //in mV, you can change this
    int InputCFDLevel = -50;             //in mV, you can change this

    int Syncrate;
    int Countrate;
    int i, j;
    int flags;
    int warnings;
    char warningstext[16384]; //must have 16384 bytes of text buffer
    int nRecords;
    unsigned int Progress;
    int stopretry = 0;

    /*
    The following variables are for programming the event filter.
    Please read the manual on what the filter parameters are about.
    Here we implement a simple "Singles Filter" which is the most obvious
    and most useful type of event filter in typical quantum optics applications.
    The idea is that photon events that are "single" in the sense that there is no
    other event in temporal proximity (within timerange) will be filtered out.
    This reduces the bus load and the file size if you write the records to a file.
    Any further (application specific) coincidence processing can then be
    done in software, either on the fly or subsequently on the stored data.
    */

    int filter_timerange = 4000;   // in picoseconds
    int filter_matchcnt = 1;       // must have at least one other event in proximity
    int filter_inverse = 0;        // normal filtering mode, see manual
    int filter_enable = 1;         // activate the filter
    int filter_usechannels = 0x3;  // bitmask for which channels are to be used
    int filter_passchannels = 0;   // bitmask for which channels to pass unfiltered
    //the following are count rate buffers for the filter test further down
    int ftestsyncrate;
    int ftestchanrates[MAXINPCHAN];
    int ftestsumrate;

    printf("\nPicoHarp 330 PHLib330 Demo Application                PicoQuant GmbH, 2024");
    printf("\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~");

    if (Mode == MODE_T2)
    {
        printf("This demo is not for use with T2 mode!\n");
        printf("\npress RETURN to exit");
        getchar();
        goto ex;
    }

    if (APICALL(PH330_GetLibraryVersion(LIB_Version)) < 0)
        goto ex;
    printf("\nLibrary version is %s\n", LIB_Version);
    if (strncmp(LIB_Version, LIB_VERSION, sizeof(LIB_VERSION)) != 0)
    {
        printf("\nWarning: The application was built for version %s.", LIB_VERSION);
    }

    if ((fpout = fopen("t3histout.txt", "w")) == NULL)
    {
        printf("\ncannot open output file\n");
        goto ex;
    }

    printf("\nSearching for PicoHarp 330 devices...");
    printf("\nDevidx     Serial     Status");

    for (i = 0; i < MAXDEVNUM; i++)
    {
        retcode = PH330_OpenDevice(i, HW_Serial);
        if (retcode == 0) //Grab any device we can open
        {
            printf("\n  %1d        %7s    open ok", i, HW_Serial);
            dev[found] = i; //keep index to devices we want to use
            found++;
        }
        else
        {
            if (retcode == PH330_ERROR_DEVICE_OPEN_FAIL)
            {
                printf("\n  %1d        %7s    no device", i, HW_Serial);
            }
            else
            {
                APICALL(PH330_GetErrorString(Errorstring, retcode));
                printf("\n  %1d        %7s    %s", i, HW_Serial, Errorstring);
            }
        }
    }

    //In this demo we will use the first device we find, i.e. dev[0].
    //You can also use multiple devices in parallel.
    //You can also check for specific serial numbers, so that you always know 
    //which physical device you are talking to.

    if (found < 1)
    {
        printf("\nNo device available.");
        goto ex;
    }
    printf("\nUsing device #%1d", dev[0]);

    printf("\nInitializing the device...");
    fflush(stdout);

    if (APICALL(PH330_Initialize(dev[0], Mode, 0)) < 0)
    {
        //in case of an obscure error (a hardware error in particular) 
        //it may be helpful to  obtain debug information like so:
        APICALL(PH330_GetDebugInfo(dev[0], debuginfobuffer));
        printf("\nDEBUGINFO:\n %s", debuginfobuffer);
        goto ex;
    }

    if (APICALL(PH330_GetHardwareInfo(dev[0], HW_Model, HW_Partno, HW_Version)) < 0)
        goto ex;

    printf("\nFound Model %s Part no %s Version %s", HW_Model, HW_Partno, HW_Version);

    if (APICALL(PH330_GetNumOfInputChannels(dev[0], &NumChannels)) < 0)
        goto ex;

    printf("\nDevice has %i input channels.", NumChannels);

    if (APICALL(PH330_SetSyncDiv(dev[0], SyncDivider)) < 0)
        goto ex;

    if (APICALL(PH330_SetSyncTrgMode(dev[0], SyncTrgMode)) < 0)
        goto ex;

    if (SyncTrgMode == TRGMODE_ETR)
    {
        if (APICALL(PH330_SetSyncEdgeTrg(dev[0], SyncTriggerLevel, SyncTiggerEdge)) < 0)
            goto ex;
    }

    if (SyncTrgMode == TRGMODE_CFD)
    {
        if (APICALL(PH330_SetSyncCFD(dev[0], SyncCFDLevel, SyncCFDZeroCross)) < 0)
            goto ex;
    }

    if (APICALL(PH330_SetSyncChannelOffset(dev[0], -10000)) < 0)  // in ps, emulate a cable delay
        goto ex;

    for (i = 0; i < NumChannels; i++) // for simplicity we use the same settings for all channels
    {
        if (APICALL(PH330_SetInputTrgMode(dev[0], i, InputTrgMode)) < 0)
            goto ex;

        if (InputTrgMode == TRGMODE_ETR)
        {
            if (APICALL(PH330_SetInputEdgeTrg(dev[0], i, InputTriggerLevel, InputTriggerEdge)) < 0)
                goto ex;
        }

        if (SyncTrgMode == TRGMODE_CFD)
        {
            if (APICALL(PH330_SetInputCFD(dev[0], i, InputCFDLevel, InputCFDZeroCross)) < 0)
                goto ex;
        }

        if (APICALL(PH330_SetInputChannelOffset(dev[0], i, 0)) < 0) // in ps, emulate a cable delay
            goto ex;

        if (APICALL(PH330_SetInputChannelEnable(dev[0], i, 1)) < 0) // we enable all channels
            goto ex;
    }

    if (Mode == MODE_T3)
    {
        if (APICALL(PH330_SetBinning(dev[0], Binning)) < 0)
            goto ex;

        if (APICALL(PH330_SetOffset(dev[0], Offset)) < 0)
            goto ex;
    }

    if (APICALL(PH330_GetResolution(dev[0], &Resolution)) < 0)
        goto ex;

    printf("\nResolution is %1.0lfps\n", Resolution);

    // now we program the event filter 

    if (APICALL(PH330_SetEventFilterChannels(dev[0], filter_usechannels, filter_passchannels)) < 0)
        goto ex;

    if (APICALL(PH330_SetEventFilterParams(dev[0], filter_timerange, filter_matchcnt, filter_inverse)) < 0)
        goto ex;

    if (APICALL(PH330_EnableEventFilter(dev[0], filter_enable)) < 0)
        goto ex;

    // filter programming ends here

    printf("\nMeasuring input rates...\n");

    // After Init allow 150 ms for valid  count rate readings
    // Subsequently you get new values after every 100ms
    Sleep(150);

    if (APICALL(PH330_GetSyncRate(dev[0], &Syncrate)) < 0)
        goto ex;

    printf("\nSyncrate=%1d/s", Syncrate);

    for (i = 0; i < NumChannels; i++) // for all channels
    {
        if (APICALL(PH330_GetCountRate(dev[0], i, &Countrate)) < 0)
            goto ex;

        printf("\nCountrate[%1d]=%1d/s", i, Countrate);
    }

    printf("\n");

    //after getting the count rates you can check for warnings
    if (APICALL(PH330_GetWarnings(dev[0], &warnings)) < 0)
        goto ex;

    if (warnings)
    {
        if (APICALL(PH330_GetWarningsText(dev[0], warningstext, warnings)) < 0)
            goto ex;

        printf("\n\n%s", warningstext);
    }

    /*
    Now we perform a filter test. This is not strictly necessary but helpful
    when the overall count rate is over the USB troughput limit and you wish to
    use the filter to alleviate this.
    The filter test consists of just a simple retrievel of input and output rates
    of the filter, so that you can assess its effect in terms of rate reduction.
    You can do this a few times in a loop, so that you can also see if there are
    significant fluctuations. However, note that for each round you will need to
    wait for at least 100ms to get new results.
    The filter test must be performed while measurement is running. In order
    to avoid FiFo buffer overruns we can use PH330_SetFilterTestMode to disable
    the transfer of measurement data into the FiFo.
    */

    if (APICALL(PH330_SetFilterTestMode(dev[0], 1)) < 0) //disable FiFo input
        goto ex;

    if (APICALL(PH330_StartMeas(dev[0], ACQTMAX)) < 0) //longest possible time, we will stop manually
        goto ex;

    Sleep(150); //allow the hardware at least 100ms time for rate counting 

    /*
    To start with, we retrieve the front end count rates. This is somewhat redundant
    here as we already retrieved them above. However, for demonstration purposes,
    this time we use a more efficient method that fetches all rates in one go.
    */
    if (APICALL(PH330_GetAllCountRates(dev[0], &ftestsyncrate, ftestchanrates)) < 0)
        goto ex;

    //We only care about the overall rates, so we sum them up here.
    ftestsumrate = 0;
    for (i = 0; i < NumChannels; i++)
        ftestsumrate += ftestchanrates[i];
    if (Mode == MODE_T2) //in this case also add the sync rate
        ftestsumrate += ftestsyncrate;
    printf("\nFront end input rate=%1d/s", ftestsumrate);

    /*
    Now we fetch the filter input rates. These are not necessarily the same as
    the front end count rates as there may already have been losses due to front end
    troughput limits. We do the same summation as above.
    */
    if (APICALL(PH330_GetFilterInputRates(dev[0], &ftestsyncrate, ftestchanrates)) < 0)
        goto ex;

    ftestsumrate = 0;
    for (i = 0; i < NumChannels; i++)
        ftestsumrate += ftestchanrates[i];
    if (Mode == MODE_T2) //in this case also add the sync rate
        ftestsumrate += ftestsyncrate;
    printf("\nFilter input rate=%1d/s", ftestsumrate);

    //Now we do the same rate retrieval and summation for the filter output.
    if (APICALL(PH330_GetFilterOutputRates(dev[0], &ftestsyncrate, ftestchanrates)) < 0)
        goto ex;

    ftestsumrate = 0;
    for (i = 0; i < NumChannels; i++)
        ftestsumrate += ftestchanrates[i];
    if (Mode == MODE_T2) //in this case also add the sync rate
        ftestsumrate += ftestsyncrate;
    printf("\nFilter output rate=%1d/s", ftestsumrate);

    if (APICALL(PH330_StopMeas(dev[0])) < 0) //test finished, stop measurement
        goto ex;

    //Testmode must be switched off again to allow a real measurement
    if (APICALL(PH330_SetFilterTestMode(dev[0], 0)) < 0) //re-enable FiFo input
        goto ex;

    // here we begin the real measurement

    for (j = 0; j < NumChannels; j++)
        fprintf(fpout, "   ch%2u ", j + 1);
    fprintf(fpout, "\n");


    printf("\n\npress RETURN to start");
    getchar();

    if (APICALL(PH330_StartMeas(dev[0], Tacq)) < 0)
        goto ex;

    if (Mode == MODE_T3)
    {
        //We need the sync period in order to calculate the true times of photon records.
        //This only makes sense in T3 mode and it assumes a stable period like from a laser.
        //Note: Two sync periods must have elapsed after PH330_StartMeas to get proper results.
        //You can also use the inverse of what you read via GetSyncRate but it depends on 
        //the actual sync rate if this is accurate enough.
        //It is OK to use the sync input for a photon detector, e.g. if you want to perform
        //something like an antibunching measurement. In that case the sync rate obviously is
        //not periodic. This means that a) you should set the sync divider to 1 (none) and
        //b) that you cannot meaningfully measure the sync period here, which probaly won't
        //matter as you only care for the time difference (dtime) of the events.
        if (APICALL(PH330_GetSyncPeriod(dev[0], &Syncperiod)) < 0)
            goto ex;

        printf("\nSync period is %lf ns\n", Syncperiod * 1e9);
    }

    printf("\nStarting data collection...\n");

    Progress = 0;
    printf("\nProgress:%12u", Progress);

    oflcorrection = 0;

    while (1)
    {
        if (APICALL(PH330_GetFlags(dev[0], &flags)) < 0)
            goto ex;

        if (flags & FLAG_FIFOFULL)
        {
            printf("\nFiFo Overrun!\n");
            goto stoptttr;
        }

        if (APICALL(PH330_ReadFiFo(dev[0], buffer, &nRecords)) < 0) //may return less!  
            goto ex;

        if (nRecords)
        {
            // Here we process the data. Note that the time this consumes prevents us
            // from getting around the loop quickly for the next Fifo read.
            // In a serious performance critical scenario you would write the data to
            // a software queue and do the processing in another thread reading from 
            // that queue.

            for (i = 0; i < nRecords; i++)
                ProcessT3(buffer[i]);

            Progress += nRecords;
            printf("\b\b\b\b\b\b\b\b\b\b\b\b%12u", Progress);
            fflush(stdout);
        }
        else //do the following only when there was no data
        {
            if (APICALL(PH330_CTCStatus(dev[0], &ctcstatus)) < 0)
                goto ex;
            if (ctcstatus)
            {
                stopretry++; //do a few more rounds as there might be some more in the FiFo
                if (stopretry > 5)
                {
                    printf("\nDone\n");
                    goto stoptttr;
                }
            }
        }

        //within this loop you can also read the count rates if needed.
        //Do it sparingly and use PH330_GetAllCountRates for speed.
    }

stoptttr:

    if (APICALL(PH330_StopMeas(dev[0])) < 0)
        goto ex;

    for (i = 0; i < T3HISTBINS; i++)
    {
        for (j = 0; j < NumChannels; j++)
            fprintf(fpout, "%7u ", histogram[j][i]);
        fprintf(fpout, "\n");
    }

ex:

    for (i = 0; i < MAXDEVNUM; i++) //no harm to close all
    {
        PH330_CloseDevice(i);
    }
    if (fpout)
    {
        fclose(fpout);
    }

    printf("\npress RETURN to exit");
    getchar();

    return 0;
}
