/************************************************************************

Demo access to PicoHarp 330 via PH330Lib.dll or libph330.so v.2.0.
The program performs a TTTR measurement based on hard-coded settings.
Event data is filtered by the device's event filter in the FPGA.
The demo shows the software calls for setting the filter parameters
and activating the filter. Resulting event data is instantly processed.
Processing consists here of simple histogramming. This allows replicating 
and verifying the filter demonstration as shown in the section on event 
filtering in the PicoHarp 330 manual. If you use identical or at least 
similar input signals and setings you should obtain similar histograms 
as shown in the manual. Observe the offset of -10000ps in the call of 
PH330_SetSyncChannelOffset. Under the assumption that the sync channel 
and the first input channel are receiving the same signal through identical
cable lengths this is necessary in order to shift the region of interest
where coincidences occur into the usable histogram range.
Note also that in order to implement the easiest way of histogramming, 
this demo is implemented for T3 mode only, although the filtering as such
works in T2 mode as well.

Michael Wahl, PicoQuant GmbH, August 2024

Note: This is a console application

Note: At the API level the input channel numbers are indexed 0..N-1
where N is the number of input channels the device has.
Upon processing we map this back to 1..N corresponding to the front
panel labelling.

Tested with the following compilers:

  - Microsoft Visual C# 2022 (Windows)
  - Mono 6.12.0 (Windows and Linux)

************************************************************************/

using System;  // for Console
using System.Text;  // for StringBuilder 
using System.IO;  // for File
using static PH330LibSpace.PH330Lib;

namespace tttrmode
{
    class tttrmode
    {
        static ulong oflcorrection = 0;
        static double Resolution = 0; // in ps
        static double Syncperiod = 0; // in s

        const int T3HISTBINS = 32768;
        static uint[,] histogram = new uint[MAXINPCHAN, T3HISTBINS];

        static void Main(string[] args)
        {
            //////////////////////// some settings you can change /////////////////////////

            int Mode = MODE_T3;   // this demo is only for T3, observe suitable Sync divider and Range!
            int Binning = 6;      // you can change this, meaningful only in T3 mode, observe limits 
            int Offset = 0;       // you can change this, meaningful only in T3 mode, observe limits 
            int Tacq = 10000;     // Measurement time in millisec, you can change this, observe limits 
            int SyncDivider = 1;  // you can change this, usually 1 in T2 mode 

            int SyncTrgMode = TRGMODE_ETR;       // you can change this to TRGMODE_CFD
                                                 // in case of SyncTrgMode == TRGMODE_ETR this will apply:
            int SyncTriggerEdge = EDGE_FALLING;  // you can change this to EDGE_RISING
            int SyncTriggerLevel = -50;          // in mV, you can change this, observe limits 
                                                 // in case of SyncTrgMode == TRGMODE_CFD this will apply:
            int SyncCFDZeroCross = -20;          // in mV, you can change this, observe limits 
            int SyncCFDLevel = -50;              // in mV, you can change this, observe limits 

            int InputTrgMode = TRGMODE_ETR;      // you can change this to TRGMODE_CFD
                                                 // in case of InputTrgMode == TRGMODE_ETR this will apply:
            int InputTriggerEdge = EDGE_FALLING; // you can change this to EDGE_RISING
            int InputTriggerLevel = -50;         // in mV, you can change this, observe limits 
                                                 // in case of InputTrgMode == TRGMODE_CFD this will apply:
            int InputCFDZeroCross = -20;         // in mV, you can change this, observe limits 
            int InputCFDLevel = -50;             // in mV, you can change this, observe limits 

            /*
            The following variables are for programming the event filter.
            Please read the manual on what the filter parameters are about.
            Here we implement a simple "Singles Filter" which is the most obvious
            and most useful type of event filter in typical quantum optics applications.
            The idea is that photon events which are "single" in the sense that there is no
            other event in temporal proximity (within timerange) will be filtered out.
            This reduces the bus load and the file size if you write the records to a file.
            Any further (application specific) coincidence processing can then be
            done in software, either on the fly or subsequently on the stored data.
            */

            int filter_timerange = 4000;   // in picoseconds
            int filter_matchcnt = 1;       // must have at least one other event in proximity
            int filter_inverse = 0;        // normal filtering mode, see manual
            int filter_enable = 1;         // activate the filter
            int filter_usechannels = 0x3;  // bitmask for which channels are to be used
            int filter_passchannels = 0;   // bitmask for which channels to pass unfiltered

            ///////////////////////////// program variables /////////////////////////////
            
            int[] dev = new int[MAXDEVNUM];
            int found = 0;
            int retcode;
            int ctcstatus = 0;

            StringBuilder LibVer = new StringBuilder(8);
            StringBuilder Serial = new StringBuilder(8);
            StringBuilder Errstr = new StringBuilder(40);
            StringBuilder Model = new StringBuilder(32);
            StringBuilder Partno = new StringBuilder(16);
            StringBuilder Version = new StringBuilder(16);
            StringBuilder Wtext = new StringBuilder(16384);

            int NumChannels = 0;
            int Syncrate = 0;
            int Countrate = 0;
            int flags = 0;
            int warnings = 0;
            long Progress = 0;
            int nRecords = 0;
            int stopretry = 0;

            uint[] buffer = new uint[TTREADMAX];

            int i,j;

            FileStream fs = null;
            StreamWriter sw = null;

            //the following are count rate buffers for the filter test further down
            int ftestsyncrate = 0;
            int[] ftestchanrates = new int[MAXINPCHAN];
            int ftestsumrate;

            ///////////////////////////// main program code /////////////////////////////
            
            for (i = 0; i < T3HISTBINS; i++)
                for (j = 0; j < MAXINPCHAN; j++)
                    histogram[j, i] = 0;

            Console.WriteLine("PicoHarp 330 PH330Lib Demo Application                 PicoQuant GmbH, 2024");
            Console.WriteLine("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~");

            if (APICALL(() => PH330_GetLibraryVersion(LibVer)) < 0)
                goto ex;

            Console.WriteLine("PH330Lib Version is " + LibVer);

            if (LibVer.ToString() != LIB_VERSION)
            {
                Console.WriteLine("This program requires PH330Lib v." + LIB_VERSION);
                goto ex;
            }

            try
            {
                fs = new FileStream("t3histout.txt", FileMode.Create, FileAccess.Write);
                sw = new StreamWriter(fs);
            }
            catch (Exception)
            {
                Console.WriteLine("Error creating file");
                goto ex;
            }

            Console.WriteLine("Searching for PicoHarp 330 devices...");
            Console.WriteLine("Devidx     Status");

            for (i = 0; i < MAXDEVNUM; i++)
            {
                retcode = PH330_OpenDevice(i, Serial);
                if (retcode == 0) //Grab any device we can open
                {
                    Console.WriteLine("  {0}        S/N {1}", i, Serial);
                    dev[found] = i; //keep index to devices we want to use
                    found++;
                }
                else
                {
                    if (retcode == PH330_ERROR_DEVICE_OPEN_FAIL)
                        Console.WriteLine("  {0}        no device", i);
                    else
                    {
                        PH330_GetErrorString(Errstr, retcode);
                        Console.WriteLine("  {0}        S/N {1}", i, Errstr);
                    }
                }
            }

            //In this demo we will use the first device we find, i.e. dev[0].
            //You can also use multiple devices in parallel.
            //You can also check for specific serial numbers, so that you always know 
            //which physical device you are talking to.

            if (found < 1)
            {
                Console.WriteLine("No device available.");
                goto ex;
            }

            Console.WriteLine("Using device {0}", dev[0]);
            Console.WriteLine();
            Console.WriteLine("Initializing the device...");

            if (APICALL(() => PH330_Initialize(dev[0], Mode, 0)) < 0)
                goto ex;

            if (APICALL(() => PH330_GetHardwareInfo(dev[0], Model, Partno, Version)) < 0) //only for information
                goto ex;
            else
                Console.WriteLine("Found Model {0} Part no {1} Version {2}", Model, Partno, Version);

            if (APICALL(() => PH330_GetNumOfInputChannels(dev[0], ref NumChannels)) < 0)
                goto ex;
            else
                Console.WriteLine("Device has {0} input channels.", NumChannels);

            if (APICALL(() => PH330_SetSyncDiv(dev[0], SyncDivider)) < 0)
                goto ex;

            if (APICALL(() => PH330_SetSyncTrgMode(dev[0], SyncTrgMode)) < 0)
                goto ex;

            if (SyncTrgMode == TRGMODE_ETR)
            {
                if (APICALL(() => PH330_SetSyncEdgeTrg(dev[0], SyncTriggerLevel, SyncTriggerEdge)) < 0)
                    goto ex;
            }
            if (SyncTrgMode == TRGMODE_CFD)
            {
                if (APICALL(() => PH330_SetSyncCFD(dev[0], SyncCFDLevel, SyncCFDZeroCross)) < 0)
                    goto ex;
            }

            if (APICALL(() => PH330_SetSyncChannelOffset(dev[0], 0)) < 0)    //in ps, can emulate a cable delay
                goto ex;

            for (i = 0; i < NumChannels; i++)  // we use the same input settings for all channels
            {
                if (APICALL(() => PH330_SetInputTrgMode(dev[0], i, InputTrgMode)) < 0)
                    goto ex;

                if (InputTrgMode == TRGMODE_ETR)
                {
                    if (APICALL(() => PH330_SetInputEdgeTrg(dev[0], i, InputTriggerLevel, InputTriggerEdge)) < 0)
                        goto ex;
                }
                if (InputTrgMode == TRGMODE_CFD)
                {
                    if (APICALL(() => PH330_SetInputCFD(dev[0], i, InputCFDLevel, InputCFDZeroCross)) < 0)
                        goto ex;
                }

                if (APICALL(() => PH330_SetInputChannelOffset(dev[0], i, 0)) < 0)  //in ps, can emulate a cable delay
                    goto ex;

                if (APICALL(() => PH330_SetInputChannelEnable(dev[0], i, 1)) < 0)
                    goto ex;
            }

            if (Mode != MODE_T2)
            {
                if (APICALL(() => PH330_SetBinning(dev[0], Binning)) < 0)
                    goto ex;

                if (APICALL(() => PH330_SetOffset(dev[0], Offset)) < 0)
                    goto ex;
            }

            if (APICALL(() => PH330_GetResolution(dev[0], ref Resolution)) < 0)
                goto ex;

            Console.WriteLine("Resolution is {0} ps", Resolution);

            #region Here this demo differs from the classic tttrmode demo
            // now we program the event filter

            if (APICALL(() => PH330_SetEventFilterChannels(dev[0], filter_usechannels, filter_passchannels)) < 0)
                goto ex;

            if (APICALL(() => PH330_SetEventFilterParams(dev[0], filter_timerange, filter_matchcnt, filter_inverse)) < 0)
                goto ex;

            if (APICALL(() => PH330_EnableEventFilter(dev[0], filter_enable)) < 0)
                goto ex;

            // filter programming ends here
            #endregion

            Console.WriteLine("Measuring input rates...\n");

            // After Init allow 150 ms for valid  count rate readings
            // Subsequently you get new values after every 100ms
            System.Threading.Thread.Sleep(150);

            if (APICALL(() => PH330_GetSyncRate(dev[0], ref Syncrate)) < 0)
                goto ex;
            Console.WriteLine("Syncrate = {0}/s", Syncrate);

            for (i = 0; i < NumChannels; i++)  // for all channels
            {
                if (APICALL(() => PH330_GetCountRate(dev[0], i, ref Countrate)) < 0)
                    goto ex;
                Console.WriteLine("Countrate[{0}] = {1}/s", i, Countrate);
            }

            Console.WriteLine();

            if (APICALL(() => PH330_GetWarnings(dev[0], ref warnings)) < 0)
                goto ex;
            if (warnings != 0)
            {
                if (APICALL(() => PH330_GetWarningsText(dev[0], Wtext, warnings)) < 0)
                    goto ex;
                Console.WriteLine("{0}", Wtext);
            }

            #region Here this demo differs from classic 'tttrdemo'

            /*
            Now we perform a filter test. This is not strictly necessary but helpful
            when the overall count rate is over the USB troughput limit and you wish to
            use the filter to alleviate this.
            The filter test consists of just a simple retrievel of input and output rates 
            of the filter(s), so that you can assess its effect in terms of rate reduction.
            You can do this a few times in a loop, so that you can also see if there are 
            significant fluctuations. However, note that for each round you will need to 
            wait for at least 100ms to get new results.
            The filter test must be performed while measurement is running. In order
            to avoid FiFo buffer overruns we can use PH330_SetFilterTestMode to disable the
            transfer of meaurement data into the FiFo.
            */

            if (APICALL(() => PH330_SetFilterTestMode(dev[0], 1)) < 0) //disable FiFo input
                goto ex;

            if (APICALL(() => PH330_StartMeas(dev[0], ACQTMAX)) < 0) //longest possible time, we will stop manually
                goto ex;

            System.Threading.Thread.Sleep(150); //allow the hardware at least 100ms time for rate counting 

            /*
            To start with, we retrieve the front end count rates. This is somewhat redundant 
            here as we already retrieved them above. However, for demonstration purposes, 
            this time we use a more efficient method that fetches all rates in one go.
            */

            if (APICALL(() => PH330_GetAllCountRates(dev[0], ref ftestsyncrate, ftestchanrates)) < 0)
                goto ex;

            ftestsumrate = 0;
            for (i = 0; i < NumChannels; i++)
                ftestsumrate += ftestchanrates[i];
            if (Mode == MODE_T2) //in this case also add the sync rate
                ftestsumrate += ftestsyncrate;
            Console.WriteLine("Front end input rate={0}/s", ftestsumrate);

            /*
            Now we fetch the filter input rates. These are not necessarily the same as
            the front end count rates as there may already have been losses due to front end
            troughput limits. We do the same summation as above.
            */

            if (APICALL(() => PH330_GetFilterInputRates(dev[0], ref ftestsyncrate, ftestchanrates)) < 0)
                goto ex;

            ftestsumrate = 0;
            for (i = 0; i < NumChannels; i++)
                ftestsumrate += ftestchanrates[i];
            if (Mode == MODE_T2) //in this case also add the sync rate
                ftestsumrate += ftestsyncrate;
            Console.WriteLine("Filter input rate={0}/s", ftestsumrate);

            // Now we do the same rate retrieval and summation for the Filter output.

            if (APICALL(() => PH330_GetFilterOutputRates(dev[0], ref ftestsyncrate, ftestchanrates)) < 0)
                goto ex;

            ftestsumrate = 0;
            for (i = 0; i < NumChannels; i++)
                ftestsumrate += ftestchanrates[i];
            if (Mode == MODE_T2) //in this case also add the sync rate
                ftestsumrate += ftestsyncrate;
            Console.WriteLine("Filter output rate={0}/s", ftestsumrate);

            if (APICALL(() => PH330_StopMeas(dev[0])) < 0) //test finished, stop measurement
                goto ex;

            // Testmode must be switched off again to allow a real measurement

            if (APICALL(() => PH330_SetFilterTestMode(dev[0], 0)) < 0) //re-enable FiFo input
                goto ex;

            // here we begin the real measurement

            for (j = 0; j < NumChannels; j++) 
                sw.Write($"   ch{j + 1,2:N0} ");
            sw.WriteLine();

            Console.WriteLine("press RETURN to start");
            Console.ReadLine();

            if (APICALL(() => PH330_StartMeas(dev[0], Tacq)) < 0)
                goto ex;

            if (Mode == MODE_T3)
            {
                //We need the sync period in order to calculate the true times of photon records.
                //This only makes sense in T3 mode and it assumes a stable period like from a laser.
                //Note: Two sync periods must have elapsed after PH330_StartMeas to get proper results.
                //You can also use the inverse of what you read via GetSyncRate but it depends on 
                //the actual sync rate if this is accurate enough.
                //It is OK to use the sync input for a photon detector, e.g. if you want to perform
                //something like an antibunching measurement. In that case the sync rate obviously is
                //not periodic. This means that a) you should set the sync divider to 1 (none) and
                //b) that you cannot meaningfully measure the sync period here, which probaly won't
                //matter as you only care for the time difference (dtime) of the events.

                if (APICALL(() => PH330_GetSyncPeriod(dev[0], ref Syncperiod)) < 0)
                    goto ex;
                Console.WriteLine("Sync period is {0} ns\n", Syncperiod * 1e9);
            }

            Console.WriteLine("Starting data collection...\n");

            #endregion

            Progress = 0;
            Console.Write("Progress: {0,12}", Progress);

            oflcorrection = 0;

            while (true)
            {
                if (APICALL(() => PH330_GetFlags(dev[0], ref flags)) < 0)
                    goto ex;

                if ((flags & FLAG_FIFOFULL) != 0)
                {
                    Console.WriteLine();
                    Console.WriteLine("FiFo Overrun!");
                    goto stoptttr;
                }

                if (APICALL(() => PH330_ReadFiFo(dev[0], buffer, ref nRecords)) < 0)  // may return less!
                    goto stoptttr;

                if (nRecords > 0)
                {
                    // Here we process the data. Note that the time this consumes prevents us
                    // from getting around the loop quickly for the next Fifo read.
                    // In a serious performance critical scenario you would write the data to
                    // a software queue and do the processing in another thread reading from 
                    // that queue.

                    if (Mode == MODE_T2)
                    {
                        Console.WriteLine("T2 mode is not supported in this demo!");
                        goto ex;
                    }
                    else
                        for (i = 0; i < nRecords; i++)
                            ProcessT3(sw, buffer[i]);

                    Progress += nRecords;
                    Console.Write("\b\b\b\b\b\b\b\b\b\b\b\b{0,12}", Progress);
                }
                else
                {
                    if (APICALL(() => PH330_CTCStatus(dev[0], ref ctcstatus)) < 0)
                        goto ex;
                    if (ctcstatus > 0)
                    {
                        stopretry++; //do a few more rounds as there might be some more in the FiFo
                        if (stopretry > 5)
                        {
                            Console.WriteLine("\nDone\n");
                            goto stoptttr;
                        }
                    }
                }
                // within this loop you can also read the count rates if needed.
            }

            stoptttr:
            Console.WriteLine();

            if (APICALL(() => PH330_StopMeas(dev[0])) < 0)
                goto ex;

            for (i = 0; i < T3HISTBINS; i++)
            {
                for (j = 0; j < NumChannels; j++)
                {
                    sw.Write($"{histogram[j, i],7:F0} ");
                }
                sw.WriteLine();
            }


            ex:

            for (i = 0; i < MAXDEVNUM; i++)  // no harm to close all
            {
                PH330_CloseDevice(i);
            }

            if (sw != null)
            {
                try
                {
                    sw.Flush();
                    sw.Close();
                    sw.Dispose();
                    sw = null;
                }
                catch (Exception e)
                {
                    Console.WriteLine("Error closing the file: " + e);
                }
            }

            Console.WriteLine("press RETURN to exit");
            Console.ReadLine();
        }

        //Got PhotonT3
        //  TimeTag: Overflow-corrected arrival time in units of the sync period 
        //  DTime: Arrival time of photon after last Sync event in units of the chosen resolution (set by binning)
        //  Channel: 1..N where N is the numer of channels the device has
        static void GotPhotonT3(StreamWriter fpout, ulong TimeTag, int Channel, int DTime)
        {
            histogram[Channel - 1, DTime]++; // histogramming
        }

        //Got MarkerT3
        //  TimeTag: Overflow-corrected arrival time in units of the sync period 
        //  Markers: Bitfield of arrived Markers, different markers can arrive at same time (same record)
        static void GotMarkerT3(StreamWriter fpout, ulong TimeTag, int Markers)
        {
            // Could switch to a new Histogram here, e.g. for a FLIM application
        }

        // HydraHarpV2 or TimeHarp260 or MultiHarp or PicoHarp 330 - T3 record data
        static void ProcessT3(StreamWriter fpout, uint TTTRRecord)
        {
            int ch, dt;
            ulong truensync;
            const int T3WRAPAROUND = 1024;

            uint nsync = (TTTRRecord >> 00) & (0xFFFFFFFF >> (32 - 10)); //the lowest 10 bits
            uint dtime = (TTTRRecord >> 10) & (0xFFFFFFFF >> (32 - 15)); //the next   15 bits
            uint channel = (TTTRRecord >> 25) & (0xFFFFFFFF >> (32 - 06)); //the next   6  bits
            uint special = (TTTRRecord >> 31) & (0xFFFFFFFF >> (32 - 01)); //the next   1  bit

            if (special == 1)
            {
                if (channel == 0x3F) //overflow
                {
                    //number of overflows is stored in nsync
                    oflcorrection += (ulong)T3WRAPAROUND * nsync;
                }
                if ((channel >= 1) && (channel <= 15)) //markers
                {
                    truensync = oflcorrection + nsync;
                    //the time unit depends on sync period
                    GotMarkerT3(fpout, truensync, (int)channel);
                }
            }
            else //regular input channel
            {
                truensync = oflcorrection + nsync;
                ch = (int)(channel + 1); //we encode the input channels as 1..N
                dt = (int)dtime;
                //truensync indicates the number of the sync period this event was in
                //the dtime unit depends on the chosen resolution (binning)
                GotPhotonT3(fpout, truensync, ch, dt);
            }
        }
    }
}
