﻿/************************************************************************

Demo access to PicoHarp 330 via PH330Lib.dll or libph330.so v.2.0.
The program performs a measurement based on hardcoded settings.
The resulting event data is instantly processed.
Processing consists here only of dissecting the binary event record
data and writing it to a text file. This is only for demo purposes.
In a real application this makes no sense as it limits throughput and
creates very large files. In practice you would more sensibly perform
some meaningful processing such as counting coincidences on the fly.

Michael Wahl, PicoQuant GmbH, August 2024

Note: This is a console application

Note: At the API level the input channel numbers are indexed 0..N-1
where N is the number of input channels the device has.
Upon processing we map this back to 1..N corresponding to the front
panel labelling.

Tested with the following compilers:

  - Microsoft Visual C# 2022 (Windows)
  - Mono 6.12.0 (Windows and Linux)

************************************************************************/

using System;  // for Console
using System.Text;  // for StringBuilder 
using System.IO;  // for File
using static PH330LibSpace.PH330Lib;

namespace tttrmode
{
    class tttrmode
    {
        static ulong oflcorrection = 0;
        static double Resolution = 0; // in ps
        static double Syncperiod = 0; // in s

        static void Main(string[] args)
        {
            //////////////////////// some settings you can change /////////////////////////

            int Mode = MODE_T2;   // you can change this, adjust other settings accordingly!
            int Binning = 4;      // you can change this, meaningful only in T3 mode, observe limits 
            int Offset = 0;       // you can change this, meaningful only in T3 mode, observe limits 
            int Tacq = 1000;      // measurement time in millisec, you can change this, observe limits 
            int SyncDivider = 1;  // you can change this, usually 1 in T2 mode 

            int SyncTrgMode = TRGMODE_ETR;       // you can change this to TRGMODE_CFD
                                                 // in case of SyncTrgMode == TRGMODE_ETR this will apply:
            int SyncTriggerEdge = EDGE_FALLING;  // you can change this to EDGE_RISING
            int SyncTriggerLevel = -50;          // in mV, you can change this, observe limits 
                                                 // in case of SyncTrgMode == TRGMODE_CFD this will apply:
            int SyncCFDZeroCross = -20;          // in mV, you can change this, observe limits 
            int SyncCFDLevel = -50;              // in mV, you can change this, observe limits 

            int InputTrgMode = TRGMODE_ETR;      // you can change this to TRGMODE_CFD
                                                 // in case of InputTrgMode == TRGMODE_ETR this will apply:
            int InputTriggerEdge = EDGE_FALLING; // you can change this to EDGE_RISING
            int InputTriggerLevel = -50;         // in mV, you can change this, observe limits 
                                                 // in case of InputTrgMode == TRGMODE_CFD this will apply:
            int InputCFDZeroCross = -20;         // in mV, you can change this, observe limits 
            int InputCFDLevel = -50;             // in mV, you can change this, observe limits 


            ///////////////////////////// program variables /////////////////////////////
            
            int[] dev = new int[MAXDEVNUM];
            int found = 0;
            int retcode;
            int ctcstatus = 0;

            StringBuilder LibVer = new StringBuilder(8);
            StringBuilder Serial = new StringBuilder(8);
            StringBuilder Errstr = new StringBuilder(40);
            StringBuilder Model = new StringBuilder(32);
            StringBuilder Partno = new StringBuilder(16);
            StringBuilder Version = new StringBuilder(16);
            StringBuilder Wtext = new StringBuilder(16384);

            int NumChannels = 0;
            int Syncrate = 0;
            int Countrate = 0;
            int flags = 0;
            int warnings = 0;
            long Progress = 0;
            int nRecords = 0;
            int stopretry = 0;

            uint[] buffer = new uint[TTREADMAX];

            int i;

            FileStream fs = null;
            StreamWriter sw = null;

            ///////////////////////////// main program code /////////////////////////////

            Console.WriteLine("PicoHarp 330 PH330Lib Demo Application                 PicoQuant GmbH, 2024");
            Console.WriteLine("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~");

            if (APICALL(() => PH330_GetLibraryVersion(LibVer)) < 0)
                goto ex;

            Console.WriteLine("PH330Lib Version is " + LibVer);

            if (LibVer.ToString() != LIB_VERSION)
            {
                Console.WriteLine("This program requires PH330Lib v." + LIB_VERSION);
                goto ex;
            }

            try
            {
                fs = new FileStream("tttrmodeout.txt", FileMode.Create, FileAccess.Write);
                sw = new StreamWriter(fs);
            }
            catch (Exception)
            {
                Console.WriteLine("Error creating file");
                goto ex;
            }

            Console.WriteLine("Searching for PicoHarp 330 devices...");
            Console.WriteLine("Devidx     Status");

            for (i = 0; i < MAXDEVNUM; i++)
            {
                //not using APICALL here as we wish to handle errors gracefully below
                retcode = PH330_OpenDevice(i, Serial);
                if (retcode == 0) //grab any device we can open
                {
                    Console.WriteLine("  {0}        S/N {1}", i, Serial);
                    dev[found] = i; //keep index to devices we want to use
                    found++;
                }
                else
                {
                    if (retcode == PH330_ERROR_DEVICE_OPEN_FAIL)
                        Console.WriteLine("  {0}        no device", i);
                    else
                    {
                        PH330_GetErrorString(Errstr, retcode);
                        Console.WriteLine("  {0}        S/N {1}", i, Errstr);
                    }
                }
            }

            if (found < 1)
            {
                Console.WriteLine("No device available.");
                goto ex;
            }

            Console.WriteLine("Using device {0}", dev[0]);
            Console.WriteLine();
            Console.WriteLine("Initializing the device...");

            if (APICALL(() => PH330_Initialize(dev[0], Mode, 0)) < 0) goto ex;

            if (APICALL(() => PH330_GetHardwareInfo(dev[0], Model, Partno, Version)) < 0) goto ex;
            Console.WriteLine("Found Model {0} Part no {1} Version {2}", Model, Partno, Version);

            if (APICALL(() => PH330_GetNumOfInputChannels(dev[0], ref NumChannels)) < 0) goto ex;
            Console.WriteLine("Device has {0} input channels.", NumChannels);

            if (APICALL(() => PH330_SetSyncDiv(dev[0], SyncDivider)) < 0) goto ex;

            if (APICALL(() => PH330_SetSyncTrgMode(dev[0], SyncTrgMode)) < 0)
                goto ex;

            if (SyncTrgMode == TRGMODE_ETR)
            {
                if (APICALL(() => PH330_SetSyncEdgeTrg(dev[0], SyncTriggerLevel, SyncTriggerEdge)) < 0)
                    goto ex;
            }
            if (SyncTrgMode == TRGMODE_CFD)
            {
                if (APICALL(() => PH330_SetSyncCFD(dev[0], SyncCFDLevel, SyncCFDZeroCross)) < 0)
                    goto ex;
            }

            if (APICALL(() => PH330_SetSyncChannelOffset(dev[0], 0)) < 0)    //in ps, can emulate a cable delay
                goto ex;

            for (i = 0; i < NumChannels; i++)  // we use the same input settings for all channels
            {
                if (APICALL(() => PH330_SetInputTrgMode(dev[0], i, InputTrgMode)) < 0)
                    goto ex;

                if (InputTrgMode == TRGMODE_ETR)
                {
                    if (APICALL(() => PH330_SetInputEdgeTrg(dev[0], i, InputTriggerLevel, InputTriggerEdge)) < 0)
                        goto ex;
                }
                if (InputTrgMode == TRGMODE_CFD)
                {
                    if (APICALL(() => PH330_SetInputCFD(dev[0], i, InputCFDLevel, InputCFDZeroCross)) < 0)
                        goto ex;
                }

                if (APICALL(() => PH330_SetInputChannelOffset(dev[0], i, 0)) < 0)  //in ps, can emulate a cable delay
                    goto ex;

                if (APICALL(() => PH330_SetInputChannelEnable(dev[0], i, 1)) < 0)
                    goto ex;
            }

            if (Mode != MODE_T2)
            {
                if (APICALL(() => PH330_SetBinning(dev[0], Binning)) < 0) goto ex;

                if (APICALL(() => PH330_SetOffset(dev[0], Offset)) < 0) goto ex;
            }

            if (APICALL(() => PH330_GetResolution(dev[0], ref Resolution)) < 0) goto ex;

            Console.WriteLine("Resolution is {0} ps", Resolution);
            Console.WriteLine();
            Console.WriteLine("Measuring input rates...\n");

            System.Threading.Thread.Sleep(150);

            if (APICALL(() => PH330_GetSyncRate(dev[0], ref Syncrate)) < 0) goto ex;
            Console.WriteLine("Syncrate = {0}/s", Syncrate);

            for (i = 0; i < NumChannels; i++)  // for all channels
            {
                if (APICALL(() => PH330_GetCountRate(dev[0], i, ref Countrate)) < 0) goto ex;
                Console.WriteLine("Countrate[{0}] = {1}/s", i, Countrate);
            }

            Console.WriteLine();

            if (APICALL(() => PH330_GetWarnings(dev[0], ref warnings)) < 0) goto ex;
            if (warnings != 0)
            {
                PH330_GetWarningsText(dev[0], Wtext, warnings);
                Console.WriteLine("{0}", Wtext);
            }

            if (Mode == MODE_T2)
                sw.Write("ev chn       time/ps\n\n");
            else
                sw.Write("ev chn  ttag/s   dtime/ps\n\n");

            Console.WriteLine("press RETURN to start");
            Console.ReadLine();

            if (APICALL(() => PH330_StartMeas(dev[0], Tacq)) < 0) goto ex;

            #region Here this demo differs from the simple tttrmode demo

            if (Mode == MODE_T3)
            {
                if (APICALL(() => PH330_GetSyncPeriod(dev[0], ref Syncperiod)) < 0) goto ex;
                Console.WriteLine("Sync period is {0} ns", Syncperiod * 1e9);
            }

            Console.WriteLine("Starting data collection...");

            Progress = 0;
            Console.Write("Progress: {0,12}", Progress);

            oflcorrection = 0;

            #endregion

            while (true)
            {
                if (APICALL(() => PH330_GetFlags(dev[0], ref flags)) < 0) goto ex;

                if ((flags & FLAG_FIFOFULL) != 0)
                {
                    Console.WriteLine();
                    Console.WriteLine("FiFo Overrun!");
                    goto stoptttr;
                }

                if (APICALL(() => PH330_ReadFiFo(dev[0], buffer, ref nRecords)) < 0) goto stoptttr;

                if (nRecords > 0)
                {
                    if (Mode == MODE_T2)
                        for (i = 0; i < nRecords; i++)
                            ProcessT2(sw, buffer[i]);
                    else
                        for (i = 0; i < nRecords; i++)
                            ProcessT3(sw, buffer[i]);

                    Progress += nRecords;
                    Console.Write("\b\b\b\b\b\b\b\b\b\b\b\b{0,12}", Progress);
                }
                else
                {
                    if (APICALL(() => PH330_CTCStatus(dev[0], ref ctcstatus)) < 0) goto ex;
                    if (ctcstatus > 0)
                    {
                        stopretry++; //do a few more rounds as there might be some more in the FiFo
                        if (stopretry > 5)
                        {
                            Console.WriteLine("\nDone\n");
                            goto stoptttr;
                        }
                    }
                }
            }

            stoptttr:
            Console.WriteLine();

            if (APICALL(() => PH330_StopMeas(dev[0])) < 0) goto ex;

            ex:

            for (i = 0; i < MAXDEVNUM; i++)  // no harm to close all
                PH330_CloseDevice(i);

            if (sw != null)
            {
                try
                {
                    sw.Flush();
                    sw.Close();
                    sw.Dispose();
                    sw = null;
                }
                catch (Exception e)
                {
                    Console.WriteLine("Error closing the file: " + e);
                }
            }

            Console.WriteLine("press RETURN to exit");
            Console.ReadLine();
        }

        //Got PhotonT2
        //  TimeTag: Overflow-corrected arrival time in units of the device's base resolution 
        //  Channel: Channel the photon arrived (0 = Sync channel, 1..N = regular timing channel)
        static void GotPhotonT2(StreamWriter fpout, ulong TimeTag, int Channel)
        {
            fpout.Write("CH {0,2:D2} {1,14:F1}\n", Channel, TimeTag * Resolution);
        }

        //Got MarkerT2
        //  TimeTag: Overflow-corrected arrival time in units of the device's base resolution 
        //  Markers: Bitfield of arrived markers, different markers can arrive at same time (same record)
        static void GotMarkerT2(StreamWriter fpout, ulong TimeTag, int Markers)
        {
            fpout.Write("MK {0,2:D2} {1,14:F1}\n", Markers, TimeTag * Resolution);
        }

        //Got PhotonT3
        //  TimeTag: Overflow-corrected arrival time in units of the sync period 
        //  DTime: Arrival time of photon after last Sync event in units of the chosen resolution (set by binning)
        //  Channel: 1..N where N is the number of channels the device has
        static void GotPhotonT3(StreamWriter fpout, ulong TimeTag, int Channel, int DTime)
        {
            //Syncperiod is in seconds
            fpout.Write("CH {0,2:D2} {1,10:F8} {2,8:F1}\n", Channel, TimeTag * Syncperiod, DTime * Resolution);
        }

        //Got MarkerT3
        //  TimeTag: Overflow-corrected arrival time in units of the sync period 
        //  Markers: Bitfield of arrived Markers, different markers can arrive at same time (same record)
        static void GotMarkerT3(StreamWriter fpout, ulong TimeTag, int Markers)
        {
            //Syncperiod is in seconds
            fpout.Write("MK {0,2:D2} {1,10:F8}\n", Markers, TimeTag * Syncperiod);
        }

        // HydraHarpV2 or TimeHarp260 or MultiHarp or PicoHarp 330 - T2 record data
        static void ProcessT2(StreamWriter fpout, uint TTTRRecord)
        {
            int ch;
            ulong truetime;
            const int T2WRAPAROUND_V2 = 33554432;

            // shift and mask out the elements of TTTRRecord
            uint timetag = (TTTRRecord >> 00) & (0xFFFFFFFF >> (32 - 25)); //the lowest 25 bits
            uint channel = (TTTRRecord >> 25) & (0xFFFFFFFF >> (32 - 06)); //the next    6 bits
            uint special = (TTTRRecord >> 31) & (0xFFFFFFFF >> (32 - 01)); //the next    1 bit

            if (special == 1)
            {
                if (channel == 0x3F) //an overflow record
                {
                    //number of overflows is stored in timetag
                    oflcorrection += (ulong)T2WRAPAROUND_V2 * timetag;
                }
                if ((channel >= 1) && (channel <= 15)) //markers
                {
                    truetime = oflcorrection + timetag;
                    //Note that actual marker tagging accuracy is only some ns.
                    ch = (int)channel;
                    GotMarkerT2(fpout, truetime, ch);
                }
                if (channel == 0) //sync
                {
                    truetime = oflcorrection + timetag;
                    ch = 0; //we encode the Sync channel as 0
                    GotPhotonT2(fpout, truetime, ch);
                }
            }
            else //regular input channel
            {
                truetime = oflcorrection + timetag;
                ch = (int)(channel + 1); //we encode the regular channels as 1..N
                GotPhotonT2(fpout, truetime, ch);
            }
        }

        // HydraHarpV2 or TimeHarp260 or MultiHarp or PicoHarp 330 - T3 record data
        static void ProcessT3(StreamWriter fpout, uint TTTRRecord)
        {
            int ch, dt;
            ulong truensync;
            const int T3WRAPAROUND = 1024;

            uint nsync = (TTTRRecord >> 00) & (0xFFFFFFFF >> (32 - 10)); //the lowest 10 bits
            uint dtime = (TTTRRecord >> 10) & (0xFFFFFFFF >> (32 - 15)); //the next   15 bits
            uint channel = (TTTRRecord >> 25) & (0xFFFFFFFF >> (32 - 06)); //the next   6  bits
            uint special = (TTTRRecord >> 31) & (0xFFFFFFFF >> (32 - 01)); //the next   1  bit

            if (special == 1)
            {
                if (channel == 0x3F) //overflow
                {
                    //number of overflows is stored in nsync
                    oflcorrection += (ulong)T3WRAPAROUND * nsync;
                }
                if ((channel >= 1) && (channel <= 15)) //markers
                {
                    truensync = oflcorrection + nsync;
                    //the time unit depends on sync period
                    GotMarkerT3(fpout, truensync, (int)channel);
                }
            }
            else //regular input channel
            {
                truensync = oflcorrection + nsync;
                ch = (int)(channel + 1); //we encode the input channels as 1..N
                dt = (int)dtime;
                //truensync indicates the number of the sync period this event was in
                //the dtime unit depends on the chosen resolution (binning)
                GotPhotonT3(fpout, truensync, ch, dt);
            }
        }
    }
}

