﻿/* 
    PH330Lib programming library for PicoHarp 330
    PicoQuant GmbH

    Ver. 2.0.0.0   Michael Wahl, August 2024

    You shouldn't normally edit this file!
*/

using System;
using System.Text;
using System.Runtime.InteropServices;  // for DllImport
using System.Runtime.CompilerServices; // for CallerMemberName

namespace PH330LibSpace
{
    public static class PH330Lib
    {
        // unfortunately WINDOWS is not predefined, must define it as a compiler directive
        // MS Visual C#: find it defined in the .csproj-file
#if WINDOWS
        public const string PH330LibSystemname = "PH330Lib.dll";
#else
        public const string PH330LibSystemname = "libph330.so"; 
#endif

        //////////////////////////// PH330Lib Constants //////////////////////////////

        public const string LIB_VERSION = "2.0";        // expected library version ("major.minor", lesser numbers not relevant)

        public const int MAXDEVNUM = 8;                 // max number of PH330 devices

        public const int MAXINPCHAN = 4;                // max number of physical input channels

        public const int MAXBINSTEPS = 24;              // max number of binning steps, 
                                                        // get actual number via PH330_GetBaseResolution()

        public const int MAXHISTLEN = 524288;           // max number of histogram bins
        public const int DFLTHISTLEN = 65536;           // default number of histogram bins

        public const int TTREADMAX = 1048576;           // number of event records that can be read by PH330_ReadFiFo
                                                        // buffer must provide space for this number of dwords

        // constants for PH330_Initialize parameter "refsource"
        public const int REFSRC_INTERNAL = 0;  // use internal clock
        public const int REFSRC_EXTERNAL_10MHZ = 1;  // use 10MHz external clock
        public const int REFSRC_EXTERNAL_100MHZ = 2;  // use 100MHz external clock
        public const int REFSRC_EXTERNAL_500MHZ = 3;  // use 500MHz external clock

        // constants for PH330_Initialize parameter "mode"
        public const int MODE_HIST = 0;
        public const int MODE_T2 = 2;
        public const int MODE_T3 = 3;

        // constants for PH330_SetMeasControl parameter "control"
        public const int MEASCTRL_SINGLESHOT_CTC = 0; //default
        public const int MEASCTRL_C1_GATED = 1;
        public const int MEASCTRL_C1_START_CTC_STOP = 2;
        public const int MEASCTRL_C1_START_C2_STOP = 3;
        public const int MEASCTRL_SW_START_SW_STOP = 6;

        // codes for PH330_SetMeasControl, PH330_SetMarkerEdges, PH330_SetSyncEdgeTrg and PH330_SetInputEdgeTrg parameters "..edge"
        public const int EDGE_RISING = 1;
        public const int EDGE_FALLING = 0;

        // limits for PH330_SetHistoLen
        public const int MINLENCODE = 0;      // 1024 histogram bins
        public const int MAXLENCODE = 9;      // 524288 histogram bins
        public const int DFLTLENCODE = 6;     // 65536 histogram bins, default, see DFLTHISTLEN

        // limits for PH330_SetSyncDiv
        public const int SYNCDIVMIN = 1;
        public const int SYNCDIVMAX = 8;

        // codes for PH330_SetSyncTrgMode and PH330_SetInputTrgMode parameter "mode"
        public const int TRGMODE_ETR = 0;      // edge trigger
        public const int TRGMODE_CFD = 1;      // constant fraction discriminator

        // limits for PH330_SetSyncEdgeTrg and PH330_SetInputEdgeTrg
        public const int TRGLVLMIN = -1500;    // mV
        public const int TRGLVLMAX = 1500;     // mV

        // limits for PH330_SetSyncCFD and PH330_SetInputCFD
        public const int CFDLVLMIN = -1500;    // mV
        public const int CFDLVLMAX = 0;        // mV
        public const int CFDZCMIN = -100;      // mV
        public const int CFDZCMAX = 0;         // mV

        // limits for PH330_SetSyncChannelOffset and PH330_SetInputChannelOffset
        public const int CHANOFFSMIN = -99999; // ps
        public const int CHANOFFSMAX = 99999;  // ps

        // limits for PH330_SetSyncDeadTime and PH330_SetInputDeadTime
        public const int EXTDEADMIN = 800;     // ps
        public const int EXTDEADMAX = 160000;  // ps

        // limits for PH330_SetOffset
        public const int OFFSETMIN = 0;         // ns
        public const int OFFSETMAX = 100000000; // ns

        // limits for PH330_StartMeas
        public const int ACQTMIN = 1;           // ms
        public const int ACQTMAX = 360000000;   // ms  (100*60*60*1000ms = 100h)

        // limits for PH330_SetStopOverflow
        public const int STOPCNTMIN = 1;
        public const uint STOPCNTMAX = 4294967295;  // 32 bit is mem max

        // limits for PH330_SetTriggerOutput
        public const int TRIGOUTMIN = 0;         // 0=off
        public const int TRIGOUTMAX = 16777215;  // in units of 100ns

        // limits for PH330_SetMarkerHoldoffTime
        public const int HOLDOFFMIN = 0;         // ns
        public const int HOLDOFFMAX = 25500;     // ns

        // limits for PH330_SetInputHysteresis
        public const int HYSTCODEMIN = 0;     // approx. 3mV
        public const int HYSTCODEMAX = 1;     // approx. 35mV

        // limits for PH330_SetOflCompression
        public const int HOLDTIMEMIN = 0;     // ms
        public const int HOLDTIMEMAX = 255;   // ms

        // limits for PH330_SetEventFilterParams and PH330_SetEventFilterChannels
        public const int MATCHCNTMIN = 1;
        public const int MATCHCNTMAX = 6;
        public const int INVERSEMIN = 0;
        public const int INVERSEMAX = 1;
        public const int TIMERANGEMIN = 0;        // ps
        public const int TIMERANGEMAX = 160000;   // ps
        public const int USECHANSMIN = 0x000;     // no channels used 
        public const int USECHANSMAX = 0x1FF;     // note: sync bit 0x100 will be ignored in T3 mode
        public const int PASSCHANSMIN = 0x000;    // no channels passed 
        public const int PASSCHANSMAX = 0x1FF;    // note: sync bit 0x100 will be ignored in T3 mode

        // bitmasks for results from PH330_GetFeatures
        public const int FEATURE_DLL = 0x0001;       // DLL License available
        public const int FEATURE_TTTR = 0x0002;      // TTTR mode available
        public const int FEATURE_MARKERS = 0x0004;   // Markers available
        public const int FEATURE_LOWRES = 0x0008;    // Long range mode available 
        public const int FEATURE_TRIGOUT = 0x0010;   // Trigger output available
        public const int FEATURE_PROG_TD = 0x0020;   // Programmable deadtime available
        public const int FEATURE_EXT_FPGA = 0x0040;  // Interface for external FPGA available
        public const int FEATURE_PROG_HYST = 0x0080; // Programmable input hysteresis available
        public const int FEATURE_EVNT_FILT = 0x0100; // Coincidence filtering available
        public const int FEATURE_INPT_MODE = 0x0200; // Programmable input mode (CFD vs. edge trigger)

        // bitmasks for results from PH330_GetFlags
        public const int FLAG_OVERFLOW = 0x0001;      // histo mode only
        public const int FLAG_FIFOFULL = 0x0002;      // TTTR mode only
        public const int FLAG_SYNC_LOST = 0x0004;     // Sync signal was lost
        public const int FLAG_REF_LOST = 0x0008;      // Reference clock was lost
        public const int FLAG_SYSERROR = 0x0010;      // hardware error, must contact support
        public const int FLAG_ACTIVE = 0x0020;        // measurement is running
        public const int FLAG_CNTS_DROPPED = 0x0040;  // counts were dropped
        public const int FLAG_SOFTERROR = 0x0080;     // software error, must contact support

        // bitmasks for results from PH330_GetWarnings
        public const int WARNING_SYNC_RATE_ZERO = 0x0001;
        public const int WARNING_SYNC_RATE_VERY_LOW = 0x0002;
        public const int WARNING_SYNC_RATE_TOO_HIGH = 0x0004;
        public const int WARNING_INPT_RATE_ZERO = 0x0010;
        public const int WARNING_INPT_RATE_TOO_HIGH = 0x0040;
        public const int WARNING_INPT_RATE_RATIO = 0x0100;
        public const int WARNING_DIVIDER_GREATER_ONE = 0x0200;
        public const int WARNING_TIME_SPAN_TOO_SMALL = 0x0400;
        public const int WARNING_OFFSET_UNNECESSARY = 0x0800;
        public const int WARNING_DIVIDER_TOO_SMALL = 0x1000;
        public const int WARNING_COUNTS_DROPPED = 0x2000;
        public const int WARNING_USB20_SPEED_ONLY = 0x4000;

        //////////////////////////// PH330Lib Methods //////////////////////////////

        // Helper method for elegant one-line API calls with error reporting
        public static int APICALL(Func<int> apiCall, [CallerMemberName] string caller = null, [CallerLineNumber] int lineNumber = 0)
        {
            int retcode = 0;
            try
            {
                retcode = apiCall();
            }
            catch (Exception e)
            {
                Console.WriteLine($"\nThe API call in {caller} at line {lineNumber} raised an exception: ");
                Console.WriteLine(e.Message);
                Console.WriteLine("press RETURN to exit");
                Console.ReadLine();
                Environment.Exit(0);
            }
            if (retcode < 0)
            {
                StringBuilder errstr = new StringBuilder(40);
                if (PH330_GetErrorString(errstr, retcode) == 0)
                    Console.WriteLine($"\nThe API call in {caller} at line {lineNumber} returned error {retcode} ({errstr})\n");
                else
                    Console.WriteLine($"\nThe API call in {caller} at line {lineNumber} returned error {retcode}\n");
            }
            return retcode;
        }

        // for debugging only
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_GetLibraryVersion(StringBuilder vers);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_GetErrorString(StringBuilder errstring, int errcode);

        [DllImport(PH330LibSystemname)]
        extern public static int PH330_OpenDevice(int devidx, StringBuilder serial);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_CloseDevice(int devidx);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_Initialize(int devidx, int mode, int refsource);

        // all functions below can only be used after PH330_Initialize

        [DllImport(PH330LibSystemname)]
        extern public static int PH330_GetHardwareInfo(int devidx, StringBuilder model, StringBuilder partno, StringBuilder version);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_GetSerialNumber(int devidx, StringBuilder serial);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_GetFeatures(int devidx, ref int features);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_GetBaseResolution(int devidx, ref double resolution, ref int binsteps);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_GetNumOfInputChannels(int devidx, ref int nchannels);

        [DllImport(PH330LibSystemname)]
        extern public static int PH330_SetSyncDiv(int devidx, int div);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_SetSyncTrgMode(int devidx, int mode);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_SetSyncEdgeTrg(int devidx, int level, int edge);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_SetSyncCFD(int devidx, int level, int zerocross);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_SetSyncChannelOffset(int devidx, int offset);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_SetSyncChannelEnable(int devidx, int enable);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_SetSyncDeadTime(int devidx, int on, int deadtime);

        [DllImport(PH330LibSystemname)]
        extern public static int PH330_SetInputTrgMode(int devidx, int channel, int mode);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_SetInputEdgeTrg(int devidx, int channel, int level, int edge);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_SetInputCFD(int devidx, int channel, int level, int zerocross);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_SetInputChannelOffset(int devidx, int channel, int offset);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_SetInputDeadTime(int devidx, int channel, int on, int deadtime);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_SetInputHysteresis(int devidx, int hystcode);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_SetInputChannelEnable(int devidx, int channel, int enable);

        [DllImport(PH330LibSystemname)]
        extern public static int PH330_SetStopOverflow(int devidx, int stop_ovfl, uint stopcount);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_SetBinning(int devidx, int binning);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_SetOffset(int devidx, int offset);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_SetHistoLen(int devidx, int lencode, out int actuallen);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_SetMeasControl(int devidx, int control, int startedge, int stopedge);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_SetTriggerOutput(int devidx, int period);

        [DllImport(PH330LibSystemname)]
        extern public static int PH330_ClearHistMem(int devidx);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_StartMeas(int devidx, int tacq);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_StopMeas(int devidx);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_CTCStatus(int devidx, ref int ctcstatus);

        [DllImport(PH330LibSystemname)]
        extern public static int PH330_GetHistogram(int devidx, uint[] chcount, int channel);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_GetAllHistograms(int devidx, uint[] chcount);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_GetResolution(int devidx, ref double resolution);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_GetSyncPeriod(int devidx, ref double period);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_GetSyncRate(int devidx, ref int syncrate);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_GetCountRate(int devidx, int channel, ref int cntrate);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_GetAllCountRates(int devidx, ref int syncrate, int[] cntrates);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_GetFlags(int devidx, ref int flags);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_GetElapsedMeasTime(int devidx, out double elapsed);

        [DllImport(PH330LibSystemname)]
        extern public static int PH330_GetWarnings(int devidx, ref int warnings);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_GetWarningsText(int devidx, StringBuilder text, int warnings);

        // for the time tagging modes only
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_SetOflCompression(int devidx, int holdtime);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_SetMarkerHoldoffTime(int devidx, int holdofftime);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_SetMarkerEdges(int devidx, int me1, int me2, int me3, int me4);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_SetMarkerEnable(int devidx, int en1, int en2, int en3, int en4);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_ReadFiFo(int devidx, uint[] buffer, ref int nactual);

        // for event filtering, time tagging modes only
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_SetEventFilterParams(int devidx, int timerange, int matchcnt, int inverse);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_SetEventFilterChannels(int devidx, int usechannels, int passchannels);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_EnableEventFilter(int devidx, int enable);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_SetFilterTestMode(int devidx, int testmode);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_GetFilterInputRates(int devidx, ref int syncrate, int[] cntrates);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_GetFilterOutputRates(int devidx, ref int syncrate, int[] cntrates);

        // for debugging only
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_GetDebugInfo(int devidx, StringBuilder debuginfo);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_GetModuleInfo(int devidx, ref int modelcode, ref int versioncode);
        [DllImport(PH330LibSystemname)]
        extern public static int PH330_SaveDebugDump(int devidx, StringBuilder filepath);

        //////////////////////////// PH330Lib Error Codes //////////////////////////////

        public const int PH330_ERROR_NONE                      =    0;

        public const int PH330_ERROR_DEVICE_OPEN_FAIL          =   -1;
        public const int PH330_ERROR_DEVICE_BUSY               =   -2;
        public const int PH330_ERROR_DEVICE_HEVENT_FAIL        =   -3;
        public const int PH330_ERROR_DEVICE_CALLBSET_FAIL      =   -4;
        public const int PH330_ERROR_DEVICE_BARMAP_FAIL        =   -5;
        public const int PH330_ERROR_DEVICE_CLOSE_FAIL         =   -6;
        public const int PH330_ERROR_DEVICE_RESET_FAIL         =   -7;
        public const int PH330_ERROR_DEVICE_GETVERSION_FAIL    =   -8;
        public const int PH330_ERROR_DEVICE_VERSION_MISMATCH   =   -9;
        public const int PH330_ERROR_DEVICE_NOT_OPEN           =  -10;
        public const int PH330_ERROR_DEVICE_LOCKED             =  -11;
        public const int PH330_ERROR_DEVICE_DRIVERVER_MISMATCH =  -12;

        public const int PH330_ERROR_INSTANCE_RUNNING          =  -16;
        public const int PH330_ERROR_INVALID_ARGUMENT          =  -17;
        public const int PH330_ERROR_INVALID_MODE              =  -18;
        public const int PH330_ERROR_INVALID_OPTION            =  -19;
        public const int PH330_ERROR_INVALID_MEMORY            =  -20;
        public const int PH330_ERROR_INVALID_RDATA             =  -21;
        public const int PH330_ERROR_NOT_INITIALIZED           =  -22;
        public const int PH330_ERROR_NOT_CALIBRATED            =  -23;
        public const int PH330_ERROR_DMA_FAIL                  =  -24;
        public const int PH330_ERROR_XTDEVICE_FAIL             =  -25;
        public const int PH330_ERROR_FPGACONF_FAIL             =  -26;
        public const int PH330_ERROR_IFCONF_FAIL               =  -27;
        public const int PH330_ERROR_FIFORESET_FAIL            =  -28;
        public const int PH330_ERROR_THREADSTATE_FAIL          =  -29;
        public const int PH330_ERROR_THREADLOCK_FAIL           =  -30;

        public const int PH330_ERROR_USB_GETDRIVERVER_FAIL     =  -32;
        public const int PH330_ERROR_USB_DRIVERVER_MISMATCH    =  -33;
        public const int PH330_ERROR_USB_GETIFINFO_FAIL        =  -34;
        public const int PH330_ERROR_USB_HISPEED_FAIL          =  -35;
        public const int PH330_ERROR_USB_VCMD_FAIL             =  -36;
        public const int PH330_ERROR_USB_BULKRD_FAIL           =  -37;
        public const int PH330_ERROR_USB_RESET_FAIL            =  -38;

        public const int PH330_ERROR_LANEUP_TIMEOUT            =  -40;
        public const int PH330_ERROR_DONEALL_TIMEOUT           =  -41;
        public const int PH330_ERROR_MB_ACK_TIMEOUT            =  -42;
        public const int PH330_ERROR_MACTIVE_TIMEOUT           =  -43;
        public const int PH330_ERROR_MEMCLEAR_FAIL             =  -44;
        public const int PH330_ERROR_MEMTEST_FAIL              =  -45;
        public const int PH330_ERROR_CALIB_FAIL                =  -46;
        public const int PH330_ERROR_REFSEL_FAIL               =  -47;
        public const int PH330_ERROR_STATUS_FAIL               =  -48;
        public const int PH330_ERROR_MODNUM_FAIL               =  -49;
        public const int PH330_ERROR_DIGMUX_FAIL               =  -50;
        public const int PH330_ERROR_MODMUX_FAIL               =  -51;
        public const int PH330_ERROR_MODFWPCB_MISMATCH         =  -52;
        public const int PH330_ERROR_MODFWVER_MISMATCH         =  -53;
        public const int PH330_ERROR_MODPROPERTY_MISMATCH      =  -54;
        public const int PH330_ERROR_INVALID_MAGIC             =  -55;
        public const int PH330_ERROR_INVALID_LENGTH            =  -56;
        public const int PH330_ERROR_RATE_FAIL                 =  -57;
        public const int PH330_ERROR_MODFWVER_TOO_LOW          =  -58;
        public const int PH330_ERROR_MODFWVER_TOO_HIGH         =  -59;
        public const int PH330_ERROR_MB_ACK_FAIL               =  -60;

        public const int PH330_ERROR_EEPROM_F01                =  -64;
        public const int PH330_ERROR_EEPROM_F02                =  -65;
        public const int PH330_ERROR_EEPROM_F03                =  -66;
        public const int PH330_ERROR_EEPROM_F04                =  -67;
        public const int PH330_ERROR_EEPROM_F05                =  -68;
        public const int PH330_ERROR_EEPROM_F06                =  -69;
        public const int PH330_ERROR_EEPROM_F07                =  -70;
        public const int PH330_ERROR_EEPROM_F08                =  -71;
        public const int PH330_ERROR_EEPROM_F09                =  -72;
        public const int PH330_ERROR_EEPROM_F10                =  -73;
        public const int PH330_ERROR_EEPROM_F11                =  -74;
        public const int PH330_ERROR_EEPROM_F12                =  -75;
        public const int PH330_ERROR_EEPROM_F13                =  -76;
        public const int PH330_ERROR_EEPROM_F14                =  -77;
        public const int PH330_ERROR_EEPROM_F15                =  -78;

        public const int PH330_ERROR_UNSUPPORTED_FUNCTION      =  -80;
        public const int PH330_ERROR_WRONG_TRGMODE             =  -81;
        public const int PH330_ERROR_BULKRDINIT_FAIL           =  -82;
        public const int PH330_ERROR_CREATETHREAD_FAIL         =  -83;
        public const int PH330_ERROR_FILEOPEN_FAIL             =  -84;
        public const int PH330_ERROR_FILEWRITE_FAIL            =  -85;
        public const int PH330_ERROR_FILEREAD_FAIL             =  -86;

        public const int PH330_ERROR_INVALID_ARGUMENT_1        = -201;
        public const int PH330_ERROR_INVALID_ARGUMENT_2        = -202;
        public const int PH330_ERROR_INVALID_ARGUMENT_3        = -203;
        public const int PH330_ERROR_INVALID_ARGUMENT_4        = -204;
        public const int PH330_ERROR_INVALID_ARGUMENT_5        = -205;
        public const int PH330_ERROR_INVALID_ARGUMENT_6        = -206;
        public const int PH330_ERROR_INVALID_ARGUMENT_7        = -207;
        public const int PH330_ERROR_INVALID_ARGUMENT_8        = -208;
    
    }
}

