﻿/************************************************************************

  Demo access to PicoHarp 330 via PH330Lib.dll or libph330.so v.2.0.
  The program performs a measurement based on hardcoded settings.
  The resulting event data is stored in a binary output file.

  Michael Wahl, PicoQuant GmbH, August 2024

  Note: This is a console application

  Note: At the API level channel numbers are indexed 0..N-1 
        where N is the number of channels the device has.

  Note: This demo writes only raw event data to the output file.
  It does not write a file header as regular .ptu files have it.

  Tested with the following compilers:

  - Microsoft Visual C# 2022 (Windows)
  - Mono 6.12.0 (Windows and Linux)

************************************************************************/

using System;  // for Console
using System.Text;  // for StringBuilder 
using System.IO;  // for File
using static PH330LibSpace.PH330Lib;

namespace tttrmode
{
    class tttrmode
    {
        static void Main(string[] args)
        {
            //////////////////////// some settings you can change /////////////////////////
            
            int Mode = MODE_T2;  // you can change this, adjust other settings accordingly!
            int Binning = 0;     // you can change this, meaningful only in T3 mode, observe limits 
            int Offset = 0;      // you can change this, meaningful only in T3 mode, observe limits 
            int Tacq = 10000;    // measurement time in millisec, you can change this, observe limits 

            int SyncDivider = 1;  // you can change this, usually 1 in T2 mode 

            int SyncTrgMode = TRGMODE_ETR;       // you can change this to TRGMODE_CFD
                                                 // in case of SyncTrgMode == TRGMODE_ETR this will apply:
            int SyncTriggerEdge = EDGE_FALLING;  // you can change this to EDGE_RISING
            int SyncTriggerLevel = -50;          // in mV, you can change this, observe limits 
                                                 // in case of SyncTrgMode == TRGMODE_CFD this will apply:
            int SyncCFDZeroCross = -20;          // in mV, you can change this, observe limits 
            int SyncCFDLevel = -50;              // in mV, you can change this, observe limits 

            int InputTrgMode = TRGMODE_ETR;      // you can change this to TRGMODE_CFD
                                                 // in case of InputTrgMode == TRGMODE_ETR this will apply:
            int InputTriggerEdge = EDGE_FALLING; // you can change this to EDGE_RISING
            int InputTriggerLevel = -50;         // in mV, you can change this, observe limits 
                                                 // in case of InputTrgMode == TRGMODE_CFD this will apply:
            int InputCFDZeroCross = -20;         // in mV, you can change this, observe limits 
            int InputCFDLevel = -50;             // in mV, you can change this, observe limits 


            ///////////////////////////// program variables /////////////////////////////

            int i, j;
            int retcode;
            int[] dev = new int[MAXDEVNUM];
            int found = 0;
            int NumChannels = 0;

            StringBuilder LibVer = new StringBuilder(8);
            StringBuilder Serial = new StringBuilder(8);
            StringBuilder Errstr = new StringBuilder(40);
            StringBuilder Model = new StringBuilder(32);
            StringBuilder Partno = new StringBuilder(16);
            StringBuilder Version = new StringBuilder(16);
            StringBuilder Wtext = new StringBuilder(16384);

            double Resolution = 0;

            int Syncrate = 0;
            int Countrate = 0;
            int ctcstatus = 0;
            int flags = 0;
            long Progress = 0;
            int nRecords = 0;
            int warnings = 0;

            uint[] buffer = new uint[TTREADMAX];

            FileStream fs = null;
            BinaryWriter bw = null;

            ///////////////////////////// main program code /////////////////////////////
            
            Console.WriteLine("PicoHarp PH330Lib Demo Application                     PicoQuant GmbH, 2024");
            Console.WriteLine("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~");

            if (APICALL(() => PH330_GetLibraryVersion(LibVer)) < 0)
                goto ex;

            Console.WriteLine("PH330Lib Version is " + LibVer);

            if (LibVer.ToString() != LIB_VERSION)
            {
                Console.WriteLine("This program requires PH330Lib v." + LIB_VERSION);
                goto ex;
            }

            try
            {
                fs = File.Create("tttrmode.out");
                bw = new BinaryWriter(fs);
            }
            catch (Exception)
            {
                Console.WriteLine("Error creating file");
                goto ex;
            }

            Console.WriteLine("Searching for PicoHarp 330 devices...");
            Console.WriteLine("Devidx     Status");

            for (i = 0; i < MAXDEVNUM; i++)
            {
                //not using APICALL here as we wish to handle errors gracefully
                retcode = PH330_OpenDevice(i, Serial);
                if (retcode == 0) //Grab any device we can open
                {
                    Console.WriteLine("  {0}        S/N {1}", i, Serial);
                    dev[found] = i; //keep index to devices we want to use
                    found++;
                }
                else
                {
                    if (retcode == PH330_ERROR_DEVICE_OPEN_FAIL)
                        Console.WriteLine("  {0}        no device", i);
                    else
                    {
                        PH330_GetErrorString(Errstr, retcode);
                        Console.WriteLine("  {0}        S/N {1}", i, Errstr);
                    }
                }
            }

            //In this demo we will use the first device we find, i.e. dev[0].
            //You can also use multiple devices in parallel.
            //You can also check for specific serial numbers, so that you always know 
            //which physical device you are talking to.

            if (found < 1)
            {
                Console.WriteLine("No device available.");
                goto ex;
            }

            Console.WriteLine("Using device {0}", dev[0]);
            Console.WriteLine("Initializing the device...");

            if (APICALL(() => PH330_Initialize(dev[0], Mode, 0)) < 0)
                goto ex;

            if (APICALL(() => PH330_GetHardwareInfo(dev[0], Model, Partno, Version)) < 0)
                goto ex;
            else
                Console.WriteLine("Found Model {0} Part no {1} Version {2}", Model, Partno, Version);

            if (APICALL(() => PH330_GetNumOfInputChannels(dev[0], ref NumChannels)) < 0)
                goto ex;
            else
                Console.WriteLine("Device has {0} input channels.", NumChannels);

            Console.WriteLine("\n\nUsing the following settings:\n");

            Console.WriteLine("Mode              : " + Mode);
            Console.WriteLine("Binning           : " + Binning);
            Console.WriteLine("Offset            : " + Offset);
            Console.WriteLine("AcquisitionTime   : " + Tacq);
            Console.WriteLine("SyncDivider       : " + SyncDivider);

            if (APICALL(() => PH330_SetSyncDiv(dev[0], SyncDivider)) < 0)
                goto ex;

            if (APICALL(() => PH330_SetSyncTrgMode(dev[0], SyncTrgMode)) < 0)
                goto ex;

            if (SyncTrgMode == TRGMODE_ETR)
            {
                if (APICALL(() => PH330_SetSyncEdgeTrg(dev[0], SyncTriggerLevel, SyncTriggerEdge)) < 0)
                    goto ex;
            }
            if (SyncTrgMode == TRGMODE_CFD)
            {
                if (APICALL(() => PH330_SetSyncCFD(dev[0], SyncCFDLevel, SyncCFDZeroCross)) < 0)
                    goto ex;
            }

            if (APICALL(() => PH330_SetSyncChannelOffset(dev[0], 0)) < 0)    //in ps, can emulate a cable delay
                goto ex;

            for (i = 0; i < NumChannels; i++)  // we use the same input settings for all channels
            {
                if (APICALL(() => PH330_SetInputTrgMode(dev[0], i, InputTrgMode)) < 0)
                    goto ex;

                if (InputTrgMode == TRGMODE_ETR)
                {
                    if (APICALL(() => PH330_SetInputEdgeTrg(dev[0], i, InputTriggerLevel, InputTriggerEdge)) < 0)
                        goto ex;
                }
                if (InputTrgMode == TRGMODE_CFD)
                {
                    if (APICALL(() => PH330_SetInputCFD(dev[0], i, InputCFDLevel, InputCFDZeroCross)) < 0)
                        goto ex;
                }

                if (APICALL(() => PH330_SetInputChannelOffset(dev[0], i, 0)) < 0)  //in ps, can emulate a cable delay
                    goto ex;

                if (APICALL(() => PH330_SetInputChannelEnable(dev[0], i, 1)) < 0)
                    goto ex;
            }

            if (Mode != MODE_T2)
            {
                if (APICALL(() => PH330_SetBinning(dev[0], Binning)) < 0)
                    goto ex;

                if (APICALL(() => PH330_SetOffset(dev[0], Offset)) < 0)
                    goto ex;
            }

            if (APICALL(() => PH330_GetResolution(dev[0], ref Resolution)) < 0)
                goto ex;

            Console.WriteLine("Resolution is {0} ps", Resolution);
            Console.WriteLine();

            // After Init allow 150 ms for valid  count rate readings
            // Subsequently you get new values after every 100ms
            System.Threading.Thread.Sleep(150);

            if (APICALL(() => PH330_GetSyncRate(dev[0], ref Syncrate)) < 0)
                goto ex;

            Console.WriteLine("Syncrate = {0}/s", Syncrate);

            for (i = 0; i < NumChannels; i++)  // for all channels
            {
                if (APICALL(() => PH330_GetCountRate(dev[0], i, ref Countrate)) < 0)
                    goto ex;

                Console.WriteLine("Countrate[{0}] = {1}/s", i, Countrate);
            }

            Console.WriteLine();

            // After getting the count rates you can check for warnings
            if (APICALL(() => PH330_GetWarnings(dev[0], ref warnings)) < 0)
                goto ex;

            if (warnings != 0)
            {
                if (APICALL(() => PH330_GetWarningsText(dev[0], Wtext, warnings)) < 0)
                    goto ex;

                Console.WriteLine("{0}", Wtext);
            }

            Progress = 0;
            Console.Write("Progress: {0,12}", Progress);

            if (APICALL(() => PH330_StartMeas(dev[0], Tacq)) < 0)
                goto ex;

            while (true)
            {
                if (APICALL(() => PH330_GetFlags(dev[0], ref flags)) < 0)
                    goto ex;

                if ((flags & FLAG_FIFOFULL) != 0)
                {
                    Console.WriteLine();
                    Console.WriteLine("FiFo Overrun!");
                    goto stoptttr;
                }

                if (APICALL(() => PH330_ReadFiFo(dev[0], buffer, ref nRecords)) < 0)
                    goto ex;

                if (nRecords > 0)
                {
                    for (j = 0; j < nRecords; j++)
                        bw.Write(buffer[j]);

                    Progress += nRecords;
                    Console.Write("\b\b\b\b\b\b\b\b\b\b\b\b{0,12}", Progress);
                }
                else
                {
                    if (APICALL(() => PH330_CTCStatus(dev[0], ref ctcstatus)) < 0)
                        goto ex;

                    if (ctcstatus > 0)
                    {
                        Console.WriteLine();
                        Console.WriteLine("Done");
                        goto stoptttr;
                    }
                }

                // within this loop you can also read the count rates if needed.
            }

            stoptttr:
            Console.WriteLine();

            if (APICALL(() => PH330_StopMeas(dev[0])) < 0)
                goto ex;

            ex:

            for (i = 0; i < MAXDEVNUM; i++)  // no harm to close all
                PH330_CloseDevice(i);

            if (bw != null)
            {
                try
                {
                    bw.Flush();
                    bw.Close();
                    bw.Dispose();
                    bw = null;
                }
                catch (Exception e)
                {
                    Console.WriteLine("Error closing the file: " + e);
                }
            }

            Console.WriteLine("press RETURN to exit");
            Console.ReadLine();
        }
    }
}
