% PH330call  Calls a routine from PH330Lib with error handling
%   
%   [retcode, ...] = PH330call(routinename, ...) calls the routine
%   specified by routinename, where ... are variable lists of arguments 
%   that must match the signature of the routine.    
%   PH330Lib must have been loaded before PH330call can be used.
%   PH330call is essentially a wrapper around the native Matlab function 
%   callib. The usage is therefore very similar, only that the library name
%   is omitted. After invoking calllib PH330call will check for errors (ret<0)
%   and if there was an error translate it to an error string.
%   Upon detecting an error PH330call will invoke error() to report the 
%   error code and error string and then terminate the program. This is 
%   useful for lean demo code but may not be ideal for production code.
%   In that case you may want to modify PH330call and, for instance,
%   replace error(..) by warning(..). Note, however, that then you
%   need to check the return code of PH330call and handle possible errors 
%   at the caller's level.
% 
%   See also LOADLIBRARY, CALLLIB.

function varargout = PH330call(varargin)

% see https://nl.mathworks.com/help/matlab/matlab_prog/support-variable-number-of-inputs.html
% and https://nl.mathworks.com/help/matlab/matlab_prog/support-variable-number-of-outputs.html

funcname = varargin{1};

nOutputs = nargout;
if(nOutputs==0) %in case we were called without assigning the return code to a variable
    nOutputs = 1;
end
varargout = cell(1,nOutputs);

[varargout{:}] = calllib('PH330Lib', varargin{:});

retcode = varargout{1};  % this is our retcode from the DLL call

if (retcode<0)
    ErrorStr   = blanks(40); % provide enough length!
    ErrorPtr   = libpointer('cstring', ErrorStr);
    [ret, ErrorStr] = calllib('PH330Lib', 'PH330_GetErrorString', ErrorPtr, retcode);
    % if you do not want the program to abort upon every error here 
    % you may want to replace error(..) by warning(..). Note that then you
    % need to check the return code of PH330call and handle possible errors 
    % at the caller's level.
    if(ret==0)
        error('PH330Lib Error %d (%s) calling %s', retcode, ErrorStr, funcname);
    else
        error('PH330Lib Error %d calling %s\n', retcode, funcname);
    end       
end






