
  Demo access to PicoHarp 330 via PH330Lib.dll or libph330.so v.2.0
% The program performs a TTTR measurement based on hardcoded settings.
% The resulting data stream is stored in a binary output file.
%
% Michael Wahl, PicoQuant, July 2024 

% Constants from ph330defin.h in CAPITALS. You cannot change these!

REQLIBVER   =   '2.0';    % the library version this program expects
MAXDEVNUM   =       8;    % max number of PH330 devices
MAXINPCHAN  =       4;    % max number of input channels
TTREADMAX   = 1048576;    % 1M event records max from one call of PH330_ReadFiFo
MAXBINSTEPS =      24;    % max binning steps, irrelevant in T2 mode
 
MODE_HIST   =       0;    % histogramming mode
MODE_T2     =       2;    % Time tagging mode T2
MODE_T3     =       3;    % Time tagging mode T3

FLAG_FIFOFULL = hex2dec('0002');

% codes for PH330_SetSyncTrgMode and PH330_SetInputTrgMode parameter "mode"
TRGMODE_ETR =       0;     % edge trigger
TRGMODE_CFD =       1;     % constant fraction discriminator

% limits for PH330_SetSyncEdgeTrg and PH330_SetInputEdgeTrg
TRGLVLMIN   =       -1500;   % mV
TRGLVLMAX   =        1500;   % mV

% limits for PH330_SetSyncCFD and PH330_SetInputCFD
CFDLVLMIN   =       -1500;   % mV
CFDLVLMAX   =           0;   % mV
CFDZCMIN    =        -100;   % mV
CFDZCMAX    =           0;   % mV

% limits for tacq in PH330_StartMeas
ACQTMIN     =          1;	   % ms
ACQTMAX     =  360000000;	   % ms  (100*60*60*1000ms = 100h)

% Errorcodes from errorcodes.h

PH330_ERROR_DEVICE_OPEN_FAIL		 = -1;

% Settings for the measurement, adjust them to your setup!

SyncTrgMode   = TRGMODE_ETR; % you can change this 
InputTrgMode  = TRGMODE_ETR; % you can change this

SyncTrgLevel  =  -50;     % mV, you can change this
SyncTrgEdge   =    0;     % mV, you can change this
InputTrgLevel =  -50;     % mV, you can change this
InputTrgEdge  =    0;     % mV, you can change this

SyncCFDLevel      =  -50;    % mV, you can change this
SyncCDFZeroCross  =  -20;    % mV, you can change this
InputCFDLevel     =  -50;    % mV, you can change this
InputCFDZeroCross =  -20;    % mV, you can change this
 
Mode          = MODE_T2;  %  you can change this

SyncDiv       =       1;  %  you can change this (observe mode!)
Binning       =       0;  %  you can change this, irrelevant in T2 mode
Tacq          =    1000;  %  ms, you can change this       
    

fprintf('\nPicoHarp 330 PH330Lib Demo Application          PicoQuant 2024\n');

if (~libisloaded('PH330Lib'))    
    %Attention: The header file name given below is case sensitive and must
    %be spelled exactly the same as the actual name on disk except the file 
    %extension. 
    %Wrong case will apparently do the load successfully but you will not
    %be able to access the library!
    %The alias is used to provide a fixed spelling for any further access via
    %calllib() etc, which is also case sensitive.
    
    %To load the right dll for the given OS we use the mexext command.
    %Note that only x86-64 is supported.
    OS = mexext;
    if strcmp('mexw64', OS)
        DLL = 'PH330Lib.dll';                        % Windows 64 bit
    elseif strcmp('mexa64', OS)
        DLL = '/usr/local/lib64/ph330/libph330.so';  % Linux 64 bit
    else
        fprintf('\nNo supported operating system\n');
        return;
    end
    loadlibrary(DLL, 'ph330lib.h', 'alias', 'PH330Lib');
else
    fprintf('Note: PH330Lib was already loaded\n');
end

if (libisloaded('PH330Lib'))
    fprintf('PH330Lib opened successfully\n');
    %libfunctionsview PH330Lib; % use this to test for proper loading
else
    fprintf('Could not open PH330Lib\n');
    return;
end
   
% Normally one uses Matlab's native calllib() to invoke DLL
% routines. To simplify things we use PH330call.m where possible.
% Type help PH330call to understand this little helper routine.
% Note that it will terminate the program in case of errors.

LibVersion    = blanks(8); % provide enough length!
LibVersionPtr = libpointer('cstring', LibVersion);
[ret, LibVersion] = PH330call('PH330_GetLibraryVersion', LibVersionPtr);	
fprintf('PH330Lib version is %s\n', LibVersion);
if ~strcmp(LibVersion,REQLIBVER)
    fprintf('This program requires PH330Lib version %s\n', REQLIBVER);
    return;
end

fid = fopen('tttrmode.out','wb');
if (fid<0)
    fprintf('Cannot open output file\n');
    return;
end

fprintf('\nSearching for PicoHarp 330 devices...');

dev = [];
found = 0;
Serial     = blanks(8); % provide enough length!
SerialPtr  = libpointer('cstring', Serial);
ErrorStr   = blanks(40); % provide enough length!
ErrorPtr   = libpointer('cstring', ErrorStr);

% in this loop we DO NOT use PH330call to allow more specific error handling 
for i=0:MAXDEVNUM-1
    [ret, Serial] = calllib('PH330Lib', 'PH330_OpenDevice', i, SerialPtr);
    if (ret==0)       % grab any PicoHarp 330 we successfully opened
        fprintf('\n  %1d        S/N %s', i, Serial);
        found = found+1;            
        dev(found)=i; % keep index to devices we may want to use
    else
        if(ret==PH330_ERROR_DEVICE_OPEN_FAIL)
            fprintf('\n  %1d        no device', i);
        else 
            [ret, ErrorStr] = calllib('PH330Lib', 'PH330_GetErrorString', ErrorPtr, ret);
            fprintf('\n  %1d        %s', i,ErrorStr);
        end
    end
end
    
% in this demo we will use the first PicoHarp 330 device we found, i.e. dev(1)
% you could also check for a specific serial number, so that you always know 
% which physical device you are talking to.

if (found<1)
	fprintf('\nNo device available. Aborted.\n');
	return; 
end

fprintf('\nUsing device #%1d',dev(1));
fprintf('\nInitializing the device...');

[ret] = PH330call('PH330_Initialize', dev(1), Mode, 0);
% Note that here we assign but then actually ignore the value ret.
% Also note the brackets. We do this for consistency with all other
% calls to PH330call where there can be more return values in the bracket. 
% PH330call always requires a correct assignment of return values
% according to the signature of the called function. This is important 
% for the variable aruments lists passed in and out from PH330call 
% (as well as for callib) to work properly. 

% the following library call is only for information
Model      = blanks(24); % provide enough length!
Partno     = blanks(8);  % provide enough length!
Version    = blanks(8);  % provide enough length!
ModelPtr   = libpointer('cstring', Model);
PartnoPtr  = libpointer('cstring', Partno);
VersionPtr = libpointer('cstring', Version);

[ret, Model, Partno, Version] = PH330call('PH330_GetHardwareInfo', dev(1), ModelPtr, PartnoPtr, VersionPtr);
fprintf('\nFound model %s part number %s version %s', Model, Partno, Version);             

NumInpChannels = int32(0);
NumChPtr = libpointer('int32Ptr', NumInpChannels);
[ret,NumInpChannels] = PH330call('PH330_GetNumOfInputChannels', dev(1), NumChPtr); 
fprintf('\nDevice has %i input channels.', NumInpChannels);             
   
[ret] = PH330call('PH330_SetSyncDiv', dev(1), SyncDiv);

if(SyncTrgMode == TRGMODE_ETR)
    [ret] = PH330call('PH330_SetSyncEdgeTrg', dev(1), SyncTrgLevel, SyncTrgEdge);
end

if(SyncTrgMode == TRGMODE_CFD)
    [ret] = PH330call('PH330_SetSyncCFD', dev(1), SyncCFDLevel, SyncCFDZeroCross);
end
 
[ret] = PH330call('PH330_SetSyncChannelOffset', dev(1), 0);

for i=0:NumInpChannels-1 % we use the same input settings for all channels
    if(InputTrgMode == TRGMODE_ETR)
        [ret] = PH330call('PH330_SetInputEdgeTrg', dev(1), i, InputTrgLevel, InputTrgEdge);
    end
    if(InputTrgMode == TRGMODE_CFD)
        [ret] = PH330call('PH330_SetInputCFD', dev(1), i, InputCFDLevel, InputCFDZeroCross);
    end    
    [ret] = PH330call('PH330_SetInputChannelOffset', dev(1), i, 0);
end

if (Mode ~= MODE_T2)    % The next 2 calls make no sense in T2 mode
    [ret] = PH330call('PH330_SetBinning', dev(1), Binning);
    [ret] = PH330call('PH330_SetOffset', dev(1), 0);
end

% PH330_GetResolution is meaningless in T2 mode but it does no harm
Resolution = 0;
ResolutionPtr = libpointer('doublePtr', Resolution);
[ret, Resolution] = PH330call('PH330_GetResolution', dev(1), ResolutionPtr);
fprintf('\nResolution=%1dps', Resolution);

pause(0.15); % after Init or SetSyncDiv you must allow 150 ms for valid new 
             % count rates, otherwise you get new values every 100 ms

% from here you could repeat the measurement (with the same settings)

Syncrate = 0;
SyncratePtr = libpointer('int32Ptr', Syncrate);
[ret, Syncrate] = PH330call('PH330_GetSyncRate', dev(1), SyncratePtr);
fprintf('\nSyncrate=%1d/s', Syncrate);
 
for i=0:NumInpChannels-1
	Countrate = 0;
	CountratePtr = libpointer('int32Ptr', Countrate);
	[ret, Countrate] = PH330call('PH330_GetCountRate', dev(1), i, CountratePtr);
	fprintf('\nCountrate%1d=%1d/s', i, Countrate);
end

Warnings = 0;
WarningsPtr = libpointer('int32Ptr', Warnings);
[ret, Warnings] = PH330call('PH330_GetWarnings', dev(1), WarningsPtr);
if (Warnings~=0)
    Warningstext = blanks(16384); % provide enough length!
    WtextPtr     = libpointer('cstring', Warningstext);
    [ret, Warningstext] = PH330call('PH330_GetWarningsText', dev(1), WtextPtr, Warnings);
    fprintf('\n\n%s',Warningstext);
end

Progress = 0;
fprintf('\nProgress:%11d',Progress);

buffer  = uint32(zeros(1,TTREADMAX));
bufferptr = libpointer('uint32Ptr', buffer);
nactual = int32(0);
nactualptr = libpointer('int32Ptr', nactual);
ctcdone = int32(0);
ctcdonePtr = libpointer('int32Ptr', ctcdone);       

[ret] = PH330call('PH330_StartMeas', dev(1), Tacq); 
       
while(1)  % the data retrieval loop, keep this slim
    
    flags = int32(0);
    flagsPtr = libpointer('int32Ptr', flags);
    [ret,flags] = PH330call('PH330_GetFlags', dev(1), flagsPtr);   
    if (bitand(uint32(flags),FLAG_FIFOFULL)) 
        fprintf('\nFiFo Overrun!\n'); 
        break;
    end

    [ret, buffer, nactual] = PH330call('PH330_ReadFiFo', dev(1), bufferptr, nactualptr);
    % consider that PH330_ReadFiFo may return zero events
    if(nactual) 
        cnt = fwrite(fid, buffer(1:nactual),'uint32');
        if(cnt ~= nactual)
            fprintf('\nfile write error\n');
            break;
        end          
		Progress = Progress + nactual;
		fprintf('\b\b\b\b\b\b\b\b\b\b\b%11d',Progress);
    else
        [ret,ctcdone] = PH330call('PH330_CTCStatus', dev(1), ctcdonePtr);     
        if (ctcdone) 
            fprintf('\nDone\n'); 
            break;
        end
    end
     
    % you can read the count rates here if needed, but do it sparingly

end % while

[ret] = PH330call('PH330_StopMeas', dev(1)); 
            
fprintf('\nBinary output data is in tttrmode.out\n');

for i=0:MAXDEVNUM-1 % no harm to close all
    [ret] = PH330call('PH330_CloseDevice', i);
end

if(fid>0) 
    fclose(fid);
end

