﻿{
  Demo access to PicoHarp 330 via PH330Lib.dll or libph330.so v.2.0
  The program performs a measurement based on hardcoded settings.
  The resulting photon event data is instantly histogrammed.
  Works in T3 mode only!

  Stefan Eilers, Michael Wahl, PicoQuant, July 2024

  Note: This is a console application

  Note: At the API level the input channel numbers are indexed 0..N-1
  where N is the number of input channels the device has.
  Upon processing we map this back to 1..N corresponding to the front
  panel labelling.

  Tested with the following compilers:

  - Delphi 11 (Windows)
  - Lazarus 3.0 RC1 (Windows)
  - Lazarus 2.2.0 / FPC 3.2.2 (Linux)
}

program tttrmode;

{$IF defined(MSWINDOWS)}
  {$APPTYPE CONSOLE}  // Windows needs this, Linux does not want it
{$ENDIF}

uses
  {$ifdef fpc}
  SysUtils,
  {$else}
  System.SysUtils,
  System.Ansistrings,
  {$ENDIF}
  PH330Lib in 'PH330Lib.pas';

const
  T3HISTBINS = 32768; //=2^15, dtime in T3 mode has 15 bits

var
  DevIdx          : array [0..MAXDEVNUM - 1] of LongInt;
  Dev             : LongInt;
  Found           : Integer = 0;
  RetCode         : LongInt;
  CTCStatus       : LongInt;
  strLibVers      : array[0..7]  of AnsiChar;
  HW_Model        : array[0..31] of AnsiChar;
  HW_Partno       : array[0..8]  of AnsiChar;
  HW_Serial       : array[0..9]  of AnsiChar;
  HW_Version      : array[0..15] of AnsiChar;
  Errorstring     : array[0..39] of AnsiChar;
  DebugInfoBuffer : array[0..16383] of AnsiChar;
  NumChannels     : LongInt;
  Mode            : LongInt = MODE_T3;
  Binning         : LongInt = 0; // you can change this (meaningless in T2 mode)
  Offset          : LongInt = 0; // normally no need to change this
  Tacq            : LongInt = 1000; // you can change this, unit is millisec

  // Sync Settings
  SyncDivider     : LongInt = 1; // you can change this
  SyncChannelOffset : LongInt =           0; // in ps, you can change this (like a cable delay)
  SyncTrgMode       : LongInt = TRGMODE_ETR; // you can change this to TRGMODE_CFD

  // in case of SyncTrgMode == TRGMODE_ETR this will apply:
  SyncTrgEdge  : LongInt = EDGE_FALLING; // you can change this to EDGE_RISING
  SyncTrgLevel : LongInt = -50; // in mV, you can change this

  // in case of SyncTrgMode == TRGMODE_CFD this will apply:
  SyncCFDZeroCross : LongInt = -50; // in mV, you can change this
  SyncCFDLevel     : LongInt = -50; // in mV, you can change this

  // Input Settings
  InputChannelOffset : LongInt =           0; // in ps, you can change this (like a cable delay)
  InputTrgMode       : LongInt = TRGMODE_ETR; // you can change this to TRGMODE_CFD

  // in case of InputTrgMode == TRGMODE_ETR this will apply:
  InputTrgEdge  : LongInt = EDGE_FALLING; // you can change this to EDGE_RISING
  InputTrgLevel : LongInt = -50;          // in mV, you can change this

  // in case of InputTrgMode == TRGMODE_CFD this will apply:
  InputCFDZeroCross : LongInt = -20;      // in mV, you can change this
  InputCFDLevel     : LongInt = -50;      // in mV, you can change this

  // other variables
  Resolution    : Double;
  SyncRate      : LongInt;
  CountRate     : LongInt;
  Flags         : LongInt;
  Warnings      : LongInt;
  WarningsText  : array[0..16343] of AnsiChar; // must have 16384 bytes of text buffer
  Records       : LongInt;
  Progress      : LongInt = 0;
  StopRetry     : LongInt = 0;
  FiFoFull      : Boolean = False;
  FileError     : Boolean = False;
  ChanIdx       : LongInt;
  Buffer        : array[0..TTREADMAX - 1] of LongWord;
  OutTextFile   : TextFile;
  i,j           : Integer;
  TimeOut       : Boolean = False;
  OflCorrection : Int64 = 0;
  Histogram     : array [0..MAXINPCHAN - 1, 0..T3HISTBINS - 1] of LongWord;

// GotPhotonT3 procedure
// NSync: Overflow-corrected arrival time in units of the sync period
// DTime: Arrival time of photon after last Sync event in units of the chosen resolution (set by binning)
// Channel: 1..N where N is the numer of channels the device has
procedure GotPhotonT3(NSync: Int64; Channel: Integer; DTime: Integer);
begin
  Inc(Histogram[Channel - 1, DTime]); // Histogramming
end;

// GotMarkerT3 rocedure
//  TimeTag: Raw TimeTag from Record * Global resolution = Real Time arrival of Marker
//  Markers: Bitfield of arrived Markers, different markers can arrive at same time (same record)
procedure GotMarkerT3(TimeTag: Int64; Markers: Integer);
begin
  // Could switch to a new Histogram here, e.g. for a FLIM application
end;

// ProcessT3 procedure
procedure ProcessT3(TTTR_RawData: Cardinal);
const
  T3WRAPAROUND = 1024;
type
  TT3DataRecords = record
    Special: Boolean;
    Channel: Byte;
    DTime: Word;
    NSync: Word;
  end;
var
  TTTR_Data: TT3DataRecords;
  TrueNSync: Integer;
begin
  TTTR_Data.NSync := Word(TTTR_RawData and $000003FF); // 10 bit of 32 bit for NSync
  TTTR_Data.DTime := Word((TTTR_RawData shr 10) and $00007FFF); // 15 bit of 32 bit for DTime
  TTTR_Data.Channel := Byte((TTTR_RawData shr 25) and $0000003F); // 6 bit of 32 bit for Channel
  TTTR_Data.Special := Boolean((TTTR_RawData shr 31) and $00000001); // 1 bit of 32 bit for Special
  if TTTR_Data.Special then // This means we have a Special record
    case TTTR_Data.Channel of
      $3F: // Overflow
        begin
          // Number of overflows is stored in timetag
          // If it is zero, it is an old style single overflow {should never happen with new Firmware}
          if TTTR_Data.NSync = 0 then
            OflCorrection := OflCorrection + T3WRAPAROUND
          else
            OflCorrection := OflCorrection + T3WRAPAROUND * TTTR_Data.NSync;
        end;
      1..15: // Markers
        begin
          TrueNSync := OflCorrection + TTTR_Data.NSync; //the time unit depends on sync period
          // Note that actual marker tagging accuracy is only some ns.
          GotMarkerT3(TrueNSync, TTTR_Data.Channel);
        end;
    end
  else // It is a regular photon record
  begin
    TrueNSync := OflCorrection + TTTR_Data.NSync;
    GotPhotonT3(TrueNSync,
    TTTR_Data.Channel + 1, // We encode the input channels as 1..N
    TTTR_Data.DTime // The dtime unit depends on the chosen resolution (binning)
    ); // TruenSync indicates the number of the sync period this event was in
  end;
end;

procedure Ex(RetCode: Integer);
begin
  if RetCode <> PH330_ERROR_NONE then
  begin
    PH330_GetErrorString(Errorstring, RetCode);
    Writeln('Error ', RetCode:3, ' = "', Trim(string(Errorstring)), '"');
  end;
  Writeln;
  {$I-}
    CloseFile(OutTextFile);
    IOResult();
  {$I+}
  Writeln('Press RETURN to exit');
  Readln;
  Halt(RetCode);
end;

procedure StopTTTR;
begin
  RetCode := PH330_StopMeas(DevIdx[0]);
  if RetCode <> PH330_ERROR_NONE then
  begin
    Writeln('PH330_StopMeas error ', RetCode:3, '. Aborted.');
    Ex(RetCode);
  end;
end;

// Main procedure
begin
  try
    Writeln;
    Writeln('PicoHarp 330 PHLib330 Demo Application              PicoQuant GmbH, 2024');
    Writeln('~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~');

    if Mode = MODE_T2 then
    begin
      Writeln('This demo is not for use with T2 mode!');
      Ex(PH330_ERROR_NONE);
    end;

    RetCode := PH330_GetLibraryVersion(strLibVers);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetLibraryVersion error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('PH330LIB version is ' + strLibVers);
    if Trim(AnsiString(strLibVers)) <> Trim(AnsiString (LIB_VERSION)) then
      Writeln('Warning: The application was built for version ' + LIB_VERSION);

    AssignFile(OutTextFile, 't3histout.txt');
    {$I-}
      Rewrite(OutTextFile);
    {$I+}
    if IOResult <> 0 then
    begin
      Writeln('Cannot open output file');
      Ex(PH330_ERROR_NONE);
    end;

    Writeln;
    Writeln('Searching for PicoHarp 330 devices...');
    Writeln('DevIdx       Serial       Status');

    for Dev := 0 to MAXDEVNUM - 1 do
    begin
      RetCode := PH330_OpenDevice(Dev, HW_Serial);
      //
      case RetCode of
        PH330_ERROR_NONE:
          begin
            // Grab any device we can open
            DevIdx[Found] := Dev; // Keep index to devices we want to use
            Inc(Found);
            Writeln(Dev, '            ', HW_Serial,'      open ok');
          end;
        PH330_ERROR_DEVICE_OPEN_FAIL:
          Writeln(Dev, '                         no device')
        else
          begin
            PH330_GetErrorString(Errorstring, RetCode);
            Writeln(Dev, '       ', HW_Serial, '    ', Trim(string(ErrorString)));
          end;
      end;
    end;

    // In this demo we will use the first device we find, i.e. dev[0].
    // You can also use multiple devices in parallel.
    // You can also check for specific serial numbers, so that you always know
    // which physical device you are talking to.

    if Found < 1 then
    begin
      Writeln('No device available.');
      Ex(PH330_ERROR_NONE);
    end;

    Writeln('Using device ', DevIdx[0]);
    Writeln('Initializing the device...');

    RetCode := PH330_Initialize (DevIdx[0], Mode, 0); // with internal clock
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_Initialize error ', RetCode:3, '. Aborted.');
      PH330_GetDebugInfo(DevIdx[0], DebugInfoBuffer);
      Writeln('Debug info: ', DebugInfoBuffer);
      Ex(RetCode);
    end;

    RetCode := PH330_GetHardwareInfo (DevIdx[0], HW_Model, HW_PartNo, HW_Version);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetHardwareInfo error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end
    else
      Writeln('Found Model ', HW_Model,'  Part no ', HW_PartNo,'  Version ', HW_Version);

    RetCode := PH330_GetNumOfInputChannels (DevIdx[0], NumChannels);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetNumOfInputChannels error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end
    else
      Writeln('Device has ', NumChannels, ' input channels.');

    Writeln;

    RetCode := PH330_SetSyncDiv (DevIdx[0], SyncDivider);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_SetSyncDiv error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    RetCode := PH330_SetSyncTrgMode(DevIdx[0], SyncTrgMode);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_SetSyncDiv error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    if SyncTrgMode = TRGMODE_ETR then
    begin
      RetCode := PH330_SetSyncEdgeTrg(DevIdx[0], SyncTrgLevel, SyncTrgEdge);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetSyncEdgeTrg error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
    end;
    if SyncTrgMode = TRGMODE_CFD then
    begin
      RetCode := PH330_SetSyncCFD(DevIdx[0], SyncCFDLevel, SyncCFDZeroCross);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetSyncCFD error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
    end;

    RetCode := PH330_SetSyncChannelOffset (DevIdx[0], SyncChannelOffset); // in ps, emulate a cable delay
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_SetSyncChannelOffset error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    for ChanIdx := 0 to NumChannels - 1 do // for simplicity we use the same settings for all channels
    begin
      RetCode := PH330_SetInputTrgMode(DevIdx[0], ChanIdx, InputTrgMode);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetInputTrgMode ', ChanIdx:2, ' error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      if InputTrgMode = TRGMODE_ETR then
      begin
        RetCode := PH330_SetInputEdgeTrg(DevIdx[0], ChanIdx, InputTrgLevel, InputTrgEdge);
        if RetCode <> PH330_ERROR_NONE then
        begin
          Writeln('PH330_SetInputEdgeTrg ', ChanIdx:2, ' error ', RetCode:3, '. Aborted.');
          Ex(RetCode);
        end;
      end;
      if InputTrgMode = TRGMODE_CFD then
      begin
        RetCode := PH330_SetInputCFD(DevIdx[0], ChanIdx, InputCFDLevel, InputCFDZeroCross);
        if RetCode <> PH330_ERROR_NONE then
        begin
          Writeln('PH330_SetInputCFD ', ChanIdx:2, ' error ', RetCode:3, '. Aborted.');
          Ex(RetCode);
        end;
      end;
      RetCode := PH330_SetInputChannelOffset(DevIdx[0], ChanIdx, InputChannelOffset); // in ps, emulate a cable delay
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetInputChannelOffset channel ', ChanIdx:2, ' error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;

      RetCode := PH330_SetInputChannelEnable(DevIdx[0], ChanIdx, 1); // We enable all channels
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetInputChannelEnable channel ', ChanIdx:2, ' error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
    end;

    if (Mode = MODE_T3) then
    begin
      RetCode := PH330_SetBinning (DevIdx[0], Binning);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetBinning error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;

      RetCode := PH330_SetOffset(DevIdx[0], Offset);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetOffset error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
    end;

    RetCode := PH330_GetResolution (DevIdx[0], Resolution);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetResolution error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('Resolution is ', Resolution:7:3, 'ps');
    Writeln('Measuring input rates...');

    // After Init allow 150 ms for valid  count rate readings
    // Subsequently you get new values after every 100ms
    Sleep (150);

    Writeln;

    RetCode := PH330_GetSyncRate(DevIdx[0], SyncRate);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetSyncRate error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('SyncRate = ', SyncRate, '/s');

    Writeln;

    for ChanIdx := 0 to NumChannels - 1 do // For all channels
    begin
      RetCode := PH330_GetCountRate(DevIdx[0], ChanIdx, CountRate);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_GetCountRate error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      Writeln('Countrate [', ChanIdx:2, '] = ', CountRate:8, '/s');
    end;

    Writeln;

    RetCode := PH330_GetWarnings(DevIdx[0], Warnings);   // After getting the count rates you can check for warnings
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetWarnings error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    if Warnings <> 0 then
    begin
      PH330_GetWarningsText(DevIdx[0], WarningsText, Warnings);
      Writeln(WarningsText);
    end;

    if Mode = Mode_T2 then
    begin
      Writeln('This demo is only for T3 mode!');
      Ex(PH330_ERROR_NONE);
    end
    else
    begin
      for ChanIdx := 0 to NumChannels - 1 do
        Write(OutTextFile, '  ch', (ChanIdx + 1):2, ' ');
    end;
    Write(OutTextFile, chr(10));

    Writeln('Press RETURN to start measurement');
    Readln;

    RetCode := PH330_StartMeas(DevIdx[0], Tacq);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_StartMeas error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('Starting data collection...');

    Progress := 0;
    OflCorrection := 0;

    for i := 0 to T3HISTBINS - 1 do
        for j := 0 to MAXINPCHAN - 1 do
            Histogram[j, i] := 0;

    repeat
      RetCode := PH330_GetFlags(DevIdx[0], Flags);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_GetFlags error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      FiFoFull := (Flags and FLAG_FIFOFULL) > 0;

      if FiFoFull then
        Writeln('  FiFo Overrun!')
      else
      begin
        RetCode := PH330_ReadFiFo(DevIdx[0], Buffer[0], Records); // may return less!
        if RetCode <> PH330_ERROR_NONE then
        begin
          Writeln('PH330_TTReadData error ', RetCode:3, '. Aborted.');
          Ex(RetCode);
        end;

        // Here we process the data. Note that the time this consumes prevents us
        // from getting around the loop quickly for the next Fifo read.
        // In a serious performance critical scenario you would write the data to
        // a software queue and do the processing in another thread reading from
        // that queue.
        if Records > 0 then
        begin
          for i := 0 to Records - 1 do
              ProcessT3(Buffer[i]);
          Progress := Progress + Records;
          Write(#8#8#8#8#8#8#8#8#8, Progress:9);
        end
        else // Do the following only when there was no data
        begin
          RetCode := PH330_CTCStatus(DevIdx[0], CTCStatus);
          if RetCode <> PH330_ERROR_NONE then
          begin
            Writeln;
            Writeln('PH330_CTCStatus error ', RetCode:3, '. Aborted.');
            Ex(RetCode);
          end;
          TimeOut := (CTCStatus <> 0);
          if TimeOut then
          begin
            Inc(StopRetry); // Do a few more rounds as there might be some more in the FiFo
            if StopRetry > 5 then
            begin
              StopTTTR;
              Writeln;
              Writeln('Done');
            end;
          end;
        end;
      end;
      // Within this loop you can also read the count rates if needed.
      // Do it sparingly and use PH330_GetAllCountRates for speed.
    until FiFoFull or TimeOut or FileError;

    Writeln;

    RetCode := PH330_StopMeas(DevIdx[0]);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_StopMeas error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    for i := 0 to T3HISTBINS - 1 do
    begin
      for ChanIdx := 0 to NumChannels - 1 do
        Write(OutTextFile, Histogram[ChanIdx, i]:7, ' ');
      Write(OutTextFile, chr(10));
    end;

    Writeln;

    PH330_CloseAllDevices;
    Ex(PH330_ERROR_NONE);

  except
  on E: Exception do
    Writeln(E.ClassName, ': ', E.Message);
  end;
end.

