{
  Demo access to PicoHarp 330 via PH330Lib.dll or libph330.so v.2.0

  THIS IS AN ADVANCED DEMO. DO NOT USE FOR YOUR FIRST EXPERIMENTS.
  Look at the variable meascontrol down below to see what it does.

  The program performs a measurement based on hardcoded settings.
  The resulting histogram is stored in an ASCII output file.

  Stefan Eilers, Michael Wahl, PicoQuant, July 2024

  Note: This is a console application

  Note: At the API level channel numbers are indexed 0..N-1
    where N is the number of channels the device has.

  Tested with the following compilers:

  - Delphi 11.1 (Windows)
  - Lazarus 3.0 RC1 (Windows)
  - Lazarus 2.2.0 / FPC 3.2.2 (Linux)
}

program histomode;

{$IF defined(MSWINDOWS)}
  {$APPTYPE CONSOLE}  //windows needs this, Linux does not want it
{$ENDIF}

uses
  {$ifdef fpc}
  SysUtils,
  {$else}
  System.SysUtils,
  System.Ansistrings,
  {$ENDIF}
  PH330Lib in 'PH330Lib.pas';

type
  THistogramCounts   = array [0..MAXHISTLEN - 1] of Cardinal;

var
  DevIdx          : array [0..MAXDEVNUM - 1] of LongInt;
  Dev             : LongInt;
  Found           : Integer = 0;
  OutTextFile     : text;
  RetCode         : LongInt;
  CTCStatus       : LongInt;
  strLibVers      : array[0..7]  of AnsiChar;
  HW_Model        : array[0..31] of AnsiChar;
  HW_Partno       : array[0..7]  of AnsiChar;
  HW_Serial       : array[0..31] of AnsiChar;
  HW_Version      : array[0..15] of AnsiChar;
  Errorstring     : array[0..39] of AnsiChar;
  DebugInfoBuffer : array[0..16383] of AnsiChar;
  NumChannels     : LongInt;
  Mode            : LongInt = MODE_HIST;
  Binning         : LongInt = 0; // you can change this
  Offset          : LongInt = 0; // normally no need to change this
  Tacq            : LongInt = 1000; // you can change this, unit is millisec

  // Sync Settings
  SyncDivider     : LongInt = 1; // you can change this
  SyncChannelOffset : LongInt =           0; // in ps, you can change this (like a cable delay)
  SyncTrgMode       : LongInt = TRGMODE_ETR; // you can change this to TRGMODE_CFD

  // in case of SyncTrgMode == TRGMODE_ETR this will apply:
  SyncTrgEdge  : LongInt = EDGE_FALLING; // you can change this to EDGE_RISING
  SyncTrgLevel : LongInt = -50; // in mV, you can change this

  // in case of SyncTrgMode == TRGMODE_CFD this will apply:
  SyncCFDZeroCross : LongInt = -50; // in mV, you can change this
  SyncCFDLevel     : LongInt = -50; // in mV, you can change this

  // Input Settings
  InputChannelOffset : LongInt =           0; // in ps, you can change this (like a cable delay)
  InputTrgMode       : LongInt = TRGMODE_ETR; // you can change this to TRGMODE_CFD

  // in case of InputTrgMode == TRGMODE_ETR this will apply:
  InputTrgEdge  : LongInt = EDGE_FALLING; // you can change this to EDGE_RISING
  InputTrgLevel : LongInt = -50;          // in mV, you can change this

  // in case of InputTrgMode == TRGMODE_CFD this will apply:
  InputCFDZeroCross : LongInt = -20;      // in mV, you can change this
  InputCFDLevel     : LongInt = -50;      // in mV, you can change this

  // this determines how meaurements are started/stopped by hardware/software
  MeasControl : LongInt
    = MEASCTRL_SINGLESHOT_CTC; // Start by software and stop when CTC expires (default)
    //= MEASCTRL_C1_GATED; // Measure while C1 is active
    //= MEASCTRL_C1_START_CTC_STOP; // Start with C1 and stop when CTC expires
    //= MEASCTRL_C1_START_C2_STOP;  // Start with C1 and stop with C2
  edge1 : LongInt = EDGE_RISING;  //Edge of C1 to start (if applicable in chosen mode)
  edge2 : LongInt = EDGE_FALLING; //Edge of C2 to stop (if applicable in chosen mode)

  // other variables
  Resolution  : Double;
  SyncRate    : LongInt;
  CountRate   : LongInt;
  IntegralCount : Double;
  Elapsed     : Double;
  Flags       : LongInt;
  Warnings    : LongInt;
  WarningsText : array[0..16343] of AnsiChar; // must have 16384 bytes of text buffer
  Cmd         : Char = #0;
  ChanIdx : LongInt;
  HistoBin : LongInt;
  HistLen  : LongInt;
  Counts : array[0..MAXINPCHAN - 1] of THistogramCounts;

procedure Ex(RetCode: Integer);
begin
  if RetCode <> PH330_ERROR_NONE then
  begin
    PH330_GetErrorString(Errorstring, RetCode);
    Writeln('Error ', RetCode:3, ' = "', Trim (string(Errorstring)), '"');
  end;
  Writeln;
  {$I-}
    CloseFile(OutTextFile);
    IOResult();
  {$I+}
  Writeln('Press RETURN to exit');
  Readln;
  Halt(RetCode);
end;

begin
  try
    Writeln;
    Writeln('PicoHarp 330 PHLib330 Demo Application              PicoQuant GmbH, 2024');
    Writeln('~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~');
    RetCode := PH330_GetLibraryVersion(strLibVers);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetLibraryVersion error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('PH330LIB version is ' + strLibVers);
    if Trim(AnsiString(strLibVers)) <> Trim(AnsiString(LIB_VERSION)) then
      Writeln('Warning: The application was built for version ' + LIB_VERSION);

    AssignFile(OutTextFile, 'histomodeout.txt');
    {$I-}
      Rewrite(OutTextFile);
    {$I+}
    if IOResult <> 0 then
    begin
      Writeln('Cannot open output file');
      Ex(PH330_ERROR_NONE);
    end;

    Writeln;
    Writeln('Searching for devices...');
    Writeln('DevIdx     Serial     Status');

    for Dev := 0 to MAXDEVNUM - 1 do
    begin
      RetCode := PH330_OpenDevice(Dev, HW_Serial);
      //
      if RetCode = PH330_ERROR_NONE then
      begin
        // Grab any device we can open
        Writeln(Dev, '          ', HW_Serial, '    open ok');
        DevIdx[Found] := Dev; // Keep index to devices we want to use
        Inc(Found);
      end
      else
      begin
        if RetCode = PH330_ERROR_DEVICE_OPEN_FAIL then
          Writeln(Dev, '                     no device')
        else
        begin
          PH330_GetErrorString(Errorstring, RetCode);
          Writeln(Dev, '       ', Trim(string(Errorstring)));
        end;
      end;
    end;

    // In this demo we will use the first device we found,
    // i.e. DevIdx[0].  You can also use multiple devices in parallel.
    // you could also check for a specific serial number, so that you
    // always know which physical device you are talking to.

    if Found < 1 then
    begin
      Writeln('No device available.');
      Ex(PH330_ERROR_NONE);
    end;

    Writeln('Using device ', DevIdx[0]);
    Writeln('Initializing the device...');

    RetCode := PH330_Initialize(DevIdx[0], Mode, 0); //Histo mode with internal clock
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_Initialize error ', RetCode:3, '. Aborted.');
      PH330_GetDebugInfo(DevIdx[0], DebugInfoBuffer);
      Writeln('Debug info: ', DebugInfoBuffer);
      Ex(RetCode);
    end;

    RetCode := PH330_GetHardwareInfo(DevIdx[0], HW_Model, HW_Partno, HW_Version); // this is only for information
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetHardwareInfo error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end
    else
      Writeln('Found Model ', HW_Model,'  Part no ', HW_Partno,'  Version ', HW_Version);

    RetCode := PH330_GetNumOfInputChannels(DevIdx[0], NumChannels);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetNumOfInputChannels error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end
    else
      Writeln('Device has ', NumChannels, ' input channels.');

    Writeln;

    RetCode := PH330_SetSyncDiv(DevIdx[0], SyncDivider);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_SetSyncDiv error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    RetCode := PH330_SetSyncTrgMode(DevIdx[0], SyncTrgMode);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_SetSyncTrgMode error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    if SyncTrgMode = TRGMODE_ETR then
    begin
      RetCode := PH330_SetSyncEdgeTrg(DevIdx[0], SyncTrgLevel, SyncTrgEdge);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetSyncEdgeTrg error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
    end;

    if SyncTrgMode = TRGMODE_CFD then
    begin
      RetCode := PH330_SetSyncCFD(DevIdx[0], SyncCFDLevel, SyncCFDZeroCross);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetSyncCFD error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
    end;

    RetCode := PH330_SetSyncChannelOffset(DevIdx[0], SyncChannelOffset); // Emulate a cable delay
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_SetSyncChannelOffset error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;


    for ChanIdx := 0 to NumChannels - 1 do // for simplicity we use the same settings for all channels
    begin
      RetCode := PH330_SetInputTrgMode(DevIdx[0], ChanIdx, InputTrgMode);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetInputTrgMode ', ChanIdx:2, ' error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      if InputTrgMode = TRGMODE_ETR then
      begin
        RetCode := PH330_SetInputEdgeTrg(DevIdx[0], ChanIdx, InputTrgLevel, InputTrgEdge);
        if RetCode <> PH330_ERROR_NONE then
        begin
          Writeln('PH330_SetInputEdgeTrg ', ChanIdx:2, ' error ', RetCode:3, '. Aborted.');
          Ex(RetCode);
        end;
      end;
      if InputTrgMode = TRGMODE_CFD then
      begin
        RetCode := PH330_SetInputCFD(DevIdx[0], ChanIdx, InputCFDLevel, InputCFDZeroCross);
        if RetCode <> PH330_ERROR_NONE then
        begin
          Writeln('PH330_SetInputCFD ', ChanIdx:2, ' error ', RetCode:3, '. Aborted.');
          Ex(RetCode);
        end;
      end;
      RetCode := PH330_SetInputChannelOffset(DevIdx[0], ChanIdx, InputChannelOffset);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetInputChannelOffset channel ', ChanIdx:2, ' error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;

      RetCode := PH330_SetInputChannelEnable(DevIdx[0], ChanIdx, 1);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetInputChannelEnable channel ', ChanIdx:2, ' error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
    end;

    RetCode := PH330_SetHistoLen(DevIdx[0], DFLTLENCODE, HistLen);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_SetHistoLen error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('Histogram length is ', HistLen);

    RetCode := PH330_SetBinning(DevIdx[0], Binning);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_SetBinning error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    RetCode := PH330_SetOffset(DevIdx[0], Offset);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_SetOffset error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    RetCode := PH330_GetResolution(DevIdx[0], Resolution);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetResolution error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('Resolution is ', Resolution:7:3, 'ps');

    // After Init allow 150 ms for valid  count rate readings
    // Subsequently you get new values after every 100ms
    Sleep(150);

    Writeln;

    RetCode := PH330_GetSyncRate(DevIdx[0], SyncRate);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetSyncRate error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('SyncRate = ', SyncRate, '/s');

    Writeln;

    for ChanIdx := 0 to NumChannels - 1 do // for all channels
    begin
      RetCode := PH330_GetCountRate(DevIdx[0], ChanIdx, CountRate);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_GetCountRate error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      Writeln('Countrate [', ChanIdx:2, '] = ', CountRate:8, '/s');
    end;

    Writeln;

    // After getting the count rates you can check for warnings
    RetCode := PH330_GetWarnings(DevIdx[0], Warnings);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetWarnings error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    if Warnings <> 0 then
    begin
      PH330_GetWarningsText(DevIdx[0], WarningsText, Warnings);
      Writeln(WarningsText);
    end;

    RetCode := PH330_SetStopOverflow (DevIdx[0], 0, 10000); // for example only
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_SetStopOverflow error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    RetCode := PH330_SetMeasControl(DevIdx[0], MeasControl, edge1, edge2);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_SetMeasControl error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    repeat
      PH330_ClearHistMem(DevIdx[0]);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_ClearHistMem error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;

      Writeln('Press RETURN to start measurement');
      Readln;

      Writeln;

      RetCode := PH330_GetSyncRate(DevIdx[0], SyncRate);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_GetSyncRate error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      Writeln('SyncRate = ', SyncRate, '/s');

      Writeln;

      for ChanIdx := 0 to NumChannels - 1 do // for all channels
      begin
        RetCode := PH330_GetCountRate(DevIdx[0], ChanIdx, CountRate);
        if RetCode <> PH330_ERROR_NONE then
        begin
          Writeln('PH330_GetCountRate error ', RetCode:3, '. Aborted.');
          Ex(RetCode);
        end;
        Writeln('Countrate [', ChanIdx:2, '] = ', CountRate:8, '/s');
      end;

      // Here you could check for warnings again

      RetCode := PH330_StartMeas(DevIdx[0], Tacq);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_StartMeas error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      if MeasControl <> MEASCTRL_SINGLESHOT_CTC then
      begin
        Writeln('Waiting for hardware start on C1...');
        CTCStatus := 1;
        repeat
          RetCode := PH330_CTCStatus(DevIdx[0], CTCStatus);
          if RetCode <> PH330_ERROR_NONE then
          begin
            Writeln('PH330_CTCStatus error ', RetCode:3, '. Aborted.');
            Ex(RetCode);
          end;
        until CTCStatus <> 1;
      end;

      if (MeasControl = MEASCTRL_SINGLESHOT_CTC) or (MeasControl = MEASCTRL_C1_START_CTC_STOP) then
        Writeln('Measuring for ', Tacq, ' milliseconds...');

      if MeasControl = MEASCTRL_C1_GATED then
        Writeln('Measuring, waiting for other C1 edge to stop...');

      if MeasControl = MEASCTRL_C1_START_C2_STOP then
        Writeln('Measuring, waiting for C2 to stop...');

      CTCStatus := 0;
      repeat
        RetCode := PH330_CTCStatus(DevIdx[0], CTCStatus);
        if RetCode <> PH330_ERROR_NONE then
        begin
          Writeln('PH330_CTCStatus error ', RetCode:3, '. Aborted.');
          Ex(RetCode);
        end;
      until CTCStatus <> 0;

      RetCode := PH330_StopMeas(DevIdx[0]);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_StopMeas error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;

      Writeln;

      for ChanIdx := 0 to NumChannels - 1 do // for all channels
      begin
        RetCode := PH330_GetHistogram(DevIdx[0], Counts[ChanIdx, 0], ChanIdx);
        if RetCode <> PH330_ERROR_NONE then
        begin
          Writeln('PH330_GetHistogram error ', RetCode:3, '. Aborted.');
          Ex(RetCode);
        end;
        IntegralCount := 0;
        for HistoBin := 0 to HistLen - 1 do
          IntegralCount := IntegralCount + Counts[ChanIdx, HistoBin];
        Writeln('Integralcount [', ChanIdx:2, '] = ', IntegralCount:9:0);
      end;

      Writeln;

      RetCode := PH330_GetElapsedMeasTime(DevIdx[0], Elapsed);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_GetElapsedMeasTime error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      Writeln('Elapsed measure time = ', Elapsed:5:0, ' ms');

      RetCode := PH330_GetFlags(DevIdx[0], Flags);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_GetFlags error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;

      if (Flags and FLAG_OVERFLOW) > 0 then
        Writeln('  Overflow.');

      Writeln('Enter c to continue or q to quit and save the count data.');
      Readln(Cmd);
    until Cmd = 'q';

    Writeln;
    Writeln('Saving data to file...');

    for HistoBin := 0 to HistLen - 1 do
    begin
      for ChanIdx := 0 to NumChannels - 1 do
        Write(OutTextFile, Counts[ChanIdx, HistoBin]:7, ' ');
      Write(OutTextFile, chr(10));
    end;

    PH330_CloseAllDevices;
    Ex(PH330_ERROR_NONE);

  except
    on E: Exception do
      Writeln(E.ClassName, ': ', E.Message);
  end;
end.

