{
  Demo access to PicoHarp 330 via PH330Lib.dll or libph330.so v.2.0
  The program performs a TTTR measurement based on hard-coded settings.
  Event data is filtered by the device's event filter in the FPGA.
  The demo shows the software calls for setting the filter parameters
  and activating the filter. Resulting event data is instantly processed.
  Processing consists here of simple histogramming. This allows replicating
  and verifying the filter demonstration as shown in the section on event
  filtering in the PicoHarp 330 manual. If you use identical or at least
  similar input signals and setings you should obtain similar histograms
  as shown in the manual. Observe the offset of -10000ps in the call of
  PH330_SetSyncChannelOffset. Under the assumption that the sync channel
  and the first input channel are receiving the same signal through identical
  cable lengths this is necessary in order to shift the region of interest
  where coincidences occur into the usable histogram range.
  Note also that in order to implement the easiest way of histogramming,
  this demo is implemented for T3 mode only, although the filtering as such
  works in T2 mode as well.

  Stefan Eilers, Michael Wahl, PicoQuant, July 2024

  Note: This is a console application.

  Note: At the API level the input channel numbers are indexed 0..N-1
  where N is the number of input channels the device has.
  Upon processing we map this back to 1..N corresponding to the front
  panel labelling.

  Tested with the following compilers:

  - Delphi 11.1 (Windows)
  - Lazarus 3.0 RC1 (Windows)
  - Lazarus 2.2.0 / FPC 3.2.2 (Linux)
}

program tttrmode;

{$IF defined(MSWINDOWS)}
  {$APPTYPE CONSOLE}  // Windows needs this, Linux does not want it
{$ENDIF}

uses
  {$ifdef fpc}
  SysUtils,
  {$else}
  System.SysUtils,
  System.Ansistrings,
  {$endif}
  PH330Lib in 'PH330Lib.pas';

const
  T3HISTBINS = 32768; //=2^15, dtime in T3 mode has 15 bits


type
  THistogramCounts   = array [0..MAXHISTLEN - 1] of Cardinal;

var
  DevIdx          : array [0..MAXDEVNUM - 1] of LongInt;
  Dev             : LongInt;
  Found           : Integer = 0;
  RetCode         : LongInt;
  CTCStatus       : LongInt;
  strLibVers      : array[0..7]  of AnsiChar;
  HW_Model        : array[0..31] of AnsiChar;
  HW_Partno       : array[0..8] of AnsiChar;
  HW_Serial       : array[0..9] of AnsiChar;
  HW_Version      : array[0..15] of AnsiChar;
  Errorstring     : array[0..39] of AnsiChar;
  DebugInfoBuffer : array[0..16383] of AnsiChar;
  NumChannels     : LongInt;
  Mode            : LongInt = MODE_T3;
  Binning         : LongInt = 0; // you can change this (meaningless in T2 mode)
  Offset          : LongInt = 0; // normally no need to change this
  Tacq            : LongInt = 1000; // you can change this, unit is millisec

  // Sync Settings
  SyncDivider     : LongInt = 1; // you can change this
  SyncChannelOffset : LongInt =           0; // in ps, you can change this (like a cable delay)
  SyncTrgMode       : LongInt = TRGMODE_ETR; // you can change this to TRGMODE_CFD

  // in case of SyncTrgMode == TRGMODE_ETR this will apply:
  SyncTrgEdge  : LongInt = EDGE_FALLING; // you can change this to EDGE_RISING
  SyncTrgLevel : LongInt = -50; // in mV, you can change this

  // in case of SyncTrgMode == TRGMODE_CFD this will apply:
  SyncCFDZeroCross : LongInt = -50; // in mV, you can change this
  SyncCFDLevel     : LongInt = -50; // in mV, you can change this

  // Input Settings
  InputChannelOffset : LongInt = 0; // in ps, you can change this (like a cable delay)
  InputTrgMode       : LongInt = TRGMODE_ETR; // you can change this to TRGMODE_CFD

  // in case of InputTrgMode == TRGMODE_ETR this will apply:
  InputTrgEdge  : LongInt = EDGE_FALLING; // you can change this to EDGE_RISING
  InputTrgLevel : LongInt = -50;          // in mV, you can change this

  // in case of InputTrgMode == TRGMODE_CFD this will apply:
  InputCFDZeroCross : LongInt = -20;      // in mV, you can change this
  InputCFDLevel     : LongInt = -50;      // in mV, you can change this

  // other variables
  Resolution   : Double;
  SyncRate     : LongInt;
  CountRate    : LongInt;
  Flags        : LongInt;
  Warnings     : LongInt;
  WarningsText : array[0..16343] of AnsiChar; // must have 16384 bytes of text buffer
  Records      : LongInt;
  Progress     : LongInt = 0;
  FiFoFull           : Boolean =   False;
  FileError          : Boolean =   False;
  ChanIdx            : LongInt;
  Buffer             : array[0..TTREADMAX - 1] of LongWord;
  OutTextFile        : Text;
  i                  : Integer;
  TimeOut            : Boolean =   False;
  OflCorrection      : Int64 = 0;
  SyncPeriod         : Double = 0;
  Histogram : array [0..MAXINPCHAN - 1, 0..T3HISTBINS - 1] of LongWord;

  {
  The following variables are for programming the event filter.
  Please read the manual on what the filter parameters are about.
  Here we implement a simple "Singles Filter" which is the most obvious
  and most useful type of event filter in typical quantum optics applications.
  The idea is that photon events that are "single" in the sense that there is no
  other event in temporal proximity (within timerange) will be filtered out.
  This reduces the bus load and the file size if you write the records to a file.
  Any further (application specific) coincidence processing can then be
  done in software, either on the fly or subsequently on the stored data.
  }

  Filter_TimeRange : LongInt = 100; // in picoseconds
  Filter_MatChCNT  : LongInt = 1; // if 1: there must be 2 or more events to pass
  Filter_Inverse   : LongInt = 0; // normal filtering mode, see manual
  Filter_Enable    : LongInt = 1; // activate the filter with 1
  Filter_UseChannels  : LongInt = $3; // bitmasks for which channels are to be used
  Filter_PassChannels : LongInt = 0; // bitmasks for which channels to pass unfiltered

  // The following are count rate buffers for the filter test further down
  FTestSyncRate  : LongInt;
  FTestChanRates : array[0..MAXINPCHAN - 1] of LongInt;
  FTestSumRate   : LongInt;

// GotPhotonT3 procedure
// NSync: Overflow-corrected arrival time in units of the sync period
// DTime: Arrival time of photon after last Sync event in units of the chosen resolution (set by binning)
// Channel: 1..N where N is the numer of channels the device has
procedure GotPhotonT3(NSync: Int64; Channel: Integer; DTime: Integer);
begin
  Inc(Histogram[Channel - 1, DTime]); // Histogramming
end;

// GotMarkerT3 procedure
// NSync: Overflow-corrected arrival time in units of the sync period
// Markers: Bitfield of arrived Markers, different markers can arrive at same time (same record)
procedure GotMarkerT3(NSync: Int64; Markers: Integer);
begin
  // Could switch to a new Histogram here, e.g. for a FLIM application
end;

// ProcessT3 procedure
procedure ProcessT3(TTTR_RawData: Cardinal);
const
  T3WRAPAROUND = 1024;
type
  TT3DataRecords = record
    Special: Boolean;
    Channel: Byte;
    DTime: Word;
    NSync: Word;
  end;
var
  TTTR_Data: TT3DataRecords;
  TrueSync: Integer;
begin
  TTTR_Data.NSync := Word(TTTR_RawData and $000003FF); // 10 bit of 32 bit for NSync (number of sync period)
  TTTR_Data.DTime := Word((TTTR_RawData shr 10) and $00007FFF); // 15 bit of 32 bit for DTime (DTime=delay from last sync in units of chosen resolution)
  TTTR_Data.Channel := Byte((TTTR_RawData shr 25) and $0000003F); // 6 bit of 32 bit for Channel
  TTTR_Data.Special := Boolean((TTTR_RawData shr 31) and $00000001); // 1 bit of 32 bit for Special
  if TTTR_Data.Special then // this means we have a Special record
    case TTTR_Data.Channel of
      $3F: // Overflow
        begin
          // Number of overflows is stored in timetag
          // If it is zero, it is an old style single overflow {should never happen with new Firmware}
          if TTTR_Data.NSync = 0 then
            OflCorrection := OflCorrection + T3WRAPAROUND
          else
            OflCorrection := OflCorrection + T3WRAPAROUND * TTTR_Data.NSync;
        end;
      1..15: // Markers
        begin
          TrueSync := OflCorrection + TTTR_Data.NSync; //the time unit depends on sync period
          // Note that actual marker tagging accuracy is only some ns.
          GotMarkerT3(TrueSync, TTTR_Data.Channel);
        end;
    end
  else
  begin // It is a regular photon record
    TrueSync := OflCorrection + TTTR_Data.NSync;
    // Truensync indicates the number of the sync period this event was in
    // The DTime unit depends on the chosen resolution (binning)
    GotPhotonT3(TrueSync,
      TTTR_Data.Channel + 1, // We encode the regular channels as 1..N
      TTTR_Data.DTime
      );
  end;
end;

procedure Ex(RetCode: Integer);
begin
  if RetCode <> PH330_ERROR_NONE then
  begin
    PH330_GetErrorString(Errorstring, RetCode);
    Writeln('Error ', RetCode:3, ' = "', Trim(string(Errorstring)), '"');
  end;
  Writeln;
  {$I-}
    CloseFile(OutTextFile);
    IOResult();
  {$I+}
  Writeln('Press RETURN to exit');
  Readln;
  Halt(RetCode);
end;

procedure StopTTTR;
begin
  RetCode := PH330_StopMeas(DevIdx[0]);
  if RetCode <> PH330_ERROR_NONE then
  begin
    Writeln('PH330_StopMeas error ', RetCode:3, '. Aborted.');
    Ex(RetCode);
  end;
end;

//main procedure
begin
  try
    Writeln;
    Writeln('PicoHarp 330 PHLib330 Demo Application              PicoQuant GmbH, 2024');
    Writeln('~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~');

    if Mode = MODE_T2 then
    begin
      Writeln('This demo is not for use with T2 mode!');
      Writeln('Press RETURN to exit');
      Ex(PH330_ERROR_NONE);
    end;

    RetCode := PH330_GetLibraryVersion(strLibVers);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetLibraryVersion error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('PH330LIB version is ' + strLibVers);
    if Trim(AnsiString(strLibVers)) <> Trim(AnsiString (LIB_VERSION)) then
      Writeln('Warning: The application was built for version ' + LIB_VERSION);

    AssignFile(OutTextFile, 't3histout.txt');
    {$I-}
      Rewrite(OutTextFile);
    {$I+}
    if IOResult <> 0 then
    begin
      Writeln('Cannot open output file.');
      Ex(PH330_ERROR_NONE);
    end;

    Writeln;
    Writeln('Searching for PicoHarp 330 devices...');
    Writeln('DevIdx     Serial     Status');

    ErrorString := '';

    for Dev := 0 to MAXDEVNUM - 1 do
    begin
      RetCode := PH330_OpenDevice(Dev, HW_Serial);
      case RetCode of
        PH330_ERROR_NONE:
          begin
            // Grab any device we can open
            DevIdx[Found] := Dev; // Keep index to devices we want to use
            Inc(Found);
            Writeln(Dev, '          ', HW_Serial,'    open ok');
          end;
        PH330_ERROR_DEVICE_OPEN_FAIL:
          Writeln(Dev, '                     no device');
        else
          begin
            PH330_GetErrorString(ErrorString, RetCode);
            Writeln(Dev, '     ', HW_Serial,'     ', Trim(string(ErrorString)));
          end;
      end;
    end;

    // In this demo we will use the first device we find, i.e. dev[0].
    // You can also use multiple devices in parallel.
    // You can also check for specific serial numbers, so that you always know
    // which physical device you are talking to.

    if Found < 1 then
    begin
      Writeln('No device available.');
      Ex(PH330_ERROR_NONE);
    end;

    Writeln('Using device ', DevIdx[0]);
    Writeln('Initializing the device...');

    RetCode := PH330_Initialize(DevIdx[0], Mode, 0);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_Initialize error ', RetCode:3, '. Aborted.');
      // In case of an obscure error (a hardware error in particular).
      // It may be helpful to  obtain debug information like so:
      PH330_GetDebugInfo(DevIdx[0], DebugInfoBuffer);
      Writeln('DebugInfo :', DebugInfoBuffer);
      Ex(RetCode);
    end;


    RetCode := PH330_GetHardwareInfo(DevIdx[0], HW_Model, HW_PartNo, HW_Version);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetHardwareInfo error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end
    else
      Writeln('Found Model ', HW_Model,'  Part no ', HW_PartNo,'  Version ', HW_Version);

    RetCode := PH330_GetNumOfInputChannels(DevIdx[0], NumChannels);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetNumOfInputChannels error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end
    else
      Writeln('Device has ', NumChannels, ' input channels.');

    Writeln;

    RetCode := PH330_SetSyncDiv(DevIdx[0], SyncDivider);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_SetSyncDiv error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    RetCode := PH330_SetSyncTrgMode(DevIdx[0], SyncTrgMode);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_SetSyncDiv error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    if SyncTrgMode = TRGMODE_ETR then
    begin
      RetCode := PH330_SetSyncEdgeTrg(DevIdx[0], SyncTrgLevel, SyncTrgEdge);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetSyncEdgeTrg error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
    end;
    if SyncTrgMode = TRGMODE_CFD then
    begin
      RetCode := PH330_SetSyncCFD(DevIdx[0], SyncCFDLevel, SyncCFDZeroCross);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetSyncCFD error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
    end;


    RetCode := PH330_SetSyncChannelOffset(DevIdx[0], SyncChannelOffset);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_SetSyncChannelOffset error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    for ChanIdx := 0 to NumChannels - 1 do // we use the same input settings for all channels
    begin
      RetCode := PH330_SetInputTrgMode(DevIdx[0], ChanIdx, InputTrgMode);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetInputTrgMode ', ChanIdx:2, ' error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      if InputTrgMode = TRGMODE_ETR then
      begin
        RetCode := PH330_SetInputEdgeTrg(DevIdx[0], ChanIdx, InputTrgLevel, InputTrgEdge);
        if RetCode <> PH330_ERROR_NONE then
        begin
          Writeln('PH330_SetInputEdgeTrg ', ChanIdx:2, ' error ', RetCode:3, '. Aborted.');
          Ex(RetCode);
        end;
      end;
      if InputTrgMode = TRGMODE_CFD then
      begin
        RetCode := PH330_SetInputCFD(DevIdx[0], ChanIdx, InputCFDLevel, InputCFDZeroCross);
        if RetCode <> PH330_ERROR_NONE then
        begin
          Writeln('PH330_SetInputCFD ', ChanIdx:2, ' error ', RetCode:3, '. Aborted.');
          Ex(RetCode);
        end;
      end;

      RetCode := PH330_SetInputChannelOffset(DevIdx[0], ChanIdx, InputChannelOffset);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetInputChannelOffset channel ', ChanIdx:2, ' error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;

      RetCode := PH330_SetInputChannelEnable(DevIdx[0], ChanIdx, 1);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetInputChannelEnable channel ', ChanIdx:2, ' error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
    end;

    if Mode = MODE_T3 then
    begin
      RetCode := PH330_SetBinning(DevIdx[0], Binning);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetBinning error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;

      RetCode := PH330_SetOffset(DevIdx[0], Offset);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetOffset error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
    end;

    RetCode := PH330_GetResolution (DevIdx[0], Resolution);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetResolution error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('Resolution is ', Resolution:7:3, 'ps');

  // Now we program the event filter

    RetCode := PH330_SetEventFilterChannels(DevIdx[0], Filter_UseChannels, Filter_PassChannels);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_SetEventFilterChannels error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    RetCode := PH330_SetEventFilterParams(DevIdx[0], Filter_TimeRange,
      Filter_MatChCNT, Filter_Inverse);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_SetEventFilterParams error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    RetCode := PH330_EnableEventFilter(DevIdx[0], Filter_Enable);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_EnableEventFilter error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    // Filter programming ends here

    Writeln('Measuring input rates...');
    Writeln;

    // After Init allow 150 ms for valid  count rate readings
    // Subsequently you get new values after every 100ms
    Sleep(150);

    RetCode := PH330_GetSyncRate(DevIdx[0], SyncRate);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetSyncRate error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('SyncRate = ', SyncRate, '/s');

    Writeln;

    for ChanIdx := 0 to NumChannels - 1 do // for all channels
    begin
      RetCode := PH330_GetCountRate(DevIdx[0], ChanIdx, CountRate);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_GetCountRate error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      Writeln('Countrate [', ChanIdx:2, '] = ', CountRate:8, '/s');
    end;

    Writeln;

    RetCode := PH330_GetWarnings(DevIdx[0], Warnings);   //after getting the count rates you can check for warnings
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetWarnings error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    if Warnings <> 0 then
    begin
      PH330_GetWarningsText(DevIdx[0], WarningsText, Warnings);
      Writeln(WarningsText);
    end;

    {
    Now we perform a filter test. This is not strictly necessary but helpful
    when the overall count rate is over the USB troughput limit and you wish to
    use the filter to alleviate this.
    The filter test consists of just a simple retrievel of input and output rates
    of the filter, so that you can assess its effect in terms of rate reduction.
    You can do this a few times in a loop, so that you can also see if there are
    significant fluctuations. However, note that for each round you will need to
    wait for at least 100ms to get new results.
    The filter test must be performed while measurement is running. In order
    to avoid FiFo buffer overruns we can use PH330_SetFilterTestMode to disable
    the transfer of measurement data into the FiFo.
    }

    RetCode := PH330_SetFilterTestMode(DevIdx[0], 1); // Disable FiFo input
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_SetFilterTestMode error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    RetCode := PH330_StartMeas(DevIdx[0], ACQTMAX); //longest possible time, we will stop manually
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_StartMeas error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    Sleep(150); //allow the hardware at least 100ms time for rate counting
    Writeln('');

    {
    To start with, we retrieve the front end count rates. This is somewhat redundant
    here as we already retrieved them above. However, for demonstration purposes,
    this time we use a more efficient method that fetches all rates in one go.
    }

    RetCode := PH330_GetAllCountRates(DevIdx[0], FTestSyncRate, FTestChanRates[0]);
      // here FTestChanRates[index] for GetAllCountRates is pointer to first position
      // of the defined array (1 Pointer function input but e.g. 8 values can get read out)
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetAllCountRates error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    // We only care about the overall rates, so we sum them up here.
    FTestSumRate := 0;
    for ChanIdx := 0 to NumChannels - 1 do
      FTestSumRate := FTestSumRate + FTestChanRates[ChanIdx];
    if Mode = MODE_T2 then // In this case also add the sync rate be course it's used as a additional channel
      FTestSumRate := FTestSumRate + FTestSyncRate;

    Writeln('Front end input rate = ', FTestSumRate);
    Writeln('');
    {
    Now we fetch the filter input rates. These are not necessarily the same as
    the front end count rates as there may already have been losses due to front end
    troughput limits. We do the same summation as above.
    }
    RetCode := PH330_GetFilterInputRates(DevIdx[0], FTestSyncRate, FTestChanRates[0]);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetFilterInputRates error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    FTestSumRate := 0;
    for ChanIdx := 0 to NumChannels - 1 do
      FTestSumRate := FTestSumRate + FTestChanRates[ChanIdx];
    if Mode = MODE_T2 then //in this case also add the sync rate
      FTestSumRate := FTestSumRate + FTestSyncRate;

    Writeln('Filter input rate = ', FTestSumRate);
    Writeln('');

    //Now we do the same rate retrieval and summation for the Main Filter output.
    RetCode := PH330_GetFilterOutputRates(DevIdx[0], FTestSyncRate, FTestChanRates[0]);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetFilteroutputRates error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    FTestSumRate := 0;
    for ChanIdx := 0 to NumChannels - 1 do
      FTestSumRate := FTestSumRate + FTestChanRates[ChanIdx];
    if Mode = MODE_T2 then //in this case also add the sync rate
      FTestSumRate := FTestSumRate + FTestSyncRate;

    Writeln('Filter output rate = ', FTestSumRate);
    Writeln('');

    RetCode := PH330_StopMeas(DevIdx[0]); // test finished, stop measurement
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_StopMeas error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    // Testmode must be switched off again to allow a real measurement
    RetCode := PH330_SetFilterTestMode(DevIdx[0], 0); //re-enable FiFo input
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_SetFilterTestMode ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    // End of main filter test

    // Here we begin the real measurement

    for ChanIdx := 0 to NumChannels - 1 do
      Write(OutTextFile, '   Ch', ChanIdx + 1, ' ');
    Write(OutTextFile, chr(10));

    Writeln('Press RETURN to start measurement');
    Readln;

    RetCode := PH330_StartMeas(DevIdx[0], Tacq);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_StartMeas error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('Measuring for ', Tacq, ' milliseconds...');

    if (Mode = MODE_T3) then
    begin
      {
      We need the sync period in order to calculate the true times of photon records.
      This only makes sense in T3 mode and it assumes a stable period like from a laser.
      Note: Two sync periods must have elapsed after PH330_StartMeas to get proper results.
      You can also use the inverse of what you read via GetSyncRate but it depends on
      the actual sync rate if this is accurate enough.
      It is OK to use the sync input for a photon detector, e.g. if you want to perform
      something like an antibunching measurement. In that case the sync rate obviously is
      not periodic. This means that a) you should set the sync divider to 1 (none) and
      b) that you cannot meaningfully measure the sync period here, which probaly won't
      matter as you only care for the time difference (dtime) of the events.
      }
      RetCode := PH330_GetSyncPeriod(DevIdx[0], SyncPeriod);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_GetSyncPeriod error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      Writeln('Sync period is ', round(SyncPeriod * 1e9),' ns');
    end;

    Writeln;
    Writeln('Starting data collection...');

    Progress := 0;
    OflCorrection := 0;

    repeat
      RetCode := PH330_GetFlags(DevIdx[0], Flags);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_GetFlags error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      FiFoFull := (Flags and FLAG_FIFOFULL) > 0;

      if FiFoFull then
        Writeln('  FiFo Overrun!')
      else
      begin
        RetCode := PH330_ReadFiFo(DevIdx[0], Buffer[0], Records); // may return less!
        if RetCode <> PH330_ERROR_NONE then
        begin
          Writeln('PH330_TTReadData error ', RetCode:3, '. Aborted.');
          Ex(RetCode);
        end;

        // Here we process the data. Note that the time this consumes prevents us
        // from getting around the loop quickly for the next Fifo read.
        // In a serious performance critical scenario you would write the data to
        // a software queue and do the processing in another thread reading from
        // that queue.
        if Records > 0 then
        begin
          for i := 0 to Records do
            ProcessT3(Buffer[i]);
          Progress := Progress + Records;
          Write(#8#8#8#8#8#8#8#8#8, Progress:9);
        end
        else // Do the following only when there was no data
        begin
          RetCode := PH330_CTCStatus(DevIdx[0], CTCStatus);
          if RetCode <> PH330_ERROR_NONE then
          begin
            Writeln;
            Writeln('PH330_CTCStatus error ', RetCode:3, '. Aborted.');
            Ex(RetCode);
          end;
          TimeOut := (CTCStatus <> 0);
          if TimeOut then
          begin
            Writeln;
            Writeln('Done');
            StopTTTR;
          end;
        end;
      end;
      // within this loop you can also read the count rates if needed.
      // Do it sparingly and use PH330_GetAllCountRates for speed.
    until FiFoFull or TimeOut or FileError;

    Writeln;

    RetCode := PH330_StopMeas(DevIdx[0]);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_StopMeas error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    for i := 0 to T3HISTBINS - 1 do
    begin
      for ChanIdx := 0 to NumChannels - 1 do
        Write(OutTextFile, Histogram[ChanIdx, i]:7, ' ');
      Write(OutTextFile, chr(10));
    end;

    Writeln;

    PH330_CloseAllDevices;
    Ex(PH330_ERROR_NONE);

  except
    on E: Exception do
      Writeln(E.ClassName, ': ', E.Message);
  end;
end.

