﻿{
  Demo access to PicoHarp 330 via PH330Lib.dll or libph330.so v.1.0
  The program performs a measurement based on hardcoded settings.
  The resulting event data is instantly processed.
  Processing consists here only of dissecting the binary event record
  data and writing it to a text file. This is only for demo purposes.
  In a real application this makes no sense as it limits throughput and
  creates very large files. In practice you would more sensibly perform
  some meaningful processing such as counting coincidences on the fly.

  Stefan Eilers, Michael Wahl, PicoQuant, July 2024

  Note: This is a console application.

  Note: At the API level the input channel numbers are indexed 0..N-1
  where N is the number of input channels the device has.
  Upon processing we map this back to 1..N corresponding to the front
  panel labelling.

  Tested with the following compilers:

  - Delphi 11.1 (Windows)
  - Lazarus 3.0 RC1 (Windows)
  - Lazarus 2.2.0 / FPC 3.2.2 (Linux)


}

program tttrmode;

{$IF defined(MSWINDOWS)}
  {$APPTYPE CONSOLE}  // Windows needs this, Linux does not want it
{$ENDIF}

uses
  {$ifdef fpc}
  SysUtils,
  {$else}
  System.SysUtils,
  System.Ansistrings,
  {$endif}
  PH330Lib in 'PH330Lib.pas';


var
  DevIdx          : array [0..MAXDEVNUM - 1] of LongInt;
  Dev             : LongInt;
  Found           : Integer = 0;
  RetCode         : LongInt;
  CTCStatus       : LongInt;
  strLibVers      : array[0..7]  of AnsiChar;
  HW_Model        : array[0..31] of AnsiChar;
  HW_Partno       : array[0..8]  of AnsiChar;
  HW_Serial       : array[0..9]  of AnsiChar;
  HW_Version      : array[0..15] of AnsiChar;
  Errorstring     : array[0..39] of AnsiChar;
  DebugInfoBuffer : array[0..16383] of AnsiChar;
  NumChannels     : LongInt;
  Mode            : LongInt = MODE_T2; // set T2 or T3 here, observe suitable Sync divider and Range!
  Binning         : LongInt = 0; // You can change this, meaningful only in T3 mode
  Offset          : LongInt = 0; // You can change this, meaningful only in T3 mode, normally 0
  Tacq            : LongInt = 1000; // Measurement time in millisec, you can change this

  // Sync Settings
  SyncDivider     : LongInt = 1; // you can change this
  SyncChannelOffset : LongInt =           0; // in ps, you can change this (like a cable delay)
  SyncTrgMode       : LongInt = TRGMODE_ETR; // you can change this to TRGMODE_CFD

  // in case of SyncTrgMode == TRGMODE_ETR this will apply:
  SyncTrgEdge  : LongInt = EDGE_FALLING; // you can change this to EDGE_RISING
  SyncTrgLevel : LongInt = -50; // in mV, you can change this

  // in case of SyncTrgMode == TRGMODE_CFD this will apply:
  SyncCFDZeroCross : LongInt = -50; // in mV, you can change this
  SyncCFDLevel     : LongInt = -50; // in mV, you can change this

  // Input Settings
  InputChannelOffset : LongInt = 0; // in ps, you can change this (like a cable delay)
  InputTrgMode       : LongInt = TRGMODE_ETR; // you can change this to TRGMODE_CFD

  // in case of InputTrgMode == TRGMODE_ETR this will apply:
  InputTrgEdge  : LongInt = EDGE_FALLING; // you can change this to EDGE_RISING
  InputTrgLevel : LongInt = -50;          // in mV, you can change this

  // in case of InputTrgMode == TRGMODE_CFD this will apply:
  InputCFDZeroCross : LongInt = -20;      // in mV, you can change this
  InputCFDLevel     : LongInt = -50;      // in mV, you can change this

  // other variables
  Resolution    : Double;
  SyncRate      : LongInt;
  CountRate     : LongInt;
  Flags         : LongInt;
  Warnings      : LongInt;
  WarningsText  : array[0..16343] of AnsiChar; // must have 16384 bytes of text buffer
  Records       : LongInt;
  Progress      : LongInt = 0;
  FiFoFull      : Boolean = False;
  FileError     : Boolean = False;
  ChanIdx       : LongInt;
  Buffer        : array[0..TTREADMAX - 1] of LongWord;
  OutputFile    : TextFile;
  i             : Integer;
  TimeOut       : Boolean = False;
  OflCorrection : Int64 = 0;
  SyncPeriod    : Double = 0;

  //GotPhotonT2 procedure
  //  TimeTag: Raw TimeTag from Record * Resolution = Real Time arrival of Photon
  procedure GotPhotonT2(TimeTag: Int64; Channel: Integer);
  begin
    Writeln(OutputFile, 'CH ', Channel, ' ', Round(TimeTag * Resolution):9);
  end;

  //GotPhotonT3 procedure
  //  DTime: Arrival time of Photon after last Sync event (T3 only)
  //    DTime * Resolution = Real time arrival of Photon after last Sync event
  //  Channel: Channel the Photon arrived (0 = Sync channel for T2 measurements)
  procedure GotPhotonT3(NSync: Int64; DTime: Integer; Channel: Integer);
  begin
    Writeln(OutputFile, 'CH ', Channel, ' ', (NSync * SyncPeriod):3:9, ' ',
      Round(DTime * Resolution)); //NSync * SyncPeriod/s
  end;

  //GotMarker
  //  TimeTag: Raw TimeTag from Record * Global resolution = Real Time arrival of Marker
  //  Markers: Bitfield of arrived Markers, different markers can arrive at same time (same record)
  procedure GotMarker(TimeTag: Int64; Markers: Integer);
  begin
    Writeln(OutputFile, ' MAR ', TimeTag, ' ', Markers);
  end;

  //ProcessT2 procedure
  //HydraHarpV2 (Version 2) or TimeHarp260 or MultiHarp record data
  procedure ProcessT2(TTTR_RawData: Cardinal);
  const
    T2WRAPAROUND_V2 = 33554432;
  type
    TT2DataRecords = record
      Special: Boolean;
      Channel: Byte;
      TimeTag: Cardinal;
    end;
  var
    TTTR_Data: TT2DataRecords;
    TrueTime: Cardinal;
  begin
    {split "RawData" into its parts}
    TTTR_Data.TimeTag := Cardinal(TTTR_RawData and $01FFFFFF); // 25 bit of 32 bit for TimeTag
    TTTR_Data.Channel := Byte((TTTR_RawData shr 25) and $0000003F); // 6 bit of 32 bit for Channel
    TTTR_Data.Special := Boolean((TTTR_RawData shr 31) and $00000001); // 1 bit of 32 bit for Special
    if TTTR_Data.Special then                  // this means we have a Special record
      case TTTR_Data.Channel of
        $3F:        // overflow
          begin
            // number of overflows is stored in timetag
            // if it is zero, it is an old style single overflow {should never happen with new Firmware}
            if TTTR_Data.TimeTag = 0 then
              OflCorrection := OflCorrection + T2WRAPAROUND_V2
            else
              OflCorrection := OflCorrection + T2WRAPAROUND_V2 * TTTR_Data.TimeTag;
          end;
        1..15:   // markers
          begin
            TrueTime := OflCorrection + TTTR_Data.TimeTag;
            // Note that actual marker tagging accuracy is only some ns.
            GotMarker(TrueTime, TTTR_Data.Channel);
          end;
        0: //sync
          begin
            TrueTime := OflCorrection + TTTR_Data.TimeTag;
            GotPhotonT2(TrueTime, 0); //we encode the sync channel as 0
          end;
      end
    else
    begin // it is a regular photon record
      TrueTime := OflCorrection + TTTR_Data.TimeTag;
      GotPhotonT2(TrueTime,
        TTTR_Data.Channel + 1 //we encode the regular channels as 1..N
        );
    end;
  end;

  //ProcessT3 procedure
  //HydraHarp or TimeHarp260 or MultiHarp record data
  procedure ProcessT3(TTTR_RawData: Cardinal);
  const
    T3WRAPAROUND = 1024;
  type
    TT3DataRecords = record
      Special: Boolean;
      Channel: Byte;
      DTime: Word;
      NSync: Word;
    end;
  var
    TTTR_Data: TT3DataRecords;
    TrueNSync: Integer;
  begin
    {split "RawData" into its parts}
    TTTR_Data.NSync := Word(TTTR_RawData and $000003FF); // 10 bit of 32 bit for NSync
    TTTR_Data.DTime := Word((TTTR_RawData shr 10) and $00007FFF); // 15 bit of 32 bit for DTime
    TTTR_Data.Channel := Byte((TTTR_RawData shr 25) and $0000003F); // 6 bit of 32 bit for Channel
    TTTR_Data.Special := Boolean((TTTR_RawData shr 31) and $00000001); // 1 bit of 32 bit for Special
    if TTTR_Data.Special then                  // this means we have a Special record
      case TTTR_Data.Channel of
        $3F:        // overflow
          begin
            // number of overflows is stored in timetag
            // if it is zero, it is an old style single overflow {should never happen with new Firmware}
            if TTTR_Data.NSync = 0 then
              OflCorrection := OflCorrection + T3WRAPAROUND
            else
              OflCorrection := OflCorrection + T3WRAPAROUND * TTTR_Data.NSync;
          end;
        1..15:   // markers
          begin
            TrueNSync := OflCorrection + TTTR_Data.NSync; //the time unit depends on sync period
            // Note that actual marker tagging accuracy is only some ns.
            GotMarker(TrueNSync, TTTR_Data.Channel);
          end;
      end
    else
    begin // it is a regular photon record
      TrueNSync := OflCorrection + TTTR_Data.NSync;
      //TruenSync indicates the number of the sync period this event was in
      GotPhotonT3(TrueNSync,
        TTTR_Data.DTime, //the dtime unit depends on the chosen resolution (binning)
        TTTR_Data.Channel + 1 //we encode the regular channels as 1..N
        );
    end;
  end;


procedure Ex(RetCode: Integer);
begin
  if RetCode <> PH330_ERROR_NONE then
  begin
    PH330_GetErrorString(ErrorString, RetCode);
    Writeln('Error ', RetCode:3, ' = "', Trim (string(ErrorString)), '"');
  end;
  Writeln;
  {$I-}
    CloseFile(OutputFile);
    IOResult();
  {$I+}
  Writeln('Press RETURN to exit');
  Readln;
  Halt(RetCode);
end;

procedure StopTTTR;
begin
  RetCode := PH330_StopMeas(DevIdx[0]);
  if RetCode <> PH330_ERROR_NONE then
  begin
    Writeln('PH330_StopMeas error ', RetCode:3, '. Aborted.');
    Ex(RetCode);
  end;
end;


// Main procedure
begin
  try
    Writeln;
    Writeln('PicoHarp 330 PHLib330 Demo Application              PicoQuant GmbH, 2024');
    Writeln('~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~');

    RetCode := PH330_GetLibraryVersion(strLibVers);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetLibraryVersion error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('PH330LIB version is ' + strLibVers);
    if Trim(Ansistring(strLibVers)) <> Trim(AnsiString (LIB_VERSION)) then
      Writeln('Warning: The application was built for version ' + LIB_VERSION);

    AssignFile(OutputFile, 'tttrmodeout.txt');
    {$I-}
      Rewrite(OutputFile);
    {$I+}
    if IOResult <> 0 then
    begin
      Writeln('cannot open output file');
      Ex(PH330_ERROR_NONE);
    end;

    Writeln;
    Writeln('Searching for PicoHarp330 devices...');
    Writeln('DevIdx     Serial     Status');

    for Dev := 0 to MAXDEVNUM - 1 do
    begin
      RetCode := PH330_OpenDevice(Dev, HW_Serial);
      //
      if RetCode = PH330_ERROR_NONE then
      begin
        // Grab any PicoHarp 330 we can open
        Writeln(Dev, '          ', HW_Serial, '    open ok');
        DevIdx[Found] := Dev; // Keep index to devices we want to use
        Inc(Found);
      end
      else
      begin
        if RetCode = PH330_ERROR_DEVICE_OPEN_FAIL then
          Writeln(Dev, HW_Serial, '                     no device')
        else
        begin
          PH330_GetErrorString(ErrorString, RetCode);
          Writeln(Dev,HW_Serial, '       ' + Trim(string(ErrorString)));
        end;
      end;
    end;

    // In this demo we will use the first device we found,
    // i.e. iDevIdx[0].  You can also use multiple devices in parallel.
    // You could also check for a specific serial number, so that you
    // always know which physical device you are talking to.

    if Found < 1 then
    begin
      Writeln('No device available.');
      Ex(PH330_ERROR_NONE);
    end;

    Writeln('Using device ', DevIdx[0]);
    Writeln('Initializing the device...');

    RetCode := PH330_Initialize (DevIdx[0], Mode, 0); //with internal clock
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_Initialize error ', RetCode:3, '. Aborted.');
      PH330_GetDebugInfo(DevIdx[0], DebugInfoBuffer);
      Writeln('Debug info: ', DebugInfoBuffer);
      Ex(RetCode);
    end;

    RetCode := PH330_GetHardwareInfo (DevIdx[0], HW_Model, HW_Partno, HW_Version); // this is only for information
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetHardwareInfo error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end
    else
      Writeln('Found Model ', HW_Model,'  Part no ', HW_Partno,'  Version ', HW_Version);

    RetCode := PH330_GetNumOfInputChannels (DevIdx[0], NumChannels);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetNumOfInputChannels error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end
    else
      Writeln('Device has ', NumChannels, ' input channels.');

    Writeln;

    RetCode := PH330_SetSyncDiv (DevIdx[0], SyncDivider);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_SetSyncDiv error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    RetCode := PH330_SetSyncTrgMode(DevIdx[0], SyncTrgMode);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_SetSyncDiv error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    if SyncTrgMode = TRGMODE_ETR then
    begin
      RetCode := PH330_SetSyncEdgeTrg(DevIdx[0], SyncTrgLevel, SyncTrgEdge);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetSyncEdgeTrg error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
    end;
    if SyncTrgMode = TRGMODE_CFD then
    begin
      RetCode := PH330_SetSyncCFD(DevIdx[0], SyncCFDLevel, SyncCFDZeroCross);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetSyncCFD error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
    end;

    RetCode := PH330_SetSyncChannelOffset (DevIdx[0], SyncChannelOffset); // in ps, emulate a cable delay
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_SetSyncChannelOffset error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    for ChanIdx := 0 to NumChannels - 1 do // for simplicity we use the same settings for all channels
    begin
      RetCode := PH330_SetInputTrgMode(DevIdx[0], ChanIdx, InputTrgMode);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetInputTrgMode ', ChanIdx:2, ' error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      if InputTrgMode = TRGMODE_ETR then
      begin
        RetCode := PH330_SetInputEdgeTrg (DevIdx[0], ChanIdx, InputTrgLevel, InputTrgEdge);
        if RetCode <> PH330_ERROR_NONE then
        begin
          Writeln('PH330_SetInputEdgeTrg channel ', ChanIdx:2, ' error ', RetCode:3, '. Aborted.');
          Ex(RetCode);
        end;
      end;
      if InputTrgMode = TRGMODE_CFD then
      begin
        RetCode := PH330_SetInputCFD(DevIdx[0], ChanIdx, InputCFDLevel, InputCFDZeroCross);
        if RetCode <> PH330_ERROR_NONE then
        begin
          Writeln('PH330_SetInputCFD ', ChanIdx:2, ' error ', RetCode:3, '. Aborted.');
          Ex(RetCode);
        end;
      end;
      RetCode := PH330_SetInputChannelOffset(DevIdx[0], ChanIdx, InputChannelOffset);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetInputChannelOffset channel ', ChanIdx:2, ' error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      RetCode := PH330_SetInputChannelEnable(DevIdx[0], ChanIdx, 1); // we enable all channels
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetInputChannelEnable channel ', ChanIdx:2, ' error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
    end;

    if (Mode <> MODE_T2) then
    begin
      RetCode := PH330_SetBinning (DevIdx[0], Binning);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetBinning error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;

      RetCode := PH330_SetOffset(DevIdx[0], Offset);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetOffset error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
    end;

    RetCode := PH330_GetResolution (DevIdx[0], Resolution);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetResolution error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('Resolution is ', Resolution:7:3, 'ps');

    // Note: After Init or SetSyncDiv you must allow > 400 ms for valid new count rate readings
    // otherwise you get new values after every 100 ms
    Sleep (150);

    Writeln;

    Writeln('Measuring input rates...');

    RetCode := PH330_GetSyncRate (DevIdx[0], SyncRate);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetSyncRate error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('SyncRate = ', SyncRate, '/s');

    Writeln;

    for ChanIdx := 0 to NumChannels - 1 do // for all channels
    begin
      RetCode := PH330_GetCountRate (DevIdx[0], ChanIdx, CountRate);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_GetCountRate error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      Writeln('Countrate [', ChanIdx:2, '] = ', CountRate:8, '/s');
    end;

    Writeln;

    RetCode := PH330_GetWarnings(DevIdx[0], Warnings);   //after getting the count rates you can check for warnings
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetWarnings error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    if Warnings <> 0 then
    begin
      PH330_GetWarningsText(DevIdx[0], WarningsText, Warnings);
      Writeln(WarningsText);
    end;

    if Mode = Mode_T2 then
    begin
      Writeln(OutputFile, 'ev chn time/ps');
    end
    else
    begin
      Writeln(OutputFile, 'ev chn ttag/s dtime/ps');
    end;

    Writeln('Press RETURN to start measurement');
    Readln;

    RetCode := PH330_StartMeas(DevIdx[0], Tacq);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_StartMeas error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('Measuring for ', Tacq, ' milliseconds...');

    if (Mode = MODE_T3) then
    begin
      // We need the sync period in order to calculate the true times of photon records.
      // This only makes sense in T3 mode and it assumes a stable period like from a laser.
      // Note: Two sync periods must have elapsed after MH_StartMeas to get proper results.
      // You can also use the inverse of what you read via GetSyncRate but it depends on
      // the actual sync rate if this is accurate enough.
      // It is OK to use the sync input for a photon detector, e.g. if you want to perform
      // something like an antibunching measurement. In that case the sync rate obviously is
      // not periodic. This means that a) you should set the sync divider to 1 (none) and
      // b) that you cannot meaningfully measure the sync period here, which probaly won't
      // matter as you only care for the time difference (dtime) of the events.

      RetCode := PH330_GetSyncPeriod(DevIdx[0], SyncPeriod);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_GetSyncPeriod error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      Writeln('Sync period is ', (SyncPeriod * 1e9):3:2,' ns');
    end;

    Progress := 0;
    OflCorrection := 0;

    Writeln;
    Writeln('Starting data collection...');

    repeat
      RetCode := PH330_GetFlags(DevIdx[0], Flags);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_GetFlags error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      FiFoFull := (Flags and FLAG_FIFOFULL) > 0;

      if FiFoFull then
        Writeln('  FiFo Overrun!')
      else
      begin
        RetCode := PH330_ReadFiFo(DevIdx[0], Buffer[0], Records); // may return less!
        if RetCode <> PH330_ERROR_NONE then
        begin
          Writeln('PH330_ReadFiFo error ', RetCode:3, '. Aborted.');
          Ex(RetCode);
        end;

        // Here we process the data. Note that the time this consumes prevents us
        // from getting around the loop quickly for the next Fifo read.
        // In a serious performance critical scenario you would write the data to
        // a software queue and do the processing in another thread reading from
        // that queue.
        if Records > 0 then
        begin
          if Mode = Mode_T2 then
            for i := 0 to Records - 1 do
              ProcessT2(Buffer[i])
          else
            for i := 0 to Records - 1 do
              ProcessT3(Buffer[i]);
          Progress := Progress + Records;
          Write(#8#8#8#8#8#8#8#8#8, Progress:9);
        end
        else // Do the following only when there was no data
        begin
          RetCode := PH330_CTCStatus(DevIdx[0], CTCStatus);
          if RetCode <> PH330_ERROR_NONE then
          begin
            Writeln;
            Writeln('PH330_CTCStatus error ', RetCode:3, '. Aborted.');
            Ex(RetCode);
          end;
          TimeOut := (CTCStatus <> 0);
          if TimeOut then
          begin
            Writeln;
            Writeln('Done');
            StopTTTR;
          end;
        end;
      end;
      // Within this loop you can also read the count rates if needed.
      // Do it sparingly and use PH330_GetAllCountRates for speed.
    until FiFoFull or TimeOut or FileError;

    Writeln;

    RetCode := PH330_StopMeas(DevIdx[0]);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_StopMeas error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    PH330_CloseAllDevices;
    Ex(PH330_ERROR_NONE);

  except
    on E: Exception do
      Writeln(E.ClassName, ': ', E.Message);
  end;
end.

