unit PH330Lib;
{                                                               }
{    PH330Lib programming library for PicoHarp 330              }
{    Stefan Eilers, Michael Wahl, PicoQuant, July 2024          }
{                                                               }

interface

const
  LIB_VERSION    =      '2.0';
{$UNDEF PLATFORM_OK}
{$IFDEF WIN64}
  LIB_NAME       =      'PH330Lib.dll'; // Windows
  {$DEFINE PLATFORM_OK}
{$ENDIF}
{$IFDEF LINUX}
  LIB_NAME       =      'libph330.so';  // Linux
  {$DEFINE PLATFORM_OK}
{$ENDIF}
{$IFNDEF PLATFORM_OK}
  {$FATAL OS platform not supported}
{$ENDIF}

  MAXDEVNUM   = 8;   // max num of USB devices
  MAXINPCHAN  = 4;   // max num of physical input channels
  MAXBINSTEPS = 24;  // max number of binning steps,
                     // get actual number via PH330_GetBaseResolution()

  MAXHISTLEN  = 524288;  // max number of histogram bins
  TTREADMAX   = 1048576; // max number of event records that can be read by PH330_ReadFiFo
                         // buffer must provide space for this number of dwords

  // symbolic constants for PH330_Initialize parameter "refsource"
  REFSRC_INTERNAL        = 0; // use internal clock
  REFSRC_EXTERNAL_10MHZ  = 1; // use 10MHz external clock
  REFSRC_EXTERNAL_100MHZ = 2; // use 100MHz external clock
  REFSRC_EXTERNAL_500MHZ = 3; // use 500MHz external clock

  // symbolic constants for PH330_Initialize parameter "mode"
  MODE_HIST = 0;
  MODE_T2   = 2;
  MODE_T3   = 3;

  // symbolic constants for PH330_SetMeasControl parameter "control"
  MEASCTRL_SINGLESHOT_CTC     = 0; //default
  MEASCTRL_C1_GATED           = 1;
  MEASCTRL_C1_START_CTC_STOP  = 2;
  MEASCTRL_C1_START_C2_STOP   = 3;
  MEASCTRL_SW_START_SW_STOP   = 6;

  // symb. const. for PH330_SetMeasControl, PH330_SetMarkerEdges,
  // PH330_SetSycEdgeTrg and PH330_SetInputEdgeTrg parameter "..edge"
  EDGE_RISING  = 1;
  EDGE_FALLING = 0;

  // limits for PH330_SetHistoLen
  MINLENCODE = 0;  // 1024 histogram bins
  MAXLENCODE = 9;  // 524288 histogram bins
  DFLTLENCODE = 6; // 65536 histogram bins, default, see DFLTHISTLEN

  // limits for PH330_SetSyncDiv
  SYNCDIVMIN = 1;
  SYNCDIVMAX = 8;

  // codes for PH330_SetSyncTrgMode and PH330_SetInputTrgMode parameter "mode"
  TRGMODE_ETR = 0; // edge trigger
  TRGMODE_CFD = 1; // constant fraction discriminator

  // limits for PH330_SetSyncEdgeTrg and PH330_SetInputEdgeTrg
  TRGLVLMIN	      =      -1500; // mV
  TRGLVLMAX	      =       1500; // mV

  CFDLVLMIN = -1500; // mV
  CFDLVLMAX =  0;    // mV
  CFDZCMIN  = -100;  // mV
  CFDZCMAX  =  0;    // mV

  // limits for PH330_SetSyncChannelOffset and PH330_SetInputChannelOffset
  CHANOFFSMIN = -99999; // ps
  CHANOFFSMAX =  99999; // ps

  // limits for PH330_SetSyncDeadTime and PH330_SetInputDeadTime
  EXTDEADMIN  = 800;    // ps
  EXTDEADMAX  = 160000; // ps

  // limits for PH330_SetOffset
  OFFSETMIN = 0;         // ns
  OFFSETMAX = 100000000; // ns

  // limits for PH330_StartMeas
  ACQTMIN = 1;         // ms
  ACQTMAX = 360000000; // ms  (100*60*60*1000ms = 100h)

  // limits for PH330_SetStopOverflow
  STOPCNTMIN = 1;
  STOPCNTMAX = 4294967295; // 32 bit is mem max

  // limits for PH330_SetTriggerOutput
  TRIGOUTMIN = 0;        // 0 = off
  TRIGOUTMAX = 16777215; // in units of 100ns

  // limits for PH330_SetMarkerHoldoffTime
  HOLDOFFMIN = 0;     // ns
  HOLDOFFMAX = 25500; // ns

  // limits for PH330_SetInputHysteresis
  HYSTCODEMIN = 0; // approx. 3mV
  HYSTCODEMAX = 1; // approx. 35mV

  // limits for PH330_SetOflCompression
  HOLDTIMEMIN = 0;   // ms
  HOLDTIMEMAX = 255; // ms

  // limits for PH330_SetEventFilterParams and PH330_SetEventFilterChannels
  MATCHCNTMIN  =           1;
  MATCHCNTMAX  =           6;
  INVERSEMIN   =           0;
  INVERSEMAX   =           1;
  TIMERANGEMIN =           0; // ps
  TIMERANGEMAX =      160000; // ps
  USECHANSMIN  =        $000; // no channels used
  USECHANSMAX  =        $1FF; // note: sync bit 0x100 will be ignored in T3 mode and in row filter
  PASSCHANSMIN =        $000; // no channels passed
  PASSCHANSMAX =        $1FF; // note: sync bit 0x100 will be ignored in T3 mode and in row filter

  // bitmasks for results from PH330_GetFeatures
  FEATURE_DLL       = $0001; // DLL License available
  FEATURE_TTTR      = $0002; // TTTR mode available
  FEATURE_MARKERS   = $0004; // Markers available
  FEATURE_LOWRES    = $0008; // Long range mode available
  FEATURE_TRIGOUT   = $0010; // Trigger output available
  FEATURE_PROG_TD   = $0020; // Programmable deadtime available
  FEATURE_EXT_FPGA  = $0040; // Interface for External FPGA available
  FEATURE_PROG_HYST = $0080; // Programmable input hysteresis available
  FEATURE_EVNT_FILT = $0100; // Coincidence filtering available
  FEATURE_INPT_MODE = $0100; // Programmable input mode (CFD vs. edge trigger)

  // bitmasks for results from PH330_GetFlags
  FLAG_OVERFLOW     =      $0001; // histo mode only
  FLAG_FIFOFULL     =      $0002; // TTTR mode only
  FLAG_SYNC_LOST    =      $0004;
  FLAG_REF_LOST     =      $0008;
  FLAG_SYSERROR     =      $0010; // hardware error, must contact support
  FLAG_ACTIVE       =      $0020; // measurement is running
  FLAG_CNTS_DROPPED =      $0040; // events dropped
  FLAG_SOFTERROR    =      $0080; // software error, must contact support

  // bitmasks for results from PH330_GetWarnings
  WARNING_SYNC_RATE_ZERO      = $0001;
  WARNING_SYNC_RATE_VERY_LOW  = $0002;
  WARNING_SYNC_RATE_TOO_HIGH  = $0004;

  WARNING_INPT_RATE_ZERO      = $0010;
  WARNING_INPT_RATE_TOO_HIGH  = $0040;

  WARNING_INPT_RATE_RATIO     = $0100;
  WARNING_DIVIDER_GREATER_ONE = $0200;
  WARNING_TIME_SPAN_TOO_SMALL = $0400;
  WARNING_OFFSET_UNNECESSARY  = $0800;

  WARNING_DIVIDER_TOO_SMALL   = $1000;
  WARNING_COUNTS_DROPPED      = $2000;
  WARNING_USB20_SPEED_ONLY    = $4000;

// functions from the shared lib/DLL

function PH330_GetLibraryVersion(Vers: PAnsiChar): LongInt; stdcall; external LIB_NAME;
function PH330_GetErrorString(ErrString: PAnsiChar; ErrCode: LongInt): LongInt; stdcall; external LIB_NAME;

function PH330_OpenDevice(DevIdx: LongInt; Serial: PAnsiChar): LongInt; stdcall; external LIB_NAME;
function PH330_CloseDevice(Devidx: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_Initialize(Devidx: LongInt; Mode: LongInt; RefSource: LongInt): LongInt; stdcall; external LIB_NAME;

// all functions below can only be used after PH330_Initialize

function PH330_GetHardwareInfo(DevIdx: LongInt; Model: PAnsiChar; PartNo: PAnsiChar; Version: PAnsiChar): LongInt; stdcall; external LIB_NAME;
function PH330_GetSerialNumber(DevIdx: LongInt; Serial: PAnsiChar): LongInt; stdcall; external LIB_NAME;
function PH330_GetFeatures(DevIdx: LongInt; var Features: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_GetBaseResolution(DevIdx: LongInt; var Resolution: Double; var BinSteps: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_GetNumOfInputChannels(DevIdx: LongInt; var NChannels: LongInt): LongInt; stdcall; external LIB_NAME;

function PH330_SetSyncDiv(DevIdx: LongInt; SyncDiv: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_SetSyncTrgMode(DevIdx: LongInt; Mode: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_SetSyncEdgeTrg(DevIdx: LongInt; Level: LongInt; Edge: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_SetSyncCFD(DevIdx: LongInt; Level: LongInt; ZeroCross: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_SetSyncChannelOffset(DevIdx: LongInt; Value: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_SetSyncChannelEnable(DevIdx: LongInt; Enable: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_SetSyncDeadTime(Devidx: LongInt; IsOn: LongInt; DeadTime: LongInt): LongInt; stdcall; external LIB_NAME;

function PH330_SetInputTrgMode(DevIdx: LongInt; Channel: LongInt; Mode: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_SetInputEdgeTrg(DevIdx: LongInt; channel: LongInt; level: LongInt; edge: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_SetInputCFD(DevIdx: LongInt; Channel: LongInt; Level: LongInt; ZeroCross: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_SetInputChannelOffset(DevIdx: LongInt; channel: LongInt; value: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_SetInputDeadTime(DevIdx: LongInt; Channel: LongInt; IsOn: LongInt; DeadTime: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_SetInputHysteresis(DevIdx: LongInt; HystCode: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_SetInputChannelEnable(DevIdx: LongInt; channel: LongInt; enable: LongInt): LongInt; stdcall; external LIB_NAME;

function PH330_SetStopOverflow(DevIdx: LongInt; stop_ovfl: LongInt; stopcount: LongWord): LongInt; stdcall; external LIB_NAME;
function PH330_SetBinning(DevIdx: LongInt; binning: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_SetOffset(DevIdx: LongInt; offset: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_SetHistoLen(DevIdx: LongInt; lencode: LongInt; var actuallen: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_SetMeasControl(DevIdx: LongInt; control: LongInt; startedge: LongInt; stopedge: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_SetTriggerOutput(DevIdx: LongInt; period: LongInt): LongInt; stdcall; external LIB_NAME;

function PH330_ClearHistMem(DevIdx: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_StartMeas(DevIdx: LongInt; tacq: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_StopMeas(DevIdx: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_CTCStatus(DevIdx: LongInt; var ctcstatus: LongInt): LongInt; stdcall; external LIB_NAME;

function PH330_GetHistogram(DevIdx: LongInt; var chcount: LongWord; channel: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_GetAllHistograms(DevIdx: LongInt; var chcount: LongWord): LongInt; stdcall; external LIB_NAME;
function PH330_GetResolution(DevIdx: LongInt; var Resolution: Double): LongInt; stdcall; external LIB_NAME;
function PH330_GetSyncPeriod(DevIdx: LongInt; var period: Double): LongInt; stdcall; external LIB_NAME;
function PH330_GetSyncRate(DevIdx: LongInt; var syncrate: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_GetCountRate(DevIdx: LongInt; channel: LongInt; var cntrate: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_GetAllCountRates(DevIdx: LongInt; var syncrate: LongInt; var cntrates: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_GetFlags(DevIdx: LongInt; var flags: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_GetElapsedMeasTime(DevIdx: LongInt; var elapsed: Double): LongInt; stdcall; external LIB_NAME;
function PH330_GetWarnings(DevIdx: LongInt; var warnings: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_GetWarningsText(DevIdx: LongInt; text: PAnsiChar; warnings: LongInt): LongInt; stdcall; external LIB_NAME;

// for the time tagging modes only
function PH330_SetOflCompression(DevIdx: LongInt; HoldTime: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_SetMarkerHoldoffTime(DevIdx: LongInt; holdofftime: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_SetMarkerEdges(DevIdx: LongInt; me1: LongInt; me2: LongInt; me3: LongInt; me4: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_SetMarkerEnable(DevIdx: LongInt; en1: LongInt; en2: LongInt; en3: LongInt; en4: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_ReadFiFo(DevIdx: LongInt; var buffer: LongWord; var nactual: LongInt): LongInt; stdcall; external LIB_NAME;

// for event filtering, time tagging modes only
function PH330_SetEventFilterParams(DevIdx: LongInt; TimeRange: LongInt; MatchCNT: LongInt; Inverse: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_SetEventFilterChannels(DevIdx: LongInt;  UseChannels: LongInt; PassChannels: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_EnableEventFilter(DevIdx: LongInt; Enable: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_SetFilterTestMode(DevIdx: LongInt; testmode: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_GetFilterInputRates(DevIdx: LongInt; var Syncrate: LongInt; var CNTRates:  LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_GetFilterOutputRates(DevIdx: LongInt; var Syncrate: LongInt; var CNTRates: LongInt): LongInt; stdcall; external LIB_NAME;

// for debugging only
function PH330_GetDebugInfo(DevIdx: LongInt; debuginfo: PAnsiChar): LongInt; stdcall; external LIB_NAME;
function PH330_GetModuleInfo(DevIdx: LongInt; ModelCode: LongInt; var VersionCode: LongInt): LongInt; stdcall; external LIB_NAME;
function PH330_SaveDebugDump(DevIdx: LongInt; filepath: PAnsiChar): LongInt; stdcall; external LIB_NAME;

// for final cleanup
procedure PH330_CloseAllDevices;

// error codes (return codes from the functions above)
const
  PH330_ERROR_NONE                      =   0;

  PH330_ERROR_DEVICE_OPEN_FAIL          =  -1;
  PH330_ERROR_DEVICE_BUSY               =  -2;
  PH330_ERROR_DEVICE_HEVENT_FAIL        =  -3;
  PH330_ERROR_DEVICE_CALLBSET_FAIL      =  -4;
  PH330_ERROR_DEVICE_BARMAP_FAIL        =  -5;
  PH330_ERROR_DEVICE_CLOSE_FAIL         =  -6;
  PH330_ERROR_DEVICE_RESET_FAIL         =  -7;
  PH330_ERROR_DEVICE_GETVERSION_FAIL    =  -8;
  PH330_ERROR_DEVICE_VERSION_MISMATCH   =  -9;
  PH330_ERROR_DEVICE_NOT_OPEN           = -10;
  PH330_ERROR_DEVICE_LOCKED             = -11;
  PH330_ERROR_DEVICE_DRIVERVER_MISMATCH = -12;

  PH330_ERROR_INSTANCE_RUNNING          = -16;
  PH330_ERROR_INVALID_ARGUMENT          = -17;
  PH330_ERROR_INVALID_MODE              = -18;
  PH330_ERROR_INVALID_OPTION            = -19;
  PH330_ERROR_INVALID_MEMORY            = -20;
  PH330_ERROR_INVALID_RDATA             = -21;
  PH330_ERROR_NOT_INITIALIZED           = -22;
  PH330_ERROR_NOT_CALIBRATED            = -23;
  PH330_ERROR_DMA_FAIL                  = -24;
  PH330_ERROR_XTDEVICE_FAIL             = -25;
  PH330_ERROR_FPGACONF_FAIL             = -26;
  PH330_ERROR_IFCONF_FAIL               = -27;
  PH330_ERROR_FIFORESET_FAIL            = -28;
  PH330_ERROR_THREADSTATE_FAIL          = -29;
  PH330_ERROR_THREADLOCK_FAIL           = -30;

  PH330_ERROR_USB_GETDRIVERVER_FAIL     = -32;
  PH330_ERROR_USB_DRIVERVER_MISMATCH    = -33;
  PH330_ERROR_USB_GETIFINFO_FAIL        = -34;
  PH330_ERROR_USB_HISPEED_FAIL          = -35;
  PH330_ERROR_USB_VCMD_FAIL             = -36;
  PH330_ERROR_USB_BULKRD_FAIL           = -37;
  PH330_ERROR_USB_RESET_FAIL            = -38;

  PH330_ERROR_LANEUP_TIMEOUT            = -40;
  PH330_ERROR_DONEALL_TIMEOUT           = -41;
  PH330_ERROR_MB_ACK_TIMEOUT            = -42;
  PH330_ERROR_MACTIVE_TIMEOUT           = -43;
  PH330_ERROR_MEMCLEAR_FAIL             = -44;
  PH330_ERROR_MEMTEST_FAIL              = -45;
  PH330_ERROR_CALIB_FAIL                = -46;
  PH330_ERROR_REFSEL_FAIL               = -47;
  PH330_ERROR_STATUS_FAIL               = -48;
  PH330_ERROR_MODNUM_FAIL               = -49;
  PH330_ERROR_DIGMUX_FAIL               = -50;
  PH330_ERROR_MODMUX_FAIL               = -51;
  PH330_ERROR_MODFWPCB_MISMATCH         = -52;
  PH330_ERROR_MODFWVER_MISMATCH         = -53;
  PH330_ERROR_MODPROPERTY_MISMATCH      = -54;
  PH330_ERROR_INVALID_MAGIC             = -55;
  PH330_ERROR_INVALID_LENGTH            = -56;
  PH330_ERROR_RATE_FAIL                 = -57;
  PH330_ERROR_MODFWVER_TOO_LOW          = -58;
  PH330_ERROR_MODFWVER_TOO_HIGH         = -59;
  PH330_ERROR_MB_ACK_FAIL               = -60;

  PH330_ERROR_EEPROM_F01                = -64;
  PH330_ERROR_EEPROM_F02                = -65;
  PH330_ERROR_EEPROM_F03                = -66;
  PH330_ERROR_EEPROM_F04                = -67;
  PH330_ERROR_EEPROM_F05                = -68;
  PH330_ERROR_EEPROM_F06                = -69;
  PH330_ERROR_EEPROM_F07                = -70;
  PH330_ERROR_EEPROM_F08                = -71;
  PH330_ERROR_EEPROM_F09                = -72;
  PH330_ERROR_EEPROM_F10                = -73;
  PH330_ERROR_EEPROM_F11                = -74;
  PH330_ERROR_EEPROM_F12                = -75;
  PH330_ERROR_EEPROM_F13                = -76;
  PH330_ERROR_EEPROM_F14                = -77;
  PH330_ERROR_EEPROM_F15                = -78;

  PH330_ERROR_UNSUPPORTED_FUNCTION      = -80;
  PH330_ERROR_WRONG_TRGMODE             = -81;
  PH330_ERROR_BULKRDINIT_FAIL           = -82;
  PH330_ERROR_CREATETHREAD_FAIL         = -83;

  PH330_ERROR_INVALID_ARGUMENT_1       = -201;
  PH330_ERROR_INVALID_ARGUMENT_2       = -202;
  PH330_ERROR_INVALID_ARGUMENT_3       = -203;
  PH330_ERROR_INVALID_ARGUMENT_4       = -204;
  PH330_ERROR_INVALID_ARGUMENT_5       = -205;
  PH330_ERROR_INVALID_ARGUMENT_6       = -206;
  PH330_ERROR_INVALID_ARGUMENT_7       = -207;
  PH330_ERROR_INVALID_ARGUMENT_8       = -208;

implementation

  procedure PH330_CloseAllDevices;
  var
    Dev : Integer;
  begin
    for Dev := 0 to MAXDEVNUM - 1 do // no harm closing all
      PH330_CloseDevice (Dev);
  end;

initialization

finalization
  PH330_CloseAllDevices;
end.
