{
  Demo access to PicoHarp 330 via PH330Lib.dll or libph330.so v.2.0
  The program performs a measurement based on hardcoded settings.
  The resulting event data is stored in a binary output file.

  Stefan Eilers, Michael Wahl, PicoQuant, July 2024

  Note: This is a console application

  Note: At the API level the input channel numbers are indexed 0..N-1
  where N is the number of input channels the device has.

  Note: This demo writes only raw event data to the output file.
  It does not write a file header as regular .ptu files have it.


  Tested with the following compilers:

  - Delphi 11.1 (Windows)
  - Lazarus 3.0 RC1 (Windows)
  - Lazarus 2.0.8 / FPC 3.0.4 (Linux)
}

program tttrmode;

{$IF defined(MSWINDOWS)}
  {$APPTYPE CONSOLE}  // Windows needs this, Linux does not want it
{$ENDIF}

uses
  {$ifdef fpc}
  SysUtils,
  {$else}
  System.SysUtils,
  System.Ansistrings,
  {$endif}
  PH330Lib in 'PH330Lib.pas';

var
  DevIdx          : array [0..MAXDEVNUM - 1] of LongInt;
  Dev             : LongInt;
  Found           : Integer = 0;
  OutFile         : file;
  RetCode         : LongInt;
  CTCStatus       : LongInt;
  strLibVers      : array[0..7]  of AnsiChar;
  HW_Model        : array[0..31] of AnsiChar;
  HW_Partno       : array[0..8] of AnsiChar;
  HW_Serial       : array[0..9] of AnsiChar;
  HW_Version      : array[0..15] of AnsiChar;
  Errorstring     : array[0..39] of AnsiChar;
  NumChannels     : LongInt;
  Mode            : LongInt = MODE_T3; // set T2 or T3 here, observe suitable Syncdivider and Range!
  Binning         : LongInt = 0; // you can change this (meaningless in T2 mode)
  Offset          : LongInt = 0; // normally no need to change this
  Tacq            : LongInt = 1000; // you can change this, unit is millisec

  // Sync Settings
  SyncDivider     : LongInt = 1; // you can change this
  SyncChannelOffset : LongInt =           0; // in ps, you can change this (like a cable delay)
  SyncTrgMode       : LongInt = TRGMODE_ETR; // you can change this to TRGMODE_CFD

  // in case of SyncTrgMode == TRGMODE_ETR this will apply:
  SyncTrgEdge  : LongInt = EDGE_FALLING; // you can change this to EDGE_RISING
  SyncTrgLevel : LongInt = -50; // in mV, you can change this

  // in case of SyncTrgMode == TRGMODE_CFD this will apply:
  SyncCFDZeroCross : LongInt = -50; // in mV, you can change this
  SyncCFDLevel     : LongInt = -50; // in mV, you can change this
  
  // Input Settings
  InputChannelOffset : LongInt =           0; // in ps, you can change this (like a cable delay)
  InputTrgMode       : LongInt = TRGMODE_ETR; // you can change this to TRGMODE_CFD

  // in case of InputTrgMode == TRGMODE_ETR this will apply:
  InputTrgEdge  : LongInt = EDGE_FALLING; // you can change this to EDGE_RISING
  InputTrgLevel : LongInt = -50;          // in mV, you can change this

  // in case of InputTrgMode == TRGMODE_CFD this will apply:
  InputCFDZeroCross : LongInt = -20;      // in mV, you can change this
  InputCFDLevel     : LongInt = -50;      // in mV, you can change this

  // other variables
  Resolution  : Double;
  SyncRate    : LongInt;
  CountRate   : LongInt;
  Flags       : LongInt;
  Warnings    : LongInt;
  WarningsText : array[0..16343] of AnsiChar; // must have 16384 bytes of text buffer
  Records     : LongInt;
  Progress    : LongInt = 0;
  Written     : LongInt;
  FiFoFull    : Boolean =   False;
  MeasDone    : Boolean =   False;
  FileError   : Boolean =   False;
  ChanIdx     : LongInt;
  Buffer      : array[0..TTREADMAX - 1] of LongWord;


procedure Ex(RetCode: Integer);
begin
  if RetCode <> PH330_ERROR_NONE then
  begin
    PH330_GetErrorString(Errorstring, RetCode);
    Writeln('Error ', RetCode:3, ' = "', Trim(string(Errorstring)), '"');
  end;
  Writeln;
  {$I-}
    CloseFile(OutFile);
    IOResult();
  {$I+}
  Writeln('Press RETURN to exit');
  Readln;
  Halt(RetCode);
end;

procedure StopTTTR;
begin
  RetCode := PH330_StopMeas(DevIdx[0]);
  if RetCode <> PH330_ERROR_NONE then
  begin
    Writeln('PH330_StopMeas error ', RetCode:3, '. Aborted.');
    Ex(RetCode);
  end;
end;

begin
  try
    Writeln;
    Writeln('PicoHarp 330 PHLib330 Demo Application              PicoQuant GmbH, 2024');
    Writeln('~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~');
    RetCode := PH330_GetLibraryVersion (strLibVers);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetLibraryVersion error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('PH330LIB version is ' + strLibVers);
    if Trim(AnsiString(strLibVers)) <> Trim(AnsiString(LIB_VERSION)) then
      Writeln('Warning: The application was built for version ' + LIB_VERSION);

    AssignFile(OutFile, 'tttrmode.out');
    {$I-}
      Rewrite(OutFile, 4);
    {$I+}
    if IOResult <> 0 then
    begin
      Writeln('cannot open output file');
      Ex(PH330_ERROR_NONE);
    end;

    Writeln;
    Writeln('Searching for PicoHarp 330 devices...');
    Writeln('DevIdx     Serial     Status');

    for Dev := 0 to MAXDEVNUM - 1 do
    begin
      RetCode := PH330_OpenDevice(Dev, HW_Serial);
      //
      if RetCode = PH330_ERROR_NONE then
      begin
        // grab any PicoHarp 330 we can open
        Writeln(Dev, '          ', HW_Serial, '    open ok');
        DevIdx[Found] := Dev; // keep index to devices we want to use
        Inc(Found);
      end
      else
      begin
        if RetCode = PH330_ERROR_DEVICE_OPEN_FAIL then
          Writeln(Dev, '          ', HW_Serial, '           no device')
        else
        begin
          PH330_GetErrorString(ErrorString, RetCode);
          Writeln(Dev, HW_Serial, '      ' + Trim(string(ErrorString)));
        end;
      end;
    end;

    // in this demo we will use the first PicoHarp330 device we found,
    // i.e. iDevIdx[0].  You can also use multiple devices in parallel.
    // you could also check for a specific serial number, so that you
    // always know which physical device you are talking to.

    if Found < 1 then
    begin
      Writeln('No device available.');
      Ex(PH330_ERROR_NONE);
    end;

    Writeln('Using device ', DevIdx[0]);
    Writeln('Initializing the device...');

    RetCode := PH330_Initialize(DevIdx[0], Mode, 0); // with internal clock
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_Initialize error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    RetCode := PH330_GetHardwareInfo(DevIdx[0], HW_Model, HW_Partno, HW_Version);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetHardwareInfo error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end
    else
    Writeln('Found Model ', HW_Model,'  Part no ', HW_Partno,'  Version ', HW_Version);

    RetCode := PH330_GetNumOfInputChannels(DevIdx[0], NumChannels);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetNumOfInputChannels error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end
    else
    Writeln('Device has ', NumChannels, ' input channels.');
    Writeln;

    RetCode := PH330_SetSyncDiv(DevIdx[0], SyncDivider);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_SetSyncDiv error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    RetCode := PH330_SetSyncTrgMode(DevIdx[0], SyncTrgMode);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_SetSyncDiv error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    if SyncTrgMode = TRGMODE_ETR then
    begin
      RetCode := PH330_SetSyncEdgeTrg(DevIdx[0], SyncTrgLevel, SyncTrgEdge);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetSyncEdgeTrg error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
    end;
    if SyncTrgMode = TRGMODE_CFD then
    begin
      RetCode := PH330_SetSyncCFD(DevIdx[0], SyncCFDLevel, SyncCFDZeroCross);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetSyncCFD error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
    end;

    RetCode := PH330_SetSyncChannelOffset(DevIdx[0], SyncChannelOffset); // this can emulate a cable delay
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_SetSyncChannelOffset error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    for ChanIdx := 0 to NumChannels - 1 do // for simplicity we use the same settings for all channels
    begin
      RetCode := PH330_SetInputTrgMode(DevIdx[0], ChanIdx, InputTrgMode);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetInputTrgMode ', ChanIdx:2, ' error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      if InputTrgMode = TRGMODE_ETR then
      begin
        RetCode := PH330_SetInputEdgeTrg (DevIdx[0], ChanIdx, InputTrgLevel, InputTrgEdge);
        if RetCode <> PH330_ERROR_NONE then
        begin
          Writeln('PH330_SetInputEdgeTrg channel ', ChanIdx:2, ' error ', RetCode:3, '. Aborted.');
          Ex(RetCode);
        end;
      end;
      if InputTrgMode = TRGMODE_CFD then
      begin
        RetCode := PH330_SetInputCFD(DevIdx[0], ChanIdx, InputCFDLevel, InputCFDZeroCross);
        if RetCode <> PH330_ERROR_NONE then
        begin
          Writeln('PH330_SetInputCFD ', ChanIdx:2, ' error ', RetCode:3, '. Aborted.');
          Ex(RetCode);
        end;
      end;
      RetCode := PH330_SetInputChannelOffset(DevIdx[0], ChanIdx, InputChannelOffset);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetInputChannelOffset channel ', ChanIdx:2, ' error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      RetCode := PH330_SetInputChannelEnable(DevIdx[0], ChanIdx, 1); // we enable all channels
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetInputChannelEnable channel ', ChanIdx:2, ' error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
    end;

    if (Mode = MODE_T3) then // the following functions are meaningless in T2 mode
    begin
      RetCode := PH330_SetBinning(DevIdx[0], Binning);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetBinning error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;

      RetCode := PH330_SetOffset(DevIdx[0], Offset);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_SetOffset error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;

      RetCode := PH330_GetResolution(DevIdx[0], Resolution);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_GetResolution error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      Writeln('Resolution is ', Resolution:7:3, 'ps');
    end;

    Writeln('Measuring input rates...');

    // Note: After Init or SetSyncDiv you must allow > 400 ms for valid new count rate readings
    // otherwise you get new values after every 100 ms
    Sleep(150);

    Writeln;

    RetCode := PH330_GetSyncRate(DevIdx[0], SyncRate);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetSyncRate error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('SyncRate = ', SyncRate, '/s');

    Writeln;

    for ChanIdx := 0 to NumChannels - 1 do // for all channels
    begin
      RetCode := PH330_GetCountRate(DevIdx[0], ChanIdx, CountRate);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_GetCountRate error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      Writeln('Countrate [', ChanIdx:2, '] = ', CountRate:8, '/s');
    end;

    Writeln;

    RetCode := PH330_GetWarnings(DevIdx[0], Warnings);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_GetWarnings error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    if Warnings <> 0 then
    begin
      PH330_GetWarningsText(DevIdx[0], WarningsText, Warnings);
      Writeln(WarningsText);
    end;

    Writeln('Press RETURN to start');
    Readln;

    Writeln('Starting data collection...');

    RetCode := PH330_StartMeas(DevIdx[0], Tacq);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_StartMeas error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;
    Writeln('Measuring for ', Tacq, ' milliseconds...');

    Progress := 0;
    Write(#8#8#8#8#8#8#8#8#8, Progress:9);

    repeat
      RetCode := PH330_GetFlags(DevIdx[0], Flags);
      if RetCode <> PH330_ERROR_NONE then
      begin
        Writeln('PH330_GetFlags error ', RetCode:3, '. Aborted.');
        Ex(RetCode);
      end;
      FiFoFull := (Flags and FLAG_FIFOFULL) > 0;

      if FiFoFull then
      begin
        Writeln ('  FiFo Overrun!');
        StopTTTR;
      end
      else
      begin
        RetCode := PH330_ReadFiFo(DevIdx[0], Buffer[0], Records); // may return less!
        if RetCode <> PH330_ERROR_NONE then
        begin
          Writeln('PH330_TTReadData error ', RetCode:3, '. Aborted.');
          Ex(RetCode);
        end;

        if Records > 0 then
        begin
          BlockWrite(OutFile, Buffer[0], Records, Written);
          if Records <> Written then
          begin
            Writeln;
            Writeln('File write error');
            FileError := True;
            StopTTTR;
          end;

          Progress := Progress + Written;
          Write(#8#8#8#8#8#8#8#8#8, Progress:9);
        end
        else // do the following only when there was no data
        begin
          RetCode := PH330_CTCStatus(DevIdx[0], CTCStatus);
          if RetCode <> PH330_ERROR_NONE then
          begin
            Writeln;
            Writeln('PH330_CTCStatus error ', RetCode:3, '. Aborted.');
            Ex(RetCode);
          end;
          MeasDone := (CTCStatus <> 0);
          if MeasDone then
          begin
            Writeln;
            Writeln('Done');
            StopTTTR;
          end;
        end;
      end;
      // Within this loop you can also read the count rates if needed.
      // Do it sparingly and use PH330_GetAllCountRates for speed.
    until FiFoFull or MeasDone or FileError;

    Writeln;

    RetCode := PH330_StopMeas(DevIdx[0]);
    if RetCode <> PH330_ERROR_NONE then
    begin
      Writeln('PH330_StopMeas error ', RetCode:3, '. Aborted.');
      Ex(RetCode);
    end;

    Writeln;

    PH330_CloseAllDevices;
    Ex(PH330_ERROR_NONE);

  except
  on E: Exception do
    Writeln(E.ClassName, ': ', E.Message);
  end;
end.

