  # Demo access to PicoHarp 330 via PH330Lib.dll or libph330.so v.2.0

  # THIS IS AN ADVANCED DEMO. DO NOT USE FOR YOUR FIRST EXPERIMENTS.
  # Look at the variable meascontrol down below to see what it does.

  # The program performs a measurement based on hardcoded settings.
  # The resulting histogram is stored in an ASCII output file.

  # Stefan Eilers, Michael Wahl, PicoQuant, July 2024 

  # Note: This is a console application

  # Note: At the API level channel numbers are indexed 0..N-1
  # where N is the number of channels the device has.

  # Tested with the following compilers:
  # - Python 3.11 (Windows)

import time
import ctypes as ct
from ctypes import byref
import os
import sys

if sys.version_info[0] < 3:
    print("[Warning] Python 2 is not fully supported. It might work, but "
          "use Python 3 if you encounter errors.\n")
    raw_input("Press RETURN to continue"); print
    
if os.name == "nt":
    phlib = ct.WinDLL("PH330Lib.dll")
else:
    phlib = ct.CDLL("libph330.so")    

# Constants from ph330defin.h, do not edit these!
LIB_VERSION = "2.0"
MAXDEVNUM = 8
MODE_HIST = 0
MAXLENCODE = 9
MAXINPCHAN = 4
MAXHISTLEN = 524288
FLAG_OVERFLOW = 0x0001
TRGMODE_ETR = 0 # edge trigger
TRGMODE_CFD = 1 # constant fraction discriminator
EDGE_FALLING = 0
EDGE_RISING = 1

# Symbolic constants for PH330_SetMeasControl parameter "meascontrol"
MEASCTRL_SINGLESHOT_CTC = 0 # default
MEASCTRL_C1_GATED = 1
MEASCTRL_C1_START_CTC_STOP = 2
MEASCTRL_C1_START_C2_STOP = 3
MEASCTRL_SW_START_SW_STOP = 6

PH330_ERROR_DEVICE_OPEN_FAIL = -1

# Measurement parameters, hard coded here since this is just a demo

mode = MODE_HIST # cannot be changed for the purpose of this demo
binning = 0 # You can change this, meaningful only in T3 mode
offset = 0 # You can change this, meaningful only in T3 mode
tacq = 1000 # Measurement time in millisec, you can change this
sync_divider = 1 # You can change this, observe mode. Explained in Manual.
sync_channel_offset = 0 # You can change this (like a cable delay)
input_channel_offset = 0 # You can change this (like a cable delay)

sync_trg_mode = TRGMODE_ETR # You can change this to TRGMODE_CFD  
# In case of SyncTrgMode is TRGMODE_ETR this will apply:
sync_trg_edge = EDGE_FALLING # You can change this to EDGE_RISING 
sync_trg_level = -50 # You can change this (in mV)         
# In case of SyncTrgMode is TRGMODE_CFD this will apply:
sync_cfd_zero_cross = -20 # In mV, you can change this
sync_cfd_level = -50 # In mV, you can change this

input_trg_mode = TRGMODE_ETR # You can change this to TRGMODE_CFD    
# In case of InputTrgMode is TRGMODE_ETR this will apply:
input_trg_edge = EDGE_FALLING # You can change this to EDGE_RISING
input_trg_level = -50 # In mV, you can change this    
# In case of InputTrgMode is TRGMODE_CFD this will apply:
input_cfd_zero_cross = -20 # In mV, you can change this
input_cfd_level = -50 # In mV, you can change this      

# Specific measurement parameters for this demo:
# un-comment ONE of the four lines below to select a mode of measurement control
meascontrol = ct.c_int(MEASCTRL_SINGLESHOT_CTC) # Start by software and stop when CTC expires (default)
# meascontrol = ct.c_int(MEASCTRL_C1_GATED) # Measure while C1 is active
# meascontrol = ct.c_int(MEASCTRL_C1_START_CTC_STOP) # Start with C1 and stop when CTC expires 
# meascontrol = ct.c_int(MEASCTRL_C1_START_C2_STOP) # Start with C1 and stop with C2
edge_1 = ct.c_int(EDGE_RISING) # Edge of C1 to start (if applicable in chosen mode)
edge_2 = ct.c_int(EDGE_FALLING) # Edge of C2 to stop (if applicable in chosen mode)
    
# Variables for flow contol and results from library calls

dev_idx = []
ctc_status = ct.c_int()
lib_version = ct.create_string_buffer(b"", 8) 
hw_model = ct.create_string_buffer(b"", 32)
hw_partno = ct.create_string_buffer(b"", 9)
hw_serial = ct.create_string_buffer(b"", 10)
hw_version = ct.create_string_buffer(b"", 16)
error_string = ct.create_string_buffer(b"", 40)
num_channels = ct.c_int()
resolution = ct.c_double() 
sync_rate = ct.c_int()
count_rate = ct.c_int()
elapsed = ct.c_double()
flags = ct.c_int()
warnings = ct.c_int()
warnings_text = ct.create_string_buffer(b"", 16384) 
hist_len = ct.c_int()
cmd = 0
counts = [(ct.c_uint * MAXHISTLEN)() for i in range(0, MAXINPCHAN)]

# Subroutines

def closeDevices():
    histogramfile.close()
    for i in range(0, MAXDEVNUM):
        phlib.PH330_CloseDevice(ct.c_int(i))
    if sys.version_info[0] < 3:
        raw_input("\nPress RETURN to exit"); print
    else:
        input("\nPress RETURN to exit"); print
    sys.exit(0)

# Helper for showing error texts and shutdown in case of error
def tryfunc(ret_code, func_name):
    if ret_code < 0:
        phlib.PH330_GetErrorString(error_string, ret_code)
        print("PH330_%s error %d (%s). Aborted." % (func_name, ret_code,
              error_string.value.decode("utf-8")))
        closeDevices()


# The main demo code starts here        

print("\nPicoHarp 330 PHLib330 Demo Application                PicoQuant GmbH, 2024")
print("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~")
    
phlib.PH330_GetLibraryVersion(lib_version)
print("Library version is %s" % lib_version.value.decode("utf-8"))
if lib_version.value.decode("utf-8") != LIB_VERSION:
    print("Warning: The application was built for version %s" % LIB_VERSION)

histogramfile = open("histomodeout.txt", "w+")

print("\nSearching for Picoharp 330 devices...")
print("Devidx     Serial        Status")


for i in range(0, MAXDEVNUM):
    ret_code = phlib.PH330_OpenDevice(ct.c_int(i), hw_serial)
    if ret_code == 0:
        print("  %1d        %s       open ok" % (i, hw_serial.value.decode("utf-8")))
        dev_idx.append(i)
    else:
        if ret_code == PH330_ERROR_DEVICE_OPEN_FAIL:
            print("  %1d                      no device" % i)
        else:
            phlib.PH330_GetErrorString(error_string, ct.c_int(ret_code))
            print("  %1d        %s" % (i, error_string.value.decode("utf8")))
            
            
# In this demo we will use the first device we find, i.e. dev_idx[0].
# You can also use multiple devices in parallel.
# You can also check for specific serial numbers, so that you always know 
# which physical device you are talking to.

if len(dev_idx) < 1:
    print("No device available.")
    closeDevices()
print("Using device #%1d" % dev_idx[0])
print("\nInitializing the device...")

# Initialize for histogramming mode with internal clock
tryfunc(phlib.PH330_Initialize(ct.c_int(dev_idx[0]), ct.c_int(MODE_HIST), ct.c_int(0)),
        "Initialize")   

# Only for information
tryfunc(phlib.PH330_GetHardwareInfo(ct.c_int(dev_idx[0]), hw_model, hw_partno, hw_version),
        "GetHardwareInfo")
print("Found Model %s Part no %s Version %s" % (hw_model.value.decode("utf-8"),
      hw_partno.value.decode("utf-8"), hw_version.value.decode("utf-8")))

tryfunc(phlib.PH330_GetNumOfInputChannels(ct.c_int(dev_idx[0]), byref(num_channels)),
        "GetNumOfInputChannels")
print("Device has %i input channels." % num_channels.value)

tryfunc(phlib.PH330_SetSyncDiv(ct.c_int(dev_idx[0]), ct.c_int(sync_divider)), "SetSyncDiv")

tryfunc(phlib.PH330_SetSyncTrgMode(ct.c_int(dev_idx[0]), sync_trg_mode), "SetSyncTrgMode")

if sync_trg_mode == TRGMODE_ETR:
    tryfunc(phlib.PH330_SetSyncEdgeTrg(ct.c_int(dev_idx[0]), ct.c_int(sync_trg_level),
            ct.c_int(sync_trg_edge)), "SetSyncEdgeTrg")
    
if sync_trg_mode == TRGMODE_CFD:
    tryfunc(phlib.PH330_SetSyncCFD(ct.c_int(dev_idx[0]), ct.c_int(sync_cfd_level),
            ct.c_int(sync_cfd_zero_cross)), "SetSyncCFD")             

tryfunc(phlib.PH330_SetSyncChannelOffset(ct.c_int(dev_idx[0]), ct.c_int(sync_channel_offset)),
        "SetSyncChannelOffset") # This can emulate a cable delay       

tryfunc(phlib.PH330_SetSyncChannelOffset(ct.c_int(dev_idx[0]), ct.c_int(sync_channel_offset)),
    "SetSyncChannelOffset")

# We use the same input settings for all channels, you can change this.
for i in range(0, num_channels.value):

    tryfunc(phlib.PH330_SetInputTrgMode(ct.c_int(dev_idx[0]), ct.c_int(i), input_trg_mode), 
            "SetInputTrgMode")

    if input_trg_mode == TRGMODE_ETR:
        tryfunc(phlib.PH330_SetInputEdgeTrg(ct.c_int(dev_idx[0]), ct.c_int(i), ct.c_int(input_trg_level),
                ct.c_int(input_trg_edge)), "SetInputEdgeTrg")
    
    if sync_trg_mode == TRGMODE_CFD:
        tryfunc(phlib.PH330_SetInputCFD(ct.c_int(dev_idx[0]), ct.c_int(i), ct.c_int(input_cfd_level),
                ct.c_int(input_cfd_zero_cross)), "SetInputCFD")   

    tryfunc(phlib.PH330_SetInputChannelOffset(ct.c_int(dev_idx[0]), ct.c_int(i),
            ct.c_int(input_channel_offset)), "SetInputChannelOffset") # This can emulate a cable delay
    
    tryfunc(phlib.PH330_SetInputChannelEnable(ct.c_int(dev_idx[0]), ct.c_int(i),
            ct.c_int(1)), "SetInputChannelEnable") # We enable all channels

tryfunc(phlib.PH330_SetBinning(ct.c_int(dev_idx[0]), ct.c_int(binning)), "SetBinning")

tryfunc(phlib.PH330_SetOffset(ct.c_int(dev_idx[0]), ct.c_int(offset)), "SetOffset")

tryfunc(phlib.PH330_SetHistoLen(ct.c_int(dev_idx[0]), ct.c_int(MAXLENCODE), byref(hist_len)), 
        "SetHistoLen")
print("\nHistogram length is %d" % hist_len.value)

tryfunc(phlib.PH330_GetResolution(ct.c_int(dev_idx[0]), byref(resolution)), 
        "GetResolution")
print("Resolution is %1.1lfps\n" % resolution.value)

# After Init allow 150 ms for valid  count rate readings
# Subsequently you get new values after every 100ms
time.sleep(0.15)

tryfunc(phlib.PH330_GetSyncRate(ct.c_int(dev_idx[0]), byref(sync_rate)), "GetSyncRate")
print("\nSync rate=%1d/s" % sync_rate.value)

for i in range(0, num_channels.value):
    tryfunc(phlib.PH330_GetCountRate(ct.c_int(dev_idx[0]), ct.c_int(i), byref(count_rate)),
            "GetCountRate")
    print("Count rate[%1d]=%1d/s" % (i, count_rate.value))

# After getting the count rates you can check for warnings
tryfunc(phlib.PH330_GetWarnings(ct.c_int(dev_idx[0]), byref(warnings)),
        "GetWarnings")
if warnings.value > 0:
    tryfunc(phlib.PH330_GetWarningsText(ct.c_int(dev_idx[0]), warnings_text, warnings), 
            "GetWarningsText")
    print("\n\n%s" % warnings_text.value.decode("utf-8"))

tryfunc(phlib.PH330_SetStopOverflow(ct.c_int(dev_idx[0]), 0, 10000),
        "SetStopOverflow") # for example only, actually 0=off

# important in this specific demo:
tryfunc(phlib.PH330_SetMeasControl(ct.c_int(dev_idx[0]), meascontrol, 
        edge_1, edge_2), "SetMeasControl")


while cmd != "q":
    tryfunc(phlib.PH330_ClearHistMem(ct.c_int(dev_idx[0])), "ClearHistMem")
    
    if sys.version_info[0] < 3:
        raw_input("\nPress RETURN to start"); print
    else:
        input("\nPress RETURN to start"); print
        
    tryfunc(phlib.PH330_GetSyncRate(ct.c_int(dev_idx[0]), byref(sync_rate)), 
            "GetSyncRate")
    print("\nSync rate=%1d/s" % sync_rate.value)
        
    for i in range(0, num_channels.value):
        tryfunc(phlib.PH330_GetCountRate(ct.c_int(dev_idx[0]), ct.c_int(i), byref(count_rate)), 
                "GetCountRate")
        print("Countrate[%1d]=%1d/s" % (i, count_rate.value))

    # Here you could check for warnings again

    tryfunc(phlib.PH330_StartMeas(ct.c_int(dev_idx[0]), ct.c_int(tacq)), 
            "StartMeas")

    if meascontrol.value != MEASCTRL_SINGLESHOT_CTC:
        print("\nWaiting for hardware start on C1...")
        ctc_status = ct.c_int(1)
        while ctc_status.value == 1:
            tryfunc(phlib.PH330_CTCStatus(ct.c_int(dev_idx[0]), byref(ctc_status)),
                "CTCStatus")
            
    if meascontrol.value == MEASCTRL_SINGLESHOT_CTC or meascontrol == MEASCTRL_C1_START_CTC_STOP:
        print("\nMeasuring for %1d milliseconds..." % tacq)
    
    if meascontrol.value == MEASCTRL_C1_GATED:
        print("\nMeasuring, waiting for other C1 edge to stop...") 
    
    if meascontrol.value == MEASCTRL_C1_START_C2_STOP:
        print("\nMeasuring, waiting for C2 to stop...")  
    print("\n")


    ctc_status = ct.c_int(0)  
    while ctc_status.value == 0:
        tryfunc(phlib.PH330_CTCStatus(ct.c_int(dev_idx[0]), byref(ctc_status)), "CTCStatus")
    
    tryfunc(phlib.PH330_StopMeas(ct.c_int(dev_idx[0])), "StopMeas")

    for i in range(0, num_channels.value):
        tryfunc(phlib.PH330_GetHistogram(ct.c_int(dev_idx[0]), byref(counts[i]), i),
        "GetHistogram")
        integral_count = 0
        for j in range(0, hist_len.value):
            integral_count = integral_count + counts[i][j]
        print("Integral count[%1d]=%1.0lf" % (i, integral_count))

    tryfunc(phlib.PH330_GetElapsedMeasTime(ct.c_int(dev_idx[0]), byref(elapsed)),
            "GetElapsedMeasTime")
    print("\nElapsed measurement time was %1.0lf ms" % elapsed.value)

    tryfunc(phlib.PH330_GetFlags(ct.c_int(dev_idx[0]), byref(flags)), "GetFlags")

    if flags.value & FLAG_OVERFLOW > 0:
        print(" Overflow")
        closeDevices()
    
    print("Enter c to continue or q to quit and save the count data.")
    cmd = input()

print('Saving data to file...')

for i in range(0, num_channels.value):
    histogramfile.write("   CH%1d" % (i + 1))
histogramfile.write("\n")

for j in range(0, hist_len.value):
    for i in range(0, num_channels.value):
        histogramfile.write("%7u" % counts[i][j])
    histogramfile.write("\n")
    
print('Saved.')
    
closeDevices()

