# Demo access to PicoHarp 330 via PH330Lib.dll or libph330.so v.2.0
# The program performs a TTTR measurement based on hard-coded settings.
# Event data is filtered by the device's event filter in the FPGA.
# The demo shows the software calls for setting the filter parameters
# and activating the filter. Resulting event data is instantly processed.
# Processing consists here of simple histogramming. This allows replicating 
# and verifying the filter demonstration as shown in the section on event 
# filtering in the PicoHarp 330 manual. If you use identical or at least 
# similar input signals and setings you should obtain similar histograms 
# as shown in the manual. Observe the offset of -10000ps in the call of 
# PH330_SetSyncchannelOffset. Under the assumption that the sync channel 
# and the first input channel are receiving the same signal through identical
# cable lengths this is necessary in order to shift the region of interest
# where coincidences occur into the usable histogram range.
# Note also that in order to implement the easiest way of histogramming, 
# this demo is implemented for T3 mode only, although the filtering as such
# works in T2 mode as well.

# Stefan Eilers, Michael Wahl, PicoQuant, July 2024 

# Note: This is a console application

# Note: At the API level the input channel numbers are indexed 0..N-1
# where N is the number of input channels the device has.
# Upon processing we map this back to 1..N corresponding to the front
# panel labelling.

# Tested with:								
# - Python 3.11 (Windows)    

import time
import ctypes as ct
from ctypes import byref
import os
import sys
import numpy as np 

if sys.version_info[0] < 3:
    print("[Warning] Python 2 is not fully supported. It might work, but "
          "use Python 3 if you encounter errors.\n")
    raw_input("Press RETURN to continue"); print

if os.name == "nt":
    phlib = ct.WinDLL("ph330lib.dll")
else:
    phlib = ct.CDLL("libph330.so")
    
# Constants from ph330defin.h, do not edit these!

LIB_VERSION = "2.0"
MAXDEVNUM = 8
MODE_T3 = 3
MAXINPCHAN = 4
TTREADMAX = 1048576
FLAG_FIFOFULL = 0x0002
TRGMODE_ETR = 0 # edge trigger
TRGMODE_CFD = 1 # constant fraction discriminator
EDGE_FALLING = 0
EDGE_RISING = 1

# Measurement parameters, hard coded here since this is just a demo

mode = MODE_T3 # This demo ist for T3 only.
binning = 0 # You can change this, meaningful only in T3 mode
offset = 0 # You can change this, meaningful only in T3 mode
tacq = 1000 # Measurement time in millisec, you can change this
sync_divider = 1 # You can change this, observe mode. Explained in Manual.
sync_channel_offset = -10000 # You can change this (like a cable delay)
input_channel_offset = 0 # You can change this (like a cable delay)

sync_trg_mode = TRGMODE_ETR # You can change this to TRGMODE_CFD 
 
sync_trg_mode = TRGMODE_ETR # You can change this to TRGMODE_CFD  
# In case of SyncTrgMode is TRGMODE_ETR this will apply:
sync_trg_edge = EDGE_FALLING # You can change this to EDGE_RISING 
sync_trg_level = -50 # You can change this (in mV)         
# In case of SyncTrgMode is TRGMODE_CFD this will apply:
sync_cfd_zero_cross = -20 # In mV, you can change this
sync_cfd_level = -50 # In mV, you can change this

input_trg_mode = TRGMODE_ETR # You can change this to TRGMODE_CFD    
# In case of InputTrgMode is TRGMODE_ETR this will apply:
input_trg_edge = EDGE_FALLING # You can change this to EDGE_RISING
input_trg_level = -50 # In mV, you can change this    
# In case of InputTrgMode is TRGMODE_CFD this will apply:
input_cfd_zero_cross = -20 # In mV, you can change this
input_cfd_level = -50 # In mV, you can change this    

# Event filter parameters, a key point in this demo

    # The following parameters are for programming the event filter.
    # Please read the manual on what the filter parameters are about.
    # Here we implement a simple "Singles Filter" which is the most obvious
    # and most useful type of event filter in typical quantum optics applications.
    # The idea is that photon events that are "single" in the sense that there is no
    # other event in temporal proximity (within timerange) will be filtered out.
    # This reduces the bus load and the file size if you write the records to a file.
    # Any further (application specific) coincidence processing can then be
    # done in software, either on the fly or subsequently on the stored data.

filter_time_range = 4000; # In picoseconds
filter_match_cnt = 1 # Must have at least one other event in proximity
filter_inverse = 0 # Normal filtering mode, see manual
filter_enable = 1 # Activate the filter with 1, deactivate with 0
filter_use_chans = 0x3 # Bitmasks for which channels are to be used
filter_pass_chans = 0 # Bitmasks for which channels to pass unfiltered
   
# Variables for flow control and values from the library calls

dev_idx = []
ctc_status = ct.c_int()
lib_version = ct.create_string_buffer(b"", 8) 
hw_model = ct.create_string_buffer(b"", 32)
hw_partno = ct.create_string_buffer(b"", 9)
hw_serial = ct.create_string_buffer(b"", 10)
hw_version = ct.create_string_buffer(b"", 16)
error_string = ct.create_string_buffer(b"", 40)
num_channels = ct.c_int()
resolution = ct.c_double() 
sync_rate = ct.c_int()
count_rate = ct.c_int()
flags = ct.c_int()
warnings = ct.c_int()
warnings_text = ct.create_string_buffer(b"", 16384) 
records = ct.c_int()
progress = 0
buffer = (ct.c_uint * TTREADMAX)()
sync_period = ct.c_double()
stop_retry = 0

# Count rate buffers for the filter test further down
f_test_sync_rate = ct.c_int()         
f_test_chan_rates = (ct.c_int*MAXINPCHAN)()
f_test_sum_rate = ct.c_int()

# Histogramming variables
T3HISTBINS = 32768 #  is 2^15, dtime in T3 mode has 15 bits, you cannot change this
histogram = np.zeros((MAXINPCHAN, T3HISTBINS)) # array with 32768 bins


# Subroutines

# Got PhotonT3
# TimeTag: Overflow-corrected arrival time in units of the sync period 
# d_time: Arrival time of photon after last Sync event in units of the chosen resolution (set by binning)
# Channel: 1..N where N is the numer of channels the device has
def GotPhotonT3(n_sync, channel, d_time):
    histogram[channel - 1, d_time] = histogram[channel - 1, d_time] + 1 # histogramming
    
# Got MarkerT3
# TimeTag: Overflow-corrected arrival time in units of the sync period 
# Markers: Bitfield of arrived Markers, different markers can arrive at same time (same record)    
def GotMarkerT3(n_sync, Markers):
    pass
    
# ProcessT3
# Dissect a single T3 mode data record and correct for timetag overflows
def ProcessT3(TTTRRecord):
    global outputfile, recNum, nRecords, oflcorrection # Markers, channel, Special, d_time
    ch = 0
    dt = 0
    true_sync = 0
    T3WRAPAROUND = 1024
    try:
        recordDatabinary = "{0:0{1}b}".format(TTTRRecord, 32)
    except:
        print("The file ended earlier than expected, at record %d/%d."\
          % (recNum, nRecords))
        sys.exit(0)
    
    special = int(recordDatabinary[0:1], base=2) # 1 bit for Special
    channel = int(recordDatabinary[1:7], base=2) # 6 bit for channel     
    d_time = int(recordDatabinary[7:22], base=2) # 15 bit for d_time   
    n_sync = int(recordDatabinary[22:32], base=2) # n_sync is number of the Sync period, 10 bit for n_sync
    
    if special==1:
        if channel == 0x3F: # Special record, including Overflow as well as Markers and Sync
        
            # number of overflows is stored in timetag
            if n_sync == 0: # if it is zero it is an old style single overflow 
                oflcorrection += T3WRAPAROUND
            else:
                oflcorrection += T3WRAPAROUND * n_sync
        if channel>=1 and channel<=15: # Markers
            true_sync = oflcorrection + T3WRAPAROUND * n_sync
            #Note that the time unit depends on sync period
            GotMarkerT3(true_sync, channel)

    else: # regular input channel
        true_sync = oflcorrection + n_sync
        ch = channel + 1 # we encode the regular channels as 1..N 
        dt = d_time
        # true_sync indicates the number of the sync period this event was in
        # the d_time unit depends on the chosen resolution (binning)
        GotPhotonT3(true_sync, ch, dt)        

def closeDevices():
    for dev in range(0, MAXDEVNUM):
        phlib.PH330_CloseDevice(ct.c_int(dev))
    sys.exit(0)

def stoptttr(): 
    ret_code = phlib.PH330_StopMeas(ct.c_int(dev_idx[0]))
    if ret_code < 0:
        print("PH330_StopMeas error %1d. Aborted." % ret_code)
    closeDevices() 
            
# Helper for showing error texts and shutdown in case of error
def tryfunc(ret_code, func_name, meas_running=False):
    if ret_code < 0:
        phlib.PH330_GetErrorString(error_string, ct.c_int(ret_code))
        print("PH330_%s error %d (%s). Aborted." % (func_name, ret_code,
              error_string.value.decode("utf-8")))
        if meas_running:
            stoptttr()
        else:
            closeDevices()

# The main demo code starts here 
            
print("\nPicoHarp 330 PHLib330 Demo Application                PicoQuant GmbH, 2024")
print("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~")
phlib.PH330_GetLibraryVersion(lib_version)
print("Library version is %s" % lib_version.value.decode("utf-8"))
if lib_version.value.decode("utf-8") != LIB_VERSION:
    print("Warning: The application was built for version %s" % LIB_VERSION)
    
if mode != MODE_T3:
    print("\nThis demo program is made for T3 mode only.")
    input("\nPress RETURN to exit"); print 
    sys.exit(0)

histogramfile = open("t3histout.txt", "w")

print("\nSearching for Picoharp 330 devices...")
print("Devidx     Serial      Status")

for i in range(0, MAXDEVNUM):
    ret_code = phlib.PH330_OpenDevice(ct.c_int(i), hw_serial)
    if ret_code == 0:
        print("  %1d        %s     open ok" % (i, hw_serial.value.decode("utf-8")))
        dev_idx.append(i)
    else:
        if ret_code == -1: # PH330_ERROR_DEVICE_OPEN_FAIL
            print("  %1d                    no device" % i)
        else:
            phlib.PH330_GetErrorString(error_string, ct.c_int(ret_code))
            print("  %1d        %s" % (i, error_string.value.decode("utf-8")))

# In this demo we will use the first device we find, i.e. dev_idx[0].
# You can also use multiple devices in parallel.
# You can also check for specific serial numbers, so that you always know 
# which physical device you are talking to.

if len(dev_idx) < 1:
    print("\nNo device available.")
    closeDevices()
print("\nUsing device #%1d" % dev_idx[0])
print("\nInitializing the device...\n")

# Initialize with chosen mode and internal clock
tryfunc(phlib.PH330_Initialize(ct.c_int(dev_idx[0]), ct.c_int(mode), ct.c_int(0)),
        "Initialize")

# Only for information
tryfunc(phlib.PH330_GetHardwareInfo(ct.c_int(dev_idx[0]), hw_model, hw_partno, hw_version),
        "GetHardwareInfo")
print("Found Model %s Part no %s Version %s" % (hw_model.value.decode("utf-8"),
      hw_partno.value.decode("utf-8"), hw_version.value.decode("utf-8")))

tryfunc(phlib.PH330_GetNumOfInputChannels(ct.c_int(dev_idx[0]), byref(num_channels)),
        "GetNumOfInputChannels")
print("Device has %i input channels." % num_channels.value)

tryfunc(phlib.PH330_SetSyncDiv(ct.c_int(dev_idx[0]), ct.c_int(sync_divider)), "SetSyncDiv")

tryfunc(phlib.PH330_SetSyncTrgMode(ct.c_int(dev_idx[0]), sync_trg_mode), "SetSyncTrgMode")

if sync_trg_mode == TRGMODE_ETR:
    tryfunc(phlib.PH330_SetSyncEdgeTrg(ct.c_int(dev_idx[0]), ct.c_int(sync_trg_level),
            ct.c_int(sync_trg_edge)), "SetSyncEdgeTrg")
    
if sync_trg_mode == TRGMODE_CFD:
    tryfunc(phlib.PH330_SetSyncCFD(ct.c_int(dev_idx[0]), ct.c_int(sync_cfd_level),
            ct.c_int(sync_cfd_zero_cross)), "SetSyncCFD")            

tryfunc(phlib.PH330_SetSyncChannelOffset(ct.c_int(dev_idx[0]), ct.c_int(sync_channel_offset)),
    "SetSyncChannelOffset") # This can emulate a cable delay

# We use the same input settings for all channels, you can change this.
for i in range(0, num_channels.value):

    tryfunc(phlib.PH330_SetInputTrgMode(ct.c_int(dev_idx[0]), ct.c_int(i), input_trg_mode), 
            "SetInputTrgMode")

    if input_trg_mode == TRGMODE_ETR:
        tryfunc(phlib.PH330_SetInputEdgeTrg(ct.c_int(dev_idx[0]), ct.c_int(i), ct.c_int(input_trg_level),
                ct.c_int(input_trg_edge)), "SetInputEdgeTrg")
    
    if sync_trg_mode == TRGMODE_CFD:
        tryfunc(phlib.PH330_SetInputCFD(ct.c_int(dev_idx[0]), ct.c_int(i), ct.c_int(input_cfd_level),
                ct.c_int(input_cfd_zero_cross)), "SetInputCFD")   

    tryfunc(phlib.PH330_SetInputChannelOffset(ct.c_int(dev_idx[0]), ct.c_int(i),
            ct.c_int(input_channel_offset)), "SetInputChannelOffset") # This can emulate a cable delay
    
    tryfunc(phlib.PH330_SetInputChannelEnable(ct.c_int(dev_idx[0]), ct.c_int(i),
            ct.c_int(1)), "SetInputChannelEnable") # We enable all channels

# Meaningful only in T3 mode
if mode == MODE_T3:
    tryfunc(phlib.PH330_SetBinning(ct.c_int(dev_idx[0]), ct.c_int(binning)), "SetBinning")
    tryfunc(phlib.PH330_SetOffset(ct.c_int(dev_idx[0]), ct.c_int(offset)), "SetOffset")
    
tryfunc(phlib.PH330_GetResolution(ct.c_int(dev_idx[0]), byref(resolution)), "GetResolution")
print("Resolution is %1.1lfps" % resolution.value)

# Event filter setup, a key point in this demo 

tryfunc(phlib.PH330_SetEventFilterChannels(ct.c_int(dev_idx[0]), ct.c_int(filter_use_chans), 
                                           ct.c_int(filter_pass_chans)), "SetEventFilterChannels")

tryfunc(phlib.PH330_SetEventFilterParams(ct.c_int(dev_idx[0]), filter_time_range, filter_match_cnt, 
                                         filter_inverse), "SetEventFilterParams")

tryfunc(phlib.PH330_EnableEventFilter(ct.c_int(dev_idx[0]), filter_enable), "EnableEventFilter")

# Filter setup ends here

# After Initialize or SetSyncDiv you must allow >100 ms for valid  count rate readings
# Subsequently you get new values after every 100ms
# The same applies to the filter test below.
time.sleep(0.15)# in s

print("\nMeasuring input rates...");

tryfunc(phlib.PH330_GetSyncRate(ct.c_int(dev_idx[0]), byref(sync_rate)), "GetSyncRate")
print("\nSyncrate=%1d/s" % sync_rate.value)

for i in range(0, num_channels.value):
    tryfunc(phlib.PH330_GetCountRate(ct.c_int(dev_idx[0]), ct.c_int(i), byref(count_rate)),
            "GetCountRate")
    print("Countrate[%1d]=%1d/s" % (i, count_rate.value))
    
print("\n")  
  
# After getting the count rates you can check for warnings
tryfunc(phlib.PH330_GetWarnings(ct.c_int(dev_idx[0]), byref(warnings)), "GetWarnings")  
if warnings.value > 0:
    tryfunc(phlib.PH330_GetWarningsText(ct.c_int(dev_idx[0]), warnings_text, warnings), 
            "GetWarningsTest")
    print(warnings_text.value.decode("utf-8"))

# Filter test, a key point in this demo

# Now we perform a filter test. This is not strictly necessary but helpful
# when the overall count rate is over the USB troughput limit and you wish to
# use the filter to alleviate this.
# The filter test consists of just a simple retrievel of input and output rates
# of the filter, so that you can assess its effect in terms of rate reduction.
# You can do this a few times in a loop, so that you can also see if there are
# significant fluctuations. However, note that for each round you will need to
# wait for at least 100ms to get new results.
# The filter test must be performed while measurement is running. In order
# to avoid FiFo buffer overruns we can use PH330_SetFilterTestMode to disable
# the transfer of measurement data into the FiFo.

print("\nFilter test running...")

tryfunc(phlib.PH330_SetFilterTestMode(ct.c_int(dev_idx[0]), ct.c_int(1)),"SetFilterTestMode") # Disable FiFo input
tryfunc(phlib.PH330_StartMeas(ct.c_int(dev_idx[0]), ct.c_int(tacq)),"StartMeas") # Longest possible time, we will stop manually
        
time.sleep(0.15) # Allow the hardware at least 100ms time for rate counting 

# To start with, we retrieve the front end count rates. This is somewhat redundant 
# here as we already retrieved them above. However, for demonstration purposes, 
# this time we use a more efficient method that fetches all rates in one go.        
tryfunc(phlib.PH330_GetAllCountRates(ct.c_int(dev_idx[0]), byref(f_test_sync_rate), f_test_chan_rates),
        "GetAllCountRates"
)

# We only care about the overall rates, so we sum them up here.
f_test_sum_rate = 0  
for i in range(0,num_channels.value):  
    f_test_sum_rate = f_test_sum_rate + f_test_chan_rates[i]   
print('\nFront end input rate = ', f_test_sum_rate, '/s')    
    
# Now we fetch the filter input rates. These are not necessarily the same as
# the front end count rates as there may already have been losses due to front end
# troughput limits. We do the same summation as above.

f_test_sum_rate = 0
tryfunc(phlib.PH330_GetFilterInputRates(ct.c_int(dev_idx[0]), byref(f_test_sync_rate), f_test_chan_rates),
        "GetRowFilteredRates"
        )
for i in range(0,num_channels.value):
    f_test_sum_rate = f_test_sum_rate + f_test_chan_rates[i]  
print('\nMain Filter input rate = ', f_test_sum_rate, '/s') 

# Now we do the same rate retrieval and summation for the filter output.
f_test_sum_rate = 0
tryfunc(phlib.PH330_GetFilterOutputRates(ct.c_int(dev_idx[0]), byref(f_test_sync_rate), f_test_chan_rates),"GetMainFilteredRates")
for i in range(0, num_channels.value):        
    f_test_sum_rate = f_test_sum_rate + f_test_chan_rates[i]
print('\nMain Filter output rate = ', f_test_sum_rate, '/s') 

tryfunc(phlib.PH330_StopMeas(ct.c_int(dev_idx[0])),"StopMeas") # Test finished, stop measurement

# Filter test mode must be switched off again to allow a real measurement
tryfunc(phlib.PH330_SetFilterTestMode(ct.c_int(dev_idx[0]), ct.c_int(0)),"SetFilterTestMode") # Re-enable FiFo input)

print("\nFilter test done")

# End of filter test

# Now we begin the real measurement

if sys.version_info[0] < 3:
    raw_input("\nPress RETURN to start"); print
else:
    input("\nPress RETURN to start"); print
    
tryfunc(phlib.PH330_StartMeas(ct.c_int(dev_idx[0]), ct.c_int(tacq)), "StartMeas")

if mode == MODE_T3:
    # We need the sync period in order to calculate the true times of photon records.
    # This only makes sense in T3 mode and it assumes a stable period like from a laser.
    # Note: Two sync periods must have elapsed after MH_StartMeas to get proper results.
    # You can also use the inverse of what you read via GetSyncRate but it depends on
    # the actual sync rate if this is accurate enough.
    # It is OK to use the sync input for a photon detector, e.g. if you want to perform
    # something like an antibunching measurement. In that case the sync rate obviously is
    # not periodic. This means that a) you should set the sync divider to 1 (none) and
    # b) that you cannot meaningfully measure the sync period here, which probaly won't
    # matter as you only care for the time difference(d_time) of the events.
    tryfunc(phlib.PH330_GetSyncPeriod(ct.c_int(dev_idx[0]), byref(sync_period)), "GetSyncPeriod")
    print("\nSync period is %lf ns\n" % (sync_period.value * 1e9 ))
    
print("\nStarting data collection...")    

progress = 0
sys.stdout.write("\nProgress:%9u" % progress)
sys.stdout.flush()

oflcorrection = 0

while True:
    tryfunc(phlib.PH330_GetFlags(ct.c_int(dev_idx[0]), byref(flags)), "GetFlags")
    
    if flags.value & FLAG_FIFOFULL > 0:
        print("\nFiFo Overrun!")
        stoptttr()
    
    tryfunc(phlib.PH330_ReadFiFo(ct.c_int(dev_idx[0]), byref(buffer), byref(records)),
            "ReadFiFo", meas_running=True)

    # Here we process the data. Note that the time this consumes prevents us
    # from getting around the loop quickly for the next Fifo read.
    # In a serious performance critical scenario you would write the data to
    # a software queue and do the processing in another thread reading from
    # that queue.
    if records.value > 0:       
        for i in range(0,records.value):
            ProcessT3(buffer[i])   
        progress += records.value
        sys.stdout.write("\rProgress:%9u" % progress)
        sys.stdout.flush()
    else: # do the following only when there was no data
        tryfunc(phlib.PH330_CTCStatus(ct.c_int(dev_idx[0]), byref(ctc_status)),
                "CTCStatus")
        if ctc_status.value > 0:
            # retry a few times to make sure the FIFO is really empty
            stop_retry = stop_retry + 1
            if stop_retry > 5:
                print("\nDone")
                break
    # Within this loop you can also read the count rates if needed.
    # Do it sparingly and use PH330_GetAllCountRates for speed.
    
# Saving the histogram data
for j in range(0, num_channels.value):
    histogramfile.write("    Ch%1d " % (j + 1))
histogramfile.write( "\n")  
for i in range(0, T3HISTBINS):
    for j in range(0, num_channels.value):
        histogramfile.write("%7u " % (histogram[j,i])) 
    histogramfile.write("\n")
        
histogramfile.close()  
      
input("\nPress RETURN to exit"); print

closeDevices()

