# Demo access to PicoHarp 330 via PH330Lib.dll or libph330.so v.2.0
# The program performs a measurement based on hardcoded settings.
# The resulting event data is instantly processed.
# Processing consists here only of dissecting the binary event record
# data and writing it to a text file. This is only for demo purposes.
# In a real application this makes no sense as it limits throughput and
# creates very large files. In practice you would more sensibly perform
# some meaningful processing such as counting coincidences on the fly.


# Stefan Eilers, Michael Wahl, PicoQuant, July 2024 

# Note: This is a console application

# Note: At the API level the input channel numbers are indexed 0..N-1
# where N is the number of input channels the device has.
# Upon processing we map this back to 1..N corresponding to the front
# panel labelling.

# Tested with:   
# Python 3.11 (Windows)   


import time
import ctypes as ct
from ctypes import byref
import os
import sys

if sys.version_info[0] < 3:
    print("[Warning] Python 2 is not fully supported. It might work, but "
          "use Python 3 if you encounter errors.\n")
    raw_input("Press RETURN to continue"); print

if os.name == "nt":
    phlib = ct.WinDLL("PH330lib.dll")
else:
    phlib = ct.CDLL("libph330.so")


# Constants from ph330defin.h, do not edit these!

LIB_VERSION = "2.0"
MAXDEVNUM = 8
MODE_T2 = 2
MODE_T3 = 3
MAXINPCHAN = 4
TTREADMAX = 1048576
FLAG_FIFOFULL = 0x0002
TRGMODE_ETR = 0 # edge trigger
TRGMODE_CFD = 1 # constant fraction discriminator
EDGE_FALLING = 0
EDGE_RISING = 1

# Measurement parameters, hard coded here since this is just a demo

mode = MODE_T2 # You can change this to T3, observe suitable sync divider and binning!
binning = 0 # You can change this, meaningful only in T3 mode
offset = 0 # You can change this, meaningful only in T3 mode
tacq = 1000 # Measurement time in millisec, you can change this
sync_divider = 1 # You can change this, observe mode. Explained in Manual.
sync_channel_offset = 0 # You can change this (like a cable delay)
input_channel_offset = 0 # You can change this (like a cable delay)

sync_trg_mode = TRGMODE_ETR # You can change this to TRGMODE_CFD  
# In case of SyncTrgMode is TRGMODE_ETR this will apply:
sync_trg_edge = EDGE_FALLING # You can change this to EDGE_RISING 
sync_trg_level = -50 # You can change this (in mV)         
# In case of SyncTrgMode is TRGMODE_CFD this will apply:
sync_cfd_zero_cross = -20 # In mV, you can change this
sync_cfd_level = -50 # In mV, you can change this

input_trg_mode = TRGMODE_ETR # You can change this to TRGMODE_CFD    
# In case of InputTrgMode is TRGMODE_ETR this will apply:
input_trg_edge = EDGE_FALLING # You can change this to EDGE_RISING
input_trg_level = -50 # In mV, you can change this    
# In case of InputTrgMode is TRGMODE_CFD this will apply:
input_cfd_zero_cross = -20 # In mV, you can change this
input_cfd_level = -50 # In mV, you can change this    

# Variables for flow contol and results from library calls

dev_idx = []
ctc_status = ct.c_int()
lib_version = ct.create_string_buffer(b"", 8) 
hw_model = ct.create_string_buffer(b"", 32)
hw_partno = ct.create_string_buffer(b"", 9)
hw_serial = ct.create_string_buffer(b"", 10)
hw_version = ct.create_string_buffer(b"", 16)
error_string = ct.create_string_buffer(b"", 40)
num_channels = ct.c_int()
resolution = ct.c_double() 
sync_rate = ct.c_int()
count_rate = ct.c_int()
flags = ct.c_int()
warnings = ct.c_int()
warnings_text = ct.create_string_buffer(b"", 16384) 
records = ct.c_int()
progress = 0
buffer = (ct.c_uint * TTREADMAX)()
sync_period = ct.c_double()
stop_retry = 0
ctc_status = ct.c_int()

# Subroutines

# Got PhotonT2
# TimeTag: Overflow-corrected arrival time in units of the device's base resolution 
# Channel: Channel the photon arrived (0 = Sync channel, 1..N = regular timing channel)
def GotPhotonT2(TimeTag, Channel):
    global out_text_file, resolution
    out_text_file.write("CH %2d %14.0lf\n" % (Channel, TimeTag * resolution.value))

# Got MarkerT2
# TimeTag: Overflow-corrected arrival time in units of the device's base resolution 
# Markers: Bitfield of arrived markers, different markers can arrive at same time (same record)   
def GotMarkerT2(TimeTag, Markers):
    global out_text_file, resolution
    out_text_file.write("MK %2d %14.0lf\n" % (Markers, TimeTag * resolution.value))

# Got PhotonT3
# TimeTag: Overflow-corrected arrival time in units of the sync period 
# DTime: Arrival time of photon after last Sync event in units of the chosen resolution (set by binning)
# Channel: 1..N where N is the numer of channels the device has
def GotPhotonT3(truensync, Channel, DTime):
    global out_text_file, sync_period, resolution
    out_text_file.write("CH %2d   %10.9lf  %8.0lf\n" % (Channel, 
                                                  truensync * sync_period.value,
                                                  DTime   * resolution.value))
    
# Got MarkerT3
# TimeTag: Overflow-corrected arrival time in units of the sync period 
# Markers: Bitfield of arrived Markers, different markers can arrive at same time (same record)    
def GotMarkerT3(truensync, Markers):
    global out_text_file, sync_period
    out_text_file.write("MK %2d %10.9lf\n" % (Markers, truensync * sync_period.value)) 
    
# ProcessT2
# Dissect a single T2 mode data record and correct for timetag overflows
def ProcessT2(TTTRRecord):
    global out_text_file, recNum, nRecords, oflcorrection, Markers, Channel, Special
    ch = 0
    truetime = 0
    T2WRAPAROUND_V2 = 33554432    
    try:   
        # The data handed out to this function is transformed to an up to 32 digits long binary number
        # and this binary is filled by zeros from the left up to 32 bit
        recordDatabinary = '{0:0{1}b}'.format(TTTRRecord,32)
    except:
        print("\nThe file ended earlier than expected, at record %d/%d."\
          % (recNum.value, nRecords.value))
        sys.exit(0)
        
    # Then the different parts of this 32 bit are splitted and handed over to the Variables       
    Special = int(recordDatabinary[0:1], base=2) # 1 bit for Special    
    Channel = int(recordDatabinary[1:7], base=2) # 6 bit for Channel
    TimeTag = int(recordDatabinary[7:32], base=2) # 25 bit for TimeTag
        
    if Special==1:
        if Channel == 0x3F: # Special record, including Overflow as well as Markers and Sync       
            # number of overflows is stored in timetag
            if TimeTag == 0: # if it is zero it is an old style single overflow 
                oflcorrection += T2WRAPAROUND_V2
            else:
                oflcorrection += T2WRAPAROUND_V2 * TimeTag
        if Channel>=1 and Channel<=15: # Markers
            truetime = oflcorrection + T2WRAPAROUND_V2 * TimeTag
            #Note that actual marker tagging accuracy is only some ns
            ch = Channel
            GotMarkerT2(truetime, ch)
        if Channel==0: # Sync
            truetime = oflcorrection + TimeTag
            ch = 0 # we encode the sync channel as 0
            GotPhotonT2(truetime, ch)
    else: # regular input channel
        truetime = oflcorrection + TimeTag
        ch = Channel + 1 # we encode the regular channels as 1..N        
        GotPhotonT2(truetime, ch)
    
# ProcessT3
# Dissect a single T3 mode data record and correct for timetag overflows
def ProcessT3(TTTRRecord):
    global out_text_file, recNum, nRecords, oflcorrection#, Markers, Channel, Special, DTime
    ch = 0
    dt = 0
    truensync = 0
    T3WRAPAROUND = 1024
    try:
        recordDatabinary = "{0:0{1}b}".format(TTTRRecord, 32)
    except:
        print("The file ended earlier than expected, at record %d/%d."\
          % (recNum, nRecords))
        sys.exit(0)
    
    Special = int(recordDatabinary[0:1], base=2) # 1 bit for Special
    Channel = int(recordDatabinary[1:7], base=2) # 6 bit for Channel     
    DTime = int(recordDatabinary[7:22], base=2) # 15 bit for DTime   
    nSync = int(recordDatabinary[22:32], base=2) # nSync is number of the Sync period, 10 bit for nSync
    
    if Special==1:
        if Channel == 0x3F: # Special record, including Overflow as well as Markers and Sync
        
            # number of overflows is stored in timetag
            if nSync == 0: # if it is zero it is an old style single overflow 
                oflcorrection += T3WRAPAROUND
            else:
                oflcorrection += T3WRAPAROUND * nSync
        if Channel>=1 and Channel<=15: # Markers
            truensync = oflcorrection + T3WRAPAROUND * nSync
            #Note that the time unit depends on sync period
            GotMarkerT3(truensync, Channel)

    else: # regular input channel
        truensync = oflcorrection + nSync
        ch = Channel + 1 # we encode the regular channels as 1..N 
        dt = DTime
        # truensync indicates the number of the sync period this event was in
        # the dtime unit depends on the chosen resolution (binning)
        GotPhotonT3(truensync, ch, dt)        


def closeDevices():
    out_text_file.close()
    for i in range(0, MAXDEVNUM):
        phlib.PH330_CloseDevice(ct.c_int(i))
    if sys.version_info[0] < 3:
        raw_input("\nPress RETURN to exit"); print
    else:
        input("\nPress RETURN to exit"); print
    sys.exit(0)


def stoptttr():
    retcode = phlib.PH330_StopMeas(ct.c_int(dev_idx[0]))
    if retcode < 0:
        print("PH330_StopMeas error %1d. Aborted." % retcode)
    closeDevices()


# Helper for showing error texts and shutdown in case of error
def tryfunc(retcode, funcName, meas_running=False):
    if retcode < 0:
        phlib.PH330_GetErrorString(error_string, ct.c_int(retcode))
        print("PH330_%s error %d (%s). Aborted." % (funcName, retcode,
              error_string.value.decode("utf-8")))
        if meas_running:
            stoptttr()
        else:
            closeDevices()
            
            
# The main demo code starts here            
                        
print("\nPicoHarp 330 PHLib330 Demo Application                PicoQuant GmbH, 2024")
print("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~")
phlib.PH330_GetLibraryVersion(lib_version)
print("Library version is %s" % lib_version.value.decode("utf-8"))
if lib_version.value.decode("utf-8") != LIB_VERSION:
    print("Warning: The application was built for version %s" % LIB_VERSION)

out_text_file = open("tttrmodeout.txt", "w")

print("\nSearching for PicoHarp 330 devices...")
print("Devidx     Serial     Status")

for i in range(0, MAXDEVNUM):
    retcode = phlib.PH330_OpenDevice(ct.c_int(i), hw_serial)
    if retcode == 0:
        print("  %1d        %s    open ok" %  (i, hw_serial.value.decode("utf-8")))
        dev_idx.append(i)
    else:
        if retcode == -1: # PH330_ERROR_DEVICE_OPEN_FAIL
            print("  %1d                   no device" % i)
        else:
            phlib.PH330_GetErrorString(error_string, ct.c_int(retcode))
            print("  %1d        %s" % (i, error_string.value.decode("utf8")))

# In this demo we will use the first device we find, i.e. dev_idx[0].
# You can in principle also use multiple devices in parallel, see manual.
# You can also check for specific serial numbers, so that you always know 
# which physical device you are talking to.

if len(dev_idx) < 1:
    print("\nNo device available.")
    closeDevices()
print("\nUsing device #%1d" % dev_idx[0])
print("\nInitializing the device...\n")

# Initialize with chosen mode and internal clock
tryfunc(phlib.PH330_Initialize(ct.c_int(dev_idx[0]), ct.c_int(mode), ct.c_int(0)),
        "Initialize")

# Only for information
tryfunc(phlib.PH330_GetHardwareInfo(dev_idx[0], hw_model, hw_partno, hw_version),
        "GetHardwareInfo")
print("Found Model %s Part no %s Version %s" % (hw_model.value.decode("utf-8"),
      hw_partno.value.decode("utf-8"), hw_version.value.decode("utf-8")))

tryfunc(phlib.PH330_GetNumOfInputChannels(ct.c_int(dev_idx[0]), byref(num_channels)),
        "GetNumOfInputChannels")
print("Device has %i input channels." % num_channels.value)

tryfunc(phlib.PH330_SetSyncDiv(ct.c_int(dev_idx[0]), ct.c_int(sync_divider)), "SetSyncDiv")

tryfunc(phlib.PH330_SetSyncTrgMode(ct.c_int(dev_idx[0]), sync_trg_mode), "SetSyncTrgMode")

if sync_trg_mode == TRGMODE_ETR:
    tryfunc(phlib.PH330_SetSyncEdgeTrg(ct.c_int(dev_idx[0]), ct.c_int(sync_trg_level),
            ct.c_int(sync_trg_edge)), "SetSyncEdgeTrg")
    
if sync_trg_mode == TRGMODE_CFD:
    tryfunc(phlib.PH330_SetSyncCFD(ct.c_int(dev_idx[0]), ct.c_int(sync_cfd_level),
            ct.c_int(sync_cfd_zero_cross)), "SetSyncCFD")            

tryfunc(phlib.PH330_SetSyncChannelOffset(ct.c_int(dev_idx[0]), ct.c_int(sync_channel_offset)),
    "SetSyncChannelOffset") # This can emulate a cable delay

# We use the same input settings for all channels, you can change this.
for i in range(0, num_channels.value):

    tryfunc(phlib.PH330_SetInputTrgMode(ct.c_int(dev_idx[0]), ct.c_int(i), input_trg_mode), 
            "SetInputTrgMode")

    if input_trg_mode == TRGMODE_ETR:
        tryfunc(phlib.PH330_SetInputEdgeTrg(ct.c_int(dev_idx[0]), ct.c_int(i), ct.c_int(input_trg_level),
                ct.c_int(input_trg_edge)), "SetInputEdgeTrg")
    
    if sync_trg_mode == TRGMODE_CFD:
        tryfunc(phlib.PH330_SetInputCFD(ct.c_int(dev_idx[0]), ct.c_int(i), ct.c_int(input_cfd_level),
                ct.c_int(input_cfd_zero_cross)), "SetInputCFD")   

    tryfunc(phlib.PH330_SetInputChannelOffset(ct.c_int(dev_idx[0]), ct.c_int(i),
            ct.c_int(input_channel_offset)), "SetInputChannelOffset") # This can emulate a cable delay
    
    tryfunc(phlib.PH330_SetInputChannelEnable(ct.c_int(dev_idx[0]), ct.c_int(i),
            ct.c_int(1)), "SetInputChannelEnable") # We enable all channels

# Meaningful only in T3 mode
if mode == MODE_T3:
    tryfunc(phlib.PH330_SetBinning(ct.c_int(dev_idx[0]), ct.c_int(binning)), "SetBinning")
    tryfunc(phlib.PH330_SetOffset(ct.c_int(dev_idx[0]), ct.c_int(offset)), "SetOffset")
    
tryfunc(phlib.PH330_GetResolution(ct.c_int(dev_idx[0]), byref(resolution)), "GetResolution")
print("Resolution is %1.1lfps" % resolution.value)

# After Initialize or SetSyncDiv you must allow >100 ms for valid  count rate readings
time.sleep(0.15)# in s

tryfunc(phlib.PH330_GetSyncRate(ct.c_int(dev_idx[0]), byref(sync_rate)), "GetSyncRate")
print("\nSyncrate=%1d/s" % sync_rate.value)

for i in range(0, num_channels.value):
    tryfunc(phlib.PH330_GetCountRate(ct.c_int(dev_idx[0]), ct.c_int(i), byref(count_rate)),
            "GetCountRate")
    print("Countrate[%1d]=%1d/s" % (i, count_rate.value))
    
# After getting the count rates you can check for warnings
tryfunc(phlib.PH330_GetWarnings(ct.c_int(dev_idx[0]), byref(warnings)), "GetWarnings")  
if warnings.value > 0:
    tryfunc(phlib.PH330_GetWarningsText(ct.c_int(dev_idx[0]), warnings_text, warnings), 
            "GetWarningsText")
    print("\n\n%s" % warnings_text.value.decode("utf-8")) 
    
if mode == MODE_T2:
    out_text_file.write("ev chn       time/ps\n\n") # column heading for T2 mode
else:
    out_text_file.write("ev chn     ttag/s      dtime/ps\n\n") # column heading for T3 mode

if sys.version_info[0] < 3:
    raw_input("\nPress RETURN to start"); print
else:
    input("\nPress RETURN to start"); print

tryfunc(phlib.PH330_StartMeas(ct.c_int(dev_idx[0]), ct.c_int(tacq)), "StartMeas")

if mode == MODE_T3:
    # We need the sync period in order to calculate the true times of photon records.
    # This only makes sense in T3 mode and it assumes a stable period like from a laser.
    # Note: Two sync periods must have elapsed after PH_StartMeas to get proper results.
    # You can also use the inverse of what you read via GetSyncRate but it depends on
    # the actual sync rate if this is accurate enough.
    # It is OK to use the sync input for a photon detector, e.g. if you want to perform
    # something like an antibunching measurement. In that case the sync rate obviously is
    # not periodic. This means that a) you should set the sync divider to 1 (none) and
    # b) that you cannot meaningfully measure the sync period here, which probaly won't
    # matter as you only care for the time difference(dtime) of the events.
    tryfunc(phlib.PH330_GetSyncPeriod(ct.c_int(dev_idx[0]), byref(sync_period)), "GetSyncPeriod")
    print("\nSync period is %lf ns\n" % (sync_period.value * 1e9))
    
print("\nStarting data collection...")    

progress = 0
sys.stdout.write("\nProgress:%9u" % progress)
sys.stdout.flush()

oflcorrection = 0

while True:
    tryfunc(phlib.PH330_GetFlags(ct.c_int(dev_idx[0]), byref(flags)), "GetFlags")
    
    if flags.value & FLAG_FIFOFULL > 0:
        print("\nFiFo Overrun!")
        stoptttr()
    
    tryfunc(phlib.PH330_ReadFiFo(ct.c_int(dev_idx[0]), byref(buffer), byref(records)),
            "ReadFiFo", meas_running=True)

    # Here we process the data. Note that the time this consumes prevents us
    # from getting around the loop quickly for the next Fifo read.
    # In a serious performance critical scenario you would write the data to
    # a software queue and do the processing in another thread reading from
    # that queue.
    if records.value > 0:        
        if mode == MODE_T2:
            for i in range(0,records.value):
                ProcessT2(buffer[i])                
        else:
            for i in range(0,records.value):
                ProcessT3(buffer[i])
            
        progress += records.value
        sys.stdout.write("\rProgress:%9u" % progress)
        sys.stdout.flush()
    else:
        tryfunc(phlib.PH330_CTCStatus(ct.c_int(dev_idx[0]), byref(ctc_status)),
                "CTCStatus")
        if ctc_status.value > 0:
            # Retry a few times to see if the fifo is really empty
            stop_retry = stop_retry + 1
            if stop_retry > 5: 
               print("\nDone")
               stoptttr()
               break
    # Within this loop you can also read the count rates if needed.
    # Do it sparingly and use PH330_GetAllCountRates for speed.

closeDevices()

