# Demo access to PicoHarp 330 via PH330Lib.dll or libph330.so v.2.0
# The program performs a measurement based on hardcoded settings.
# The resulting event data is stored in a binary output file.
							
# Stefan Eilers, Michael Wahl, PicoQuant, July 2024 
															
# Note: This is a console application
										   
# Note: At the API level the input channel numbers are indexed 0..N-1
# where N is the number of input channels the device has.

# Note: This demo writes only raw event data to the output file.
# It does not write a file header as regular .ptu files have it.

# Tested with the following compilers:
# - Python 3.11 (Windows)

import time
import ctypes as ct
from ctypes import byref
import os
import sys

if sys.version_info[0] < 3:
    print("[Warning] Python 2 is not fully supported. It might work, but "
          "use Python 3 if you encounter errors.\n")
    raw_input("Press RETURN to continue"); print

if os.name == "nt":
    phlib = ct.WinDLL("PH330Lib.dll")
else:
    phlib = ct.CDLL("libph330.so")


# Constants from ph330defin.h, do not edit these!

LIB_VERSION = "2.0"
MAXDEVNUM = 8
MODE_T2 = 2
MODE_T3 = 3
TTREADMAX = 1048576
FLAG_FIFOFULL = 0x0002
TRGMODE_ETR = 0 
TRGMODE_CFD = 1
EDGE_FALLING = 0
EDGE_RISING = 1

# Measurement parameters, hard coded here since this is just a demo

mode = MODE_T2 # Set T2 or T3 here, observe suitable Syncdivider and Range!
binning = 0 # You can change this, meaningful only in T3 mode
offset = 0 # You can change this, meaningful only in T3 mode, see manual
tacq = 1000 # Measurement time in millisec, you can change this
sync_divider = 1 # You can change this, observe mode, see manual.
sync_channel_offset = 0 # You can change this (like a cable delay)
input_channel_offset = 0  # You can change this (like a cable delay)
 
sync_trg_mode = TRGMODE_ETR # You can change this to TRGMODE_CFD  
# In case of SyncTrgMode is TRGMODE_ETR this will apply:
sync_trg_edge = EDGE_FALLING # You can change this to EDGE_RISING 
sync_trg_level = -50 # You can change this (in mV)         
# In case of SyncTrgMode is TRGMODE_CFD this will apply:
sync_cfd_zero_cross = -20 # In mV, you can change this
sync_cfd_level = -50 # In mV, you can change this

input_trg_mode = TRGMODE_ETR # You can change this to TRGMODE_CFD    
# In case of InputTrgMode is TRGMODE_ETR this will apply:
input_trg_edge = EDGE_FALLING # You can change this to EDGE_RISING
input_trg_level = -50 # In mV, you can change this    
# In case of InputTrgMode is TRGMODE_CFD this will apply:
input_cfd_zero_cross = -20 # In mV, you can change this
input_cfd_level = -50 # In mV, you can change this    
    
# Variables for flow contol and results from library calls

dev_idx = []
ctc_status = ct.c_int()
lib_version = ct.create_string_buffer(b"", 8) 
hw_model = ct.create_string_buffer(b"", 32)
hw_partno = ct.create_string_buffer(b"", 9)
hw_serial = ct.create_string_buffer(b"", 10)
hw_version = ct.create_string_buffer(b"", 16)
error_string = ct.create_string_buffer(b"", 40)
num_channels = ct.c_int()
resolution = ct.c_double() 
sync_rate = ct.c_int()
count_rate = ct.c_int()
flags = ct.c_int()
warnings = ct.c_int()
warningstext = ct.create_string_buffer(b"", 16384) 
records = ct.c_int()
progress = 0
stop_retry = 0
buffer = (ct.c_uint * TTREADMAX)()
warnings = ct.c_int()
warnings_text = ct.create_string_buffer(b"", 16384) 

# Subroutines

def closeDevices():
    out_file.close()
    for i in range(0, MAXDEVNUM):
        phlib.PH330_CloseDevice(ct.c_int(i))
    if sys.version_info[0] < 3:
        raw_input("\nPress RETURN to exit"); print
    else:
        input("\nPress RETURN to exit"); print
    sys.exit(0)

def stoptttr():
    retcode = phlib.PH330_StopMeas(ct.c_int(dev_idx[0]))
    if retcode < 0:
        print("PH330_StopMeas error %1d. Aborted." % retcode)
    closeDevices()

# Helper for showing error texts and shutdown in case of error
def tryfunc(retcode, funcName, measRunning=False):
    if retcode < 0:
        phlib.PH330_GetErrorString(error_string, ct.c_int(retcode))
        print("PH330_%s error %d (%s). Aborted." % (funcName, retcode,
              error_string.value.decode("utf-8")))
        if measRunning:
            stoptttr()
        else:
            closeDevices()


# The main demo code starts here     

print("\nPicoHarp 330 PHLib330 Demo Application                PicoQuant GmbH, 2024")
print("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~")
    
phlib.PH330_GetLibraryVersion(lib_version)
print("Library version is %s" % lib_version.value.decode("utf-8"))
if lib_version.value.decode("utf-8") != LIB_VERSION:
    print("Warning: The application was built for version %s" % LIB_VERSION)

out_file = open("tttrmode.out", "wb+")


print("\nSearching for Picoharp 330 devices...")
print("Devidx     Serial     Status")

for i in range(0, MAXDEVNUM):
    retcode = phlib.PH330_OpenDevice(ct.c_int(i), hw_serial)
    if retcode == 0:
        print("  %1d        %s    open ok" %  (i, hw_serial.value.decode("utf-8")))
        dev_idx.append(i)
    else:
        if retcode == -1: # PH330_ERROR_DEVICE_OPEN_FAIL
            print("  %1d                   no device" % i)
        else:
            phlib.PH330_GetErrorString(error_string, ct.c_int(retcode))
            print("  %1d        %s" % (i, error_string.value.decode("utf-8")))

# In this demo we will use the first device we find, i.e. dev_idx[0].
# You can also use multiple devices in parallel, see manual.
# You can also check for specific serial numbers, so that you always know 
# which physical device you are talking to.

if len(dev_idx) < 1:
    print("No device available.")
    closeDevices()
print("Using device #%1d" % dev_idx[0])
print("\nInitializing the device...")

# Initialize with chosen mode and internal clock
tryfunc(phlib.PH330_Initialize(ct.c_int(dev_idx[0]), ct.c_int(mode), ct.c_int(0)),
        "Initialize")

# Only for information
tryfunc(phlib.PH330_GetHardwareInfo(ct.c_int(dev_idx[0]), hw_model, hw_partno, hw_version),
        "GetHardwareInfo")
print("Found Model %s Part no %s Version %s" % (hw_model.value.decode("utf-8"),
      hw_partno.value.decode("utf-8"), hw_version.value.decode("utf-8")))

tryfunc(phlib.PH330_GetNumOfInputChannels(ct.c_int(dev_idx[0]), byref(num_channels)),
        "GetNumOfInputChannels")
print("Device has %i input channels." % num_channels.value)

tryfunc(phlib.PH330_SetSyncDiv(ct.c_int(dev_idx[0]), ct.c_int(sync_divider)), "SetSyncDiv")

tryfunc(phlib.PH330_SetSyncTrgMode(ct.c_int(dev_idx[0]), sync_trg_mode), "SetSyncTrgMode")

if sync_trg_mode == TRGMODE_ETR:
    tryfunc(phlib.PH330_SetSyncEdgeTrg(ct.c_int(dev_idx[0]), ct.c_int(sync_trg_level),
            ct.c_int(sync_trg_edge)), "SetSyncEdgeTrg")
    
if sync_trg_mode == TRGMODE_CFD:
    tryfunc(phlib.PH330_SetSyncCFD(ct.c_int(dev_idx[0]), ct.c_int(sync_cfd_level),
            ct.c_int(sync_cfd_zero_cross)), "SetSyncCFD")            

tryfunc(phlib.PH330_SetSyncChannelOffset(ct.c_int(dev_idx[0]), ct.c_int(sync_channel_offset)),
    "SetSyncChannelOffset") # This can emulate a cable delay

# We use the same input settings for all channels, you can change this.
for i in range(0, num_channels.value):

    tryfunc(phlib.PH330_SetInputTrgMode(ct.c_int(dev_idx[0]), ct.c_int(i), input_trg_mode), 
            "SetInputTrgMode")

    if input_trg_mode == TRGMODE_ETR:
        tryfunc(phlib.PH330_SetInputEdgeTrg(ct.c_int(dev_idx[0]), ct.c_int(i), ct.c_int(input_trg_level),
                ct.c_int(input_trg_edge)), "SetInputEdgeTrg")
    
    if sync_trg_mode == TRGMODE_CFD:
        tryfunc(phlib.PH330_SetInputCFD(ct.c_int(dev_idx[0]), ct.c_int(i), ct.c_int(input_cfd_level),
                ct.c_int(input_cfd_zero_cross)), "SetInputCFD")   

    tryfunc(phlib.PH330_SetInputChannelOffset(ct.c_int(dev_idx[0]), ct.c_int(i),
            ct.c_int(input_channel_offset)), "SetInputChannelOffset") # This can emulate a cable delay
    
    tryfunc(phlib.PH330_SetInputChannelEnable(ct.c_int(dev_idx[0]), ct.c_int(i),
            ct.c_int(1)), "SetInputChannelEnable") # We enable all channels

# Meaningful only in T3 mode
if mode == MODE_T3:
    tryfunc(phlib.PH330_SetBinning(ct.c_int(dev_idx[0]), ct.c_int(binning)), "SetBinning")
    tryfunc(phlib.PH330_SetOffset(ct.c_int(dev_idx[0]), ct.c_int(offset)), "SetOffset")
    
tryfunc(phlib.PH330_GetResolution(ct.c_int(dev_idx[0]), byref(resolution)), "GetResolution")
print("Resolution is %1.1lfps" % resolution.value)

# Note: after Init or SetSyncDiv you must allow >100 ms for valid  count rate readings
time.sleep(0.2)

tryfunc(phlib.PH330_GetSyncRate(ct.c_int(dev_idx[0]), byref(sync_rate)), "GetSyncRate")
print("\nSyncrate=%1d/s" % sync_rate.value)

for i in range(0, num_channels.value):
    tryfunc(phlib.PH330_GetCountRate(ct.c_int(dev_idx[0]), ct.c_int(i), byref(count_rate)),
            "GetCountRate")
    print("Countrate[%1d]=%1d/s" % (i, count_rate.value))

# After getting the count rates you can check for warnings
tryfunc(phlib.PH330_GetWarnings(ct.c_int(dev_idx[0]), byref(warnings)), "GetWarnings")
if warnings.value != 0:
    tryfunc(phlib.PH330_GetWarningsText(ct.c_int(dev_idx[0]), warnings_text, warnings),
            "GetWarningsText")
    print("\n\n%s" % warnings_text.value.decode("utf-8")) 

sys.stdout.write("\nProgress:%9u" % progress)
sys.stdout.flush()

tryfunc(phlib.PH330_StartMeas(ct.c_int(dev_idx[0]), ct.c_int(tacq)), "StartMeas")

while True:
    tryfunc(phlib.PH330_GetFlags(ct.c_int(dev_idx[0]), byref(flags)), "GetFlags")
    
    if flags.value & FLAG_FIFOFULL > 0:
        print("\nFiFo Overrun!")
        stoptttr()
    
    tryfunc(phlib.PH330_ReadFiFo(ct.c_int(dev_idx[0]), byref(buffer), byref(records)),
            "ReadFiFo", measRunning=True )

    # We could just iterate through our buffer with a for loop, however,
    # this is slow and might cause a FIFO overrun. So instead, we shrink
    # the buffer to its appropriate length with array slicing, which gives
    # us a python list. This list then needs to be converted back into
    # a ctype array which can be written at once to the output file
    if records.value > 0:
        out_file.write((ct.c_uint*records.value)(*buffer[0:records.value]))
        progress += records.value
        sys.stdout.write("\rProgress:%9u" % progress)
        sys.stdout.flush()
    else:
        tryfunc(phlib.PH330_CTCStatus(ct.c_int(dev_idx[0]), byref(ctc_status)),
                "CTCStatus")
        if ctc_status.value > 0: 
            # retry a few times to make sure the FIFO is really empty 
            stop_retry = stop_retry + 1 
            if stop_retry > 5: 
                print("\nDone")
                stoptttr()
                break            
    # Within this loop you can also read the count rates if needed.
    # Do it sparingly and use PH330_GetAllCountRates for speed.
    
closeDevices()
