/*********************************************************************
 * PicoQuant PH330 Rust Wrapper Demo
 *
 * This file demonstrates wrapping the PicoQuant PH330 DLL in Rust.
 * Each function from the C DLL is wrapped and provided with Rust data types.
 * The access to the C DLL functions is unsafe, and C data types are used
 * briefly before converting them to Rust code.
 *
 * For more information, please refer to:
 * https://doc.rust-lang.org/book/ch19-01-unsafe-rust.html
 *
 * Using the Rust approach, all functions return a Result type, which can be
 * either Ok or Err. The Ok variant contains the return value of the specific function.
 * This enforces error handling for any potential issues when calling the functions.
 *
 * For quick prototyping, you can use the unwrap() function, which either returns the
 * value from the function or panics. Here’s an example of how to use it:
 *
 * ph330_function(arg1, arg2).unwrap();
 *
 * This is a common method in Rust for writing prototyping code.
 *********************************************************************/

extern crate libc;

use libc::c_char;
use libc::c_double;
use libc::c_int;
use std::ffi::CStr;

use crate::error_codes::ErrorCode;
use std::convert::TryFrom;

// Declare the C function
extern "C" {
    fn PH330_GetLibraryVersion(vers: *mut c_char) -> i32;
    fn PH330_GetErrorString(errstring: *mut c_char, errcode: i32) -> i32;

    fn PH330_OpenDevice(devidx: i32, serial: *mut c_char) -> i32;
    fn PH330_CloseDevice(devidx: i32) -> i32;
    fn PH330_Initialize(devidx: i32, mode: i32, refsource: i32) -> i32;

    fn PH330_GetHardwareInfo(
        devidx: i32,
        model: *mut c_char,
        partno: *mut c_char,
        version: *mut c_char,
    ) -> i32;
    fn PH330_GetSerialNumber(devidx: i32, serial: *mut c_char) -> i32;
    fn PH330_GetFeatures(devidx: i32, features: *mut c_int) -> i32;
    fn PH330_GetBaseResolution(devidx: i32, resolution: *mut c_double, binsteps: *mut c_int)
        -> i32;
    fn PH330_GetNumOfInputChannels(devidx: i32, nchannels: *mut c_int) -> i32;

    fn PH330_SetSyncDiv(devidx: i32, div: i32) -> i32;
    fn PH330_SetSyncTrgMode(devidx: i32, mode: i32) -> i32;
    fn PH330_SetSyncEdgeTrg(devidx: i32, level: i32, edge: i32) -> i32;
    fn PH330_SetSyncCFD(devidx: i32, level: i32, zerocross: i32) -> i32;
    fn PH330_SetSyncChannelOffset(devidx: i32, offset: i32) -> i32;
    fn PH330_SetSyncChannelEnable(devidx: i32, enable: i32) -> i32;
    fn PH330_SetSyncDeadTime(devidx: i32, on: i32, deadtime: i32) -> i32;

    fn PH330_SetInputTrgMode(devidx: i32, channel: i32, mode: i32) -> i32;
    fn PH330_SetInputEdgeTrg(devidx: i32, channel: i32, level: i32, edge: i32) -> i32;
    fn PH330_SetInputCFD(devidx: i32, channel: i32, level: i32, zerocross: i32) -> i32;
    fn PH330_SetInputChannelOffset(devidx: i32, channel: i32, offset: i32) -> i32;
    fn PH330_SetInputDeadTime(devidx: i32, channel: i32, on: i32, deadtime: i32) -> i32;
    fn PH330_SetInputHysteresis(devidx: i32, hystcode: i32) -> i32;
    fn PH330_SetInputChannelEnable(devidx: i32, channel: i32, enable: i32) -> i32;

    fn PH330_SetStopOverflow(devidx: i32, stop_ovfl: i32, stopcount: u32) -> i32;
    fn PH330_SetBinning(devidx: i32, binning: i32) -> i32;
    fn PH330_SetOffset(devidx: i32, offset: i32) -> i32;
    fn PH330_SetHistoLen(devidx: i32, lencode: i32, actuallen: *mut i32) -> i32;
    fn PH330_SetMeasControl(devidx: i32, control: i32, startedge: i32, stopedge: i32) -> i32;
    fn PH330_SetTriggerOutput(devidx: i32, period: i32) -> i32;

    fn PH330_ClearHistMem(devidx: i32) -> i32;
    fn PH330_StartMeas(devidx: i32, tacq: i32) -> i32;
    fn PH330_StopMeas(devidx: i32) -> i32;
    fn PH330_CTCStatus(devidx: i32, ctcstatus: *mut i32) -> i32;

    fn PH330_GetHistogram(devidx: i32, chcount: *mut u32, channel: i32) -> i32;
    fn PH330_GetAllHistograms(devidx: i32, chcount: *mut u32) -> i32;
    fn PH330_GetResolution(devidx: i32, resolution: *mut f64) -> i32;
    fn PH330_GetSyncPeriod(devidx: i32, period: *mut f64) -> i32;
    fn PH330_GetSyncRate(devidx: i32, syncrate: *mut i32) -> i32;
    fn PH330_GetCountRate(devidx: i32, channel: i32, cntrate: *mut i32) -> i32;
    fn PH330_GetAllCountRates(devidx: i32, syncrate: *mut i32, cntrates: *mut i32) -> i32;
    fn PH330_GetFlags(devidx: i32, flags: *mut i32) -> i32;
    fn PH330_GetElapsedMeasTime(devidx: i32, elapsed: *mut f64) -> i32;

    fn PH330_GetWarnings(devidx: i32, warnings: *mut i32) -> i32;
    fn PH330_GetWarningsText(devidx: i32, text: *mut i8, warnings: i32) -> i32;

    fn PH330_SetOflCompression(devidx: i32, holdtime: i32) -> i32;
    fn PH330_SetMarkerHoldoffTime(devidx: i32, holdofftime: i32) -> i32;
    fn PH330_SetMarkerEdges(devidx: i32, me1: i32, me2: i32, me3: i32, me4: i32) -> i32;
    fn PH330_SetMarkerEnable(devidx: i32, en1: i32, en2: i32, en3: i32, en4: i32) -> i32;
    fn PH330_ReadFiFo(devidx: i32, buffer: *mut u32, nactual: *mut i32) -> i32;

    fn PH330_SetEventFilterParams(devidx: i32, timerange: i32, matchcnt: i32, inverse: i32) -> i32;
    fn PH330_SetEventFilterChannels(devidx: i32, usechannels: i32, passchannels: i32) -> i32;
    fn PH330_EnableEventFilter(devidx: i32, enable: i32) -> i32;
    fn PH330_SetFilterTestMode(devidx: i32, testmode: i32) -> i32;
    fn PH330_GetFilterInputRates(devidx: i32, syncrate: *mut i32, cntrates: *mut i32) -> i32;
    fn PH330_GetFilterOutputRates(devidx: i32, syncrate: *mut i32, cntrates: *mut i32) -> i32;

    fn PH330_GetDebugInfo(devidx: i32, debuginfo: *mut i8) -> i32;
    fn PH330_GetModuleInfo(devidx: i32, modelcode: *mut i32, versioncode: *mut i32) -> i32;
}

pub enum Mode {
    Hist = 0,
    T2 = 2,
    T3 = 3,
}

pub enum Refsource {
    Internal = 0,
    External10mhz = 1,
    External100mhz = 2,
    External500mhz = 3,
}

pub struct Ph330HardwareInfo {
    pub model: String,
    pub part_number: String,
    pub version: String,
}

pub enum MeasCtrl {
    MeasctrlSingleshotCtc = 0,
    MeasctrlC1Gated = 1,
    MeasctrlC1StartCtcStop = 2,
    MeasctrlC1StartC2Stop = 3,
    MeasctrlSWStartSwStop = 6,
}

pub enum Flags {
    FlagOverflow = 0x0001, // histo mode only
    FlagFifoFull = 0x0002, // TTTR mode only
    FlagSyncLost = 0x0004,
    FlagRefLost = 0x0008,
    FlagSysError = 0x0010,    // hardware error, must contact support
    FlagActive = 0x0020,      // measurement is running
    FlagCntsDropped = 0x0040, // counts were dropped
    FlagSoftError = 0x0080,   // software error, must contact support
}

//bitmasks for results from PH330_GetWarnings
pub enum Warnings {
    WarningSyncRateZero = 0x0001,
    WarningSyncRateVeryLow = 0x0002,
    WarningSyncRateTooHigh = 0x0004,
    WarningInptRateZero = 0x0010,
    WarningInptRateTooHigh = 0x0040,
    WarningInptRateRatio = 0x0100,
    WarningDividerGreaterOne = 0x0200,
    WarningTimeSpanTooSmall = 0x0400,
    WarningOffsetUnnecessary = 0x0800,
    WarningDividerTooSmall = 0x1000,
    WarningCountsDropped = 0x2000,
    WarningUsb20SpeedOnly = 0x4000,
}


pub enum Edge {
    Rising = 1,
    Falling = 0,
}

pub enum TriggerMode {
    Edge = 0, // edge trigger
    Cfd = 1,  // constant fraction discriminator
}

// limits
pub const MAXDEVNUM: i32 = 8; // max number of USB devices
pub const MAXINPCHAN: i32 = 2; // max number of physical input channels
pub const BINSTEPSMAX: i32 = 24; // max number of binning steps;
                                 // get actual number via PH330_GetBaseResolution()

pub const MAXHISTLEN: i32 = 524288; // max number of histogram bins
pub const DFLTHISTLEN: i32 = 65536; // default number of histogram bins

pub const TTREADMAX: i32 = 1048576; // number of event records that can be read by PH330_ReadFiFo
pub const TRGLVLMIN: i32 = -1500; // mV
pub const TRGLVLMAX: i32 = 1500; // mV

//limits for PH330_SetSyncCFD and PH330_SetInputCFD
pub const CFDLVLMIN: i32 = -1500; // mV
pub const CFDLVLMAX: i32 = 0; // mV
pub const CFDZCMIN: i32 = -100; // mV
pub const CFDZCMAX: i32 = 0; // mV                     // buffer must provide space for this number of dwords
                             //limits for PH330_SetSyncChannelOffset and PH330_SetInputChannelOffset
pub const CHANOFFSMIN: i32 = -99999; // ps
pub const CHANOFFSMAX: i32 = 99999; // ps

//limits for PH330_SetSyncDeadTime and PH330_SetInputDeadTime
pub const EXTDEADMIN: i32 = 800; // ps
pub const EXTDEADMAX: i32 = 160000; // ps

//limits for PH330_SetOffset
pub const OFFSETMIN: i32 = 0; // ns
pub const OFFSETMAX: i32 = 100000000; // ns

//limits for PH330_StartMeas
pub const ACQTMIN: i32 = 1; // ms
pub const ACQTMAX: i32 = 360000000; // ms  (100*60*60*1000ms = 100h)

//limits for PH330_SetStopOverflow
pub const STOPCNTMIN: i32 = 1;
pub const STOPCNTMAX: u32 = 4294967295; // 32 bit is mem max

//limits for PH330_SetTriggerOutput
pub const TRIGOUTMIN: i32 = 0; // 0=off
pub const TRIGOUTMAX: i32 = 16777215; // in units of 100ns

//limits for PH330_SetMarkerHoldoffTime
pub const HOLDOFFMIN: i32 = 0; // ns
pub const HOLDOFFMAX: i32 = 25500; // ns

//limits for PH330_SetInputHysteresis
pub const HYSTCODEMIN: i32 = 0; // approx. 3mV
pub const HYSTCODEMAX: i32 = 1; // approx. 35mV

//limits for PH330_SetOflCompression
pub const HOLDTIMEMIN: i32 = 0; // ms
pub const HOLDTIMEMAX: i32 = 255; // ms

//limits for PH330_SetEventFilterParams and PH330_SetEventFilterChannels
pub const MATCHCNTMIN: i32 = 1;
pub const MATCHCNTMAX: i32 = 6;
pub const INVERSEMIN: i32 = 0;
pub const INVERSEMAX: i32 = 1;
pub const TIMERANGEMIN: i32 = 0; // ps
pub const TIMERANGEMAX: i32 = 160000; // ps
pub const USECHANSMIN: i32 = 0x000; // no channels used
pub const USECHANSMAX: i32 = 0x10F; // note: sync bit 0x100 will be ignored in T3 mode
pub const PASSCHANSMIN: i32 = 0x000; // no channels passed
pub const PASSCHANSMAX: i32 = 0x10F; // note: sync bit 0x100 will be ignored in T3 mode


pub fn ph330_get_library_version() -> Result<String, ErrorCode> {
    // Define the size of the buffer
    let buffer_size = 10;
    // Allocate a buffer of the defined size
    let mut buffer: Vec<c_char> = vec![0; buffer_size];

    unsafe {
        let ret_code = PH330_GetLibraryVersion(buffer.as_mut_ptr());
        match ret_code {
            0 => {
                // Convert the buffer to a Rust string
                let c_str = CStr::from_ptr(buffer.as_ptr());
                let r_str = c_str.to_str().expect("Invalid UTF-8 string");
                Ok(r_str.to_string())
            }
            _ => {
                match ErrorCode::try_from(ret_code) {
                    Ok(error_code) => Err(error_code),
                    Err(_) => Err(ErrorCode::InvalidArgument), // cannot parse error, should not happen
                }
            }
        }
    }
}

pub fn ph330_get_error_string(errcode: i32) -> Result<String, ErrorCode> {
    // Define the size of the buffer
    let buffer_size = 256;
    // Allocate a buffer of the defined size
    let mut buffer: Vec<c_char> = vec![0; buffer_size];
    unsafe {
        let ret_code = PH330_GetErrorString(buffer.as_mut_ptr(), errcode);
        match ret_code {
            0 => {
                let c_str = CStr::from_ptr(buffer.as_ptr());
                let r_str = c_str.to_str().expect("Invalid UTF-8 string");
                Ok(r_str.to_string())
            }
            _ => {
                match ErrorCode::try_from(ret_code) {
                    Ok(error_code) => Err(error_code),
                    Err(_) => Err(ErrorCode::InvalidArgument), // cannot parse error, should not happen
                }
            }
        }
    }
}

pub fn ph330_open_device(devidx: i32) -> Result<String, ErrorCode> {
    // Define the size of the buffer
    let buffer_size = 256;
    // Allocate a buffer of the defined size
    let mut buffer: Vec<c_char> = vec![0; buffer_size];
    unsafe {
        let ret_code = PH330_OpenDevice(devidx, buffer.as_mut_ptr());
        match ret_code {
            0 => {
                let c_str = CStr::from_ptr(buffer.as_ptr());
                let r_str = c_str.to_str().expect("Invalid UTF-8 string");
                Ok(r_str.to_string())
            }
            _ => {
                match ErrorCode::try_from(ret_code) {
                    Ok(error_code) => Err(error_code),
                    Err(_) => Err(ErrorCode::InvalidArgument), // cannot parse error, should not happen
                }
            }
        }
    }
}

pub fn ph330_close_device(devidx: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_CloseDevice(devidx);
        match ret_code {
            0 => Ok(()),
            _ => {
                match ErrorCode::try_from(ret_code) {
                    Ok(error_code) => Err(error_code),
                    Err(_) => Err(ErrorCode::InvalidArgument), // cannot parse error, should not happen
                }
            }
        }
    }
}

pub fn ph330_initialize(devidx: i32, mode: Mode, ref_source: Refsource) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_Initialize(devidx, mode as i32, ref_source as i32);
        match ret_code {
            0 => Ok(()),
            _ => {
                match ErrorCode::try_from(ret_code) {
                    Ok(error_code) => Err(error_code),
                    Err(_) => Err(ErrorCode::InvalidArgument), // cannot parse error, should not happen
                }
            }
        }
    }
}

pub fn ph330_get_hardware_info(devidx: i32) -> Result<Ph330HardwareInfo, ErrorCode> {
    let buffer_size = 256;
    let mut info = Ph330HardwareInfo {
        model: String::new(),
        part_number: String::new(),
        version: String::new(),
    };
    let mut buffers: Vec<Vec<c_char>> = vec![vec![0; buffer_size]; 3]; // initialize inner vectors

    unsafe {
        let ret_code = PH330_GetHardwareInfo(
            devidx,
            buffers[0].as_mut_ptr(),
            buffers[1].as_mut_ptr(),
            buffers[2].as_mut_ptr(),
        );
        match ret_code {
            0 => {
                for (i, buffer) in buffers.iter().enumerate() {
                    let c_str = CStr::from_ptr(buffer.as_ptr());
                    let r_str = c_str.to_str().expect("Invalid UTF-8 string");

                    match i {
                        0 => info.model = r_str.to_string(),
                        1 => info.part_number = r_str.to_string(),
                        2 => info.version = r_str.to_string(),
                        _ => unreachable!(),
                    }
                }
                Ok(info)
            }
            _ => {
                match ErrorCode::try_from(ret_code) {
                    Ok(error_code) => Err(error_code),
                    Err(_) => Err(ErrorCode::InvalidArgument), // cannot parse error, should not happen
                }
            }
        }
    }
}

pub fn ph330_get_serial_number(devidx: i32) -> Result<String, ErrorCode> {
    let buffer_size = 256;
    let mut buffer: Vec<c_char> = vec![0; buffer_size];
    unsafe {
        let ret_code = PH330_GetSerialNumber(devidx, buffer.as_mut_ptr());
        match ret_code {
            0 => {
                let c_str = CStr::from_ptr(buffer.as_ptr());
                let r_str = c_str.to_str().expect("Invalid UTF-8 string");
                Ok(r_str.to_string())
            }
            _ => {
                match ErrorCode::try_from(ret_code) {
                    Ok(error_code) => Err(error_code),
                    Err(_) => Err(ErrorCode::InvalidArgument), // cannot parse error, should not happen
                }
            }
        }
    }
}

pub fn ph330_get_features(devidx: i32) -> Result<i32, ErrorCode> {
    let mut buffer_features: i32 = 0;
    unsafe {
        let ret_code = PH330_GetFeatures(devidx, &mut buffer_features as *mut i32);
        match ret_code {
            0 => Ok(buffer_features),
            _ => {
                match ErrorCode::try_from(ret_code) {
                    Ok(error_code) => Err(error_code),
                    Err(_) => Err(ErrorCode::InvalidArgument), // cannot parse error, should not happen
                }
            }
        }
    }
}

pub fn ph330_get_base_resolution(devidx: i32) -> Result<(i32, f64), ErrorCode> {
    let mut buffer_res: f64 = 0.0;
    let mut buffer_steps: i32 = 0;

    unsafe {
        let ret_code = PH330_GetBaseResolution(
            devidx,
            &mut buffer_res as *mut f64,
            &mut buffer_steps as *mut i32,
        );
        match ret_code {
            0 => Ok((buffer_steps, buffer_res)),
            _ => {
                match ErrorCode::try_from(ret_code) {
                    Ok(error_code) => Err(error_code),
                    Err(_) => Err(ErrorCode::InvalidArgument), // cannot parse error, should not happen
                }
            }
        }
    }
}

pub fn ph330_get_number_of_input_channels(devidx: i32) -> Result<i32, ErrorCode> {
    let mut nchannels: i32 = 0;
    unsafe {
        let ret_code = PH330_GetNumOfInputChannels(devidx, &mut nchannels as *mut i32);
        match ret_code {
            0 => Ok(nchannels),
            _ => {
                match ErrorCode::try_from(ret_code) {
                    Ok(error_code) => Err(error_code),
                    Err(_) => Err(ErrorCode::InvalidArgument), // cannot parse error, should not happen
                }
            }
        }
    }
}

pub fn ph330_set_sync_divider(devidx: i32, divider: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_SetSyncDiv(devidx, divider);
        match ret_code {
            0 => Ok(()),
            _ => {
                match ErrorCode::try_from(ret_code) {
                    Ok(error_code) => Err(error_code),
                    Err(_) => Err(ErrorCode::InvalidArgument), // cannot parse error, should not happen
                }
            }
        }
    }
}

pub fn ph330_set_sync_trigger_mode(devidx: i32, mode: TriggerMode) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_SetSyncTrgMode(devidx, mode as i32);
        match ret_code {
            0 => Ok(()),
            _ => {
                match ErrorCode::try_from(ret_code) {
                    Ok(error_code) => Err(error_code),
                    Err(_) => Err(ErrorCode::InvalidArgument), // cannot parse error, should not happen
                }
            }
        }
    }
}

pub fn ph330_set_sync_edge_trigger(devidx: i32, level: i32, edge: Edge) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_SetSyncEdgeTrg(devidx, level, edge as i32);
        match ret_code {
            0 => Ok(()),
            _ => {
                match ErrorCode::try_from(ret_code) {
                    Ok(error_code) => Err(error_code),
                    Err(_) => Err(ErrorCode::InvalidArgument), // cannot parse error, should not happen
                }
            }
        }
    }
}

pub fn ph330_get_sync_rate(devidx: i32) -> Result<i32, ErrorCode> {
    let mut syncrate: i32 = 0;
    unsafe {
        let ret_code = PH330_GetSyncRate(devidx, &mut syncrate);
        match ret_code {
            0 => Ok(syncrate),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_get_elapsed_meas_time(devidx: i32) -> Result<f64, ErrorCode> {
    let mut elapsed: f64 = 0.0;
    unsafe {
        let ret_code = PH330_GetElapsedMeasTime(devidx, &mut elapsed);
        match ret_code {
            0 => Ok(elapsed),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_get_warnings(devidx: i32) -> Result<i32, ErrorCode> {
    let mut warnings: i32 = 0;
    unsafe {
        let ret_code = PH330_GetWarnings(devidx, &mut warnings);
        match ret_code {
            0 => Ok(warnings),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_set_sync_cfd(devidx: i32, level: i32, zerocross: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_SetSyncCFD(devidx, level, zerocross);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_set_sync_channel_offset(devidx: i32, offset: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_SetSyncChannelOffset(devidx, offset);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_set_sync_channel_enable(devidx: i32, enable: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_SetSyncChannelEnable(devidx, enable);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_set_sync_dead_time(devidx: i32, on: i32, deadtime: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_SetSyncDeadTime(devidx, on, deadtime);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_set_input_trigger_mode(
    devidx: i32,
    channel: i32,
    mode: TriggerMode,
) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_SetInputTrgMode(devidx, channel, mode as i32);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_set_input_edge_trigger(
    devidx: i32,
    channel: i32,
    level: i32,
    edge: Edge,
) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_SetInputEdgeTrg(devidx, channel, level, edge as i32);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_set_input_cfd(
    devidx: i32,
    channel: i32,
    level: i32,
    zerocross: i32,
) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_SetInputCFD(devidx, channel, level, zerocross);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_set_input_channel_offset(
    devidx: i32,
    channel: i32,
    offset: i32,
) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_SetInputChannelOffset(devidx, channel, offset);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_set_input_dead_time(
    devidx: i32,
    channel: i32,
    on: i32,
    deadtime: i32,
) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_SetInputDeadTime(devidx, channel, on, deadtime);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_set_input_hysteresis(devidx: i32, hystcode: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_SetInputHysteresis(devidx, hystcode);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_set_input_channel_enable(
    devidx: i32,
    channel: i32,
    enable: i32,
) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_SetInputChannelEnable(devidx, channel, enable);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_set_stop_overflow(
    devidx: i32,
    stop_ovfl: i32,
    stopcount: u32,
) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_SetStopOverflow(devidx, stop_ovfl, stopcount);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_set_binning(devidx: i32, binning: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_SetBinning(devidx, binning);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_set_offset(devidx: i32, offset: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_SetOffset(devidx, offset);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_set_histo_len(devidx: i32, lencode: i32) -> Result<i32, ErrorCode> {
    let mut actuallen: i32 = 0;
    unsafe {
        let ret_code = PH330_SetHistoLen(devidx, lencode, &mut actuallen);
        match ret_code {
            0 => Ok(actuallen),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_set_meas_control(
    devidx: i32,
    control: MeasCtrl,
    startedge: Edge,
    stopedge: Edge,
) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code =
            PH330_SetMeasControl(devidx, control as i32, startedge as i32, stopedge as i32);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_set_trigger_output(devidx: i32, period: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_SetTriggerOutput(devidx, period);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_clear_hist_mem(devidx: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_ClearHistMem(devidx);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_start_meas(devidx: i32, tacq: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_StartMeas(devidx, tacq);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_stop_meas(devidx: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_StopMeas(devidx);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_get_ctc_status(devidx: i32) -> Result<i32, ErrorCode> {
    let mut ctcstatus: i32 = 0;
    unsafe {
        let ret_code = PH330_CTCStatus(devidx, &mut ctcstatus);
        match ret_code {
            0 => Ok(ctcstatus),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_get_histogram(devidx: i32, channel: i32) -> Result<u32, ErrorCode> {
    let mut chcount: u32 = 0;
    unsafe {
        let ret_code = PH330_GetHistogram(devidx, &mut chcount, channel);
        match ret_code {
            0 => Ok(chcount),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_get_all_histograms(devidx: i32) -> Result<Vec<u32>, ErrorCode> {
    let ret_code = ph330_get_number_of_input_channels(devidx);
    match ret_code {
        Ok(number_of_input_channels) => {
            let mut chcount: Vec<u32> = vec![0; number_of_input_channels as usize]; // Ensure the correct type and size
            unsafe {
                let ret_code = PH330_GetAllHistograms(devidx, chcount.as_mut_ptr());
                match ret_code {
                    0 => Ok(chcount),
                    _ => match ErrorCode::try_from(ret_code) {
                        Ok(error_code) => Err(error_code),
                        Err(_) => Err(ErrorCode::InvalidArgument),
                    },
                }
            }
        }
        Err(error_code) => Err(error_code),
    }
}

pub fn ph330_get_resolution(devidx: i32) -> Result<f64, ErrorCode> {
    let mut resolution: f64 = 0.0;
    unsafe {
        let ret_code = PH330_GetResolution(devidx, &mut resolution);
        match ret_code {
            0 => Ok(resolution),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_get_sync_period(devidx: i32) -> Result<f64, ErrorCode> {
    let mut period: f64 = 0.0;
    unsafe {
        let ret_code = PH330_GetSyncPeriod(devidx, &mut period);
        match ret_code {
            0 => Ok(period),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_get_count_rate(devidx: i32, channel: i32) -> Result<i32, ErrorCode> {
    let mut cntrate: i32 = 0;
    unsafe {
        let ret_code = PH330_GetCountRate(devidx, channel, &mut cntrate);
        match ret_code {
            0 => Ok(cntrate),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_get_all_count_rates(devidx: i32) -> Result<(i32, Vec<i32>), ErrorCode> {
    let mut syncrate: i32 = 0;
    let ret = ph330_get_number_of_input_channels(devidx);

    match ret {
        Ok(number_of_channels) => {
            let mut cntrates: Vec<i32> = vec![0; number_of_channels as usize]; // Corrected to use `number_of_channels`
            unsafe {
                let ret_code = PH330_GetAllCountRates(devidx, &mut syncrate, cntrates.as_mut_ptr());
                match ret_code {
                    0 => Ok((syncrate, cntrates)),
                    _ => match ErrorCode::try_from(ret_code) {
                        Ok(error_code) => Err(error_code),
                        Err(_) => Err(ErrorCode::InvalidArgument),
                    },
                }
            }
        }
        Err(error_code) => Err(error_code), // Return the specific error code
    }
}

pub fn ph330_get_flags(devidx: i32) -> Result<i32, ErrorCode> {
    let mut flags: i32 = 0;
    unsafe {
        let ret_code = PH330_GetFlags(devidx, &mut flags);
        match ret_code {
            0 => Ok(flags),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_get_warnings_text(devidx: i32, warnings: i32) -> Result<String, ErrorCode> {
    let mut buffer: Vec<c_char> = vec![0; 256];
    unsafe {
        let ret_code = PH330_GetWarningsText(devidx, buffer.as_mut_ptr(), warnings);
        match ret_code {
            0 => {
                let c_str = CStr::from_ptr(buffer.as_ptr());
                let r_str = c_str.to_str().expect("Invalid UTF-8 string");
                Ok(r_str.to_string())
            }
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_set_ofl_compression(devidx: i32, holdtime: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_SetOflCompression(devidx, holdtime);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_set_marker_holdoff_time(devidx: i32, holdofftime: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_SetMarkerHoldoffTime(devidx, holdofftime);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_set_marker_edges(
    devidx: i32,
    me1: Edge,
    me2: Edge,
    me3: Edge,
    me4: Edge,
) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_SetMarkerEdges(devidx, me1 as i32, me2 as i32, me3 as i32, me4 as i32);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_set_marker_enable(
    devidx: i32,
    en1: i32,
    en2: i32,
    en3: i32,
    en4: i32,
) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_SetMarkerEnable(devidx, en1, en2, en3, en4);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_read_fifo(devidx: i32) -> Result<Vec<u32>, ErrorCode> {
    let mut nactual: i32 = 0;
    let mut buffer: Vec<u32> = Vec::with_capacity(TTREADMAX as usize);
    unsafe {
        let ret_code = PH330_ReadFiFo(devidx, buffer.as_mut_ptr(), &mut nactual);
        match ret_code {
            0 => {
                buffer.set_len(nactual as usize);
                Ok(buffer)
            }
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_set_event_filter_params(
    devidx: i32,
    timerange: i32,
    matchcnt: i32,
    inverse: i32,
) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_SetEventFilterParams(devidx, timerange, matchcnt, inverse);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_set_event_filter_channels(
    devidx: i32,
    usechannels: i32,
    passchannels: i32,
) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_SetEventFilterChannels(devidx, usechannels, passchannels);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_enable_event_filter(devidx: i32, enable: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_EnableEventFilter(devidx, enable);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_set_filter_test_mode(devidx: i32, testmode: i32) -> Result<(), ErrorCode> {
    unsafe {
        let ret_code = PH330_SetFilterTestMode(devidx, testmode);
        match ret_code {
            0 => Ok(()),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_get_filter_input_rates(devidx: i32) -> Result<(i32, Vec<i32>), ErrorCode> {
    let mut syncrate: i32 = 0;
    let ret_code = ph330_get_number_of_input_channels(devidx);

    match ret_code {
        Ok(number_of_channels) => {
            let mut cntrates: Vec<i32> = vec![0; number_of_channels as usize];
            unsafe {
                let ret_code =
                    PH330_GetFilterInputRates(devidx, &mut syncrate, cntrates.as_mut_ptr());
                match ret_code {
                    0 => Ok((syncrate, cntrates)),
                    _ => match ErrorCode::try_from(ret_code) {
                        Ok(error_code) => Err(error_code),
                        Err(_) => Err(ErrorCode::InvalidArgument),
                    },
                }
            }
        }
        Err(error_code) => Err(error_code),
    }
}

pub fn ph330_get_filter_output_rates(devidx: i32) -> Result<(i32, Vec<i32>), ErrorCode> {
    let mut syncrate: i32 = 0;
    let ret_code = ph330_get_number_of_input_channels(devidx);

    match ret_code {
        Ok(number_of_channels) => {
            let mut cntrates: Vec<i32> = vec![0; number_of_channels as usize];
            unsafe {
                let ret_code =
                    PH330_GetFilterOutputRates(devidx, &mut syncrate, cntrates.as_mut_ptr());
                match ret_code {
                    0 => Ok((syncrate, cntrates)),
                    _ => match ErrorCode::try_from(ret_code) {
                        Ok(error_code) => Err(error_code),
                        Err(_) => Err(ErrorCode::InvalidArgument),
                    },
                }
            }
        }
        Err(error_code) => Err(error_code),
    }
}

pub fn ph330_get_debug_info(devidx: i32) -> Result<String, ErrorCode> {
    let mut buffer: Vec<c_char> = vec![0; 256];
    unsafe {
        let ret_code = PH330_GetDebugInfo(devidx, buffer.as_mut_ptr());
        match ret_code {
            0 => {
                let c_str = CStr::from_ptr(buffer.as_ptr());
                let r_str = c_str.to_str().expect("Invalid UTF-8 string");
                Ok(r_str.to_string())
            }
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}

pub fn ph330_get_module_info(devidx: i32) -> Result<(i32, i32), ErrorCode> {
    let mut modelcode: i32 = 0;
    let mut versioncode: i32 = 0;
    unsafe {
        let ret_code = PH330_GetModuleInfo(devidx, &mut modelcode, &mut versioncode);
        match ret_code {
            0 => Ok((modelcode, versioncode)),
            _ => match ErrorCode::try_from(ret_code) {
                Ok(error_code) => Err(error_code),
                Err(_) => Err(ErrorCode::InvalidArgument),
            },
        }
    }
}
