/************************************************************************

  PicoQuant Unified TTTR (PTU)    File Access Demo in C/C++

  This is demo code. Use at your own risk. No warranties.

  Tested with MS Visual Studio 2015 and Mingw 4.5, GCC 5.4.0 (Linux)

  Marcus Sackrow, PicoQuant GmbH, December 2013
  Michael Wahl, PicoQuant GmbH, revised July 2014


************************************************************************/

#include  <stdio.h>
#include  <stddef.h>
#include  <stdlib.h>
#include  <time.h>
#include  <string.h>
#include  <stdint.h>
#include  <wchar.h>

// some important Tag Idents (TTagHead.Ident) that we will need to read the most common content of a PTU file
// check the output of this program and consult the tag dictionary if you need more
#define TTTRTagTTTRRecType "TTResultFormat_TTTRRecType"
#define TTTRTagNumRecords  "TTResult_NumberOfRecords"  // Number of TTTR Records in the File;
#define FileTagEnd         "Header_End"                // Always appended as last tag (BLOCKEND)
// Imaging tags
#define TTTRTagImgIdent     "ImgHdr_Ident"     // Which scanner was used for measurement
#define TTTRTagImgPixX      "ImgHdr_PixX"      // Number of Pixels in x direction
#define TTTRTagImgPixY      "ImgHdr_PixY"      // Number of Pixels in y direction
#define TTTRTagImgLineStart "ImgHdr_LineStart" // Marker for the line start
#define TTTRTagImgLineStop  "ImgHdr_LineStop"  // Marker for the line stop
#define TTTRTagImgFrame     "ImgHdr_Frame"     // Marker for the frame change
#define TTTRTagImgBidirect  "ImgHdr_BiDirect"  // If the Image is recorded in bidirectional mode
#define TTTRTagImgPixRes    "ImgHdr_PixResol"  // Pixel Resolution in µm

// TagTypes  (TTagHead.Typ)
#define tyEmpty8      0xFFFF0008
#define tyBool8       0x00000008
#define tyInt8        0x10000008
#define tyBitSet64    0x11000008
#define tyColor8      0x12000008
#define tyFloat8      0x20000008
#define tyTDateTime   0x21000008
#define tyFloat8Array 0x2001FFFF
#define tyAnsiString  0x4001FFFF
#define tyWideString  0x4002FFFF
#define tyBinaryBlob  0xFFFFFFFF

// RecordTypes
#define rtPicoHarpT3     0x00010303    // (SubID = $00 ,RecFmt: $01) (V1), T-Mode: $03 (T3), HW: $03 (PicoHarp)
#define rtPicoHarpT2     0x00010203    // (SubID = $00 ,RecFmt: $01) (V1), T-Mode: $02 (T2), HW: $03 (PicoHarp)
#define rtHydraHarpT3    0x00010304    // (SubID = $00 ,RecFmt: $01) (V1), T-Mode: $03 (T3), HW: $04 (HydraHarp)
#define rtHydraHarpT2    0x00010204    // (SubID = $00 ,RecFmt: $01) (V1), T-Mode: $02 (T2), HW: $04 (HydraHarp)
#define rtHydraHarp2T3   0x01010304    // (SubID = $01 ,RecFmt: $01) (V2), T-Mode: $03 (T3), HW: $04 (HydraHarp)
#define rtHydraHarp2T2   0x01010204    // (SubID = $01 ,RecFmt: $01) (V2), T-Mode: $02 (T2), HW: $04 (HydraHarp)
#define rtTimeHarp260NT3 0x00010305    // (SubID = $00 ,RecFmt: $01) (V2), T-Mode: $03 (T3), HW: $05 (TimeHarp260N)
#define rtTimeHarp260NT2 0x00010205    // (SubID = $00 ,RecFmt: $01) (V2), T-Mode: $02 (T2), HW: $05 (TimeHarp260N)
#define rtTimeHarp260PT3 0x00010306    // (SubID = $00 ,RecFmt: $01) (V1), T-Mode: $02 (T3), HW: $06 (TimeHarp260P)
#define rtTimeHarp260PT2 0x00010206    // (SubID = $00 ,RecFmt: $01) (V1), T-Mode: $02 (T2), HW: $06 (TimeHarp260P)
#define rtMultiHarpT3    0x00010307    // (SubID = $00 ,RecFmt: $01) (V1), T-Mode: $02 (T3), HW: $07 (MultiHarp)
#define rtMultiHarpT2    0x00010207    // (SubID = $00 ,RecFmt: $01) (V1), T-Mode: $02 (T2), HW: $07 (MultiHarp)

#pragma pack(8) //structure alignment to 8 byte boundaries

// A Tag entry
struct TgHd{
  char Ident[32];     // Identifier of the tag
  int Idx;            // Index for multiple tags or -1
  unsigned int Typ;  // Type of tag ty..... see const section
    long long TagValue; // Value of tag.
} TagHead;

 enum TLinePos {lpInBetween, lpInLine} LinePos;

// TDateTime (in file) to time_t (standard C) conversion

const int EpochDiff = 25569; // days between 30/12/1899 and 01/01/1970
const int SecsInDay = 86400; // number of seconds in a day

time_t TDateTime_TimeT(double Convertee)
{
  time_t Result((long)(((Convertee) - EpochDiff) * SecsInDay));
  return Result;
}

FILE *fpin,*fpout;
long long RecNum;
long long oflcorrection = 0;
long long truensync, truetime;
int m, c;
unsigned int dlen = 0;
unsigned int cnt_0=0, cnt_1=0;

// Imaging
int ScannerIdent;                         // Which scanner was used
int StartMarker, StopMarker, FrameMarker; // Marker for image creation
int PixX, PixY;                           // Size of the image
bool IsBidir;                             // Bidirectional scanning?
double PixRes;                            // Size of a single pixel in µm (x and y direction)

//Image
long long *image;

// temporary space for a line of photons
typedef struct
{
  long long TimeTag;
  int Channel;
  int DTime;
} TPhoton;
TPhoton *TempPhotons;
int CurTemp;
int TempSize;

long long StartPos;
int CurLine;

//Got Photon
//  TimeTag: Raw TimeTag from Record, arrival time of Photon
//  DTime: Arrival time of Photon after last Sync event (T3 only)
//  Channel: Channel the Photon arrived (0 = Sync channel for T2 measurements)
void GotPhoton(long long TimeTag, int Channel, int DTime)
{
  // Only care about photons after Start before Stop marker, all other Photons are ignored
  if (LinePos == lpInLine)
  {
    // Enough space in the temp area, if not extend it
	  if (CurTemp >= TempSize)
	  {
      TempSize = CurTemp + 10000;
      TempPhotons = (TPhoton *)realloc(TempPhotons, TempSize * sizeof(TPhoton));      
	  }
    // we only save the photon to a temp area we need to wait for the line end marker
    // to determine its pixel position
    TempPhotons[CurTemp].TimeTag = TimeTag; // We only need the TimeTag
    TempPhotons[CurTemp].Channel = Channel; // but for further analysis we copy all
    TempPhotons[CurTemp].DTime = DTime;     // Values to the temp
    //
    CurTemp++;
  }
}

//Got Marker
//  TimeTag: Raw TimeTag from Record, arrival time of marker
//  Markers: Bit field of arrived Markers, different markers can arrive at same time (same record)
void GotMarker(long long TimeTag, int Markers)
{
  // Attention: the markers can arrive in the very same record,
  // especially the Stop Line marker and Frame marker often come together
  // Therefore the ordering of marker checking is important

  // Check if the Line Stop arrived, only make sense inside a line -> else just ignore it
  if ((((1 << StopMarker) & Markers) != 0) && (LinePos = lpInLine))
  {
  	// do the processing!
    double Dist = TimeTag - StartPos; // calculate the Distance between start and end marker
	  double SizePix;
	  if (ScannerIdent == 1)
	    SizePix = Dist / (PixX + 1); // for E710 the marker is one Pixel off
	  else
      SizePix = Dist / PixX;
	  double NextPixel = StartPos + SizePix; // Time Tag of the Next Pixel
    int PixelNum = 0; // Current pixel to fill    
    // main sorting routine
    for (int i = 0;  i < CurTemp; i++)
    {      
	    // calculate the pixel position
	    while (TempPhotons[i].TimeTag >= NextPixel)
      {
        PixelNum++; // next pixel
        NextPixel = NextPixel + SizePix; // start of next pixel
      }
	    // make sure we are inside the image array
      if ((CurLine >= PixY) || (PixelNum >= PixX))
        break;
	    int CurPixel;
	    // Correct pixel number for bidirectional scan if needed
      if ((IsBidir) && ((CurLine % 2) == 1))
        CurPixel = PixX - PixelNum - 1;         // revert every 2nd line if bidirectional scan
      else
        CurPixel = PixelNum;
	    // One could here filter by channel or process the photon data further more,
      // the pixel position is CurPixel = x, Curline = y and the data is in
      // TempPhotons[i].Channel and TempPhotons[i].DTime
      //
      // As demonstration we count the photons in the pixel to get an intensity image
      image[(CurLine * PixX) + CurPixel]++;
	  }
	  // the line is finished -> Next line
	  CurLine++;
	  // the line is ended -> we are outside the line again
    LinePos = lpInBetween;
  }
    // Check if the FrameMarker arrived
  if ((FrameMarker >= 0) && (((1 << FrameMarker) & Markers) != 0))
  {
    // a frame marker just restart from the beginning, so we reset the line position
    CurLine = 0;
    LinePos = lpInBetween;
  }
  
  // Check for the Start Marker, only make sense outside a line -> else just ignore it
  if ((((1 << StartMarker) & Markers) != 0) && (LinePos == lpInBetween))
  {
    // a Line Starts here
    LinePos = lpInLine; // inside a line from now on
    CurTemp = 0;        // Reset Temp area
    StartPos = TimeTag; // remember start for the calculation
  }
}

//Got Overflow
//  Count: Some TCSPC provide Overflow compression = if no Photons between overflow you get one record for multiple Overflows
void GotOverflow(int Count)
{
  // nothing to do here
}

// PicoHarp T3 input
void ProcessPHT3(unsigned int TTTRRecord)
{
  const int T3WRAPAROUND = 65536;
  union
  {
    unsigned int allbits;
    struct
    {
    unsigned numsync  :16;
    unsigned dtime    :12;
    unsigned channel  :4;
    } bits;
    struct
    {
    unsigned numsync  :16;
    unsigned markers  :12;
    unsigned channel  :4;
    } special;
  } Record;

  Record.allbits = TTTRRecord;
    if(Record.bits.channel==0xF) //this means we have a special record
  {
    if(Record.special.markers==0) //not a marker means overflow
    {
      GotOverflow(1);
      oflcorrection += T3WRAPAROUND; // unwrap the time tag overflow
    }
    else
    {
      truensync = oflcorrection + Record.bits.numsync;
      m = Record.special.markers;
      GotMarker(truensync, m);
    }
  } else
  {
    if(
        (Record.bits.channel==0) //Should never occur in T3 Mode
      ||(Record.bits.channel>4) //Should not occur with current routers
      )
    {
      printf("\nIllegal Channel: #%1d %1u",dlen,Record.bits.channel);
    }

    truensync = oflcorrection + Record.bits.numsync;
    m = Record.bits.dtime;
    c = Record.bits.channel;
    GotPhoton(truensync, c, m);
    dlen++;
  }
};

// PicoHarp T2 input
void ProcessPHT2(unsigned int TTTRRecord)
{
  const int T2WRAPAROUND = 210698240;
  union
  {
    unsigned int allbits;
    struct
    {
    unsigned time   :28;
    unsigned channel  :4;
    } bits;

  } Record;
  unsigned int markers;
  Record.allbits = TTTRRecord;
  if(Record.bits.channel==0xF) //this means we have a special record
  {
    //in a special record the lower 4 bits of time are marker bits
    markers=Record.bits.time&0xF;
    if(markers==0) //this means we have an overflow record
    {
      oflcorrection += T2WRAPAROUND; // unwrap the time tag overflow
      GotOverflow(1);
    }
    else //a marker
    {
      //Strictly, in case of a marker, the lower 4 bits of time are invalid
      //because they carry the marker bits. So one could zero them out.
      //However, the marker resolution is only a few tens of nanoseconds anyway,
      //so we can just ignore the few picoseconds of error.
      truetime = oflcorrection + Record.bits.time;
            GotMarker(truetime, markers);
    }
  }
  else
  {
    if((int)Record.bits.channel > 4) //Should not occur
    {
      printf(" Illegal Chan: #%I64u %1u\n",RecNum,Record.bits.channel);
    }
    else
    {
      if(Record.bits.channel==0) cnt_0++;
      if(Record.bits.channel>=1) cnt_1++;

      truetime = oflcorrection + Record.bits.time;
      m = Record.bits.time;
      c = Record.bits.channel;
      GotPhoton(truetime, c, m);
    }
  }
}

// HydraHarp/TimeHarp260 T2 input
void ProcessHHT2(unsigned int TTTRRecord, int HHVersion)
{
  const int T2WRAPAROUND_V1 = 33552000;
  const int T2WRAPAROUND_V2 = 33554432;
  union{
    unsigned int allbits;
    struct{ unsigned timetag  :25;
        unsigned channel  :6;
        unsigned special  :1; // or sync, if channel==0
        } bits;
    } T2Rec;
  T2Rec.allbits = TTTRRecord;

  if(T2Rec.bits.special==1)
    {
      if(T2Rec.bits.channel==0x3F) //an overflow record
      {
    if(HHVersion == 1)
    {
      oflcorrection += (uint64_t)T2WRAPAROUND_V1;
      GotOverflow(1);
    }
    else
    {
      //number of overflows is stored in timetag
      if(T2Rec.bits.timetag==0) //if it is zero it is an old style single overflow
      {
          GotOverflow(1);
        oflcorrection += (uint64_t)T2WRAPAROUND_V2;  //should never happen with new Firmware!
      }
      else
      {
        oflcorrection += (uint64_t)T2WRAPAROUND_V2 * T2Rec.bits.timetag;
        GotOverflow(T2Rec.bits.timetag);
      }
    }
      }

      if((T2Rec.bits.channel>=1)&&(T2Rec.bits.channel<=15)) //markers
      {
        truetime = oflcorrection + T2Rec.bits.timetag;
        //Note that actual marker tagging accuracy is only some ns.
    m = T2Rec.bits.channel;
    GotMarker(truetime, m);
      }

      if(T2Rec.bits.channel==0) //sync
      {
        truetime = oflcorrection + T2Rec.bits.timetag;
    GotPhoton(truetime, 0, 0);
      }
    }
    else //regular input channel
    {
    truetime = oflcorrection + T2Rec.bits.timetag;
    c = T2Rec.bits.channel + 1;
    GotPhoton(truetime, c, 0);
    }

}

// HydraHarp/TimeHarp260 T3 input
void ProcessHHT3(unsigned int TTTRRecord, int HHVersion)
{
  const int T3WRAPAROUND = 1024;
  union {
    unsigned int allbits;
    struct  {
      unsigned nsync    :10;  // numer of sync period
      unsigned dtime    :15;    // delay from last sync in units of chosen resolution
      unsigned channel  :6;
      unsigned special  :1;
    } bits;
  } T3Rec;
  T3Rec.allbits = TTTRRecord;
  if(T3Rec.bits.special==1)
  {
    if(T3Rec.bits.channel==0x3F) //overflow
    {
      //number of overflows is stored in nsync
      if((T3Rec.bits.nsync==0) || (HHVersion==1)) //if it is zero or old version it is an old style single overflow
      {
        oflcorrection += (uint64_t)T3WRAPAROUND;
        GotOverflow(1); //should never happen with new Firmware!
      }
      else
      {
        oflcorrection += (uint64_t)T3WRAPAROUND * T3Rec.bits.nsync;
        GotOverflow(T3Rec.bits.nsync);
      }
    }
    if((T3Rec.bits.channel>=1)&&(T3Rec.bits.channel<=15)) //markers
    {
      truensync = oflcorrection + T3Rec.bits.nsync;
      //the time unit depends on sync period which can be obtained from the file header
      c = T3Rec.bits.channel;
      GotMarker(truensync, c);
    }
  }
  else //regular input channel
    {
      truensync = oflcorrection + T3Rec.bits.nsync;
      //the nsync time unit depends on sync period which can be obtained from the file header
      //the dtime unit depends on the resolution and can also be obtained from the file header
      c = T3Rec.bits.channel;
      m = T3Rec.bits.dtime;
      GotPhoton(truensync, c, m);
    }
}

int main(int argc, char* argv[])
{
  image = NULL;
  char Magic[8];
  char Version[8];
  char Buffer[40];
  char* AnsiBuffer;
  wchar_t* WideBuffer;
  int Result;
    
  StartMarker = -1;
  StopMarker = -1;
  FrameMarker = -1;
  // 0 = no image
  PixX = 0;
  PixY = 0;
  IsBidir = false; // if Tag not given = monodirectional
  //
  LinePos = lpInBetween;
  CurTemp = 0;
  
  TempSize = 10000;
  TempPhotons = (TPhoton *)malloc(TempSize * sizeof(TPhoton));
  
  long long NumRecords = -1;
  long long RecordType = 0;
  int RecordLength; // Length of an Record, by default 4 bytes, But LIN Camera has 8 bytes


  printf("\nPicoQuant Unified TTTR (PTU) Image File Demo");
  printf("\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n");

  if((argc<3)||(argc>3))
  {
   printf("usage: ptuimagedemo infile oufile\n");
   printf("infile is a Unified TTTR ptu file (binary)\n");
   printf("outfile is ASCII matrix of the intensity image\n");
   getchar();
   exit(-1);
  }
  if((fpin=fopen(argv[1],"rb"))==NULL)
      {printf("\n ERROR! Input file cannot be opened, aborting.\n"); goto ex;}


  if((fpout=fopen(argv[2],"w"))==NULL)
   {printf("\n ERROR! Output file cannot be opened, aborting.\n"); goto ex;}

  printf("\n Loading data from %s \n", argv[1]);
  printf("\n Writing output to %s \n", argv[2]);

  Result = fread( &Magic, 1, sizeof(Magic) ,fpin);
  if (Result!= sizeof(Magic))
  {
    printf("\nerror reading header, aborted.");
      goto close;
  }
  Result = fread(&Version, 1, sizeof(Version) ,fpin);
  if (Result!= sizeof(Version))
    {
    printf("\nerror reading header, aborted.");
      goto close;
  }
  if (strncmp(Magic, "PQTTTR", 6))
  {
    printf("\nWrong Magic, this is not a PTU file.");
    goto close;
  }
  
  // read tagged header
  do
  {
    // This loop is very generic. It reads all header items and displays the identifier and the
    // associated value, quite independent of what they mean in detail.
    // Only some selected items are explicitly retrieved and kept in memory because they are
    // needed to subsequently interpret the TTTR record data.

    Result = fread( &TagHead, 1, sizeof(TagHead) ,fpin);
    if (Result!= sizeof(TagHead))
    {
      printf("\nIncomplete File.");
      goto close;
    }

    strcpy(Buffer, TagHead.Ident);
    if (TagHead.Idx > -1)
      sprintf(Buffer, "%s(%d)", TagHead.Ident,TagHead.Idx);
    switch (TagHead.Typ)
    {
      case tyEmpty8:
        break;
      case tyBool8:
        if (strcmp(TagHead.Ident, TTTRTagImgBidirect)==0) // Number of records
                    IsBidir = bool(TagHead.TagValue);
        break;
      case tyInt8:
        // get some Values we need to analyse records
        if (strcmp(TagHead.Ident, TTTRTagNumRecords)==0) // Number of records
                    NumRecords = TagHead.TagValue;
        if (strcmp(TagHead.Ident, TTTRTagTTTRRecType)==0) // TTTR RecordType
                    RecordType = TagHead.TagValue;
		    if (strcmp(TagHead.Ident, TTTRTagImgIdent)==0) // Scanner Ident
                    ScannerIdent = TagHead.TagValue;			
		    if (strcmp(TagHead.Ident, TTTRTagImgPixX)==0)
                    PixX = TagHead.TagValue;
		    if (strcmp(TagHead.Ident, TTTRTagImgPixY)==0)
                    PixY = TagHead.TagValue;
		    if (strcmp(TagHead.Ident, TTTRTagImgLineStart)==0)
                    StartMarker = TagHead.TagValue - 1;
		    if (strcmp(TagHead.Ident, TTTRTagImgLineStop)==0)
                    StopMarker = TagHead.TagValue - 1;
		    if (strcmp(TagHead.Ident, TTTRTagImgFrame)==0)
                    FrameMarker = TagHead.TagValue - 1;
        break;
      case tyBitSet64:
        break;
      case tyColor8:
        break;
      case tyFloat8:
		    if (strcmp(TagHead.Ident, TTTRTagImgPixRes)==0) // pixel resolution in x and y
                    PixRes = *(double*)&(TagHead.TagValue); // in um			
        break;
      case tyFloat8Array:
        // only seek the Data, if one needs the data, it can be loaded here
        fseek(fpin, (long)TagHead.TagValue, SEEK_CUR);
        break;
      case tyTDateTime:
        time_t CreateTime;
        CreateTime = TDateTime_TimeT(*((double*)&(TagHead.TagValue)));
        break;
      case tyAnsiString:
        AnsiBuffer = (char*)calloc((size_t)TagHead.TagValue,1);
        Result = fread(AnsiBuffer, 1, (size_t)TagHead.TagValue, fpin);
        if (Result!= TagHead.TagValue)
        {
          printf("\nIncomplete File.");
          free(AnsiBuffer);
          goto close;
        }
        free(AnsiBuffer);
        break;
      case tyWideString:
        WideBuffer = (wchar_t*)calloc((size_t)TagHead.TagValue,1);
        Result = fread(WideBuffer, 1, (size_t)TagHead.TagValue, fpin);
        if (Result!= TagHead.TagValue)
        {
          printf("\nIncomplete File.");
          free(WideBuffer);
          goto close;
        }
        free(WideBuffer);
        break;
      case tyBinaryBlob:
        // only seek the Data, if one needs the data, it can be loaded here
        fseek(fpin, (long)TagHead.TagValue, SEEK_CUR);
        break;
      default:
        printf("Illegal Type identifier found! Broken file?");
        goto close;
    }
  }
  while((strncmp(TagHead.Ident, FileTagEnd, sizeof(FileTagEnd))));
// End Header loading

// Special case for converted old SPT32 files, did not have this information
	if ((ScannerIdent == 1) && ((StartMarker < 0) || (StopMarker < 0)))
	{
	  StartMarker = 2;
	  StopMarker = 1;
	};
	// Check if all needed Image data are there
	if ((PixX <= 0) || (PixY <= 0) || (StartMarker < 0) || (StopMarker < 0))
	{
	  printf("Not a valid image, some informations missing\n");
	  goto close;
	}
	fprintf(fpout, "PTU Image File:\n");
	fprintf(fpout, "Size: %i x %i pixels ", PixX, PixY);
	// some older files does not have the pixel size included
	if (PixRes == 0)
	  fprintf(fpout, "No size informations\n");
	else
	  fprintf(fpout, " = %f um x %f um\n",  PixX * PixRes, PixY * PixRes);


	// Init resulting Image
	image = (long long *)malloc(PixX * PixY * sizeof(long long)); // create the counting image Array
  memset(image, 0, PixX * PixY * sizeof(long long));

  // TTTR Record type
  RecordLength = 4; // all Record Length 4 until now
  switch (RecordType)
  {
    case rtPicoHarpT2:
      fprintf(fpout, "PicoHarp T2 data\n");
      break;
    case rtPicoHarpT3:
      fprintf(fpout, "PicoHarp T3 data\n");
      break;
    case rtHydraHarpT2:
      fprintf(fpout, "HydraHarp V1 T2 data\n");
      break;
    case rtHydraHarpT3:
      fprintf(fpout, "HydraHarp V1 T3 data\n");
      break;
    case rtHydraHarp2T2:
      fprintf(fpout, "HydraHarp V2 T2 data\n");
      break;
    case rtHydraHarp2T3:
      fprintf(fpout, "HydraHarp V2 T3 data\n");
      break;
	  case rtTimeHarp260NT3:
      fprintf(fpout, "TimeHarp260N T3 data\n");
      break;
	  case rtTimeHarp260NT2:
      fprintf(fpout, "TimeHarp260N T2 data\n");
      break;
    case rtTimeHarp260PT3:
      fprintf(fpout, "TimeHarp260P T3 data\n");
      break;
	  case rtTimeHarp260PT2:
      fprintf(fpout, "TimeHarp260P T2 data\n");
      break;
    case rtMultiHarpT3:
      fprintf(fpout, "MultiHarp T3 data\n");
      break;
	  case rtMultiHarpT2:
      fprintf(fpout, "MultiHarp T2 data\n");
      break;
  default:
    printf("Unknown record type: 0x%X\n 0x%X\n ", RecordType);
    goto close;
  }
  //
  unsigned int TTTRRecord;
  uint64_t TTTRRecord64;
  for(RecNum=0;RecNum<NumRecords;RecNum++)
  {
    Result = fread(&TTTRRecord, 1, sizeof(TTTRRecord) ,fpin);
    if (Result!= sizeof(TTTRRecord))
    {
      printf("\nUnexpected end of input file!");
      break;
    }
   
    switch (RecordType)
    {
    case rtPicoHarpT2:
      ProcessPHT2(TTTRRecord);
      break;
    case rtPicoHarpT3:
      ProcessPHT3(TTTRRecord);
      break;
    case rtHydraHarpT2:
      ProcessHHT2(TTTRRecord, 1);
      break;
    case rtHydraHarpT3:
      ProcessHHT3(TTTRRecord, 1);
      break;
    case rtMultiHarpT2:
	  case rtHydraHarp2T2:
	  case rtTimeHarp260NT2:
	  case rtTimeHarp260PT2:
      ProcessHHT2(TTTRRecord, 2);
      break;
	  case rtMultiHarpT3:
	  case rtHydraHarp2T3:
	  case rtTimeHarp260NT3:
	  case rtTimeHarp260PT3:
      ProcessHHT3(TTTRRecord, 2);
      break;
    default:
      goto close;
    }
  }
  //
  // write the image to output file as integer matrix
  for (int y = 0; y < PixY; y++)
  {
	  for (int x = 0; x < PixX; x++)
    {
      if (x > 0)
        fprintf(fpout, "\t");
      fprintf(fpout, "%li", image[(y * PixX) + x]);
    }
    fprintf(fpout, "\n");
  }
close:
  if (TempPhotons)
    free(TempPhotons);
  if (image)
    free(image);
  fclose(fpin);
  fclose(fpout);

ex:
  printf("\n any key...");
  getchar();
  exit(0);
  return(0);
}