 {
  PicoQuant Unified TTTR (PTU) Image reconstruction Demo in Pascal

  This is demo code. Use at your own risk. No warranties.

  Tested with Delphi XE10.1 and Lazarus 1.6 (Freepascal 3.1.1)

  Marcus Sackrow, PicoQuant GmbH, January 2019
}

program PTUImageDemo;
{$apptype console}
uses
  Classes,
  SysUtils,
  StrUtils,
  Math;

const
  MAX_TAGIDENT_LENGTH      = 31;
  // TagTypes  (TTagHead.Typ)
  tyEmpty8                 = $FFFF0008;
  tyBool8                  = $00000008;
  tyInt8                   = $10000008;
  tyBitSet64               = $11000008;
  tyColor8                 = $12000008;
  tyFloat8                 = $20000008;
  tyTDateTime              = $21000008;
  tyFloat8Array            = $2001FFFF;
  tyAnsiString             = $4001FFFF;
  tyWideString             = $4002FFFF;
  tyBinaryBlob             = $FFFFFFFF;

  // selected Tag Idents (TTagHead.Name) we will need to interpret the subsequent record data
  // check the tag dictionary if you need more
  TTTRTagTTTRRecType       = 'TTResultFormat_TTTRRecType';
  TTTRTagNumRecords        = 'TTResult_NumberOfRecords'; // Number of TTTR Records in the File;
  FileTagEnd               = 'Header_End';               // has always to be appended as last tag (BLOCKEND)
  // Image Tags
  TTTRTagImgIdent          = 'ImgHdr_Ident';     // Which scanner was used for measurement
  TTTRTagImgPixX           = 'ImgHdr_PixX';      // Number of Pixels in x direction
  TTTRTagImgPixY           = 'ImgHdr_PixY';      // Number of Pixels in y direction
  TTTRTagImgLineStart      = 'ImgHdr_LineStart'; // Marker for the line start
  TTTRTagImgLineStop       = 'ImgHdr_LineStop';  // Marker for the line stop
  TTTRTagImgFrame          = 'ImgHdr_Frame';     // Marker for the frame change
  TTTRTagImgBidirect       = 'ImgHdr_BiDirect';  // If the Image is recorded in bidirectional mode
  TTTRTagImgPixRes         = 'ImgHdr_PixResol';  // Pixel Resolution in m

  // RecordTypes
  rtPicoHarpT3     = $00010303;    // (SubID = $00 ,RecFmt: $01) (V1), T-Mode: $03 (T3), HW: $03 (PicoHarp)
  rtPicoHarpT2     = $00010203;    // (SubID = $00 ,RecFmt: $01) (V1), T-Mode: $02 (T2), HW: $03 (PicoHarp)
  rtHydraHarpT3    = $00010304;    // (SubID = $00 ,RecFmt: $01) (V1), T-Mode: $03 (T3), HW: $04 (HydraHarp)
  rtHydraHarpT2    = $00010204;    // (SubID = $00 ,RecFmt: $01) (V1), T-Mode: $02 (T2), HW: $04 (HydraHarp)
  rtHydraHarp2T3   = $01010304;    // (SubID = $01 ,RecFmt: $01) (V2), T-Mode: $03 (T3), HW: $04 (HydraHarp)
  rtHydraHarp2T2   = $01010204;    // (SubID = $01 ,RecFmt: $01) (V2), T-Mode: $02 (T2), HW: $04 (HydraHarp)
  rtTimeHarp260NT3 = $00010305;    // (SubID = $01 ,RecFmt: $01) (V2), T-Mode: $03 (T3), HW: $05 (TimeHarp260N)
  rtTimeHarp260NT2 = $00010205;    // (SubID = $01 ,RecFmt: $01) (V2), T-Mode: $02 (T2), HW: $05 (TimeHarp260N)
  rtTimeHarp260PT3 = $00010306;    // (SubID = $01 ,RecFmt: $01) (V2), T-Mode: $03 (T3), HW: $06 (TimeHarp260P)
  rtTimeHarp260PT2 = $00010206;    // (SubID = $01 ,RecFmt: $01) (V2), T-Mode: $02 (T2), HW: $06 (TimeHarp260P)
  rtMultiHarpT3    = $00010307;    // (SubID = $01 ,RecFmt: $01) (V2), T-Mode: $03 (T3), HW: $07 (MultiHarp)
  rtMultiHarpT2    = $00010207;    // (SubID = $01 ,RecFmt: $01) (V2), T-Mode: $02 (T2), HW: $07 (MultiHarp)
type
  // Tag Entry
  TTagHead = packed record
    Name: array[0..MAX_TAGIDENT_LENGTH] of AnsiChar;  // Identifier of the tag
    Idx: LongInt;                                     // Index for multiple tags or -1
    Typ: Cardinal;                                    // Type of tag ty..... see const section
    TagValue: Int64;                                  // Value of tag.
  end;

  TLinePos = (lpInBetween, lpInLine);

var
  InpFile: file;
  OutFile: TextFile;
  Magic: array[0..19] of AnsiChar;
  Version: array[0..19] of AnsiChar;

  TagHead: TTagHead;
  Res: Integer;
  AnsiTemp: PAnsiChar;
  WideTemp: PWideChar;
  StrTemp: string;

  NumRecords: Int64;
  RecordType: Int64;
  RecordLength: Integer; // Length of one Record, default is 4 bytes until now
  n: Int64;
  TTTRRecord: Cardinal;
  OflCorrection: Int64 = 0;

  // Imaging
  ScannerIdent: Integer; // Which scanner was used
  StartMarker, StopMarker, FrameMarker: Integer; // Die Marker for image creation
  PixX, PixY: Integer; // Size of the image
  IsBidir: Boolean;    // Bidirectional scanning?
  PixRes: Double;      // Size of a single pixel in m (x and y direction)

  Image: array of array of Int64; // the actual Image counts

  x,y: Integer;  // Counter variables for the for loops
  TempPhotons: array of record // storage for a line of Photons
    TimeTag: Int64;
    Channel: Byte;
    DTime: Word;
  end;

  CurTemp: Integer; // Position in the TempPhotons to write
  LinePos: TLinePos; // are we inside a line -> only record photons inside a line

  StartPos: Int64; // Position of the last StartMarker
  CurLine: Integer; // Current Line number

//*********************** Image creation *****************************************

//Got Photon
//  TimeTag: Raw TimeTag from Record, arrival time of Photon
//  DTime: Arrival time of Photon after last Sync event (T3 only)
//  Channel: Channel the Photon arrived (0 = Sync channel for T2 measurements)
procedure GotPhoton(TimeTag: Int64; DTime: Integer; Channel: Integer);
begin
  // Only care about photons after Start before Stop marker, all other Photons are ignored
  if LinePos = lpInLine then
  begin
    // Enough space in the temp area, if not extend it
    if CurTemp > High(TempPhotons) then
      SetLength(TempPhotons, CurTemp + 10000);
    // we only save the photon to a temp area we need to wait for the line end marker
    // to determine its pixel position
    TempPhotons[CurTemp].TimeTag := TimeTag; // We only need the TimeTag
    TempPhotons[CurTemp].Channel := Channel; // but for further analysis we copy all
    TempPhotons[CurTemp].DTime := DTime;     // Values to the temp
    //
    Inc(CurTemp);
  end;
end;

//Got Marker
//  TimeTag: Raw TimeTag from Record, arrival time of marker
//  Markers: Bitfield of arrived Markers, different markers can arrive at same time (same record)
procedure GotMarker(TimeTag: Int64; Markers: Integer);
var
  Dist: Int64;
  SizePix: Double;
  i: Integer;
  NextPixel: Double;
  PixelNum: Integer;
  CurPixel: Integer;
begin
  // attentions the markers can arrive in the very same record.
  // especially the Stop Line marker and Frame marker often come together
  // Therefore the ordering of marker checking is important

  // Check if the Line Stop arrived, only make sense inside a line -> else just ignore it
  if (((1 shl StopMarker) and Markers) <> 0) and (LinePos = lpInLine) then
  begin
    // do the processing!
    Dist := TimeTag - StartPos; // calculate the Distance between start and end marker
    // Size of a single pixel (in TimeTags, must be a float value to prevent rounding errors)
    if ScannerIdent = 1 then
      SizePix := Dist / (PixX + 1) // for E710 the marker is one Pixel off
    else
      SizePix := Dist / PixX;
    NextPixel := StartPos + SizePix; // Time Tag of the the Next Pixel
    PixelNum := 0; // Current pixel to fill
    // main sorting routine
    for i := 0 to (CurTemp - 1) do
    begin
      // calculate the pixel position
      while TempPhotons[i].TimeTag >= NextPixel do
      begin
        Inc(PixelNum); // next pixel
        NextPixel := NextPixel + SizePix; // start of next pixel
      end;
      // make sure we are inside the image array
      if (CurLine >= PixY) or (PixelNum >= PixX) then
        Break;
      // Correct for bidirectional scan if needed
      if IsBidir and ((CurLine mod 2) = 1) then
        CurPixel := PixX - PixelNum - 1         // revert every 2nd line if bidirectional scan
      else
        CurPixel := PixelNum;
      // One could here filter by channel or process the photon data further more,
      // the pixel position is CurPixel = x, Curline = y and the data is in
      // TempPhotons[i].Channel and TempPhotons[i].DTime
      //
      // As demonstration we count the photons in the pixel to get an intensity image
      Inc(Image[CurPixel, CurLine]);
    end;
    // the line is finished -> Next line
    Inc(CurLine);
    // the line is ended -> we are outside the line again
    LinePos := lpInBetween;
  end;
  // Check if the FrameMarker arrived
  if (FrameMarker >= 0) and (((1 shl FrameMarker) and Markers) <> 0) then
  begin
    // a frame marker just restart from the beginning, so we reset the line position
    CurLine := 0;
    LinePos := lpInBetween;
  end;
  // Check for the Start Marker, only make sense outside a line -> else just ignore it
  if ((1 shl StartMarker) and Markers <> 0) and (LinePos = lpInBetween) then
  begin
    // a Line Starts here
    LinePos := lpInLine; // inside a line from now on
    CurTemp := 0;        // Reset Temp area
    StartPos := TimeTag; // remember start for the calculation
  end;
end;

//Got Overflow
//  Count: Some TCSPC provide Overflow compression = if no Photons between overflow you get one record for multiple Overflows
procedure GotOverflow(Count: Integer);
begin
  // nothing to do for overflows
end;
//
//******************************************************************************


//******************** TTTR-Record inspection **********************************
//

  // HydraHarp T3 Input
procedure ProcessHHT3(TTTR_RawData: Cardinal; Version: Integer);
  const
    T3WRAPAROUND = 1024;
  type
     T_TTTRData = record
       Special: Boolean;
       Channel: Byte;
       DTime,
       NSync: Word;
    end;
  var
    TTTR_Data: T_TTTRData;
    TrueNSync: Int64;
  begin
    with TTTR_Data do
    begin
      {split "RawData" into its parts}
      NSync   := Word   ( TTTR_RawData         and $000003FF);
      DTime   := Word   ((TTTR_RawData shr 10) and $00007FFF);
      Channel := Byte   ((TTTR_RawData shr 25) and $0000003F);
      Special := Boolean((TTTR_RawData shr 31) and $00000001);
      if Special then                                       // this means we have a Special record
      begin
        if (Channel = $3F) then                               // overflow
        begin
          case Version of
            1: begin
              GotOverflow(1);
              OflCorrection := OflCorrection + T3WRAPAROUND;
            end;
            2: begin
              // number of overflows is stored in nsync
              // if it is zero, it is an old style single overflow {should never happen with new Firmware}
              GotOverflow(ifthen(NSync = 0, 1, NSync));
              OflCorrection := OflCorrection + Int64(T3WRAPAROUND) * ifthen(NSync = 0, 1, NSync);
            end;
          end;
        end else
        if ((Channel > 0) and (Channel <= 15)) then          //markers
        begin
          TrueNSync := OflCorrection + NSync;
          GotMarker(TrueNSync, Channel);
        end
      end else
      begin                                                 //regular input Channel
        TrueNSync := OflCorrection + NSync;
        GotPhoton(TrueNSync, DTime, Channel + 1);
      end;
    end;
  end;

// HydraHarp T2 input
procedure ProcessHHT2(TTTR_RawData: Cardinal; Version: Integer);
const
  T2WRAPAROUND_V1 = 33552000;
  T2WRAPAROUND_V2 = 33554432;
type
  TDataRecords = record
    Special: Boolean;
    Channel: Byte;
    DTime: Cardinal;
  end;
var
  TTTR_Data: TDataRecords;
begin
  with TTTR_Data do
  begin
    {split "RawData" into its parts}
    DTime   := Cardinal ( TTTR_RawData         and $01FFFFFF);
    Channel := Byte     ((TTTR_RawData shr 25) and $0000003F);
    Special := Boolean  ((TTTR_RawData shr 31) and $00000001);
    if (Special)then                  // this means we have a Special record
    begin
      if (Channel = $3F) then        // overflow
      begin
        case Version of
          1: begin
            GotOverflow(1);
            OflCorrection := OflCorrection + T2WRAPAROUND_V1;
          end;
          2: begin
            // number of overflows is stored in time tag
            // if it is zero, it is an old style single overflow {should never happen with new Firmware}
            GotOverflow(ifthen(DTime = 0, 1, DTime));
            OflCorrection := OflCorrection + Int64(T2WRAPAROUND_V2) * ifthen(DTime = 0, 1, DTime);
          end;
        end;
      end else
        if (Channel = 0) then        // sync
        begin
          GotPhoton(OflCorrection + DTime, 0, 0);
        end else
          if (Channel <= 15) then   // markers
          begin
            // Note that actual marker tagging accuracy is only some ns.
            GotMarker(OflCorrection + DTime, Channel);
          end;
    end else
    begin // it is a regular photon record
      GotPhoton(OflCorrection + DTime, 0, Channel + 1);
    end;
  end;
end;

// PicoHarp T3 input
procedure ProcessPHT3(Value: Cardinal);
const
  T3WRAPAROUND = 65536;
type
  T_TTTRData = record
    NumSync,
    Data: Word;
  end;
  TDataRecords = record
    Channel: Byte;
    case Boolean of
      True:  (DTime: Word);
      False: (Markers: Word);
  end;
var
  TTTR_RawData: T_TTTRData;
  TTTR_Data: TDataRecords;
  TrueNSync: Int64;
begin
  TTTR_RawData := T_TTTRData(Value);
  with TTTR_Data, TTTR_RawData do
  begin
      {split joined parts of "RawData"}
      Markers := word ( TTTR_RawData.Data         and $00000FFF);   // resp. DTime
      Channel := byte ((TTTR_RawData.Data shr 12) and $0000000F);
      if (Channel = $0F) then                                // this means we have a special record
      begin
        if (Markers = 0) then                                // missing marker means overflow
        begin
          GotOverflow(1);
          OflCorrection := OflCorrection + T3WRAPAROUND;     // unwrap the time tag overflow
        end else
        begin // a marker
          TrueNSync := OflCorrection + NumSync;
          GotMarker(TrueNSync, Markers);
        end
      end
      else begin // a photon record
        if ((Channel = 0)             // Should never occur in T3 Mode
          or (Channel > 4) ) then     // Should not occur with current routers
        begin
          WriteLn('illegal Channel: ', Channel);
        end;
        TrueNSync := OflCorrection + NumSync;
        GotPhoton(TrueNSync, DTime, Channel);
      end;
    end;
end;

// PicoHarp T2 input
procedure ProcessPHT2(TTTR_RawData: Cardinal);
const
  T2WRAPAROUND = 210698240;
type
  TDataRecords = record
    Channel: Byte;
    DTime: Cardinal;
  end;
var
  TTTR_Data: TDataRecords;
  Time: Int64;
  Markers: Cardinal;
begin
  with TTTR_Data
    do begin
      {split "RawData" into its parts}
      DTime   := Cardinal ( TTTR_RawData         and $0FFFFFFF);
      Channel := Byte     ((TTTR_RawData shr 28) and $0000000F);

      if (Channel = $0F) then             // this means we have a special record
      begin
        //in a special record the lower 4 bits of DTime are marker bits
        Markers := DTime and $0F;
        if (Markers = 0) then             //this means we have an overflow record
        begin
          GotOverflow(1);
          OflCorrection := OflCorrection + T2WRAPAROUND;  // unwrap the Time tag overflow
        end else
        begin                             //it is a marker record
          Time := OflCorrection + DTime;
          GotMarker(Time, Markers);
          // Strictly, in case of a marker, the lower 4 bits of DTime are invalid
          // because they carry the marker bits. So one could zero them out.
          // However, the marker resolution is only a few tens of nanoseconds anyway,
          // so we can just ignore the few picoseconds of error.
          // Due to the lower Time resolution markers may therefore appear
          // in the file slightly out of order with respect to regular event records.
          // This is by design. markers are designed only for relatively coarse
          // synchronization requirements such as image scanning.
        end;
      end else
      begin                               // it is a photon record
        if (Channel > 4) //Should not occur
        then begin
          WriteLn(' Illegal Chan: ', Channel);
        end;
        Time := OflCorrection + DTime;
        GotPhoton(Time, 0, Channel);
      end;
    end;
end;

//
//******************************************************************************


//*********************** Main Program *****************************************
begin
try
  NumRecords := -1;
  RecordType := 0;
  
  Res := 0;
  Magic[0] := #0;
  Version[0] := #0;
  TagHead.Name[0] := #0;
  TTTRRecord := 0;
  CurLine := 0;
  PixRes := 0;

  // Init image variables
  // Init Marker with -1 = not available
  StartMarker := -1;
  StopMarker := -1;
  FrameMarker := -1;
  // 0 = no image
  PixX := 0;
  PixY := 0;
  IsBidir := False; // if Tag not given = mono-directional
  //
  LinePos := lpInBetween;
  CurTemp := 0;
  SetLength(TempPhotons, 10000);
  
  WriteLn('PicoQuant Unified TTTR (PTU) Mode Imaging Demo');
  WriteLn('~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~');

  if (ParamCount <> 2) then
  begin
    WriteLn('  Usage: ptuimagedemo infile outfile');
    WriteLn;
    WriteLn('  infile  is a binary mode file (*.ptu)');
    WriteLn('  outfile will be ASCII matrix of the intensity');
    Exit;
  end;

  // Open Input File
  AssignFile(InpFile, ParamStr(1));
  {$I-}
    Reset(InpFile, 1);
  {$I+}
  if (IOResult <> 0) then
  begin
    WriteLn('Cannot open input file '+ ParamStr(1) + ' error code: ' + IntToStr(IOResult));
    Exit;
  end;
  WriteLn('Loading data from ', ParamStr(1));

  // Open Output file
  AssignFile(OutFile, ParamStr(2));
  {$I-}
    ReWrite(OutFile);
  {$I+}
  if (IOResult <> 0)
  then begin
    WriteLn ('Cannot open output file ' + ParamStr(2) + ' error code: ' + IntToStr(IOResult));
    CloseFile(InpFile);
    Exit;
  end;
  WriteLn ('Writing output to ', ParamStr(2));

// Start Header loading
  try
    // get Magic and TagFile Version
    FillChar(Magic[0], Length(Magic), #0);
    FillChar(Version[0], Length(Version), #0);
    BlockRead(InpFile, Magic[0], 8, Res);
    BlockRead(InpFile, Version[0], 8, Res);
    // Check Magic for TTTR File
    if Magic <> 'PQTTTR' then
    begin
      Writeln('Wrong Magic, this is not a PTU file.');
      Exit;
    end;
    repeat
      // Read tagged Header
      BlockRead(InpFile, TagHead, SizeOf(TagHead), Res);
      if Res < SizeOf(TagHead) then
      begin
        WriteLn('Incomplete File.');
        Exit;
      end;
      StrTemp := string(TagHead.Name);
      if TagHead.Idx > - 1 then
        StrTemp := StrTemp + '(' + IntToStr(TagHead.Idx) + ')';
      // Inspect Value by Type
      case TagHead.typ of
        tyEmpty8: begin                    // Empty8
          //
        end;
        tyBool8: begin                     // Bool8
          if TagHead.Name = TTTRTagImgBidirect then
            IsBidir := LongBool(TagHead.TagValue);
        end;
        tyInt8: begin                      // Int8
          if TagHead.Name = TTTRTagNumRecords then // Number of records
            NumRecords := TagHead.TagValue;
          if TagHead.Name = TTTRTagTTTRRecType then // TTTR RecordType
            RecordType := TagHead.TagValue;
          if TagHead.Name = TTTRTagImgIdent then
            ScannerIdent := TagHead.TagValue;
          if TagHead.Name = TTTRTagImgPixX then
            PixX := TagHead.TagValue;
          if TagHead.Name = TTTRTagImgPixY then
            PixY := TagHead.TagValue;
          if TagHead.Name = TTTRTagImgLineStart then
            StartMarker := TagHead.TagValue - 1;
          if TagHead.Name = TTTRTagImgLineStop then
            StopMarker := TagHead.TagValue - 1;
          if TagHead.Name = TTTRTagImgFrame then
            FrameMarker := TagHead.TagValue - 1;
        end;
        tyBitSet64: begin                  // BitSet64
          //
        end;
        tyColor8: begin                    // Color8
          //
        end;
        tyFloat8: begin                    // Float8
          if TagHead.Name = TTTRTagImgPixRes then
            PixRes := Double((@TagHead.TagValue)^);
        end;
        tyFloat8Array: begin               // FloatArray
          Seek(InpFile, FilePos(InpFile) + TagHead.TagValue);
        end;
        tyTDateTime: begin                 //TDateTime
          //
        end;
        tyAnsiString: begin                // AnsiString
          GetMem(AnsiTemp, TagHead.TagValue);
          try
            BlockRead(InpFile, AnsiTemp^, TagHead.TagValue, Res);
          finally
            Freemem(AnsiTemp);
          end;
        end;
        tyWideString:begin                 // WideString
          GetMem(WideTemp, TagHead.TagValue);
          try
            BlockRead(InpFile, WideTemp^, TagHead.TagValue, Res);
          finally
            Freemem(WideTemp);
          end;
        end;
        tyBinaryBlob: begin               // BinaryBlob
          // only seek the Data, if one needs the data, it can be loaded here or remember the position
          // and length for later reading
          Seek(InpFile, FilePos(InpFile) + TagHead.TagValue);
        end;
        else begin                         // Unknown Type
          Writeln('Illegal Type identifier found! Broken file?');
          Exit;
        end;
      end; //Case
      // FileTagEnd marks the end of Headerarea -> after this tags the TTTR record begin
    until (Trim(string(TagHead.Name)) = FileTagEnd);
// End Header loading

// Special case for converted old SPT32 files, did not have this information
    if (ScannerIdent = 1) and ((StartMarker < 0) or (StopMarker < 0)) then
    begin
      StartMarker := 2;
      StopMarker := 1;
    end;
// Check if all needed Image data are there
    if (PixX <= 0) or (PixY <= 0) or (StartMarker < 0) or (StopMarker < 0) then
    begin
      writeln('Not a valid image, some informations missing');
      Exit;
    end;
    WriteLn(OutFile, 'PTU Image File:');
    Write(OutFile, 'Size: ', PixX, ' x ', PixY, ' pixels ');
    // some older files does not have the pixel size included
    if PixRes = 0 then
      writeln(OutFile, 'No size informations')
    else
      writeln(OutFile, ' = ' + FloatToStrF(PixX * PixRes, ffFixed, 8, 1), 'm x ', FloatToStrF(PixY * PixRes, ffFixed, 8, 1), 'm');

    // Init resulting Image
    SetLength(Image, PixX, PixY); // create the counting image Array
    for y := 0 to PixY - 1 do
      for x := 0 to PixX - 1 do
        Image[x, y] := 0;

// Start TTTR Record section
    RecordLength := 4; // all use 4 Bytes until now
    // print TTTR Record type
    case RecordType of
      rtPicoHarpT3: WriteLn(OutFile, 'PicoHarp T3 data');
      rtPicoHarpT2: WriteLn(OutFile, 'PicoHarp T2 data');
      rtHydraHarpT3: WriteLn(OutFile, 'HyraHarp V1 T3 data');
      rtHydraHarpT2: WriteLn(OutFile, 'HydraHarp V1 T2 data');
      rtHydraHarp2T3: WriteLn(OutFile, 'HyraHarp V2 T3 data');
      rtHydraHarp2T2: WriteLn(OutFile, 'HydraHarp V2 T2 data');
      rtTimeHarp260NT3: WriteLn(OutFile, 'TimeHarp260N T3 data');
      rtTimeHarp260NT2: WriteLn(OutFile, 'TimeHarp260N T2 data');
      rtTimeHarp260PT3: WriteLn(OutFile, 'TimeHarp260P T3 data');
      rtTimeHarp260PT2: WriteLn(OutFile, 'TimeHarp260P T2 data');
      rtMultiHarpT3: WriteLn(OutFile, 'MultiHarp T3 data');
      rtMultiHarpT2: WriteLn(OutFile, 'MultiHarp T2 data');
      else
        begin
          WriteLn('unknown Record type: $' + IntToHex(RecordType, 8));
          Exit;
        end;
    end;
// read the tttr Records
    n := -1;
    while (n+1 < NumRecords) do
    begin
      inc(n);
      if (n mod 100000 = 0) then
        if (n mod 10000000 = 0) then
          write('+')
        else
          write('-');
      // Read Records
      BlockRead(InpFile, TTTRRecord , SizeOf(TTTRRecord), Res);
      if (Res <> RecordLength) then
      begin
        writeln('Unexpected end of input file!');
        Exit;
      end;
      // Analyse record
      case RecordType of
          // PicoHarp
        rtPicoHarpT3: ProcessPHT3(TTTRRecord);
        rtPicoHarpT2: ProcessPHT2(TTTRRecord);
          // HydraHarp V1
        rtHydraHarpT2: ProcessHHT2(TTTRRecord, 1);
        rtHydraHarpT3: ProcessHHT3(TTTRRecord, 1);
          // HydraHarp V2 + TimeHarp260N+P, MultiHarp
        rtMultiHarpT2,
        rtHydraHarp2T2,
        rtTimeHarp260NT2,
        rtTimeHarp260PT2: ProcessHHT2(TTTRRecord, 2);
        rtMultiHarpT3,
        rtHydraHarp2T3,
        rtTimeHarp260NT3,
        rtTimeHarp260PT3: ProcessHHT3(TTTRRecord, 2);
      end;
    end;
    SetLength(TempPhotons, 0); // we do not need the temp area anymore
    //
    // write the image to output file as integer matrix
    for y := 0 to PixY - 1 do
    begin
      for x := 0 to PixX - 1 do
      begin
        if x > 0 then
          write(OutFile, #9);
        write(OutFile, Image[x,y]);
      end;
      writeln(OutFile);
    end;
    //
  finally
    CloseFile(InpFile);
    CloseFile(OutFile);
  end;
finally
  writeln;
  writeln;
  writeln ('press RETURN');
  {$R-}
    readln;
  {$R+}
end;
end.

